/* msm-video.c
 *
 * Copyright (c) 2009, Code Aurora Forum. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of Code Aurora nor
 *       the names of its contributors may be used to endorse or promote
 *       products derived from this software without specific prior written
 *       permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NON-INFRINGEMENT ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <unistd.h>
#include <fcntl.h>

#include <sys/ioctl.h>
#include <X11/extensions/Xv.h>

#include "xf86.h"
#include "msm.h"
#include "xf86xv.h"
#include "fourcc.h"
#include "msm_fourcc.h"
#include "msm-render.h"
#include "msm-drm.h"

extern struct cmsghdr *cmptr;

#define ALIGN(_v, _d) (((_v) + ((_d) - 1)) & ~((_d) - 1))

#define ISPLANAR(_id) ( (_id) != FOURCC_RGB565 \
			 && (_id) != FOURCC_UYVY \
			 && (_id) != FOURCC_YUY2 )

#define MAX_STRETCH_FACTOR 4		/* Maximum MDP stretchblt factor */
#define MAX_SHRINK_FACTOR  4		/* Maximum MDP shrinkblt factor  */
#define SHRINK_LIMIT       8	  	/* Minimum shrink size           */
#define BITS_TO_BYTES      3

static XF86VideoEncodingRec DummyEncoding[1] = {
   {0, "XV_IMAGE", 1280, 720, {1, 1}}
};

static XF86VideoFormatRec Formats[] = {
   {8, PseudoColor}, {15, TrueColor}, {16, TrueColor}, {24, TrueColor}
};

static XF86ImageRec Images[] = {
   XVIMAGE_UYVY,			/* MDP_YCRYCB_H2V1 */
   XVIMAGE_NV12,			/* MDP_Y_CRCB_H2V1 */
   XVIMAGE_NV21,			/* MDP_Y_CBCR_H2V1 */
   XVIMAGE_YUY2,
   XVIMAGE_I420,
   XVIMAGE_YV12,
   XVIMAGE_RGB565,
};

static XF86AttributeRec Attributes[] = {
   {XvSettable | XvGettable, 0, 1, "XV_HWCODEC"},
};

static Atom xvHWCodec;

typedef struct
{
    int HWCodecFlag;
    struct msm_offscreen_area *area;
    int width;
    int height;
} MSMPortPrivRec, *MSMPortPrivPtr;

static int
MSMGetFormat(int id)
{
   switch (id) {
   case FOURCC_UYVY:
   case FOURCC_YUY2:
      return MDP_YCRYCB_H2V1;

   case FOURCC_RGB565:
      return MDP_RGB_565;

   case FOURCC_NV12:
   case FOURCC_I420:
      return MDP_Y_CRCB_H2V2;

   case FOURCC_YV12:
   case FOURCC_NV21:
      return MDP_Y_CBCR_H2V2;
   }

   return -1;
}


static MSMPtr latest_pMsm = NULL;

static void
copy(unsigned char *dst, unsigned char *src, int width,
     int height, int stride)
{
   if (latest_pMsm->FastVideoMemCopy)
   {
      BOOL blockSignalsForVFP = !(latest_pMsm->NoSigBlock);
      // The following code assumes that dst and src pointers are not aliased.
      if ((width % 2) == 0) {
         // Use the fast software-blit code if possible (16bpp);
         // it assumes widths in multiples of 16-bit pixels.
         swBlit_NoOverlap(dst, src, width / 2, height,
                          stride, stride, 16, blockSignalsForVFP);
      }
      else {
         // Otherwise, use 8bpp code.
         swBlit_NoOverlap(dst, src, width, height,
                          stride, stride, 8, blockSignalsForVFP);
      }
   }
   else {
      while (height--) {
         memcpy(dst, src, width);
         dst += stride;
         src += stride;
      }
   }
}

static void
MSMCopyPseudoPlanar(unsigned char *dst, unsigned char *src,
		    int srcX, int srcY,
		    int srcW, int srcH, int width, int height)
{
   unsigned int yoffset = (srcY * ALIGN(width, 2)) + srcX;

   unsigned int uvoffset = (height * ALIGN(width, 2)) + yoffset;

   copy(dst + yoffset, src + yoffset, srcW, srcH, ALIGN(width, 2));
   copy(dst + uvoffset, src + uvoffset, srcW, srcH, ALIGN(width, 2));
}

static void
MSMCopyPacked(unsigned char *dst, unsigned char *src,
	      int srcX, int srcY, int srcW, int srcH, int width, int height)
{
   /* dword align the destination pitch so that it matches
    * what we do for planar */

   unsigned int offset = (srcY * ALIGN(width, 2) * 2) + (srcX * 2);

   copy(dst + offset, src + offset, srcW * 2, srcH, ALIGN(width, 2) * 2);
}

static void
MSMCopyAndSwapPacked(unsigned char *dst, unsigned char *src,
		     int srcX, int srcY, int srcW, int srcH, int width,
		     int height)
{
    unsigned int offset = (srcY * ALIGN(width,2) << 1) + (srcX << 1);
    unsigned char *sptr = src + offset;
    unsigned char *dptr = dst + offset;
    int x, y;

    int block = srcX >> 3;

    for(y = 0; y < srcH; y++) {

	for(x = 0; x < block; x++) {
		dptr[(x << 1) + 1] = sptr[(x << 1) + 0];
		dptr[(x << 1) + 0] = sptr[(x << 1) + 1];
		dptr[(x << 1) + 3] = sptr[(x << 1) + 2];
		dptr[(x << 1) + 2] = sptr[(x << 1) + 3];
		dptr[(x << 1) + 5] = sptr[(x << 1) + 4];
		dptr[(x << 1) + 4] = sptr[(x << 1) + 5];
		dptr[(x << 1) + 7] = sptr[(x << 1) + 6];
		dptr[(x << 1) + 6] = sptr[(x << 1) + 7];

		dptr[(x << 1) + 9] = sptr[(x << 1) + 8];
		dptr[(x << 1) + 8] = sptr[(x << 1) + 9];
		dptr[(x << 1) + 11] = sptr[(x << 1) + 10];
		dptr[(x << 1) + 10] = sptr[(x << 1) + 11];
		dptr[(x << 1) + 13] = sptr[(x << 1) + 12];
		dptr[(x << 1) + 12] = sptr[(x << 1) + 13];
		dptr[(x << 1) + 15] = sptr[(x << 1) + 14];
		dptr[(x << 1) + 14] = sptr[(x << 1) + 15];
	}

	for(x = block << 3; x < srcX; x++) {
		dptr[(x << 1) + 1] = sptr[(x << 1) + 0];
		dptr[(x << 1) + 0] = sptr[(x << 1) + 1];
	}

	dptr += ALIGN(width, 2) << 1;
	sptr += ALIGN(width, 2) << 1;
   }
}

static void
copy2(unsigned char *dst, unsigned char *usrc, unsigned char *vsrc,
      int width, int height, int pitch)
{
   int drawh = height >> 1;

   int draww = width >> 1;

   while (--drawh) {
      int w;

      for (w = 0; w < draww; w++) {
	 dst[(w * 2)] = usrc[w];
	 dst[(w * 2) + 1] = vsrc[w];
      }

      usrc += (pitch >> 1);
      vsrc += (pitch >> 1);

      dst += pitch;
   }
}

static void
MSMCopyPlanar(unsigned char *dst, unsigned char *src, int id,
	      int srcX, int srcY, int srcW, int srcH, int width, int height)
{
   int pitch = ALIGN(width, 2);

   unsigned int yoffset = (srcY * pitch) + srcX;

   unsigned int uvoffset = (height * pitch) + yoffset;

   unsigned char *usrc = src + (height * pitch);

   unsigned char *vsrc = usrc + ((height >> 1) * (pitch >> 1));

   copy(dst + yoffset, src + yoffset, srcW, srcH, pitch);
   copy2(dst + uvoffset, usrc, vsrc, srcW, srcH, pitch);
}

static int
MSMDoBlit(MSMPtr pMsm, MSMPortPrivPtr pPriv,
	  int srcFd, int srcOffset, int id, int drawX, int drawY,
	  int drawW, int drawH,
	  int width, int height, RegionPtr clipBoxes, DrawablePtr pDraw)
{
    PixmapPtr pxDst;
    ScreenPtr pScreen = pDraw->pScreen;
    BoxPtr pbox = REGION_RECTS(clipBoxes);
    int nbox = REGION_NUM_RECTS(clipBoxes);
    int i;

    hwBlitFlush(pMsm);

    if (pDraw->type == DRAWABLE_WINDOW) {
	pxDst = (*pScreen->GetWindowPixmap)((WindowPtr)pDraw);
	if (!pMsm->useDRI2)
		exaMoveInPixmap(pxDst);
    } else {
	pxDst = (PixmapPtr)pDraw;
    }

    for (i = 0; i < nbox; i++) {
	int sx,sy,sw,sh;
	int dx,dy,dw,dh;
	int newMaxSrcSize;

	MSMBlitSurface srcSurface, dstSurface;
	MSMBlitRect srcRect, dstRect;
	MSMBlitRec blit;

	srcSurface.width = ALIGN(width, 2);
	srcSurface.height = ALIGN(height,2);

	srcSurface.format = MSMGetFormat(id);
	srcSurface.pitch = srcSurface.width * 2;

	if (srcFd > 0) {
	    srcSurface.flags = MSM_BLIT_PMEM;
	    srcSurface.priv[0] = srcFd;
	    srcSurface.priv[1] = srcOffset;
	}
	else {
	    if (pPriv->area->type == MSM_OFFSCREEN_GEM) {
		srcSurface.flags = MSM_BLIT_GEM;
		srcSurface.priv[0] = (unsigned long) pPriv->area->priv;
	    }
	    else {
		srcSurface.flags = MSM_BLIT_FB;
		srcSurface.priv[0] = ((ExaOffscreenArea *) pPriv->area->priv)->offset;
	    }
	}

	dstSurface.width = pxDst->drawable.width;
	dstSurface.height = pxDst->drawable.height;
	dstSurface.format = MDP_FB_FORMAT;
	dstSurface.pitch = msm_pixmap_get_pitch(pxDst);

	if (msm_pixmap_in_gem(pxDst)) {
	    dstSurface.flags = MSM_BLIT_GEM;
	    dstSurface.priv[0] = (unsigned long) msm_get_pixmap_bo(pxDst);
	}
	else {
	    dstSurface.flags = MSM_BLIT_FB;
	    dstSurface.priv[0] = exaGetPixmapOffset(pxDst);
	}

	dx = pbox->x1 + (pxDst->drawable.x - pxDst->screen_x);
	dy = pbox->y1 + (pxDst->drawable.y - pxDst->screen_y);
	sx = pbox->x1 - drawX;
	sy = pbox->y1 - drawY;
	sw = dw = pbox->x2 - pbox->x1;
	sh = dh = pbox->y2 - pbox->y1;

	if (drawW != width && (sw > 2))  {
	    sx = (sx * width) / drawW;
	    sw = (sw * width) / drawW;
	}

	if (drawH != height) {
	    sy = (sy * height) / drawH;
	    sh = (sh * height) / drawH;
	}

	if (sx + sw > width)
	    sw = width - sx;

	if (sy + sh > height)
	    sh = height - sy;

	if (ISPLANAR(id)) {
	    if (sx & 1) {
		sx &= ~1;
		sw++;
	    }

	    if (sy & 1) {
		sy &= ~1;
		sh++;
	    }

	    sw &= ~1;
	    sh &= ~1;
	}


	/* FIXME:
	   It occurred to me that these could be done as a series of blits,
	   stretching or shrinking as much as possible until the final
	   size is reached.  Lots of copying and work, but it'd look nice.
	*/

	/* Account for MDP stretch boundaries.
	   Cannot shrink to less than 15-25% reliably */
	/* Check for horizontal shrinks */

	if ((sw > 0) && (sh > 0)) {
	    if ((sw > dw) && ((sw/dw) >= MAX_SHRINK_FACTOR)) {
		newMaxSrcSize = dw;
		if (dw > SHRINK_LIMIT) {
		    newMaxSrcSize = dw * (MAX_SHRINK_FACTOR - 1);
		} else {
		    /* No shrink at all if width is very small */
		    newMaxSrcSize = dw;
		}

		/* Cut off left and right edges leaving middle */
		sx = sx + ((sw - newMaxSrcSize) >> 1);
		sw = newMaxSrcSize;
	    }

	    /* Check for vertical shrinks */
	    if ((sh > dh) && ((sh/dh) >= MAX_SHRINK_FACTOR)) {
		if (dh > SHRINK_LIMIT) {
		    newMaxSrcSize = dh * (MAX_SHRINK_FACTOR - 1);
		} else {
		    /* No shrink at all if height is very small */
		    newMaxSrcSize = dh;
		}
		/* Cut off top and bottom edges leaving middle */
		sy = sy + ((sh - newMaxSrcSize) >> 1);
		sh = newMaxSrcSize;
	    }

	    /* Clamp out-of-range horizontal stretches */
	    if ((dw > sw) && ((dw/sw) >= MAX_STRETCH_FACTOR)) {
		dw = width * MAX_STRETCH_FACTOR;
	    }

	    /* Clamp out-of-range vertical stretches */
	    if ((dh > sh) && ((dh/sh) >= MAX_STRETCH_FACTOR)) {
		dh = height * MAX_STRETCH_FACTOR;
	    }
	}

	srcRect.x = sx;
	srcRect.y = sy;
	srcRect.w = sw;
	srcRect.h = sh;

	dstRect.x = dx;
	dstRect.y = dy;
	dstRect.w = dw;
	dstRect.h = dh;

	pbox++;

	if (sw < 2 || sh < 2)
	    continue;

	blit.src = &srcSurface;
	blit.dst = &dstSurface;
	blit.srcRect = &srcRect;
	blit.dstRect = &dstRect;

	hwBlit(pMsm, &blit, id != FOURCC_RGB565 ? MDP_DITHER : 0);
   }

    hwBlitFlush(pMsm);

    /* Update dirty regions for compositor */
    DamageDamageRegion(pDraw, clipBoxes);

    return Success;
}

static int
MSMPutHWCodecImage(ScrnInfoPtr pScrn,
		   short srcX, short srcY, short drawX, short drawY,
		   short srcW, short srcH, short drawW, short drawH,
		   int id, unsigned char *buf,
		   short width, short height, Bool sync, RegionPtr clipBoxes,
		   pointer data, DrawablePtr pDraw)
{
   MSMPtr pMsm = MSMPTR(pScrn);

   MSMPortPrivPtr pPriv = (MSMPortPrivPtr) data;

   unsigned int *udata = (unsigned int *)buf;

   if (drawW == 0 || drawH == 0)
      return Success;

   return MSMDoBlit(pMsm, pPriv, pMsm->pfd, udata[1], id, drawX, drawY,
		    drawW, drawH, width, height, clipBoxes, pDraw);
}

static int
MSMPutImage(ScrnInfoPtr pScrn,
	    short srcX, short srcY, short drawX, short drawY,
	    short srcW, short srcH, short drawW, short drawH,
	    int id, unsigned char *buf,
	    short width, short height, Bool sync, RegionPtr clipBoxes,
	    pointer data, DrawablePtr pDraw)
{
   MSMPtr pMsm = MSMPTR(pScrn);
   MSMPortPrivPtr pPriv = (MSMPortPrivPtr) data;

   int size;

   /* Nothing to do here */

   if (srcW == 0 || srcH == 0 || drawW == 0 || drawH == 0)
      return Success;

   if (pPriv->HWCodecFlag || id == FOURCC_NV21) {
      pPriv->HWCodecFlag = TRUE;
      return MSMPutHWCodecImage(pScrn, srcX, srcY, drawX, drawY,
				srcW, srcH, drawW, drawH,
				id, buf, width, height, sync,
				clipBoxes, data, pDraw);
   }

   if (!pMsm->accel) {
	ErrorF("Cannot do software codecs without EXA support\n");
	return BadAlloc;
   }

   if (ISPLANAR(id))
      size = (ALIGN(width, 2) * ALIGN(height, 2)) * 2;
   else
      size = (ALIGN(width, 2) * 2) * height;

   if (pPriv->area == NULL || width > pPriv->width || height > pPriv->height) {

       if (pPriv->area != NULL)
	   msm_free_offscreen_memory(pScrn->pScreen, pPriv->area);

       pPriv->area = msm_alloc_offscreen_memory(pScrn->pScreen, size);

       if (pPriv->area == NULL)
	   return BadAlloc;

       pPriv->width = width;
       pPriv->height = height;
   }

   // FIXME: Rather than change a lot of function prototypes,
   //        just save a pointer to the MSM driver for the copy fundtions.

   latest_pMsm = pMsm;

   /* Clip against the actual source pixmap width and height - we can't
      trust that srcW and srcH are correct */

   if (width < srcW)
	srcW = width;

   if (height < srcH)
	srcH = height;

   switch (id) {
   case FOURCC_UYVY:
   case FOURCC_RGB565:
      MSMCopyPacked(pPriv->area->ptr, buf, srcX, srcY, srcW, srcH, width, height);
      break;
   case FOURCC_YUY2:
      MSMCopyAndSwapPacked(pPriv->area->ptr,
			   buf, srcX, srcY, srcW, srcH, width, height);
      break;
   case FOURCC_NV12:
   case FOURCC_NV21:
       MSMCopyPseudoPlanar(pPriv->area->ptr, buf, srcX, srcY, srcW, srcH, width, height);
      break;
   case FOURCC_I420:
   case FOURCC_YV12:
       MSMCopyPlanar(pPriv->area->ptr, buf, id, srcX, srcY, srcW, srcH, width, height);
       break;
   }

   return MSMDoBlit(pMsm, pPriv, -1, 0,
		    id, drawX, drawY, drawW, drawH, width, height,
		    clipBoxes, pDraw);
}

static int
MSMQueryImageAttributes(ScrnInfoPtr pScrn,
			int id,
			unsigned short *w, unsigned short *h,
			int *pitches, int *offsets)
{
   int ypitch;

   if (!ISPLANAR(id)) {

      /* Packed */

      if (pitches)
	 pitches[0] = ALIGN(*w, 2) * 2;

      if (offsets)
	 offsets[0] = 0;

      return ALIGN(*w, 2) * 2 * *h;
   } else if (id == FOURCC_NV12 || id == FOURCC_NV21) {
      /* Pseudo planar */

      /* Adjust the width to be word aligned */
      *w = ALIGN(*w, 2);

      /* The number of lines needs to be even */
      *h = ALIGN(*h, 2);

      /* Figure out the pitches for the segments */

      ypitch = ALIGN(*w, 2);

      if (pitches) {
	 pitches[0] = ypitch;
	 pitches[1] = ypitch;
      }

      /* Calculate the offsets of the chunks */

      if (offsets) {
	 offsets[0] = 0;
	 offsets[1] = (*h * ypitch);
      }

      /* Return the total size of the chunk */
      return (*h * ypitch) + (*h * ypitch);
   } else {
      /* True planar */

      /* Adjust the width to be word aligned */
      *w = ALIGN(*w, 2);

      /* The number of lines needs to be even */
      *h = ALIGN(*h, 2);

      /* Figure out the pitches for the segments */

      ypitch = ALIGN(*w, 2);

      if (pitches) {
	 pitches[0] = ypitch;
	 pitches[1] = ypitch >> 1;
	 pitches[2] = ypitch >> 1;
      }

      /* Calculate the offsets of the chunks */

      if (offsets) {
	 offsets[0] = 0;
	 offsets[1] = (*h * ypitch);
	 offsets[2] = ((*h >> 1) * (ypitch >> 1));
      }

      return (*h * ypitch) + (((*h >> 1) * (ypitch >> 1)) * 2);
   }
}

static void
MSMQueryBestSize(ScrnInfoPtr pScrn, Bool motion,
		 short vidW, short vidH, short drawW, short drawH,
		 unsigned int *retW, unsigned int *retH, pointer data)
{
   /* Allow any size window */

   *retW = drawW;
   *retH = drawH;
}

static void
MSMStopVideo(ScrnInfoPtr pScrn, pointer data, Bool exit)
{
   MSMPtr pMsm = MSMPTR(pScrn);

   MSMPortPrivPtr pPriv = (MSMPortPrivPtr) data;

   if (!pPriv->HWCodecFlag && pPriv->area != NULL)
       msm_free_offscreen_memory(pScrn->pScreen, pPriv->area);

   if (exit && pPriv->HWCodecFlag && pMsm->pfd > 0) {
      close(pMsm->pfd);
      pMsm->pfd = -1;

      if (cmptr) {
         free(cmptr);
         cmptr=NULL;
      }

      pPriv->HWCodecFlag = FALSE;
   }

   pPriv->area = NULL;
   pPriv->width = 0;
   pPriv->height = 0;
}

static int
MSMSetPortAttribute(ScrnInfoPtr pScrni,
		    Atom attribute, INT32 value, pointer data)
{
   MSMPortPrivPtr pPriv = (MSMPortPrivPtr) data;

   if (attribute == xvHWCodec)
      pPriv->HWCodecFlag = value ? 1 : 0;

   return Success;
}

static int
MSMGetPortAttribute(ScrnInfoPtr pScrni,
		    Atom attribute, INT32 * value, pointer data)
{
   MSMPortPrivPtr pPriv = (MSMPortPrivPtr) data;

   if (attribute == xvHWCodec)
      *value = pPriv->HWCodecFlag;
   else
      return BadMatch;

   return Success;
}

static XF86VideoAdaptorPtr
MSMInitAdaptor(ScreenPtr pScreen)
{
   ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];

   MSMPtr pMsm = MSMPTR(pScrn);

   XF86VideoAdaptorRec *adapt;

   MSMPortPrivPtr pPriv;

   int i;

   adapt = xcalloc(1, sizeof(XF86VideoAdaptorRec) +
		   (sizeof(DevUnion) * pMsm->xvports) +
		   (sizeof(MSMPortPrivRec) * pMsm->xvports));

   if (adapt == NULL)
      return NULL;

   adapt->type = XvWindowMask | XvInputMask | XvImageMask;
   adapt->flags = VIDEO_OVERLAID_IMAGES | VIDEO_CLIP_TO_VIEWPORT;

   adapt->name = "MSM";
   adapt->nEncodings = 1;
   adapt->pEncodings = DummyEncoding;

   adapt->nFormats = ARRAY_SIZE(Formats);
   adapt->pFormats = Formats;

   adapt->nPorts = pMsm->xvports;

   /* This is a list of private pointers - located immediately
    * after the adapt struture */

   adapt->pPortPrivates = (DevUnion *) (&adapt[1]);

   /* The actual set of private sructures begins afer the
    * DevUnion component */

   pPriv = (MSMPortPrivRec *) (&adapt->pPortPrivates[pMsm->xvports]);

   for (i = 0; i < pMsm->xvports; i++) {
      adapt->pPortPrivates[i].ptr = (void *)&pPriv[i];
   }

   adapt->nAttributes = ARRAY_SIZE(Attributes);
   adapt->pAttributes = Attributes;

   adapt->nImages = ARRAY_SIZE(Images);
   adapt->pImages = Images;

   adapt->StopVideo = MSMStopVideo;
   adapt->SetPortAttribute = MSMSetPortAttribute;
   adapt->GetPortAttribute = MSMGetPortAttribute;
   adapt->QueryBestSize = MSMQueryBestSize;
   adapt->PutImage = MSMPutImage;
   adapt->QueryImageAttributes = MSMQueryImageAttributes;

   xvHWCodec = MakeAtom("XV_HWCODEC", sizeof("XV_HWCODEC") - 1, TRUE);

   return adapt;
}

void
MSMInitVideo(ScreenPtr pScreen)
{
   ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];

   XF86VideoAdaptorPtr adapt = MSMInitAdaptor(pScreen);

   XF86VideoAdaptorPtr *list, *newlist;

   int count;

   if (adapt == NULL)
      return;

   count = xf86XVListGenericAdaptors(pScrn, &list);

   newlist = xalloc((count + 1) * sizeof(XF86VideoAdaptorPtr *));
   if (newlist == NULL)
      return;

   if (count > 0)
      memcpy(newlist, list, count * sizeof(XF86VideoAdaptorPtr *));

   newlist[count++] = adapt;

   xf86XVScreenInit(pScreen, newlist, count);

   xfree(newlist);
}
