/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

//----------------------------------------------------------------------------
//
//  File:       Credentials.cpp
//
//  Notes:      code to retrieve AMT credentials
//
//----------------------------------------------------------------------------

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "Credentials.h"
#include "registryWrapper.h"
#include "CryptUtils.h"
#include <fstream>

using namespace std;

string GetRegistryValue(const char* regBase, const char* regKey, const char* val);
CredentialsRegInfo::CredentialsRegInfo(const std::string &base,
				       const std::string &key,
				       const std::string &user,
				       const std::string &pass,
				       const std::string &cert,
				       const std::string &cainfo,
				       const std::string &capath,
				       const std::string &sslKey) :
  regBase(base), regKey(key), regUser(user), regPass(pass), regCert(cert),regCainfo(cainfo), regCapath(capath), regSSLkey(sslKey)
{
}


template <class T>
void EraseString(T &string, size_t size) {
	for (size_t i = 0; i < size; i++) {
		string[i] = '\0';
	}
}

Credentials *RetrieveCredentials(const CredentialsRegInfo &credRegInfo)
{
	Credentials *cred = new Credentials;

	const char* regBase = credRegInfo.regBase.c_str();
	const char* regKey = credRegInfo.regKey.c_str();

	cred->username   = GetRegistryValue(regBase, regKey, credRegInfo.regUser.c_str());
	cred->password   = GetRegistryValue(regBase, regKey, credRegInfo.regPass.c_str());
	cred->certificate= GetRegistryValue(regBase, regKey, credRegInfo.regCert.c_str());
	cred->cainfo     = GetRegistryValue(regBase, regKey, credRegInfo.regCainfo.c_str());
	cred->capath     = GetRegistryValue(regBase, regKey, credRegInfo.regCapath.c_str());
	cred->sslkey     = GetRegistryValue(regBase, regKey, credRegInfo.regSSLkey.c_str());

	return cred;
}

string GetRegistryValue(const char* regBase, const char* regKey, const char* val)
{
  string retVal = "";
  UNSREG_DATATYPE type;

  unsigned char *encryptedVal = NULL;
  unsigned long encryptedValLen = 0;
  UNSREG_DATATYPE regEncryptedValLen = 0;
  unsigned char *value = NULL;

  if (!GetRegistryData(NULL, &regEncryptedValLen, &type, regBase, regKey, val)) {
    goto err;
  }

  if (regEncryptedValLen > 0) {
    encryptedVal = new unsigned char[regEncryptedValLen];

    if (encryptedVal == NULL) {
      goto err;
    }

    if (!GetRegistryData(encryptedVal, &regEncryptedValLen, &type, regBase,
			 regKey, val)) {
      goto err;
    }
  }

  encryptedValLen = (unsigned long)regEncryptedValLen;

  if (!Decrypt(encryptedVal, &encryptedValLen, (void **)&value)) {
    goto err;
  }

  if (value)
    retVal = string((char *)value, encryptedValLen);

 err:
  delete []encryptedVal;

#ifdef _WIN32
  LocalFree(value);
#else
  delete []value;
#endif
  return string(retVal);
}

bool StoreCredentials(Credentials *cred, const CredentialsRegInfo &credRegInfo)
{
	unsigned char *encryptedUser = NULL;
	unsigned char *encryptedPass = NULL;
	unsigned char *encryptedCert = NULL;
	unsigned char *encryptedSSLKey = NULL;
	unsigned char *encryptedCainfo = NULL;
	unsigned char *encryptedCapath = NULL;

	bool res = false;
	const char* regBase = credRegInfo.regBase.c_str();
	const char* regKey = credRegInfo.regKey.c_str();
	const char* regUser = credRegInfo.regUser.c_str();
	const char* regPass = credRegInfo.regPass.c_str();
	const char* regCert = credRegInfo.regCert.c_str();
	const char* regCainfo = credRegInfo.regCainfo.c_str();
	const char* regCapath = credRegInfo.regCapath.c_str();
	const char* regSSLKey = credRegInfo.regSSLkey.c_str();


	unsigned long encryptedUserLen = (unsigned long)cred->username.size();
	unsigned long encryptedPassLen = (unsigned long)cred->password.size();
	unsigned long encryptedCertLen = (unsigned long)cred->certificate.size();
	unsigned long encryptedCapathLen = (unsigned long)cred->capath.size();
	unsigned long encryptedCainfoLen = (unsigned long)cred->cainfo.size();
	unsigned long encryptedSSLKeyLen = (unsigned long)cred->sslkey.size();

	if (!Encrypt(cred->username.c_str(), &encryptedUserLen, (void **)&encryptedUser) ||
	    !Encrypt(cred->password.c_str(), &encryptedPassLen, (void **)&encryptedPass) ||
	    !Encrypt(cred->certificate.c_str(), &encryptedCertLen, (void**)&encryptedCert) ||
	    !Encrypt(cred->capath.c_str(), &encryptedCapathLen, (void**)&encryptedCapath) ||
	    !Encrypt(cred->cainfo.c_str(), &encryptedCainfoLen, (void**)&encryptedCainfo) ||
	    !Encrypt(cred->sslkey.c_str(), &encryptedSSLKeyLen, (void**)&encryptedSSLKey)) {
		goto out;
	}

	if (!SetRegistryData(encryptedUser, (UNSREG_DATATYPE)encryptedUserLen, REG_BINARY,
			     regBase, regKey, regUser, false)) {
	  goto out;
	}

	if (!SetRegistryData(encryptedPass, (UNSREG_DATATYPE)encryptedPassLen, REG_BINARY,
			     regBase, regKey, regPass, false)) {
	  goto out;
	}

	if (!SetRegistryData(encryptedCert, (UNSREG_DATATYPE)encryptedCertLen, REG_BINARY,
			     regBase, regKey, regCert, false)) {
	  goto out;
	}

	if (!SetRegistryData(encryptedCapath, (UNSREG_DATATYPE)encryptedCapathLen, REG_BINARY,
			     regBase, regKey, regCapath, false)) {
	  goto out;
	}

	if (!SetRegistryData(encryptedCainfo, (UNSREG_DATATYPE)encryptedCainfoLen, REG_BINARY,
			     regBase, regKey, regCainfo, false)) {
	  goto out;
	}
	if (!SetRegistryData(encryptedSSLKey, (UNSREG_DATATYPE)encryptedSSLKeyLen, REG_BINARY,
			     regBase, regKey, regSSLKey, false)) {
	  goto out;
	}
	res = true;

out:
#ifdef _WIN32
	LocalFree(encryptedUser);
	LocalFree(encryptedPass);
	LocalFree(encryptedCert);
	LocalFree(encryptedCapath);
	LocalFree(encryptedCainfo);
	LocalFree(encryptedSSLKey);
#else
	delete []encryptedUser;
	delete []encryptedPass;
	delete []encryptedCert;
	delete []encryptedCapath;
	delete []encryptedCainfo;
	delete []encryptedSSLKey;
#endif

	return res;
}

