/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "AMTStatusThread.h"
#include "UNSEventLogger.h"
#include "UNSConfig.h"

#ifdef _WIN32
#include "heci/Windows/PTHICommand.h"
#else
#include "heci/Linux/PTHICommand.h"
#endif
#include "StatusCodeDefinitions.h"

#include <sstream>
#include <string>

using namespace std;

const unsigned long AMTStatusThread::DEFAULT_CHECK_AMT_STATUS_TIMEOUT = 10 * SECOND;
const unsigned long AMTStatusThread::HECI_DOWNTIME_ON_RESUME = 10 * SECOND;

// Constructor
AMTStatusThread::AMTStatusThread(UNSEventLogger *pLogger /*= NULL*/) :
	PeriodicActionThread("AMTStatusCheck", DEFAULT_CHECK_AMT_STATUS_TIMEOUT, HECI_DOWNTIME_ON_RESUME)
{
	m_pLogger = pLogger;
	m_nNumPollsDisabled = 0;
	m_State = INITIAL_STATE;
}

//Destructor
AMTStatusThread::~AMTStatusThread()
{
	m_AMTStatus.close();
}

// Initialize this thread with a given action interval time.
bool AMTStatusThread::init(const UNSConfig &Config)
{
	m_UNSConfig = Config;

	m_AMTStatus.init(true);

	setActionInterval(m_UNSConfig.CheckAMTStatusTimeout());
	if (!m_AMTStatus.isOk())
	{
		stringstream ss;
		ss << "IPCAMTStatus object is not ok: last error=" << m_AMTStatus.lastError();
		if (m_pLogger)
			m_pLogger->ErrorLog(ss.str().c_str());
	}
	return true;
}

bool AMTStatusThread::IsAMTEnabledByHECI(bool bLogging /*= false*/)
{
#ifdef _DEBUG
	//In debug mode, we just return the enablement that was set in the configuration.
	if (m_UNSConfig.debugMode())
		return m_UNSConfig.debugModeAMTEnabled();
#endif

	PTHICommand pthiCommand;
	AMT_PROVISIONING_STATE provisioningState;
	bool bAMTEnabled = false;

	if (!pthiCommand.PTHIClient.Init())
	{
		debugLog("Could not initialize PTHI client");
		return bAMTEnabled;
	}
	if (pthiCommand.GetProvisioningState(&provisioningState) == AMT_STATUS_SUCCESS)
	{
		if (provisioningState == PROVISIONING_STATE_IN ||
			provisioningState == PROVISIONING_STATE_POST)
		{
			debugLog("Got status from HECI: AMT is IN/POST provisioning");
			bAMTEnabled = true;
		}
		else
		{
			debugLog("Got status from HECI: AMT is PRE provisioning");
		}
	}
	else
	{
		debugLog("Got status from HECI: HECI command failed.");
		//If pthi command failed, because HECI could not be talked to, it is assumed
		//that AMT is disabled.
		if (bLogging && m_pLogger)
			m_pLogger->HECICommandFailedError();
	}

	pthiCommand.PTHIClient.Deinit();

	return bAMTEnabled;
}

void AMTStatusThread::debugLog(const char *msg)
{
	if (m_pLogger && m_UNSConfig.TraceLogEnabled())
		m_pLogger->DebugLog(msg);
}

AMTStatusThread::State AMTStatusThread::SetAMTEnabled(bool bEnabled, bool bLogging /*= false*/)
{
	if (!m_AMTStatus.SetEnablement(bEnabled ? IPCAMTStatus::STATUS_ENABLED : IPCAMTStatus::STATUS_DISABLED))
	{
		if (bLogging && m_pLogger)
			m_pLogger->SetAMTStatusFailedError();
	}
	stringstream ss;
	ss << "Setting AMT status: " << (bEnabled ? "Enabled" : "Disabled");
	debugLog(ss.str().c_str());
	return bEnabled ? ENABLED_STATE : DISABLED_STATE;
}


void AMTStatusThread::action()
{
	switch (m_State)
	{
		case INITIAL_STATE:
			m_State = OnInitialState();
			break;
		case ENABLED_STATE:
			m_State = OnEnabledState();
			break;
		case POSSIBLY_DISABLED_STATE:
			m_State = OnPossiblyDisabledState();
			break;
		case DISABLED_STATE:
			m_State = OnDisabledState();
			break;
	}
}

void AMTStatusThread::cleanup()
{
}

AMTStatusThread::State AMTStatusThread::OnInitialState()
{
	if (IsAMTEnabledByHECI(true)) //Do this call with logging
	{
		return SetAMTEnabled(true, true); //Do this call with logging
	}
	else
	{
		return SetAMTEnabled(false, true); //Do this call with logging
	}
}

AMTStatusThread::State AMTStatusThread::OnEnabledState()
{
	if (IsAMTEnabledByHECI())
	{
		return ENABLED_STATE;
	}
	else
	{
		m_nNumPollsDisabled = 1; //We got "disabled" status from HECI once.
		return OnPossiblyDisabledState(false);
	}
}

AMTStatusThread::State AMTStatusThread::OnPossiblyDisabledState(bool bNeedCheck /*=true*/)
{
	//bNeedCheck tells us if we need to check the status by HECI. When it's false,
	//the status received from HECI is assumed to be disabled.
	if (bNeedCheck && IsAMTEnabledByHECI())
	{
		return SetAMTEnabled(true);
	}
	else
	{
		if (m_nNumPollsDisabled++ == m_UNSConfig.NumPollsForDisabled())
		{
			//We reached the threshold for setting "disabled" status.
			stringstream ss;
			ss << "Got 'disabled' status from HECI " << (m_nNumPollsDisabled-1) << " times."
			   << " Setting AMT status to 'disabled'.";
			debugLog(ss.str().c_str());
			return SetAMTEnabled(false);
		}
		else
		{
			//We haven't reached the threshold yet.
			stringstream ss;
			ss << "Got 'disabled' status from HECI " << (m_nNumPollsDisabled-1) << " times."
				<< " Still not changing AMT status to 'disabled'.";
			debugLog(ss.str().c_str());
			return POSSIBLY_DISABLED_STATE;
		}
	}
}

AMTStatusThread::State AMTStatusThread::OnDisabledState()
{
	if (IsAMTEnabledByHECI())
	{
		return SetAMTEnabled(true);
	}
	else
	{
		return DISABLED_STATE;
	}
}
