/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License version 3, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
 * SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

// local
#include "application_manager.h"
#include "proc_info.h"
#include "application.h"
#include "desktopfilereader.h"
#include "dbuswindowstack.h"
#include "taskcontroller.h"
#include "upstart/applicationcontroller.h"

// unity-mir
#include "qmirserverapplication.h"
#include "shellserverconfiguration.h"
#include "sessionlistener.h"
#include "sessionauthorizer.h"
#include "initialsurfaceplacementstrategy.h"
#include "taskcontroller.h"
#include "logging.h"

// mir
#include <mir/scene/depth_id.h>
#include <mir/shell/session.h>
#include <mir/shell/focus_controller.h>
#include <mir/shell/surface.h>
#include <mir/graphics/display.h>
#include <mir/graphics/display_buffer.h>
#include <mircommon/mir/geometry/rectangles.h>

// Qt
#include <QCoreApplication>

// std
#include <csignal>

// Default element sizes
#define SIDE_STAGE_WIDTH_GU 40
#define TABLET_MODE_MINIMUM_WIDTH_GU 100

namespace msh = mir::shell;

using namespace unity::shell::application;

namespace unitymir
{

namespace
{

QSize get_display_size(const std::shared_ptr<mir::graphics::Display> & display) {
    // Obtain display size
    mir::geometry::Rectangles view_area;
    display->for_each_display_buffer(
        [&view_area](const mir::graphics::DisplayBuffer & db)
        {
            view_area.add(db.view_area());
        });

    return QSize(
        view_area.bounding_rectangle().size.width.as_uint32_t(),
        view_area.bounding_rectangle().size.height.as_uint32_t()
        );
}


void connectToSessionListener(ApplicationManager * manager, SessionListener * listener)
{
    QObject::connect(listener, &SessionListener::sessionStarting,
                     manager, &ApplicationManager::onSessionStarting);
    QObject::connect(listener, &SessionListener::sessionStopping,
                     manager, &ApplicationManager::onSessionStopping);
    QObject::connect(listener, &SessionListener::sessionFocused,
                     manager, &ApplicationManager::onSessionFocused, Qt::QueuedConnection);
    QObject::connect(listener, &SessionListener::sessionUnfocused,
                     manager, &ApplicationManager::onSessionUnfocused);
    QObject::connect(listener, &SessionListener::sessionCreatedSurface,
                     manager, &ApplicationManager::onSessionCreatedSurface);
}

void connectToSessionAuthorizer(ApplicationManager * manager, SessionAuthorizer * authorizer)
{
    QObject::connect(authorizer, &SessionAuthorizer::requestAuthorizationForSession,
                     manager, &ApplicationManager::authorizeSession, Qt::BlockingQueuedConnection);
}


void connectToPlacementStrategy(ApplicationManager * manager, InitialSurfacePlacementStrategy * strategy)
{
    QObject::connect(strategy, &InitialSurfacePlacementStrategy::requestPlacementForSession,
                     manager, &ApplicationManager::placeSession, Qt::DirectConnection);

}

void connectToTaskController(ApplicationManager * manager, TaskController * controller)
{
    QObject::connect(controller, &TaskController::processStartReport,
                     manager, &ApplicationManager::onProcessStartReportReceived);
    QObject::connect(controller, &TaskController::processStopped,
                     manager, &ApplicationManager::onProcessStopped);
    QObject::connect(controller, &TaskController::requestFocus,
                     manager, &ApplicationManager::onFocusRequested);
    QObject::connect(controller, &TaskController::requestResume,
                     manager, &ApplicationManager::onResumeRequested);

}
}

ApplicationManager* ApplicationManager::Factory::Factory::create()
{
    QMirServerApplication* mirServerApplication = dynamic_cast<QMirServerApplication*>(QCoreApplication::instance());
    if (mirServerApplication == nullptr) {
        LOG("Need to use QMirServerApplication");
        QCoreApplication::quit();
        return nullptr;
    }

    ShellServerConfiguration * mirServer = mirServerApplication->server();

    QSize displaySize{get_display_size(mirServer->the_display())};

    QSharedPointer<upstart::ApplicationController> appController(new upstart::ApplicationController());
    QSharedPointer<TaskController> taskController(new TaskController(nullptr, appController));
    QSharedPointer<DesktopFileReader::Factory> fileReaderFactory(new DesktopFileReader::Factory());
    QSharedPointer<ProcInfo> procInfo(new ProcInfo());

    // FIXME: We should use a QSharedPointer to wrap this ApplicationManager object, which requires us
    // to use the data() method to pass the raw pointer to the QML engine. However the QML engine appears
    // to take ownership of the object, and deletes it when it wants to. This conflicts with the purpose
    // of the QSharedPointer, and a double-delete results. Trying QQmlEngine::setObjectOwnership on the 
    // object no effect, which it should. Need to investigate why.
    ApplicationManager* appManager = new ApplicationManager(
                                             taskController,
                                             fileReaderFactory,
                                             procInfo,
                                             mirServer->the_focus_controller(),
                                             displaySize
                                         );

    connectToSessionListener(appManager, mirServer->sessionListener());
    connectToSessionAuthorizer(appManager, mirServer->sessionAuthorizer());
    connectToPlacementStrategy(appManager, mirServer->placementStrategy());
    connectToTaskController(appManager, taskController.data());

    return appManager;
}

ApplicationManager* ApplicationManager::singleton()
{
    static ApplicationManager* instance;
    if (!instance) {
        Factory appFactory;
        instance = appFactory.create();
    }
    return instance;
}

ApplicationManager::ApplicationManager(
        const QSharedPointer<TaskController>& taskController,
        const QSharedPointer<DesktopFileReader::Factory>& desktopFileReaderFactory,
        const QSharedPointer<ProcInfo>& procInfo,
        const std::shared_ptr<mir::shell::FocusController>& controller,
        const QSize& displaySize,
        QObject *parent)
    : ApplicationManagerInterface(parent)
    , m_focusedApplication(nullptr)
    , m_mainStageApplication(nullptr)
    , m_sideStageApplication(nullptr)
    , m_msApplicationToBeFocused(nullptr)
    , m_ssApplicationToBeFocused(nullptr)
    , m_lifecycleExceptions(QStringList() << "com.ubuntu.music")
    , m_focusController(controller)
    , m_dbusWindowStack(new DBusWindowStack(this))
    , m_taskController(taskController)
    , m_desktopFileReaderFactory(desktopFileReaderFactory)
    , m_procInfo(procInfo)
    , m_gridUnitPx(8)
    , m_fenceNext(false)
    , m_displaySize(displaySize)
    , m_panelHeight(54)
{
    DLOG("ApplicationManager::ApplicationManager (this=%p)", this);

    // Setup panel height
    QByteArray gridUnitString = qgetenv("GRID_UNIT_PX");
    if (!gridUnitString.isEmpty()) {
        bool ok;
        int value = gridUnitString.toInt(&ok);
        if (ok) {
            m_gridUnitPx = value;
        }
    }

    int densityPixelPx = qFloor( (float)m_gridUnitPx / 8 );

    m_panelHeight = 3 * m_gridUnitPx + 2 * densityPixelPx;
}

ApplicationManager::~ApplicationManager()
{
    DLOG("ApplicationManager::~ApplicationManager");
}

int ApplicationManager::panelHeight()
{
    return m_panelHeight;
}

int ApplicationManager::rowCount(const QModelIndex &parent) const
{
    return !parent.isValid() ? m_applications.size() : 0;
}

QVariant ApplicationManager::data(const QModelIndex &index, int role) const
{
    if (index.row() >= 0 && index.row() < m_applications.size()) {
        Application *application = m_applications.at(index.row());
        switch (role) {
            case RoleAppId:
                return QVariant::fromValue(application->appId());
            case RoleName:
                return QVariant::fromValue(application->name());
            case RoleComment:
                return QVariant::fromValue(application->comment());
            case RoleIcon:
                return QVariant::fromValue(application->icon());
            case RoleStage:
                return QVariant::fromValue((int)application->stage());
            case RoleState:
                return QVariant::fromValue((int)application->state());
            case RoleFocused:
                return QVariant::fromValue(application->focused());
            case RoleScreenshot:
                return QVariant::fromValue(application->screenshot());
            default:
                return QVariant();
        }
    } else {
        return QVariant();
    }
}

Application* ApplicationManager::get(int index) const
{
    DLOG("ApplicationManager::get (this=%p, index=%i, count=%i)", this, index, m_applications.count());
    if (index < 0 || index >= m_applications.count()) {
        return nullptr;
    }
    return m_applications.at(index);
}

Application* ApplicationManager::findApplication(const QString &appId) const
{
    for (Application *app : m_applications) {
        if (app->appId() == appId) {
            return app;
        }
    }
    return nullptr;
}

bool ApplicationManager::requestFocusApplication(const QString &appId)
{
    DLOG("ApplicationManager::requestFocusApplication (this=%p, appId=%s)", this, qPrintable(appId));
    Application *application = findApplication(appId);

    if (!application) {
        DLOG("No such running application '%s'", qPrintable(appId));
        return false;
    }

    if (application == m_focusedApplication) {
        DLOG("Application %s is already focused", qPrintable(appId));
        return false;
    }

    // If there is a currently focused application, first update the screenshot
    // for it and only emit focusRequested() when the screenshot is ready.
    Application *currentlyFocusedApplication = findApplication(focusedApplicationId());
    if (currentlyFocusedApplication) {
        m_nextFocusedAppId = appId;
        currentlyFocusedApplication->updateScreenshot();
    } else {
        Q_EMIT focusRequested(appId);
    }
    return true;
}

QString ApplicationManager::focusedApplicationId() const
{
    if (m_focusedApplication) {
        return m_focusedApplication->appId();
    } else {
        return QString();
    }
}

bool ApplicationManager::suspended() const
{
    return m_suspended;
}

void ApplicationManager::setSuspended(bool suspended)
{
    if (suspended == m_suspended) {
        return;
    }
    m_suspended = suspended;
    Q_EMIT suspendedChanged();

    if (m_suspended) {
        suspendApplication(m_mainStageApplication);
        suspendApplication(m_sideStageApplication);
    } else {
        resumeApplication(m_mainStageApplication);
        resumeApplication(m_sideStageApplication);
    }
}

void ApplicationManager::suspendApplication(Application *application)
{
    if (application == nullptr)
        return;

    updateScreenshot(application->appId());

    DLOG("ApplicationManager::suspend(this=%p, application(%p)->appId(%s) )",this, application, qPrintable(application->appId()));
    // Present in exceptions list, return.
    if (!m_lifecycleExceptions.filter(application->appId().section('_',0,0)).empty())
        return;

    if (application->state() == Application::Running)
        application->setState(Application::Suspended);
}

void ApplicationManager::resumeApplication(Application *application)
{
    if (application == nullptr)
        return;

    if (application->state() != Application::Running)
        application->setState(Application::Running);
}

bool ApplicationManager::focusApplication(const QString &appId)
{
    Application *application = findApplication(appId);
    DLOG("ApplicationManager::focusApplication (this=%p, application=%p, appId=%s)", this, application, qPrintable(appId));

    if (!application) {
        DLOG("No such running application '%s'", qPrintable(appId));
        return false;
    }
   
    if (application->stage() == Application::MainStage && m_sideStageApplication)
        suspendApplication(m_sideStageApplication);
    
    if (application->stage() == Application::MainStage)
        m_msApplicationToBeFocused = application;
    else
        m_ssApplicationToBeFocused = application;
    
    if (application->state() == Application::Stopped) {
        // Respawning this app, move to end of application list so onSessionStarting works ok
        // FIXME: this happens pretty late, shell could request respawn earlier
        application->setState(Application::Running);
        int from = m_applications.indexOf(application);
        move(from, m_applications.length()-1);
    } else {
        if (application->session()) {
            m_focusController->set_focus_to(application->session());
            int from = m_applications.indexOf(application);
            move(from, 0);
        }
    }

    // FIXME(dandrader): lying here. The operation is async. So we will only know whether
    // the focusing was successful once the server replies. Maybe the API in unity-api should
    // reflect that?
    return true;
}

void ApplicationManager::unfocusCurrentApplication()
{
    DLOG("ApplicationManager::unfocusCurrentApplication (this=%p)", this);

    suspendApplication(m_sideStageApplication);
    suspendApplication(m_mainStageApplication);

    // Clear both stages
    m_msApplicationToBeFocused = nullptr;
    m_ssApplicationToBeFocused = nullptr;
    m_focusController->set_focus_to(nullptr); //FIXME(greyback)
}

Application* ApplicationManager::startApplication(const QString &appId,
                                                  const QStringList &arguments)
{
    return startApplication(appId, NoFlag, arguments);
}

Application *ApplicationManager::startApplication(const QString &appId, ExecFlags flags,
                                                  const QStringList &arguments)
{
    DLOG("ApplicationManager::startApplication (this=%p, appId=%s)", this, qPrintable(appId));

    if (!m_taskController->start(appId, arguments)) {
        LOG("Asking Upstart to start application '%s' failed", qPrintable(appId));
        return nullptr;
    }

    {
        Application * application = findApplication(appId);
        if (application) {
            DLOG("ApplicationManager::startApplication - application already "
                 "exists: (this=%p, app=%p, appId=%s)",
                 this, application, qPrintable(appId));
        }
    }

    Application* application = new Application(
                m_taskController,
                m_desktopFileReaderFactory->createInstanceForAppId(appId),
                Application::Starting,
                arguments,
                this);
    if (!application->isValid()) {
        DLOG("Unable to instantiate application with appId '%s'", qPrintable(appId));
        return nullptr;
    }

    // override stage if necessary
    if (application->stage() == Application::SideStage && flags.testFlag(ApplicationManager::ForceMainStage)) {
        application->setStage(Application::MainStage);
    }

    add(application);
    return application;
}

void ApplicationManager::onProcessStartReportReceived(const QString &appId, const bool failure)
{
    DLOG("ApplicationManager::onProcessStartReportReceived (this=%p, appId=%s, failure=%c)",
         this, qPrintable(appId), (failure) ? 'Y' : 'N');

    if (failure) {
        DLOG("ApplicationManager::onProcessStartReportReceived handling failure:");
        stopStartingApplication(appId);
        return;
    }

    Application *application = findApplication(appId);

    if (!application) { // if shell did not start this application, but upstart did
        application = new Application(
                    m_taskController,
                    m_desktopFileReaderFactory->createInstanceForAppId(appId),
                    Application::Starting,
                    QStringList(), this);
        if (!application->isValid()) {
            DLOG("Unable to instantiate application with appId '%s'", qPrintable(appId));
            return;
        }

        // override stage if necessary (i.e. side stage invalid on phone)
        if (application->stage() == Application::SideStage &&
                m_displaySize.width() < (TABLET_MODE_MINIMUM_WIDTH_GU * m_gridUnitPx)) {
            application->setStage(Application::MainStage);
        }

        add(application);
        Q_EMIT focusRequested(appId);
    }
    else {
        DLOG("ApplicationManager::onProcessStartReportReceived application already found: (app=%p, appId=%s)", application, qPrintable(appId));
    }
}

bool ApplicationManager::stopApplication(const QString &appId)
{
    Application *application = findApplication(appId);
    DLOG("ApplicationManager::stopApplication (this=%p, application=%p, appId=%s)", this, application, qPrintable(appId));

    if (!application) {
        DLOG("No such running application '%s'", qPrintable(appId));
        return false;
    }

    checkFocusOnRemovedApplication(application);

    remove(application);
    m_dbusWindowStack->WindowDestroyed(0, application->appId());

    bool result = m_taskController->stop(application->appId());

    LOG_IF(result == false, "FAILED to ask Upstart to stop application '%s'", qPrintable(application->appId()));

    if (!result && application->pid() > 0) {
        DLOG("TaskController::stop UAL failed. Sending SIGTERM to %i", application->pid());
        kill(application->pid(), SIGTERM);
    }

    delete application;

    // FIXME(dandrader): lying here. The operation is async. So we will only know whether
    // the focusing was successful once the server replies. Maybe the API in unity-api should
    // reflect that?
    return result;
}

bool ApplicationManager::updateScreenshot(const QString &appId)
{
    Application *application = findApplication(appId);

    if (!application) {
        DLOG("No such running application '%s'", qPrintable(appId));
        return false;
    }

    application->updateScreenshot();
    QModelIndex appIndex = findIndex(application);
    Q_EMIT dataChanged(appIndex, appIndex, QVector<int>() << RoleScreenshot);
    return true;
}

void ApplicationManager::stopStartingApplication(const QString &appId)
{
    Application *application = findApplication(appId);

    if (application && application->state() == Application::Starting) {
        shutdownApplication(application);
    }
    else if (application) {
        DLOG("ApplicationManager::stopStartingApplication start failure report received - but application=%p, appId=%s is not in Starting state",application, qPrintable(appId));
    }
}

void ApplicationManager::onProcessStopped(const QString &appId, const bool unexpected)
{
    Application *application = findApplication(appId);
    DLOG("ApplicationManager::onProcessStopped (this=%p, application=%p, appId=%s)", this, application, qPrintable(appId));

    // if shell did not stop the application, but upstart says it died, we assume the process has been
    // killed, so it can be respawned later. Only exception is if that application is focused or running
    // as then it most likely crashed. Update this logic when upstart gives some failure info.
    if (application) {
        shutdownApplication(application);
    }

    if (unexpected) {
        // TODO: pop up a message box/notification?
        LOG("ApplicationManager: application '%s' died unexpectedly!", qPrintable(appId));
    }
}

void ApplicationManager::shutdownApplication(Application* application)
{
    bool removeApplication = checkFocusOnRemovedApplication(application);

    if (application->state() == Application::Running || application->state() == Application::Starting) {
        // Application probably crashed, else OOM killer struck. Either way state wasn't saved
        // so just remove application
        removeApplication = true;
    } else if (application->state() == Application::Suspended) {
        application->setState(Application::Stopped);
        application->setSession(nullptr);
    }

    if (removeApplication) {
        remove(application);
        m_dbusWindowStack->WindowDestroyed(0, application->appId());
        delete application;
    }

}

void ApplicationManager::onFocusRequested(const QString& appId)
{
    DLOG("ApplicationManager::onFocusRequested (this=%p, appId=%s)", this, qPrintable(appId));

    Q_EMIT focusRequested(appId);
}

void ApplicationManager::onResumeRequested(const QString& appId)
{
    Application *application = findApplication(appId);
    DLOG("ApplicationManager::onResumeRequested (this=%p, application=%p, appId=%s)", this, application, qPrintable(appId));


    if (!application) {
        DLOG("ApplicationManager::onResumeRequested: No such running application '%s'", qPrintable(appId));
        return;
    }

    // If app Stopped, trust that upstart-app-launch respawns it itself, and AppManager will
    // be notified of that through the onProcessStartReportReceived slot. Else resume.
    if (application->state() == Application::Suspended) {
        application->setState(Application::Running);
    }
}

void ApplicationManager::screenshotUpdated()
{
    if (sender()) {
        Application *application = static_cast<Application*>(sender());
        QModelIndex appIndex = findIndex(application);
        Q_EMIT dataChanged(appIndex, appIndex, QVector<int>() << RoleScreenshot);

        DLOG("ApplicationManager::screenshotUpdated: Received new screenshot for %s", qPrintable(application->appId()));
    } else {
        DLOG("ApplicationManager::screenshotUpdated: Received screenshotUpdated signal but application has disappeard.");
    }

    // Now that the screenshot is ready, check if there
    // is a pending app focusing request.
    if (!m_nextFocusedAppId.isEmpty()) {
        Q_EMIT focusRequested(m_nextFocusedAppId);
        m_nextFocusedAppId.clear();
    }
}

/************************************* Mir-side methods *************************************/

void ApplicationManager::authorizeSession(const quint64 pid, bool &authorized)
{
    authorized = false; //to be proven wrong

    DLOG("ApplicationManager::authorizeSession (this=%p, pid=%lld)", this, pid);

    for (Application *app : m_applications) {
        if (app->state() == Application::Starting
                && m_taskController->appIdHasProcessId(app->appId(), pid)) {
            app->setPid(pid);
            DLOG("ApplicationManager::authorizeSession - connecting: application=%p and pid=%lld", app, pid);
            authorized = true;
            return;
        }
    }

    /*
     * Hack: Allow applications to be launched externally, but must be executed with the
     * "desktop_file_hint" parameter attached. This exists until upstart-app-launch can
     * notify shell it is starting an application and so shell should allow it. Also reads
     * the --stage parameter to determine the desired stage
     */
    std::unique_ptr<ProcInfo::CommandLine> info = m_procInfo->commandLine(pid);
    if (!info) {
        DLOG("ApplicationManager REJECTED connection from app with pid %lld as unable to read process command", pid);
        return;
    }

    if (info->startsWith("maliit-server") || info->contains("qt5/libexec/QtWebProcess")
        || info->startsWith("/usr/bin/signon-ui")) {
        authorized = true;
        m_fenceNext = true;
        return;
    }

    boost::optional<QString> desktopFileName{ info->getParameter("--desktop_file_hint=") };

    if (!desktopFileName) {
        LOG("ApplicationManager REJECTED connection from app with pid %lld as no desktop_file_hint specified", pid);
        return;
    }

    DLOG("Process supplied desktop_file_hint, loading '%s'", desktopFileName.get().toLatin1().data());

    // FIXME: right now we support --desktop_file_hint=appId for historical reasons. So let's try that in
    // case we didn't get an existing .desktop file path
    DesktopFileReader* desktopData;
    if (QFileInfo(desktopFileName.get()).exists()) {
        desktopData = m_desktopFileReaderFactory->createInstanceForDesktopFile(QFileInfo(desktopFileName.get()));
    } else {
        desktopData = m_desktopFileReaderFactory->createInstanceForAppId(desktopFileName.get());
    }

    if (!desktopData->loaded()) {
        delete desktopData;
        LOG("ApplicationManager REJECTED connection from app with pid %lld as desktop_file_hint file not found", pid);
        return;
    }

    // some naughty applications use a script to launch the actual application. Check for the
    // case where shell actually launched the script.
    Application *application = findApplication(desktopData->appId());
    if (application && application->state() == Application::Starting) {
        DLOG("Process with pid %lld appeared, attached to existing entry '%s' in application lists",
             pid, application->appId().toLatin1().data());
        delete desktopData;
        application->setSessionName(application->appId());
        application->setPid(pid);
        authorized = true;
        return;
    }

    // if stage supplied in CLI, fetch that
    Application::Stage stage = Application::MainStage;
    boost::optional<QString> stageParam = info->getParameter("--stage_hint=");

    if (stageParam && stageParam.get() == "side_stage") {
        stage = Application::SideStage;
    }

    DLOG("Existing process with pid %lld appeared, adding '%s' to application lists", pid, desktopData->name().toLatin1().data());

    QStringList arguments(info->asStringList());
    application = new Application(m_taskController, desktopData, Application::Starting, arguments, this);
    application->setPid(pid);
    application->setStage(stage);
    add(application);
    authorized = true;
}

void ApplicationManager::placeSession(msh::Session const* session, uint32_t &x, uint32_t &y)
{
    Application* application = findApplicationWithSession(session);
    DLOG("ApplicationManager::placeSession (this=%p, application=%p, session=%p, name=%s)", this, application, session, session?(session->name().c_str()):"null");
    

    // Application defaults 
    x = 0;
    y = m_panelHeight;

    // Shell client override
    if (!session) {
        y = 0;
        return;
    }

    // Fullscreen applications override
    if (application && application->fullscreen())
        y = 0;

    // SideStage override
    if (application && application->stage() == Application::SideStage)
        x = m_displaySize.width() - (SIDE_STAGE_WIDTH_GU * m_gridUnitPx);
    
    DLOG("ApplicationManager::placeSession (x=%d, y=%d)", x, y);
}

void ApplicationManager::onSessionStarting(const std::shared_ptr<msh::Session>& session)
{
    DLOG("ApplicationManager::onSessionStarting (this=%p, application=%s)", this, session?session->name().c_str():"null");

    if (m_fenceNext) {
        m_fenceNext = false;
        return;
    }

    Application* application = findApplicationWithPid(session->process_id());
    if (application && application->state() != Application::Running) {
        application->setSession(session);
        if (application->stage() == Application::MainStage)
            m_msApplicationToBeFocused = application;
        else
            m_ssApplicationToBeFocused = application;
    } else {
        DLOG("ApplicationManager::onSessionStarting - unauthorized application!!");
    }
}

void ApplicationManager::onSessionStopping(const std::shared_ptr<msh::Session>& session)
{
    // in case application closed not by hand of shell, check again here:
    Application* application = findApplicationWithSession(session);

    DLOG("ApplicationManager::onSessionStopping (this=%p, application=%p, appId=%s, session name=%s)", this, application, application?qPrintable(application->appId()):"null", session?session->name().c_str():"null");

    if (application) {
        bool removeApplication = true;

        if (application->state() != Application::Starting) {
            application->setState(Application::Stopped);
            application->setSession(nullptr);
            m_dbusWindowStack->WindowDestroyed(0, application->appId());
            if (application != m_focusedApplication) {
                removeApplication = false;
            }
        }

        if (m_mainStageApplication == application)
            m_mainStageApplication = nullptr;
        
        if (m_sideStageApplication == application)
            m_sideStageApplication = nullptr;

        if (removeApplication) {
            // TODO(greyback) What to do?? Focus next app, or unfocus everything??
            m_focusedApplication = nullptr;
            remove(application);
            delete application;
            Q_EMIT focusedApplicationIdChanged();
        }
    }
}

void ApplicationManager::onSessionFocused(const std::shared_ptr<msh::Session>& session)
{
    Application* application = findApplicationWithSession(session);
    DLOG("ApplicationManager::onSessionFocused (this=%p, application=%p, appId=%s, session name=%s)", this, application, application?qPrintable(application->appId()):"null", session?session->name().c_str():"null");

    // Don't give application focus until it has created it's surface, when it is set as state "Running"
    // and only notify shell of focus changes that it actually expects
    if (application && application->state() != Application::Starting && 
        (application == m_msApplicationToBeFocused ||
         application == m_ssApplicationToBeFocused)
            && application != m_focusedApplication) {
        setFocused(application);
        QModelIndex appIndex = findIndex(application);
        Q_EMIT dataChanged(appIndex, appIndex, QVector<int>() << RoleFocused);
    } else {
        if (application == nullptr) {
            DLOG("Invalid application focused, discarding the event");
            if (nullptr != m_focusedApplication)
                focusApplication(m_focusedApplication->appId());
        }
    }
}

void ApplicationManager::onSessionUnfocused()
{
    DLOG("ApplicationManager::onSessionUnfocused (this=%p, application=%p)", this, m_focusedApplication);
    if (nullptr != m_focusedApplication) {
        Q_ASSERT(m_focusedApplication->focused());
        m_focusedApplication->setFocused(false);
        
        suspendApplication(m_focusedApplication);

        m_focusedApplication = nullptr;
        Q_EMIT focusedApplicationIdChanged();
        m_dbusWindowStack->FocusedWindowChanged(0, QString(), 0);

        QModelIndex appIndex = findIndex(m_focusedApplication);
        Q_EMIT dataChanged(appIndex, appIndex, QVector<int>() << RoleFocused << RoleState);
    }
}

void ApplicationManager::onSessionCreatedSurface(const msh::Session * session,
                                                 const std::shared_ptr<msh::Surface> & surface)
{
    DLOG("ApplicationManager::onSessionCreatedSurface (this=%p)", this);
    Q_UNUSED(surface);

    Application* application = findApplicationWithSession(session);
    if (application && application->state() == Application::Starting) {
        m_dbusWindowStack->WindowCreated(0, application->appId());
        // only when Session creates a Surface will we actually mark it focused
        setFocused(application);
        QModelIndex appIndex = findIndex(application);
        Q_EMIT dataChanged(appIndex, appIndex, QVector<int>() << RoleFocused);
    }
}

void ApplicationManager::setFocused(Application *application)
{
    DLOG("ApplicationManager::setFocused (application=%p, appId=%s)", application, qPrintable(application->appId()));

    if (application == m_focusedApplication)
        return;

    // set state of previously focused app to suspended
    if (m_focusedApplication && m_lifecycleExceptions.filter(m_focusedApplication->appId().section('_',0,0)).empty()) {
        Application *lastApplication = applicationForStage(application->stage());
        suspendApplication(lastApplication);
    }

    if (application->stage() == Application::MainStage)
        m_mainStageApplication = application;
    else
        m_sideStageApplication = application;

    m_focusedApplication = application;
    m_focusedApplication->setFocused(true);
    m_focusedApplication->setState(Application::Running);
    move(m_applications.indexOf(application), 0);
    Q_EMIT focusedApplicationIdChanged();
    m_dbusWindowStack->FocusedWindowChanged(0, application->appId(), application->stage());
}

Application* ApplicationManager::findApplicationWithSession(const std::shared_ptr<msh::Session> &session)
{
    return findApplicationWithSession(session.get());
}

Application* ApplicationManager::findApplicationWithSession(const msh::Session *session)
{
    for (Application *app : m_applications) {
        if (app->session().get() == session) {
            return app;
        }
    }
    return nullptr;
}

Application* ApplicationManager::findApplicationWithPid(const qint64 pid)
{
    if (pid <= 0)
        return nullptr;

    for (Application *app : m_applications) {
        if (app->m_pid == pid) {
            return app;
        }
    }
    return nullptr;
}

Application* ApplicationManager::applicationForStage(Application::Stage stage)
{
    DLOG("ApplicationManager::focusedApplicationForStage(this=%p)", this);

    if (stage == Application::MainStage)
        return m_mainStageApplication;
    else
        return m_sideStageApplication;
}

void ApplicationManager::add(Application* application)
{
    DASSERT(application != nullptr);
    DLOG("ApplicationManager::add (this=%p, application='%s')", this, qPrintable(application->name()));

    connect(application, &Application::screenshotChanged, this, &ApplicationManager::screenshotUpdated);

    beginInsertRows(QModelIndex(), m_applications.size(), m_applications.size());
    m_applications.append(application);
    endInsertRows();
    Q_EMIT countChanged();
    Q_EMIT applicationAdded(application->appId());
}

void ApplicationManager::remove(Application *application)
{
    DASSERT(application != nullptr);
    DLOG("ApplicationManager::remove (this=%p, application='%s')", this, qPrintable(application->name()));

    if (application == m_sideStageApplication)
        m_sideStageApplication = nullptr;
    if (application == m_mainStageApplication)
        m_mainStageApplication = nullptr;

    int i = m_applications.indexOf(application);
    if (i != -1) {
        beginRemoveRows(QModelIndex(), i, i);
        m_applications.removeAt(i);
        endRemoveRows();
        Q_EMIT applicationRemoved(application->appId());
        Q_EMIT countChanged();
    }
}

void ApplicationManager::move(int from, int to) {
    DLOG("ApplicationManager::move (this=%p, from=%d, to=%d)", this, from, to);
    if (from == to) return;

    if (from >= 0 && from < m_applications.size() && to >= 0 && to < m_applications.size()) {
        QModelIndex parent;
        /* When moving an item down, the destination index needs to be incremented
           by one, as explained in the documentation:
           http://qt-project.org/doc/qt-5.0/qtcore/qabstractitemmodel.html#beginMoveRows */
        beginMoveRows(parent, from, from, parent, to + (to > from ? 1 : 0));
        m_applications.move(from, to);
        endMoveRows();
    }
}

QModelIndex ApplicationManager::findIndex(Application* application)
{
    for (int i = 0; i < m_applications.size(); ++i) {
        if (m_applications.at(i) == application) {
            return index(i);
        }
    }

    return QModelIndex();
}

bool ApplicationManager::checkFocusOnRemovedApplication(Application * application)
{
    if (application == m_focusedApplication) {
        // TODO(greyback) What to do?? Focus next app, or unfocus everything??
        m_focusedApplication = nullptr;
        Q_EMIT focusedApplicationIdChanged();
        return true;
    }
    return false;
}

} // namespace unitymir
