/***********************************************************************
* File name:  util.c                                                   *
* Written by: Bruce Kall                                               *
* Created: 2/2000 (portions extracted from tpconfig.c)                 *
* Language: C                                                          *
* Version: 1.0                                                         *
* Purpose:  Utility functions for manipulating touchpads.              *
*                                                                      *
***********************************************************************/

/* Copyright (C) 1997  C. Scott Ananian <cananian@alumni.princeton.edu>
 * Copyright (c) 1998-2001 Bruce Kalk <kall@compass.com>
 *
 * Currently maintained by: Bruce Kall, <kall@compass.com>
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   $Id: utils.c,v 1.9 2002/02/22 20:35:42 bruce Exp $
 */
 

/*$Log: utils.c,v $
 *Revision 1.9  2002/02/22 20:35:42  bruce
 *Changed version number in message
 *
 *Revision 1.8  2001/04/23 20:37:25  bruce
 *Changed version string.
 *
 *Revision 1.7  2000/11/15 15:41:34  bruce
 *Added to time out message about kernel patch.
 *
 *Revision 1.6  2000/11/15 15:29:53  bruce
 *Changed last modified message
 *
 *Revision 1.5  2000/10/31 18:57:29  cph
 *Change timeout logic to use gettimeofday rather than deprecated ftime.
 *Simplify debugging messages to make high-volume dump easier to scan.
 *
 *Revision 1.4  2000/10/31 18:04:24  cph
 *Simplify byte-level I/O.  Fix alignment bug in copyright output.
 *
 *Revision 1.3  2000/10/31 15:20:04  cph
 *Add convenient byte-level I/O.
 *
 *Revision 1.2  2000/10/23 16:27:53  bruce
 *Removed extraneuos code.
 *
 *Revision 1.1  2000/09/28 13:44:18  bruce
 *Initial revision
 **/

static char rcsid[]="$Header: /home/bruce/linux-stuff/tpconfig/tpconfig-3.1.3/RCS/utils.c,v 1.9 2002/02/22 20:35:42 bruce Exp $";

#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <assert.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/timeb.h>
#include <getopt.h>
#include <termios.h>
#include "tpconfig.h"

#define TRUE 1
#define FALSE 0
#define TIMEOUT 3.0 /* wait for up to 3 seconds for a reply*/

/* General Rourines for PS2 port */
int ps2_write(int fd,unsigned char *buffer,int num_bytes)
{ /* ps2_write */
int bytes_written;
struct timeval start;
struct timeval now;
int at;
int left;
int i;
 
if (DEBUG_LEVEL == DEBUG_HIGH)
  printf ("PS2_write: %d bytes\n", num_bytes);
/* Use the temporary buffer to store the data as it arrives */
gettimeofday(&start, 0);
at = 0;
left = num_bytes;
if (DEBUG_LEVEL == DEBUG_HIGH)
  {
    printf ("PS2_write:");
    for (i = 0; i < num_bytes; i++)
      printf(" %#2x", *(buffer + i));
    printf("\n");
  }

do
  {
  bytes_written = write(fd,(buffer + at),left);
  if(bytes_written > 0)
    {
    left = left - bytes_written;
    at = at + bytes_written;
    }
  gettimeofday(&now, 0);
  if ((((((double) now.tv_sec) * 1000000.0 + ((double) now.tv_usec))
	- (((double) start.tv_sec) * 1000000.0 + ((double) start.tv_usec)))
       / 1000000.0)
      > TIMEOUT)
    {
      printf ("\nTimed out waiting to write to device.\n");
      exit (-1);
    }
  }
while(left > 0);

if (DEBUG_LEVEL == DEBUG_HIGH)
  printf ("PS2_write: done\n");
 
return(at);
} /* ps2_write */
 
int ps2_read(int fd,unsigned char *buffer,int num_bytes)
{ /* ps2_read */
int bytes_read;
struct timeval start;
struct timeval now;
int at;
int left;
int i;
 
if (DEBUG_LEVEL == DEBUG_HIGH)
  printf ("PS2_read: %d bytes\n", num_bytes);
/* Use the temporary buffer to store the data as it arrives */
gettimeofday(&start, 0);
at = 0;
left = num_bytes;
do
  {
  bytes_read = read(fd,(buffer + at),left);
  if(bytes_read > 0)
    {
    left = left - bytes_read;
    at = at + bytes_read;
    }
  gettimeofday(&now, 0);
  if ((((((double) now.tv_sec) * 1000000.0 + ((double) now.tv_usec))
	- (((double) start.tv_sec) * 1000000.0 + ((double) start.tv_usec)))
       / 1000000.0)
      > TIMEOUT)
    {
      printf("\nTimed out waiting to read from device (is pc_keyb.c kernel patch installed?)\n");
      exit(-1);
    }
  }
while(left > 0);
if (DEBUG_LEVEL == DEBUG_HIGH)
  {
    printf("PS2_read:");
    for (i = 0; i < at; i++)
      printf (" %#2x", *(buffer + i));
    printf ("\n");
    printf ("PS2_read: read %d\n", at);
  }
return(at);
} /* ps2_read */
 


void version_info(void)
{ /* version_info */
printf( "\nSynaptics Touchpad and ALPS GlidePad/Glidepoint configuration tool\n");
printf("                   version: " VERSION "\n\n");
} /* version_info */


void copyright(void) 
{ /* copyright */
  printf("\n========================================================================\n");
  printf("=                                                                      =\n");
  printf("=                tpconfig   version: %-9s                         =\n", VERSION);
  printf("=                                                                      =\n");
  printf("= Synaptics Touchpad and ALPS GlidePad/Stickpointer configuration tool =\n");
  printf("=                                                                      =\n");
  printf("= Copyright (C) 1997 C. Scott Ananian<cananian@alumni.princeton.edu>   =\n");
  printf("= Copyright (C) 1998-2001 Bruce Kall <kall@compass.com>                =\n");
  printf("= Last Modified (Version 3.1.3) by Bruce Kall, 2/22/2002              =\n");
  printf("=                                                                      =\n");
  printf("= tpconfig comes with ABSOLUTELY NO WARRANTY.  This is free software,  =\n");
  printf("= and you are welcome to redistribute it under the terms of the GPL.   =\n");
  printf("=                                                                      =\n");
  printf("========================================================================\n\n");
} /* copyright */

/* write a byte to the ps/2 port, handling ACK */
void
putbyte (int fd, byte b) 
{
  if (DEBUG_LEVEL > DEBUG_LOW)
    printf ("putbyte: write %#02x\n", b);
  ps2_write (fd, &b, 1);
  getbyte_expected (fd, AUX_ACK, "putbyte");
}

/* read a byte from the ps/2 port */
byte
getbyte (int fd)
{
  byte b;
  ps2_read (fd, &b, 1);
  if (DEBUG_LEVEL > DEBUG_LOW)
    printf ("ps2_read_byte: read %#02x\n", b);
  return b;
}

void
getbyte_expected (int fd, byte expected, const char * what)
{
  byte b = getbyte (fd);
  if (DEBUG_LEVEL && b != expected)
    printf ("Read %#02x, expected %#02x in %s.\n", b, expected, what);
  assert (b == expected);
}
