/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev$
 * $Date$
 */
#include "Window.hpp"

#include "Help.hpp"
#include "OpenRecentMenu.hpp"
#include "WindowMenu.hpp"
#include "../reg/Components.hpp"
#include "../../gnome/gnome.hpp"
#include "../meta.hpp"
#include "../image/ImageManipulation.hpp"

#include "wx/accel.h"
#include "wx/config.h"
#include "wx/button.h"
#include "wx/sizer.h"
#include "wx/textdlg.h"
#include "wx/filedlg.h"
#include "wx/msgdlg.h"
#include "wx/dcmemory.h"
#include "wx/cshelp.h"
#include "wx/filename.h"
#include "wx/stdpaths.h"

#define INDII_STANDALONE_FLAGS wxCAPTION|wxMINIMIZE_BOX|wxMAXIMIZE_BOX|wxCLOSE_BOX|wxRESIZE_BORDER
#define INDII_EMBED_FLAGS wxCAPTION|/*wxSTAY_ON_TOP|*/wxRESIZE_BORDER

using namespace indii;

Window::Window(wxWindow* parent, const Mode mode, WindowObserver* obs) :
    wxFrame(parent, wxID_ANY, _(PACKAGE_NAME), wxDefaultPosition,
    wxDefaultSize, (mode == STANDALONE) ? INDII_STANDALONE_FLAGS :
    INDII_EMBED_FLAGS), obs(obs), exiting(false), mode(mode), manager(NULL),
    main(NULL), notebook(NULL), doc(NULL), infoBar(NULL) {
  main = new wxPanel(this, wxID_ANY, wxDefaultPosition, wxDefaultSize, wxTAB_TRAVERSAL|wxBORDER_NONE);
  infoBar = new InfoBar(this);
  dlgAbout = new About(this, mode == STANDALONE ? Components::APP : Components::PLUGIN);
  if (mode == STANDALONE) {
    notebook = new wxAuiNotebook(main, wxID_ANY, wxDefaultPosition,
        wxDefaultSize, wxAUI_NB_DEFAULT_STYLE|wxBORDER_NONE);
  } else {
    SetReturnCode(wxID_CANCEL);
  }
}

Window::~Window() {
  //
}

void Window::openAndAdd(const wxString& file) {
  wxFileName filename(file);
  Document* doc = open(file);
  notebook->AddPage(doc, filename.GetFullName(), true);
  doc->getPreviewPane()->getPreviewImage()->zoomFit();
  OpenRecentMenu::AddFile(file);
  enable();
}

int Window::GetReturnCode() {
  return returnCode;
}

void Window::SetReturnCode(const int returnCode) {
  this->returnCode = returnCode;
}

void Window::OnSystemClose(wxCloseEvent& evt) {
  int ret, unsaved = 0;
  bool vetoed = false;
  size_t i;
  Document* doc;
  
  if (mode == STANDALONE) {
    /* count number of unsaved images */
    for (i = 0; i < notebook->GetPageCount(); ++i) {
      doc = dynamic_cast<Document*>(notebook->GetPage(i));
      if (!doc->saved()) {
        ++unsaved;
      }
    }

    /* prompt about unsaved changes as appropriate */
    if (unsaved > 1) {
      #ifdef __WXOSX__
      ret = wxMessageBox(_("If you don't review your images, all your "
          "changes will be lost."), wxString(_("You have ")) << unsaved <<
          wxString(_(" images with unsaved changes. Do you want to review "
          "these changes before quitting?")),
          wxYES_NO|wxCANCEL|wxICON_QUESTION, this);
      #else
      ret = wxMessageBox(wxString(_("You have ")) << unsaved <<
          _(" images with unsaved changes. Do you want to review "
          "these changes before quitting?\n\nIf you don't review your images, "
          "all your changes will be lost."), _("Save changes?"),
          wxYES_NO|wxCANCEL|wxICON_QUESTION, this);
      #endif
    } else {
      ret = wxYES;
    }

    if (ret == wxYES) {
      /* check for save */
      for (i = 0; i < notebook->GetPageCount() && !vetoed; ++i) {
        doc = dynamic_cast<Document*>(notebook->GetPage(i));
        if (!doc->saved() && evt.CanVeto()) {
          notebook->SetSelection(i);
          ret = wxMessageBox(_("Save changes to image before closing?"),
              _("Save changes?"), wxYES_NO|wxCANCEL|wxICON_QUESTION, this);
          if (ret == wxYES) {

            if (!doc->hasPath()) {
              doc->saveAs();
            } else {
              doc->save();
            }
            if (!doc->saved()) {
              evt.Veto();
              vetoed = true;
            }
          } else if (ret == wxCANCEL) {
            evt.Veto();
            vetoed = true;
          }
        }
      }
    } else if (ret == wxCANCEL) {
      evt.Veto();
      vetoed = true;
    }
  }

  if (!vetoed) {
    if (obs != NULL) {
      obs->notifyClose(this, exiting);
    }
    
    /* save AUI perspective */
    wxConfig config(_(PACKAGE_TARNAME), _("indii.org"));
    wxString persp = manager->SavePerspective();
    config.Write(_("Perspective_v2"), persp);
    config.Write(_("x"), GetPosition().x);
    config.Write(_("y"), GetPosition().y);
    config.Write(_("width"), GetSize().GetWidth());
    config.Write(_("height"), GetSize().GetHeight());
    manager->UnInit();
    
    Destroy();
  } else {
    exiting = false;
  }
}

void Window::OnAllowNotebookDnD(wxAuiNotebookEvent& evt) {
  evt.Allow();
}

void Window::OnMenuOpen(wxMenuEvent& evt) {
  if (evt.GetMenu() == mnFile) {
    mnRecent->OnShow(evt);
  } else if (evt.GetMenu() == mnView) {
    if (mode == STANDALONE && notebook->GetPageCount() > 0) {
      #ifdef URBANLIGHTSCAPE
      Document* doc;
      doc = dynamic_cast<Document*>(notebook->GetPage(notebook->GetSelection()));
      bool overlay = doc->getPreviewPane()->getPreviewImage()->getOverlay();
      mnView->Check(ID_BUT_OVERLAY, overlay);
      #endif
    }
  } else if (evt.GetMenu() == mnWindow) {
    mnWindow->OnShow(evt);
  } else if (evt.GetMenu() == mnPopupWindow) {
    mnPopupWindow->OnShow(evt);
  }
}

void Window::OnButtonOpen(wxCommandEvent& evt) {
  wxString openPath;
  {
    wxConfig config(_(PACKAGE_TARNAME), _("indii.org"));
    if (!config.Read(_("OpenPath"), &openPath)) {
      openPath = wxStandardPaths::Get().GetDocumentsDir();
    }
  }

  wxString path = wxFileSelector(wxFileSelectorPromptStr, openPath, wxEmptyString,
      wxEmptyString, wildcards(), wxFD_OPEN|wxFD_CHANGE_DIR|wxFD_FILE_MUST_EXIST|
      wxFD_PREVIEW, this);
  if (!path.empty()) {
    if (wxImage::CanRead(path)) {
      wxFileName filename(path);
      if (filename.IsOk() && filename.FileExists()) {
        openAndAdd(path);
        openPath = filename.GetPath();
        {
          wxConfig config(_(PACKAGE_TARNAME), _("indii.org"));
          config.Write(_("OpenPath"), openPath);
        }
      } else {
        wxMessageBox(_("The selected file does not exist. Please try another."),
            _("File does not exist"), wxOK|wxICON_ERROR, this);
      }
    } else {
      wxMessageBox(_("The selected file is of an unsupported type. Please try another."),
          _("Unsupported type"), wxOK|wxICON_ERROR, NULL);
    }
  }
  enable();
}

void Window::OnOpenRecent(wxCommandEvent& evt) {
  int id = evt.GetId();
  if (id >= 300 && id < 310) {
    wxString path(OpenRecentMenu::GetFile(id));
    wxLogNull logNo;
    if (wxImage::CanRead(path)) {
      wxFileName filename(path);
      if (!path.empty() && filename.IsOk() && filename.FileExists()) {
        openAndAdd(path);
      } else {
        wxMessageBox(_("The selected file does not exist. Please try another."),
            _("File does not exist"), wxOK|wxICON_ERROR, this);
        OpenRecentMenu::RemoveFile(path);
      }
    } else {
      wxMessageBox(_("The selected file is of an unsupported type. Please try another."),
          _("Unsupported type"), wxOK|wxICON_ERROR, NULL);
      OpenRecentMenu::RemoveFile(path);
    }
  }
  enable();
}

void Window::OnClearRecent(wxCommandEvent& evt) {
  mnRecent->OnClearRecent(evt);
}

void Window::OnButtonClose(wxCommandEvent& evt) {
  int ret;
  bool vetoed = false;
  Document* doc = dynamic_cast<Document*>(
      notebook->GetPage(notebook->GetSelection()));
  if (!doc->saved()) {
    ret = wxMessageBox(_("Save changes to image before closing?"),
        _("Save changes?"), wxYES_NO|wxCANCEL|wxICON_QUESTION, this);
    if (ret == wxYES) {
      if (!doc->hasPath()) {
        doc->saveAs();
      } else {
        doc->save();
      }
      if (!doc->saved()) {
        vetoed = true;
      }
    } else if (ret == wxCANCEL) {
      vetoed = true;
    }
  }
  if (!vetoed) {
    notebook->DeletePage(notebook->GetSelection());
    enable();
  }
}

void Window::OnButtonSave(wxCommandEvent& evt) {
  Document* doc = dynamic_cast<Document*>(
      notebook->GetPage(notebook->GetSelection()));
  if (!doc->hasPath()) {
    doc->saveAs();
    if (doc->saved()) {
      wxFileName filename(doc->getPath());
      notebook->SetPageText(notebook->GetSelection(), filename.GetFullName());
    }
  } else {
    doc->save();
  }
}

void Window::OnButtonSaveAs(wxCommandEvent& evt) {
  Document* doc = dynamic_cast<Document*>(
      notebook->GetPage(notebook->GetSelection()));
  doc->saveAs();
  if (doc->saved()) {
    wxFileName filename(doc->getPath());
    notebook->SetPageText(notebook->GetSelection(), filename.GetFullName());
  }
}

void Window::OnButtonExit(wxCommandEvent& evt) {
  exiting = true;
  Close();
}

void Window::OnButtonOk(wxCommandEvent& evt) {
  SetReturnCode(wxID_OK);
  Close();
}

void Window::OnButtonCancel(wxCommandEvent& evt) {
  SetReturnCode(wxID_CANCEL);
  Close();
}

void Window::OnButtonAbout(wxCommandEvent& evt) {
  dlgAbout->ShowModal();
}

void Window::OnButtonDefault(wxCommandEvent& evt) {
  Document* doc;
  if (mode == STANDALONE) {
    doc = dynamic_cast<Document*>(notebook->GetPage(notebook->GetSelection()));
  } else {
    doc = this->doc;
  }
  doc->getModel()->setDefaults();
}

void Window::OnButZoomOut(wxCommandEvent& evt) {
  Document* doc;
  if (mode == STANDALONE) {
    doc = dynamic_cast<Document*>(notebook->GetPage(notebook->GetSelection()));
  } else {
    doc = this->doc;
  }
  doc->getPreviewPane()->getPreviewImage()->zoomOut();
}

void Window::OnButZoomIn(wxCommandEvent& evt) {
  Document* doc;
  if (mode == STANDALONE) {
    doc = dynamic_cast<Document*>(notebook->GetPage(notebook->GetSelection()));
  } else {
    doc = this->doc;
  }
  doc->getPreviewPane()->getPreviewImage()->zoomIn();
}

void Window::OnButZoomNormal(wxCommandEvent& evt) {
  Document* doc;
  if (mode == STANDALONE) {
    doc = dynamic_cast<Document*>(notebook->GetPage(notebook->GetSelection()));
  } else {
    doc = this->doc;
  }
  doc->getPreviewPane()->getPreviewImage()->zoomNormal();
}

void Window::OnButZoomFit(wxCommandEvent& evt) {
  Document* doc;
  if (mode == STANDALONE) {
    doc = dynamic_cast<Document*>(notebook->GetPage(notebook->GetSelection()));
  } else {
    doc = this->doc;
  }
  doc->getPreviewPane()->getPreviewImage()->zoomFit();
}

void Window::OnButtonWindow(wxCommandEvent& evt) {
  mnPopupWindow->update();
  this->PopupMenu(mnPopupWindow);
}

void Window::OnButtonOverlay(wxCommandEvent& evt) {
  Document* doc;
  if (mode == STANDALONE) {
    doc = dynamic_cast<Document*>(notebook->GetPage(notebook->GetSelection()));
  } else {
    doc = this->doc;
  }
  PreviewImage* previewImage = doc->getPreviewPane()->getPreviewImage();
  bool overlay = !previewImage->getOverlay();
  previewImage->showOverlay(overlay);
  previewImage->refresh();
  this->GetToolBar()->ToggleTool(ID_BUT_OVERLAY, overlay);
}

void Window::OnButtonHelp(wxCommandEvent& evt) {
  Help::show();
}

void Window::OnPageClose(wxAuiNotebookEvent& evt) {
  int ret;
  bool vetoed = false;
  Document* doc = dynamic_cast<Document*>(
      notebook->GetPage(notebook->GetSelection()));
  if (!doc->saved()) {
    ret = wxMessageBox(_("Save changes to image before closing?"),
        _("Save changes?"), wxYES_NO|wxCANCEL|wxICON_QUESTION, this);
    if (ret == wxYES) {
      if (!doc->hasPath()) {
        doc->saveAs();
      } else {
        doc->save();
      }
      if (!doc->saved()) {
        evt.Veto();
        vetoed = true;
      }
    } else if (ret == wxCANCEL) {
      evt.Veto();
      vetoed = true;
    }
  }
  if (!vetoed) {
    enable(notebook->GetPageCount() > 1); // page not yet removed, but will be
  }
}

void Window::OnPageChanged(wxAuiNotebookEvent& evt) {
  Document* doc;
  doc = dynamic_cast<Document*>(notebook->GetPage(notebook->GetSelection()));
  setModel(doc->getModel());
  PreviewImage* previewImage = doc->getPreviewPane()->getPreviewImage();
  #ifdef URBANLIGHTSCAPE
  previewImage->showOverlay(this->GetToolBar()->GetToolState(ID_BUT_OVERLAY));
  #endif
  previewImage->clear();
  previewImage->refresh();
}

void Window::init(Document* doc) {
  if (mode == STANDALONE) {
    /* menu bar */
    wxMenuBar* mbMenu = new wxMenuBar();

    mnFile = new wxMenu();
    mnRecent = new OpenRecentMenu();
    #ifdef __WXOSX__
    /* cloverleaf seems missing in default shortcuts, so explicitly encode */
    mnFile->Append(wxID_OPEN, _("&Open...\tCtrl-O"));
    #ifndef MAC_APP_STORE
    mnFile->AppendSubMenu(mnRecent, _("Open Recent"));
    #endif
    mnFile->AppendSeparator();
    mnFile->Append(wxID_CLOSE, _("&Close\tCtrl-W"));
    mnFile->Append(wxID_SAVE, _("&Save\tCtrl-S"));
    mnFile->Append(wxID_SAVEAS, _("Save &As...\tShift-Ctrl-S"));
    mnFile->Append(wxID_EXIT); // moved to program menu
    #else
    mnFile->Append(wxID_OPEN);
    mnFile->AppendSubMenu(mnRecent, _("Open &Recent"));
    mnFile->AppendSeparator();
    mnFile->Append(wxID_CLOSE);
    mnFile->Append(wxID_SAVE);
    mnFile->Append(wxID_SAVEAS);
    mnFile->AppendSeparator();
    mnFile->Append(wxID_EXIT);
    #endif
    mbMenu->Append(mnFile, _("&File"));

    mnEdit = new wxMenu();
    mnEdit->Append(ID_BUT_DEFAULT, _("&Restore Defaults"));

    mnView = new wxMenu();
    #ifdef __WXOSX__
    mnView->Append(wxID_ZOOM_IN, _("Zoom &In\tCtrl-+"));
    mnView->Append(wxID_ZOOM_OUT, _("Zoom &Out\tCtrl--"));
    mnView->Append(wxID_ZOOM_100, _("&Normal"));
    mnView->Append(wxID_ZOOM_FIT, _("&Fit"));
    #else
    mnView->Append(wxID_ZOOM_IN);
    mnView->Append(wxID_ZOOM_OUT);
    mnView->Append(wxID_ZOOM_100);
    mnView->Append(wxID_ZOOM_FIT);
    #endif
    #ifdef URBANLIGHTSCAPE
    mnView->AppendSeparator();
    mnView->AppendCheckItem(ID_BUT_OVERLAY, _("Over&lay\tCtrl-L"));
    #endif

    mnWindow = new WindowMenu(this);

    mnHelp = new wxMenu();
    #ifdef __WXOSX__
    mnHelp->Append(wxID_HELP, _T("&Help\tCtrl-?"));
    #else
    mnHelp->Append(wxID_HELP);
    #endif
    mnHelp->Append(wxID_ABOUT);

    mbMenu->Append(mnEdit, _("&Edit"));
    mbMenu->Append(mnView, _("&View"));
    mbMenu->Append(mnWindow, _("&Window"));
    mbMenu->Append(mnHelp, _("&Help"));

    this->SetMenuBar(mbMenu);
  }
  mnPopupWindow = new WindowMenu(this);

  /* toolbar */
  gnome_scale_images(this->GetContentScaleFactor());
  wxToolBar* tbTop = this->CreateToolBar(wxNO_BORDER|wxTB_FLAT|wxTB_HORIZONTAL);
  tbTop->SetToolBitmapSize(wxSize(32,32));
  if (mode == STANDALONE) {
    tbTop->AddTool(wxID_OPEN, _("Open"), *_img_gnome_document_open, _("Open"));
    tbTop->AddTool(wxID_SAVE, _("Save"), *_img_gnome_document_save, _("Save"));
    tbTop->AddTool(wxID_SAVEAS, _("Save As"), *_img_gnome_document_save_as, _("Save As"));
    tbTop->AddSeparator();
  }
  tbTop->AddTool(ID_BUT_DEFAULT, _("Restore Defaults"), *_img_gnome_system_run, _("Restore Defaults"));
  tbTop->AddSeparator();
  tbTop->AddTool(wxID_ZOOM_IN, _("Zoom In"), *_img_gnome_zoom_in, _("Zoom In"));
  tbTop->AddTool(wxID_ZOOM_OUT, _("Zoom Out"), *_img_gnome_zoom_out, _("Zoom Out"));
  tbTop->AddTool(wxID_ZOOM_100, _("Actual Size"), *_img_gnome_zoom_original, _("Actual Size"));
  tbTop->AddTool(wxID_ZOOM_FIT, _("Zoom to Fit"), *_img_gnome_zoom_best_fit, _("Zoom to Fit"));
  tbTop->AddSeparator();
  #ifdef URBANLIGHTSCAPE
  tbTop->AddCheckTool(ID_BUT_OVERLAY, _("Toggle Overlay"), *_img_gnome_window_new_32, *_img_gnome_window_new, _("Toggle Overlay"));
  // ^ first image is *_32 as otherwise appears shrunk in one quarter on Mac OS X
  tbTop->ToggleTool(ID_BUT_OVERLAY, true);
  #endif
  if (mode == EMBED) {
    tbTop->AddTool(ID_BUT_WINDOW, _("Windows"), *_img_gnome_preferences_system_windows, _("Windows"));
  }

  #ifdef __WXOSX__
  /* help system causing crash under Photoshop on Cocoa, to do with menu perhaps? */
  //if (mode != EMBED) {
    tbTop->AddTool(wxID_HELP, _("Help"), *_img_gnome_help_browser, _("Help"));
  //}
  #else
  tbTop->AddTool(wxID_HELP, _("Help"), *_img_gnome_help_browser, _("Help"));
  #endif
  tbTop->AddTool(wxID_ABOUT, _("About"), *_img_gnome_dialog_information, _("About"));
  #ifdef __WXOSX__
  //if (mode != EMBED) {
    #ifndef WX_28
    tbTop->AddStretchableSpace();
    tbTop->AddControl(new wxContextHelpButton(tbTop, wxID_CONTEXT_HELP, wxDefaultPosition,
        wxDefaultSize, wxBU_AUTODRAW|wxBORDER_NONE), _T("Context help"));
    #else
    tbTop->AddSeparator();
    tbTop->AddControl(new wxContextHelpButton(tbTop));
    #endif
  //}
  #else
  #ifndef WX_28
  tbTop->AddStretchableSpace();
  tbTop->AddControl(new wxContextHelpButton(tbTop, wxID_CONTEXT_HELP, wxDefaultPosition,
      wxDefaultSize, wxBU_AUTODRAW|wxBORDER_NONE), _T("Context help"));
  #else
  tbTop->AddSeparator();
  tbTop->AddControl(new wxContextHelpButton(tbTop));
  #endif
  #endif
  tbTop->Realize();

  /* info bar */
  wxBoxSizer* sizer = new wxBoxSizer(wxVERTICAL);
  sizer->Add(infoBar, 0, wxEXPAND);
  sizer->Add(main, 1, wxEXPAND);

  /* AUI manager */
  manager = new wxAuiManager(main);

  int x, y, width, height;

  /* notebook */
  wxAuiPaneInfo paneInfo;
  paneInfo.DefaultPane();
  paneInfo.CaptionVisible(false);
  paneInfo.Centre();
  paneInfo.PaneBorder(false);

  if (mode == STANDALONE) {
    manager->AddPane(notebook, paneInfo);
    // doc added later, after other panes...
  } else {
    assert (doc != NULL);
    manager->AddPane(doc, paneInfo);
  }

  /* panes */
  initPanes();

  /* add first document to notebook, if given */
  if (mode == STANDALONE && doc != NULL) {
    notebook->AddPage(doc, _T("Unsaved"));
  }

  /* load AUI perspective */
  wxConfig config(_(PACKAGE_TARNAME), _("indii.org"));
  wxString persp;
  if (config.Read(_("Perspective_v2"), &persp)) {
    manager->LoadPerspective(persp);
  }

  /* main view seems to disappear occasionally (observed on both wxGTK and
   * wxOSX), so make sure it's shown */
  if (mode == STANDALONE) {
    manager->GetPane(notebook).Show();
  } else {
    manager->GetPane(doc).Show();
  }

  /* dialog buttons -- loaded after restoring perspective to ensure they're
   * always placed appropriately */
  if (mode == EMBED) {
    wxPanel* buttonPanel = new wxPanel(main);
    wxStdDialogButtonSizer* buttonSizer = new wxStdDialogButtonSizer();

    wxButton* butOk = new wxButton(buttonPanel, wxID_OK);
    buttonSizer->AddButton(butOk);
    wxButton* butCancel = new wxButton(buttonPanel, wxID_CANCEL);
    buttonSizer->AddButton(butCancel);
    #ifndef __WXOSX__
    wxButton* butHelp = new wxButton(buttonPanel, wxID_HELP);
    buttonSizer->AddButton(butHelp);
    #endif
    buttonSizer->Realize();

    buttonPanel->SetSizerAndFit(buttonSizer);
    width = buttonPanel->GetMinSize().GetWidth();
    height = buttonPanel->GetMinSize().GetHeight();

    wxAuiPaneInfo paneInfo;
    paneInfo.DefaultPane();
    #ifndef WX_28
    paneInfo.Name(_("Buttons"));
    #else
    paneInfo.Name(wxString("Buttons", wxConvUTF8));
    #endif
    paneInfo.MinSize(wxSize(width, height));
    paneInfo.BestSize(wxSize(width, height));
    paneInfo.CaptionVisible(false);
    paneInfo.Resizable(false);
    paneInfo.Bottom();
    paneInfo.PaneBorder(false);
    paneInfo.Position(0);
    paneInfo.Layer(10);

    manager->AddPane(buttonPanel, paneInfo);
  }

  SetSizer(sizer);
  SetReturnCode(wxID_CANCEL);

  long val;
  width = intround(0.8f*wxSystemSettings::GetMetric(wxSYS_SCREEN_X));
  height = intround(0.8f*wxSystemSettings::GetMetric(wxSYS_SCREEN_Y));
  if (config.Read(_("width"), &val)) {
    width = static_cast<int>(val);
  }
  if (config.Read(_("height"), &val)) {
    height = static_cast<int>(val);
  }
  SetSize(wxSize(width, height));
  CentreOnScreen();
  x = GetPosition().x;
  y = GetPosition().y;
  if (config.Read(_("x"), &val)) {
    x = static_cast<int>(val);
  }
  if (config.Read(_("y"), &val)) {
    y = static_cast<int>(val);
  }
  SetPosition(wxPoint(x, y));
  manager->Update();
  
  if (doc != NULL) {
    doc->getPreviewPane()->getPreviewImage()->zoomFit();
  }
}

void Window::initPanes() {
  //
}

void Window::enable() {
  enable((mode == EMBED && doc != NULL) ||
      (mode == STANDALONE && notebook->GetPageCount() > 0));
}

void Window::enable(const bool enable) {
  wxToolBar* tbTop = this->GetToolBar();
  if (mode == STANDALONE) {
    tbTop->EnableTool(wxID_SAVE, enable);
    tbTop->EnableTool(wxID_SAVEAS, enable);
  }
  tbTop->EnableTool(wxID_ZOOM_OUT, enable);
  tbTop->EnableTool(wxID_ZOOM_IN, enable);
  tbTop->EnableTool(wxID_ZOOM_100, enable);
  tbTop->EnableTool(wxID_ZOOM_FIT, enable);
  tbTop->EnableTool(ID_BUT_DEFAULT, enable);
  #ifdef URBANLIGHTSCAPE
  tbTop->EnableTool(ID_BUT_OVERLAY, enable);
  #endif
  
  if (mode == STANDALONE) {
    wxMenuBar* mbMenu = this->GetMenuBar();
    mbMenu->Enable(wxID_CLOSE, enable);
    mbMenu->Enable(wxID_SAVE, enable);
    mbMenu->Enable(wxID_SAVEAS, enable);
    mbMenu->Enable(wxID_ZOOM_OUT, enable);
    mbMenu->Enable(wxID_ZOOM_IN, enable);
    mbMenu->Enable(wxID_ZOOM_100, enable);
    mbMenu->Enable(wxID_ZOOM_FIT, enable);
    #ifdef URBANLIGHTSCAPE
    mbMenu->Enable(ID_BUT_OVERLAY, enable);
    #endif
    mbMenu->Enable(ID_BUT_DEFAULT, enable);

    if (doc != NULL) {
      doc->getPreviewPane()->getPreviewImage()->refresh();
    }
  }
}

void Window::addPane(const wxString& name) {
  if (mode == STANDALONE) {
    mnWindow->add(name);
  } else {
    mnPopupWindow->add(name);
  }
}

void Window::OnSelect(wxCommandEvent& evt) {
  if (mode == STANDALONE) {
    mnWindow->OnSelect(evt);
  } else {
    mnPopupWindow->OnSelect(evt);
  }
}

void Window::OnDefault(wxCommandEvent& evt) {
  layoutDefault();
}

BEGIN_EVENT_TABLE(Window, wxFrame)
//EVT_AUINOTEBOOK_ALLOW_DND(wxID_ANY, Window::OnAllowNotebookDnD)

#ifndef __WXOSX__
/* these events handled in OpenRecentMenu on Mac */
EVT_MENU(ID_CLEAR_RECENT, Window::OnClearRecent)

/* these events handled in WindowMenu on Mac */
EVT_MENU_OPEN(Window::OnMenuOpen)
EVT_MENU(ID_SELECT, Window::OnSelect)
EVT_MENU(ID_SELECT + 1u, Window::OnSelect)
EVT_MENU(ID_SELECT + 2u, Window::OnSelect)
EVT_MENU(ID_SELECT + 3u, Window::OnSelect)
EVT_MENU(ID_SELECT + 4u, Window::OnSelect)
EVT_MENU(ID_SELECT + 5u, Window::OnSelect)
EVT_MENU(ID_SELECT + 6u, Window::OnSelect)
EVT_MENU(ID_SELECT + 7u, Window::OnSelect)
EVT_MENU(ID_SELECT + 8u, Window::OnSelect)
EVT_MENU(ID_SELECT + 9u, Window::OnSelect)
EVT_MENU(ID_SELECT + 10u, Window::OnSelect)
EVT_MENU(ID_SELECT + 11u, Window::OnSelect)
EVT_MENU(ID_SELECT + 12u, Window::OnSelect)
EVT_MENU(ID_SELECT + 13u, Window::OnSelect)
EVT_MENU(ID_SELECT + 14u, Window::OnSelect)
EVT_MENU(ID_SELECT + 15u, Window::OnSelect)
EVT_MENU(ID_DEFAULTS, Window::OnDefault)
#endif

EVT_CLOSE(Window::OnSystemClose)

EVT_TOOL(wxID_OPEN, Window::OnButtonOpen)
EVT_TOOL(wxID_CLOSE, Window::OnButtonClose)
EVT_TOOL(wxID_SAVE, Window::OnButtonSave)
EVT_TOOL(wxID_SAVEAS, Window::OnButtonSaveAs)
EVT_TOOL(wxID_EXIT, Window::OnButtonExit)
EVT_TOOL(wxID_ZOOM_OUT, Window::OnButZoomOut)
EVT_TOOL(wxID_ZOOM_IN, Window::OnButZoomIn)
EVT_TOOL(wxID_ZOOM_100, Window::OnButZoomNormal)
EVT_TOOL(wxID_ZOOM_FIT, Window::OnButZoomFit)
EVT_TOOL(ID_BUT_DEFAULT, Window::OnButtonDefault)
EVT_TOOL(wxID_ABOUT, Window::OnButtonAbout)
EVT_TOOL(wxID_HELP, Window::OnButtonHelp)
EVT_TOOL(ID_BUT_OVERLAY, Window::OnButtonOverlay)
EVT_TOOL(ID_BUT_WINDOW, Window::OnButtonWindow)

EVT_MENU(wxID_OPEN, Window::OnButtonOpen)
EVT_MENU(wxID_CLOSE, Window::OnButtonClose)
EVT_MENU(wxID_SAVE, Window::OnButtonSave)
EVT_MENU(wxID_SAVEAS, Window::OnButtonSaveAs)
EVT_MENU(wxID_EXIT, Window::OnButtonExit)
EVT_MENU(wxID_ZOOM_OUT, Window::OnButZoomOut)
EVT_MENU(wxID_ZOOM_IN, Window::OnButZoomIn)
EVT_MENU(wxID_ZOOM_100, Window::OnButZoomNormal)
EVT_MENU(wxID_ZOOM_FIT, Window::OnButZoomFit)
EVT_MENU(ID_BUT_OVERLAY, Window::OnButtonOverlay)
EVT_MENU(ID_BUT_DEFAULT, Window::OnButtonDefault)
EVT_MENU(wxID_ABOUT, Window::OnButtonAbout)
EVT_MENU(wxID_HELP, Window::OnButtonHelp)

EVT_BUTTON(wxID_OK, Window::OnButtonOk)
EVT_BUTTON(wxID_CANCEL, Window::OnButtonCancel)
EVT_BUTTON(wxID_HELP, Window::OnButtonHelp)

EVT_MENU(300, Window::OnOpenRecent)
EVT_MENU(301, Window::OnOpenRecent)
EVT_MENU(302, Window::OnOpenRecent)
EVT_MENU(303, Window::OnOpenRecent)
EVT_MENU(304, Window::OnOpenRecent)
EVT_MENU(305, Window::OnOpenRecent)
EVT_MENU(306, Window::OnOpenRecent)
EVT_MENU(307, Window::OnOpenRecent)

EVT_AUINOTEBOOK_PAGE_CLOSE(wxID_ANY, Window::OnPageClose)
EVT_AUINOTEBOOK_PAGE_CHANGED(wxID_ANY, Window::OnPageChanged)

END_EVENT_TABLE()
