[comment {-*- tcl -*- doctools manpage}]
[manpage_begin rest n 1.0.1]
[moddesc   {A framework for RESTful web services}]
[titledesc {define REST web APIs and call them inline or asychronously}]
[require Tcl 8.5]
[require rest [opt 1.0.1]]
[description]
[para]

There are 2 types of usage this package supports: simple calls, and complete interfaces. In an interface you specify a set of rules and then the package builds commands which correspond to the REST methods. These commands can have many options such as input and output transformations and data type specific formatting. This results in a cleaner and simpler script. On the other hand, a simple call is easier and quicker to implement but less featureful. It takes the url and a few options on the command and returns the result directly. Any formatting or checking is up to rest of the script.

Simple usage
In simple usage you make calls using http method procedures and then check or process the returned data yourself
[list_begin definitions]
[call [cmd ::rest::simple] [arg url] [arg query] [opt config] [opt body]]
[call [cmd ::rest::get] [arg url] [arg query] [opt config] [opt body]]
[call [cmd ::rest::post] [arg url] [arg query] [opt config] [opt body]]
[call [cmd ::rest::head] [arg url] [arg query] [opt config] [opt body]]
[call [cmd ::rest::put] [arg url] [arg query] [opt config] [opt body]]
[call [cmd ::rest::delete] [arg url] [arg query] [opt config] [opt body]]
[list_end]
The above commands are all equivalent except for the http method used. If you use [cmd simple] then the method should be specified as an option in the [opt config] dict, otherwise it defaults to [const get]. If a body is needed then the config dict must be present, however it may be empty.

[example {
    set appid APPID
    set search tcl
    set res [rest::get http://boss.yahooapis.com/ysearch/web/v1/$search [list appid $appid]]
    set res [rest::format_json $res]
}]

[example {
    set res [rest::simple http://twitter.com/statuses/update.json \
        [list status $text] \
        {
          method post
          auth {basic user password}
          format json
        }
    ]
}]
the options supported in the config dict are as follows
headers
cookie
auth
format
method
content-type

Interface usage
An interface to a REST API consists of a series of definitions of REST calls contained in an array. The array name becomes a namespace containing the defined commands. Each array element defines the name of the call and takes the form of a dict, aka key/value pairs. These keys are the defined configuration options below.
After creating the definitions simply call rest::create_interface on the array to create the commands.

[example {
package require rest

set yweather(forecast) {
   url http://weather.yahooapis.com/forecastrss
   req_args { p: }
   opt_args { u: }
}

rest::create_interface yweather

puts [yweather::forecast -p 94089]
}]

::${name}::basic_auth [arg u] [arg p]

::${name}::set_static_args [opt args]]

[list_begin definitions]

[call [cmd ::rest::save] [arg name] [arg file]]
saves a copy of the dynamically created procs to a file for later loading

[call [cmd ::rest::describe] [arg name]]
print a description of defined api calls

[call [cmd ::rest::parameters] [arg url] [opt args]]
parse a url query string into a dict

[call [cmd ::rest::parse_opts] [arg static] [arg required] [arg optional] [arg string]]

[call [cmd ::rest::substitute] [arg string] [opt var]]
take a string and substitute real values for any option identifiers

[call [cmd ::rest::create_interface] [arg name]]
TOKENS
     the value is substituted into the url at call time. tokens in the form of %name:default_value% will be an optional argument with a default value.

url
    the target of the http request
description
    a string which describes the call. used only for [call describe]
body
    indicates if arguments are required for the request body or not. should
    be one of none, optional, required, argument or mime_multipart. default is optional.
    if [const argument] is used then the option is parsed as a list where the second value is the name
    of a option. the body will then be used as the value for that option.
    if the value is [const mime_multipart] then the body is required and interpreted as each argument
    representing one part of a mime multipart document. each argument should be a 2 item list with the first being
    a list of header keys and values, and the second being the mime part body.

[example {
set ygeo(parse) {
    url http://wherein.yahooapis.com/v1/document
    method post
    body { arg documentContent }
}
ygeo::parse "san jose ca"
# "san jose ca" will be interpreted as if it were specified as the -documentContent option
}]

[example {
set gdocs(upload) {
    url http://docs.google.com/feeds/default/private/full
    body mime_multipart
}
gdocs::upload [list {Content-Type application/atom+xml} $xml] [list {Content-Type image/jpeg} $filedata]
}]

method
    The HTTP method to call on the url. The default is GET.
copy
    this copies the definition of a previously defined call. after copying you can override selected options by defining them again.
unset
    removes the named option. useful when using copy of another definition.
headers
    the value must be another dict containing header fields and their values. The default is to not add any additional headers.
content-type
    Specifies the content type for the request data.
req_args
    a list of the required arguments. names ending in a colon will require a value.
opt_args
    arguments that may be present but are not required.
static_args
    arguments that are always the same. no sense in troubling the user with these. A leading - is allowed but not required to maintain consistancy with the command line.
auth
    should be one of basic or sign. if basic is used you can configure basic auth with the proc auth_basic which takes 2 arguments, the username and password.
    if sign is specified then the value must be a list with the second element being the name of a proc which will be called to perform the request signing.
[example {
set delicious(updated) {
    url https://api.del.icio.us/v1/posts/update
    auth basic
}

rest::create_interface flickr

flickr::basic_auth username password
}]

[example {
set flickr(auth.getToken) {
   url http://api.flickr.com/services/rest/
   req_args { api_key: secret: }
   auth { sign do_signature }
}

rest::create_interface flickr

proc ::flickr::do_signature {query} {
    # perform some operations on the query here
    return $query
}
}]
callback
    If this option is present then the method will be created as an async call. An async call will return immediately with the value of the http token. The event loop must be active to use this option. The value of this option is the name of a proc which is invoked when the HTTP call is complete. The proc receives three arguments, the name of the calling procedure, the status of the result (one of OK or ERROR), and the data associated with the result.
    the http request header is available via [call uplevel token token]
cookie
    a list of cookies to be passed in the http header. this is just a shortcut to the headers option
input_transform
    commands which take the variable $query and transform it in some manner before returning a new value. return value must be a dict which will be passed to http::formatQuery
    the request body is accessible via [call upvar body body]
format or result
    defines the format of the returned data. should be one of discard, raw, json, xml, or tdom. the default is auto which should auto detect between xml and json. rss is formated as
    a special case of xml.
pre_transform
    this value takes the form of a proc which should perform some action on $result and return a value. it is run on the result before the output (xml/json/etc) transformation is done.
    the http request header is available via [call uplevel token token]
result
    may have the value xml, json, tdom, raw, or auto. the default is auto and should auto detect json or xml results and transform them into a tcl list. this is here if you want to specify it explicitly.
post_transform
    this value takes the form of a proc which should perform some action on $result and return a value. it is run on the result after the output transformation but before returning the value to the calling procedure.
    the http request header is available via [call uplevel token token]
check_result
    this value should be a list of 2 expressions either of which may be empty. the first expression is checks the OK condition, it must return true when the result is satisfactory. the second expression is the error condition, it must return false unless there is an error.

[list_end]

[section INCLUDED]
functional but incomplete implementations are included for the following services:
flickr
twitter
yahoo boss
yahoo weather
google calendar
facebook
del.icio.us
read the file or source it and use [cmd describe] for more information. also see the
developers documentation on the respective sites.

[include ../common-text/tls-security-notes.inc]

[vset CATEGORY rest]
[include ../doctools2base/include/feedback.inc]
[manpage_end]
