// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


// mathops.h
//
// Two classes - Vector3 and SolidAngle - to make 3-d math operations easier

#ifndef _VECTOR3_H
#define _VECTOR3_H

#include "../../lib/compat.h"
#include "mathdefs.h"
#include <cmath>
#include <cstdlib>
#include <climits>

// Euler angles for the conversions between celestial and galactic
//  coordinates (presuming J2000 celestial coordinate system), from the
//  appendix of http://dx.doi.org/10.1086/424960 :
//
//  A = 3.36603341 rad (12h 51m 26.282s) =  RA of North Galactic Pole
//  D = 0.47347879 rad (27d 07' 42.01")  = Dec of North Galactic Pole
//  T = 0.57477183 rad (32.932 d)        = Galactic Longitude at the
//                                         first point where Celestial
//                                         and Galactic Equators cross
//                                         (\theta^{J2000.0} - 90 degrees)
//  Corresponds to RA = 18h 51m 26.282s, Dec = 0.
//
// Constants are given in sine and cosine form to decrease the number
//  of floating-point calculations.

const double COS_A = -0.97491873,
             SIN_A = -0.22256118,
             COS_D =  0.88998740,
             SIN_D =  0.45598512,
             COS_T =  0.83931637,
             SIN_T =  0.54364330;


class SolidAngle;  // forward declaration for below

// The Vector3 class defines the standard 3-D vector operations, plus
//  functions to convert into spherical coordinates and between
//  celestial and galactic systems.

class Vector3 { 
 private:
  double x, y, z; 
 public:
  Vector3() : x(0), y(0), z(0) { }
  Vector3(double X, double Y, double Z) : x(X), y(Y), z(Z) { }

  Vector3 &operator += (Vector3 v);
  Vector3 &operator -= (Vector3 v);
  Vector3 operator + (Vector3 v) const;
  Vector3 operator - (Vector3 v) const;
  Vector3 operator * (double c) const;
  friend Vector3 operator * (double c, Vector3 v);
  Vector3 operator / (double c) const;

  Vector3 operator - (void) const { return operator*(-1.0); }
  Vector3 operator + (void) const { return *this;           }
  
  bool operator == (Vector3 v) const
    { return x == v.x && y == v.y && z == v.z; }
  bool operator != (Vector3 v) const
    { return ! (*this == v); }

  double magnitude() const;
  Vector3 unitvector() const;
  double dot(Vector3 v) const;
  Vector3 cross(Vector3 v) const;
  SolidAngle toSpherical() const;
  Vector3 toGalactic() const;
  Vector3 toCelestial() const;

  double getX() const { return x; }
  double getY() const { return y; }
  double getZ() const { return z; }
};

// The SolidAngle class is used to hold latitude and longitude in a convenient
//  structure, and allow conversion to Cartesian coordinates and between
//  celestial and galactic systems.

class SolidAngle {
 private:
  double theta /* latitude */,   // both in radians, of course
         phi   /* longitude */;
 public:
  SolidAngle() : theta(0), phi(0) { }
  SolidAngle(double Phi, double Theta) : theta(Theta), phi(Phi)
  {
    while (phi >= 2 * M_PI) phi -= (2 * M_PI);
    while (phi < 0.0)       phi += (2 * M_PI);
    if (theta > M_PI_2) theta = M_PI_2;
    if (theta < -M_PI_2) theta = -M_PI_2;
  }

  bool operator == (SolidAngle a) const
    { return theta == a.theta && phi == a.phi; }
  bool operator != (SolidAngle a) const
    { return ! (*this == a); }

  Vector3 toCartesian(double radius) const;
  Vector3 toUnitV() const { return toCartesian(1.0); }
  SolidAngle toGalactic() const;
  SolidAngle toCelestial() const;

  double getTheta() const { return theta; }
  double getPhi()   const { return phi;   }
};


#endif // #define _VECTOR3_H
