#include "specclass.h"
#include "strings.h"
#include "mathdefs.h"
#include <map>

using std::string;

// These are only supposed to handle numbers in the range [0, 6].
static string int_to_roman(double MKtype)
{
  string result;
  switch (FLOOR(MKtype)) {
    case 0: result = "0";   case 1: result = "I";  case 2: result = "II";
    case 3: result = "III"; case 4: result = "IV"; case 5: result = "V";
    case 6: result = "VI";
    default: result = "";
  }
  return result;
}

static double roman_to_int(const string &roman_num)
{
  if (starstrings::compare_n(roman_num, "VI", 2))	return 6.0;
  else if (starstrings::compare_n(roman_num, "V", 1))	return 5.0;
  else if (starstrings::compare_n(roman_num, "III", 3))	return 3.0;
  else if (starstrings::compare_n(roman_num, "II", 2))	return 2.0;
  else if (starstrings::compare_n(roman_num, "IV", 2))	return 4.0;
  else if (starstrings::compare_n(roman_num, "Ia", 2))	return 0.5;
  else if (starstrings::compare_n(roman_num, "Ib", 2))	return 1.5;
  else if (starstrings::compare_n(roman_num, "I", 1))	return 1.0;
  else return -1;
}

int SpecClass::class_to_int(char major)
{
  static bool initialized = false;
  static std::map<char, int> table;
  major = starstrings::toupper(major);
  if (! initialized) {
    table['O'] = table['W'] = 0;
    table['B'] = 1;
    table['A'] = 2;
    table['F'] = 3;
    table['G'] = 4;
    table['K'] = table['R'] = 5;
    table['M'] = table['N'] = table['S'] = table['C'] = 6;
    initialized = true;
  }
  int temp = table[major];
  if ((! temp) && major != 'O' && major != 'W') return -1;
  else return temp;
}


// Minor and MK components of spectral type are not determined unless
// specifically requested by initialize() function, below.
SpecClass::SpecClass(const string &s)
: sSpecstring(s), sMajor(0), sMinor(ERROR_VAL), sMKtype(ERROR_VAL),
  sSpecial("")
{
  if (! starstrings::isempty(sSpecstring))
    sMajor = starstrings::toupper(s[s.find_first_not_of(WHITESPACE "ds")]);
}

SpecClass::SpecClass(const SpecClass &sc)
: sSpecstring(sc.sSpecstring), sMajor(sc.sMajor), sMinor(sc.sMinor),
  sMKtype(sc.sMKtype), sSpecial(sc.sSpecial)
{ }

SpecClass::SpecClass(char major, double minor, double MKtype, string special)
: sMajor(major), sMinor(minor), sMKtype(MKtype), sSpecial(special)
{
  sSpecstring = string() + major + starstrings::ftoa(minor, 2) + " "
    + int_to_roman(MKtype) + sSpecial;
}


SpecClass & SpecClass::operator = (const SpecClass &sc)
{
  if (this != &sc) {
    sSpecstring = sc.sSpecstring; sMajor = sc.sMajor;
    sMinor = sc.sMinor; sMKtype = sc.sMKtype; sSpecial = sc.sSpecial;
  }
  return *this;
}


color_t SpecClass::color() const
{
  switch (sMajor) {
    case 'W' : return WOLF_RAYET_COLOR;
    case 'D' : return D_COLOR;
    case '*' : return NON_STELLAR_COLOR;
    default : break;
  }
  int classno = class_to_int(sMajor);
  if (classno >= 0) return CLASS_COLORS[classno];
  else return DEFAULT_COLOR;
}

void SpecClass::initialize()
{
  size_t begin, end;
  starstrings::stripspace(sSpecstring);
  starstrings::utf8ize(sSpecstring);

  // calculate minor part of spectral type
  begin = sSpecstring.find_first_of(DIGITS);
  end = sSpecstring.find_first_not_of(DIGITS ".", begin);
  sMinor = (begin != end) ?
    starmath::atof(sSpecstring.substr(begin, end - begin)) : 10;
  if (end >= sSpecstring.size()) return;
 
  // calculate luminosity class
  begin = sSpecstring.find_first_of("IV", end);
  end = sSpecstring.find_first_not_of("IVab", begin);
  bool flagsplit = false;

  if (begin >= sSpecstring.size() || begin == end) return;
  sMKtype = roman_to_int(sSpecstring.substr(begin, end - begin));
  if (sSpecstring[end] == '/' || sSpecstring[end] == '-') {
    flagsplit = true;
    end++;
  }
  if (flagsplit && sMKtype >= 0) sMKtype += 0.5;

  if (end < sSpecstring.size()) {
    sSpecial = sSpecstring.substr(end);
    starstrings::stripspace(sSpecial);
  }
}


// Lookup tables of data for use by absmag, diameter, temperature functions.
// These tables contain linearly interpolated data derived from Appendix E of
// Carroll and Ostlie, _An Introduction to Modern Astrophysics_ (1996).
// Data are in the form { value(minor == 0), d(value)/d(minor) } for each
// spectral class, at MK types V, III, and Iab respectively.

// Effective temperatures, in Kelvins (column 2 of tables in Appendix E)
static const double temp_table[3][7][2] = {
  {
    { 59000, -3000 }, { 30000, -2048 }, { 9520, -232 }, { 7200, -117 },
    { 6030,  -78 },   { 5250,  -140 },  { 3850, -151 }
  },
  {
    { 56000, -2700 }, { 29000, -1890 }, { 10100, -295 }, { 7150, -130 },
    { 5850,  -110 },  { 4750,  -95 },   { 3800,  -93 }
  },
  {
    { 54600, -2860 }, { 26000, -1627 }, { 9730, -203 }, { 7700, -215 },
    { 5550,  -113 },  { 4420,  -77 },   { 3650, -175 }
  }
};

// Absolute visual magnitudes, M_v (column 8 of tables)
static const double mag_table[3][7][2] = {  
  {
    { -7.4, 0.34 }, { -4.0, 0.46 }, { 0.6, 0.21 }, { 2.7, 0.17 },
    { 4.4,  0.15 }, { 5.9,  0.29 }, { 8.8, 0.9 }
  },
  {
    { -7.5, 0.24 }, { -5.1, 0.51 }, { 0.0, 0.15 }, { 1.5, -0.05 },
    { 1.0, -0.03 }, { 0.7, -0.11 }, { -0.4, 0.03 }
  },
  {
    { -6.8, 0.02 }, { -6.4, 0.01 }, { -6.3, -0.03 }, { -6.6, 0.02 },
    { -6.4, 0.04 }, { -6.0, 0.04 }, { -5.6, 0.0 }
  }
};

// Bolometric correction (column 7 of tables), BC == M_bolometric - M_v
static const double bolcorr_table[3][7][2] = {
  {
    { -5.64,  0.248 }, { -3.16,  0.286 }, { -0.30, 0.021 }, { -0.09, -0.009 },
    { -0.18, -0.013 }, { -0.31, -0.107 }, { -1.38, -0.34 }
  },
  {
    { -5.22, 0.234 }, { -2.88,  0.246 }, { -0.42, 0.031 }, { -0.11, -0.009 },
    { -0.20, -0.03 }, { -0.50, -0.075 }, { -1.25, -0.247 }
  },
  {
    { -5.25,  0.276 }, { -2.49,  0.208 }, { -0.41, 0.040 }, { -0.01, -0.014 },
    { -0.15, -0.035 }, { -0.50, -0.079 }, { -1.29, -0.435 }
  }
};


// lookup(): This function looks up values for the current spectral class
// in the given table.  If MK type is between array elements, it interpolates.
// If no appropriate value is found, it returns a very large negative number
// to mark the error.

#define LOOKUP_RESULT(i) \
	(table[(5 - (i)) / 2][class_to_int(sMajor)][0] + \
	table[(5 - (i)) / 2][class_to_int(sMajor)][1] * sMinor)

double SpecClass::lookup(const double table[3][7][2]) const
{
  int major_num = class_to_int(sMajor);
  if (major_num < 0 || major_num > 6) return ERROR_VAL;

  if (fabs(sMKtype - 5.0) < 0.1)
    return LOOKUP_RESULT(5);
  else if (fabs(sMKtype - 3.0) < 0.1)
    return LOOKUP_RESULT(3);
  else if (fabs(sMKtype - 1.0) < 0.1)
    return LOOKUP_RESULT(1);

  // if MKtype isn't exactly I, III or V, interpolate linearly between them
  else if (sMKtype >= 0.0 && sMKtype < 3.0)
    return ((3 - sMKtype) * LOOKUP_RESULT(1) +
		(sMKtype - 1) * LOOKUP_RESULT(3)) / 2;
  else if (sMKtype <= 6.0 && sMKtype > 3.0)
    return ((5 - sMKtype) * LOOKUP_RESULT(3) +
		(sMKtype - 3) * LOOKUP_RESULT(5)) / 2;
  else
    return ERROR_VAL;
}


// absmag(): returns a guess for the absolute magnitude of a star with
// the current spectral class.  Returns a large negative number if an
// error occurs.
double SpecClass::absmag() const
{
  if (sMKtype < 0.0) {
    // in this case, no MK type is available; 
    //  try to save a few special cases from being thrown out
    if (sMajor == 'D') return +14;
    else return ERROR_VAL;
  }
  return lookup(mag_table);
}


// temperature(): returns the expected temperature in Kelvins
// of a star of this spectral class.
double SpecClass::temperature() const
{
  double temp = lookup(temp_table);
  return (temp <= 0.0) ? ERROR_VAL : temp;
}


// diameter(): return the expected diameter, in light-years, of a star of
// this spectral class with the given absolute magnitude.  We require the
// bolometric magnitude, M_bolo == BC + M_v, in order to get the correct
// diameter.
double SpecClass::diameter(double absmag, bool bolometric) const
{
  if (sMajor == 'D') return +1.5e-9; // white dwarfs ~ diameter of Earth
	
  double temp = temperature();
  double bolcorr = bolometric ? 0.0 : lookup(bolcorr_table);

  if (temp <= 0.0 || bolcorr < -10.0 || absmag < -25.0)
    return ERROR_VAL;
  
  // Now use the Stefan-Boltzmann law:
  // R* = RSun * sqrt(L* / LSun) * (TSun / T*)^2
  // but L* / LSun = 10^(0.4 * (MSun - M*))
  // so R* = RSun * 10^(0.2 * (MSun - M*)) * (TSun / T*)^2

  using std::pow;
  return SUN_DIAMETER * pow(SUN_TEMP / temp, 2) *
	  pow(10.0, (SUN_BOLO_MAG - absmag - bolcorr) / 5);
}

