/************************************************************************
 *
 * Copyright (C) 2014-2018 IRCAD France
 * Copyright (C) 2014-2018 IHU Strasbourg
 *
 * This file is part of Sight.
 *
 * Sight is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Sight is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sight. If not, see <https://www.gnu.org/licenses/>.
 *
 ***********************************************************************/

#include "fwRenderOgre/SRender.hpp"

#include "fwRenderOgre/IAdaptor.hpp"
#include "fwRenderOgre/OffScreenRenderWindowInteractorManager.hpp"
#include "fwRenderOgre/registry/Adaptor.hpp"
#include "fwRenderOgre/Utils.hpp"

#include <fwCom/Signal.hxx>
#include <fwCom/Slots.hxx>

#define FW_PROFILING_DISABLED
#include <fwCore/Profiling.hpp>

#include <fwData/mt/ObjectWriteLock.hpp>

#include <fwRuntime/ConfigurationElementContainer.hpp>
#include <fwRuntime/utils/GenericExecutableFactoryRegistrar.hpp>

#include <fwServices/helper/Config.hpp>
#include <fwServices/macros.hpp>

#include <OGRE/OgreEntity.h>
#include <OGRE/OgreNode.h>
#include <OGRE/Overlay/OgreOverlayContainer.h>
#include <OGRE/Overlay/OgreOverlayManager.h>
#include <OGRE/OgreSceneManager.h>
#include <OGRE/OgreSceneNode.h>

#include <stack>

fwServicesRegisterMacro( ::fwRender::IRender, ::fwRenderOgre::SRender, ::fwData::Composite );

namespace fwRenderOgre
{

//-----------------------------------------------------------------------------

const std::string SRender::s_OGREBACKGROUNDID = "ogreBackground";

//-----------------------------------------------------------------------------

static const ::fwServices::IService::KeyType s_OFFSCREEN_INOUT = "offScreen";

//-----------------------------------------------------------------------------

const ::fwCom::Signals::SignalKeyType SRender::s_COMPOSITOR_UPDATED_SIG = "compositorUpdated";

//-----------------------------------------------------------------------------

const ::fwCom::Slots::SlotKeyType SRender::s_COMPUTE_CAMERA_ORIG_SLOT     = "computeCameraParameters";
const ::fwCom::Slots::SlotKeyType SRender::s_COMPUTE_CAMERA_CLIPPING_SLOT = "computeCameraClipping";
const ::fwCom::Slots::SlotKeyType SRender::s_REQUEST_RENDER_SLOT          = "requestRender";

static const ::fwCom::Slots::SlotKeyType s_ADD_OBJECTS_SLOT    = "addObject";
static const ::fwCom::Slots::SlotKeyType s_CHANGE_OBJECTS_SLOT = "changeObject";
static const ::fwCom::Slots::SlotKeyType s_REMOVE_OBJECTS_SLOT = "removeObjects";

//-----------------------------------------------------------------------------

SRender::SRender() noexcept :
    m_interactorManager(nullptr),
    m_overlayTextPanel(nullptr),
    m_renderMode(RenderMode::AUTO),
    m_fullscreen(false)
{
    m_ogreRoot = ::fwRenderOgre::Utils::getOgreRoot();

    newSignal<CompositorUpdatedSignalType>(s_COMPOSITOR_UPDATED_SIG);

    newSlot(s_COMPUTE_CAMERA_ORIG_SLOT, &SRender::resetCameraCoordinates, this);
    newSlot(s_COMPUTE_CAMERA_CLIPPING_SLOT, &SRender::computeCameraClipping, this);
    newSlot(s_REQUEST_RENDER_SLOT, &SRender::requestRender, this);
}

//-----------------------------------------------------------------------------

SRender::~SRender() noexcept
{
    m_ogreRoot = nullptr;
}

//-----------------------------------------------------------------------------

void SRender::configuring()
{
    const ConfigType config = this->getConfigTree();

    SLM_ERROR_IF("Only one scene must be configured.", config.count("scene") != 1);
    const ConfigType sceneCfg = config.get_child("scene");

    const size_t nbInouts = config.count("inout");
    SLM_ASSERT("This service accepts at most one inout.", nbInouts <= 1);

    if(nbInouts == 1)
    {
        const std::string key = config.get<std::string>("inout.<xmlattr>.key", "");
        m_offScreen = (key == s_OFFSCREEN_INOUT);

        SLM_ASSERT("'" + key + "' is not a valid key. Only '" + s_OFFSCREEN_INOUT +"' is accepted.", m_offScreen);

        m_width  = sceneCfg.get<unsigned int>("<xmlattr>.width", m_width);
        m_height = sceneCfg.get<unsigned int>("<xmlattr>.height", m_height);
        m_flip   = sceneCfg.get<bool>("<xmlattr>.flip", m_flip);

    }
    else // no offscreen rendering.
    {
        this->initialize();
    }

    m_fullscreen = sceneCfg.get<bool>("<xmlattr>.fullscreen", false);

#ifdef __APPLE__
    // TODO: fix fullscreen rendering on macOS.
    SLM_ERROR("Fullscreen is broken on macOS (as of macOS 10.14 and Qt 5.11.2 and Ogre 1.11.4, "
              "it is therefore disabled.");
    m_fullscreen = false;
#endif

    const std::string renderMode = sceneCfg.get<std::string>("<xmlattr>.renderMode", "auto");
    if (renderMode == "auto")
    {
        m_renderMode = RenderMode::AUTO;
    }
    else if (renderMode == "always")
    {
        m_renderMode = RenderMode::ALWAYS;
    }
    else if (renderMode == "sync")
    {
        m_renderMode = RenderMode::SYNC;
    }
    else
    {
        SLM_ERROR("Unknown rendering mode '" + renderMode + "'");
    }

    auto adaptorConfigs = sceneCfg.equal_range("adaptor");
    for( auto it = adaptorConfigs.first; it != adaptorConfigs.second; ++it )
    {
        const std::string uid = it->second.get<std::string>("<xmlattr>.uid");
        auto& registry        = ::fwRenderOgre::registry::getAdaptorRegistry();
        registry[uid] = this->getID();
    }
}

//-----------------------------------------------------------------------------

void SRender::starting()
{
    SLM_TRACE_FUNC();

    bool bHasBackground = false;

    if (!m_offScreen)
    {
        this->create();
    }
    const ConfigType config = this->getConfigTree();

    SLM_ERROR_IF("Only one scene must be configured.", config.count("scene") != 1);

    const ConfigType sceneCfg = config.get_child("scene");

    auto layerConfigs = sceneCfg.equal_range("layer");
    for( auto it = layerConfigs.first; it != layerConfigs.second; ++it )
    {
        this->configureLayer(it->second);
    }
    auto bkgConfigs = sceneCfg.equal_range("background");
    for( auto it = bkgConfigs.first; it != bkgConfigs.second; ++it )
    {
        OSLM_ERROR_IF("A background has already been set, overriding it...", bHasBackground);
        try
        {
            this->configureBackgroundLayer(it->second);
        }
        catch (std::exception& e)
        {
            OSLM_ERROR("Error configuring background for layer '" + this->getID() + "': " + e.what());
        }

        bHasBackground = true;
    }

    if(!bHasBackground)
    {
        // Create a default black background
        ::fwRenderOgre::Layer::sptr ogreLayer = ::fwRenderOgre::Layer::New();
        ogreLayer->setRenderService(::fwRenderOgre::SRender::dynamicCast(this->shared_from_this()));
        ogreLayer->setID("backgroundLayer");
        ogreLayer->setDepth(0);
        ogreLayer->setWorker(m_associatedWorker);
        ogreLayer->setBackgroundColor("#000000", "#000000");
        ogreLayer->setBackgroundScale(0, 0.5);
        ogreLayer->setHasDefaultLight(false);

        m_layers[s_OGREBACKGROUNDID] = ogreLayer;
    }

    if(m_offScreen)
    {
        // Instantiate the manager that help to communicate between this service and the widget
        m_interactorManager = ::fwRenderOgre::OffScreenRenderWindowInteractorManager::New(m_width, m_height);
        m_interactorManager->setRenderService(this->getSptr());
        m_interactorManager->createContainer( nullptr, m_renderMode != RenderMode::ALWAYS, m_fullscreen );
    }
    else
    {
        // Instantiate the manager that help to communicate between this service and the widget
        m_interactorManager = ::fwRenderOgre::IRenderWindowInteractorManager::createManager();
        m_interactorManager->setRenderService(this->getSptr());
        m_interactorManager->createContainer( this->getContainer(), m_renderMode != RenderMode::ALWAYS, m_fullscreen );
    }

    // Initialize resources to load overlay scripts.
    ::Ogre::ResourceGroupManager::getSingleton().initialiseAllResourceGroups();

    std::istringstream overlays(sceneCfg.get<std::string>("<xmlattr>.overlays", ""));

    for(std::string overlayName; std::getline(overlays, overlayName, ';'); )
    {
        ::Ogre::Overlay* overlay = ::Ogre::OverlayManager::getSingleton().getByName(overlayName);

        if(overlay)
        {
            m_enabledOverlays.insert(overlay);
        }
        else
        {
            SLM_ERROR("Can't find overlay : " + overlayName);
        }
    }

    m_interactorManager->setEnabledOverlays(m_enabledOverlays);

    for (auto it : m_layers)
    {
        ::fwRenderOgre::Layer::sptr layer = it.second;
        layer->setRenderTarget(m_interactorManager->getRenderTarget());
        layer->createScene();
    }

    // Everything is started now, we can safely create connections and thus receive interactions from the widget
    m_interactorManager->connectToContainer();
}

//-----------------------------------------------------------------------------

void SRender::stopping()
{
    this->makeCurrent();

    m_connections.disconnect();

    for (auto it : m_layers)
    {
        ::fwRenderOgre::Layer::sptr layer = it.second;
        layer->destroyScene();
    }
    m_layers.clear();

    m_interactorManager->disconnectInteractor();
    m_interactorManager.reset();

    if(!m_offScreen)
    {
        this->destroy();
    }
}

//-----------------------------------------------------------------------------

void SRender::updating()
{
}

//-----------------------------------------------------------------------------

void SRender::makeCurrent()
{
    m_interactorManager->makeCurrent();
}

//-----------------------------------------------------------------------------

void SRender::configureLayer(const ConfigType& _cfg )
{
    const ConfigType attributes             = _cfg.get_child("<xmlattr>");
    const std::string id                    = attributes.get<std::string>("id", "");
    const std::string compositors           = attributes.get<std::string>("compositors", "");
    const std::string transparencyTechnique = attributes.get<std::string>("transparency", "");
    const std::string numPeels              = attributes.get<std::string>("numPeels", "");
    const std::string stereoMode            = attributes.get<std::string>("stereoMode", "");
    const std::string defaultLight          = attributes.get<std::string>("defaultLight", "");

    SLM_ASSERT( "'id' required attribute missing or empty", !id.empty() );
    SLM_ASSERT( "Unknown 3D mode : " << stereoMode,
                stereoMode.empty() || stereoMode == "no" || stereoMode == "AutoStereo5" || stereoMode == "AutoStereo8" ||
                stereoMode == "Stereo");

    const int layerDepth = attributes.get<int>("depth");
    SLM_ASSERT("Attribute 'depth' must be greater than 0", layerDepth > 0);

    ::fwRenderOgre::Layer::sptr ogreLayer = ::fwRenderOgre::Layer::New();
    compositor::Core::StereoModeType layerStereoMode =
        stereoMode == "AutoStereo5" ? compositor::Core::StereoModeType::AUTOSTEREO_5 :
        stereoMode == "AutoStereo8" ? compositor::Core::StereoModeType::AUTOSTEREO_8 :
        stereoMode == "Stereo"      ? compositor::Core::StereoModeType::STEREO :
        compositor::Core::StereoModeType::NONE;

    ogreLayer->setRenderService(::fwRenderOgre::SRender::dynamicCast(this->shared_from_this()));
    ogreLayer->setID(id);
    ogreLayer->setDepth(layerDepth);
    ogreLayer->setWorker(m_associatedWorker);
    ogreLayer->setCoreCompositorEnabled(true, transparencyTechnique, numPeels, layerStereoMode);
    ogreLayer->setCompositorChainEnabled(compositors);

    if(!defaultLight.empty() && defaultLight == "no")
    {
        ogreLayer->setHasDefaultLight(false);
    }

    // Finally, the layer is pushed in the map
    m_layers[id] = ogreLayer;
}

//-----------------------------------------------------------------------------

void SRender::configureBackgroundLayer(const ConfigType& _cfg )
{
    SLM_ASSERT( "'id' required attribute missing or empty", !this->getID().empty() );
    const ConfigType attributes = _cfg.get_child("<xmlattr>");

    ::fwRenderOgre::Layer::sptr ogreLayer = ::fwRenderOgre::Layer::New();
    ogreLayer->setRenderService(::fwRenderOgre::SRender::dynamicCast(this->shared_from_this()));
    ogreLayer->setID(s_OGREBACKGROUNDID);
    ogreLayer->setDepth(0);
    ogreLayer->setWorker(m_associatedWorker);
    ogreLayer->setHasDefaultLight(false);

    if (attributes.count("topColor") && attributes.count("bottomColor"))
    {
        std::string topColor = attributes.get<std::string>("topColor");
        std::string botColor = attributes.get<std::string>("bottomColor");

        ogreLayer->setBackgroundColor(topColor, botColor);
    }

    if (attributes.count("topScale") && attributes.count("bottomScale"))
    {
        const float topScaleVal = attributes.get<float>("topScale");
        const float botScaleVal = attributes.get<float>("bottomScale");

        ogreLayer->setBackgroundScale(topScaleVal, botScaleVal);
    }

    m_layers[s_OGREBACKGROUNDID] = ogreLayer;
}

//-----------------------------------------------------------------------------

void SRender::requestRender()
{
    if( this->isShownOnScreen() )
    {
        if ( m_renderMode == RenderMode::SYNC )
        {
            m_interactorManager->renderNow();
        }
        else
        {
            m_interactorManager->requestRender();
        }

        if(m_offScreen)
        {
            FW_PROFILE("Offscreen rendering");

            ::fwData::Image::sptr image = this->getInOut< ::fwData::Image >(s_OFFSCREEN_INOUT);
            SLM_ASSERT("Offscreen image not found.", image);

            auto offScreenInteractor = OffScreenRenderWindowInteractorManager::dynamicCast(m_interactorManager);
            {
                ::fwData::mt::ObjectWriteLock lock(image);
                ::fwRenderOgre::Utils::convertFromOgreTexture(offScreenInteractor->getRenderTexture(), image, m_flip);
            }

            auto sig = image->signal< ::fwData::Object::ModifiedSignalType >(::fwData::Object::s_MODIFIED_SIG);
            sig->asyncEmit();
        }
    }
}

//-----------------------------------------------------------------------------

void SRender::resetCameraCoordinates(const std::string& _layerId)
{
    auto layer = m_layers.find(_layerId);

    if(layer != m_layers.end())
    {
        layer->second->resetCameraCoordinates();
    }

    this->requestRender();
}

//-----------------------------------------------------------------------------

void SRender::computeCameraClipping()
{
    for (auto it : m_layers)
    {
        ::fwRenderOgre::Layer::sptr layer = it.second;
        layer->resetCameraClippingRange();
    }
}

//-----------------------------------------------------------------------------

::Ogre::OverlayContainer* SRender::getOverlayTextPanel()
{
    static std::mutex overlayManagerLock;

    overlayManagerLock.lock();
    if(m_overlayTextPanel == nullptr)
    {
        auto& overlayManager = ::Ogre::OverlayManager::getSingleton();

        m_overlayTextPanel =
            static_cast< ::Ogre::OverlayContainer* >(overlayManager.createOverlayElement("Panel",
                                                                                         this->getID() + "_GUI"));
        m_overlayTextPanel->setMetricsMode(::Ogre::GMM_PIXELS);
        m_overlayTextPanel->setPosition(0, 0);
        m_overlayTextPanel->setDimensions(1.0f, 1.0f);

        ::Ogre::Overlay* uiOverlay = overlayManager.create(this->getID() + "_UIOverlay");
        uiOverlay->add2D(m_overlayTextPanel);

        m_enabledOverlays.insert(uiOverlay);

        m_interactorManager->setEnabledOverlays(m_enabledOverlays);
    }
    overlayManagerLock.unlock();

    return m_overlayTextPanel;
}

//-----------------------------------------------------------------------------

void SRender::render()
{
    this->requestRender();
}

//-----------------------------------------------------------------------------

bool SRender::isShownOnScreen()
{
    if( m_offScreen )
    {
        return true;
    }
    return this->getContainer()->isShownOnScreen();
}

// ----------------------------------------------------------------------------

::Ogre::SceneManager* SRender::getSceneManager(const ::std::string& sceneID)
{
    ::fwRenderOgre::Layer::sptr layer = this->getLayer(sceneID);
    return layer->getSceneManager();
}

// ----------------------------------------------------------------------------

::fwRenderOgre::Layer::sptr SRender::getLayer(const ::std::string& sceneID)
{
    OSLM_ASSERT("Empty sceneID", !sceneID.empty());
    OSLM_ASSERT("Layer ID "<< sceneID <<" does not exist", m_layers.find(sceneID) != m_layers.end());

    ::fwRenderOgre::Layer::sptr layer = m_layers.at(sceneID);

    return layer;
}

// ----------------------------------------------------------------------------

::fwRenderOgre::SRender::LayerMapType SRender::getLayers()
{
    return m_layers;
}

// ----------------------------------------------------------------------------

::fwRenderOgre::IRenderWindowInteractorManager::sptr SRender::getInteractorManager() const
{
    return m_interactorManager;
}

// ----------------------------------------------------------------------------

} //namespace fwRenderOgre
