// Copyright (C) 1999-2012
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <iostream>
#include <sstream>
#include <iomanip>
using namespace std;

#include "file.h"
#include "outfile.h"
#include "outchannel.h"
#include "outsocket.h"

int FitsFile::saveFitsPrimHeader(OutFitsStream& str)
{
  // write fake primary header
  char buf[FTY_BLOCK];
  memset(buf,' ',FTY_BLOCK);

  char* hdu = buf;

  memcpy(hdu,"SIMPLE  = ",10);
  memcpy(hdu+32-3,"T /",3);
  hdu += FTY_CARDLEN;

  memcpy(hdu,"BITPIX  = ",10);
  memcpy(hdu+32-3,"8 /",3);
  hdu += FTY_CARDLEN;

  memcpy(hdu,"NAXIS   = ",10);
  memcpy(hdu+32-3,"0 /",3);
  hdu += FTY_CARDLEN;

  memcpy(hdu,"END",3);
  str.write(buf, FTY_BLOCK);

  return FTY_BLOCK;
}

int FitsFile::saveFitsHeader(OutFitsStream& str, int depth)
{
  int cnt =0;
  char buf[FTY_CARDLEN];

  memset(buf,' ',FTY_CARDLEN);
  memcpy(buf,"SIMPLE  = ",10);
  memcpy(buf+32-3,"T /",3);
  str.write(buf, FTY_CARDLEN);
  cnt += FTY_CARDLEN;

  char* ptr = head()->cards()+FTY_CARDLEN;
  char* end = head()->cards() + head()->headbytes();
  while (ptr<end) {
    if (!strncmp(ptr,"NAXIS ",6)) {
      memset(buf,' ',FTY_CARDLEN);
      memcpy(buf,"NAXIS   = ",10);
      if (!depth)
	memcpy(buf+32-3,"2 /",3);
      else
	memcpy(buf+32-3,"3 /",3);
      str.write(buf, FTY_CARDLEN);
    } 
    else if (!strncmp(ptr,"NAXIS2",6)) {
      str.write(ptr, FTY_CARDLEN);

      if (depth) {
	ostringstream ddstr;
	ddstr << depth << " /" << ends;
	const char* ddptr = ddstr.str().c_str();
	int ll = strlen(ddptr);

	memset(buf,' ',FTY_CARDLEN);
	memcpy(buf,"NAXIS3  = ",10);
	memcpy(buf+32-ll, ddptr, ll);
	str.write(buf, FTY_CARDLEN);
	cnt += FTY_CARDLEN;
      }
    } 
    else if (!strncmp(ptr,"NAXIS3",6)) {
      // skip
      cnt -= FTY_CARDLEN;
    } 
    else if (!strncmp(ptr,"NAXIS4",6)) {
      // skip
      cnt -= FTY_CARDLEN;
    } 
    else if (!strncmp(ptr,"NAXIS5",6)) {
      // skip
      cnt -= FTY_CARDLEN;
    } 
    else if (!strncmp(ptr,"PCOUNT",6)) {
      // skip
      cnt -= FTY_CARDLEN;
    } 
    else if (!strncmp(ptr,"GCOUNT",6)) {
      // skip
      cnt -= FTY_CARDLEN;
    } 
    else if (!strncmp(ptr,"END   ",6)) {
      // skip
      cnt -= FTY_CARDLEN;
    }
    else
      str.write(ptr, FTY_CARDLEN);

    ptr+=FTY_CARDLEN;
    cnt += FTY_CARDLEN;
  }

  // final END
  memset(buf,' ',FTY_CARDLEN);
  memcpy(buf,"END",3);
  str.write(buf, FTY_CARDLEN);
  cnt += FTY_CARDLEN;

  cnt += saveFitsPad(str,cnt,' ');

  return cnt;
}

int FitsFile::saveFitsXtHeader(OutFitsStream& str, int depth)
{
  // write xtension header
  // the header may be an xtension, or primary, so lets force a
  // first line of 'SIMPLE' and then write the rest of the header
  int cnt =0;
  char buf[FTY_CARDLEN];

  memset(buf,' ',FTY_CARDLEN);
  memcpy(buf,"XTENSION= 'IMAGE   '",20);
  str.write(buf, FTY_CARDLEN);
  cnt += FTY_CARDLEN;

  char* ptr = head()->cards()+FTY_CARDLEN;
  char* end = head()->cards() + head()->headbytes();
  while (ptr<end) {
    if (!strncmp(ptr,"NAXIS ",6)) {
      memset(buf,' ',FTY_CARDLEN);
      memcpy(buf,"NAXIS   = ",10);
      if (!depth)
	memcpy(buf+32-3,"2 /",3);
      else
	memcpy(buf+32-3,"3 /",3);
      str.write(buf, FTY_CARDLEN);
    } 
    else if (!strncmp(ptr,"NAXIS2",6)) {
      str.write(ptr, FTY_CARDLEN);

      if (depth) {
	ostringstream ddstr;
	ddstr << depth << " /" << ends;
	const char* ddptr = ddstr.str().c_str();
	int ll = strlen(ddptr);

	memset(buf,' ',FTY_CARDLEN);
	memcpy(buf,"NAXIS3  = ",10);
	memcpy(buf+32-ll, ddptr, ll);
	str.write(buf, FTY_CARDLEN);
	cnt += FTY_CARDLEN;
      }

      memset(buf,' ',FTY_CARDLEN);
      memcpy(buf,"PCOUNT  = ",10);
      memcpy(buf+32-3,"0 /",3);
      str.write(buf, FTY_CARDLEN);
      cnt += FTY_CARDLEN;

      memset(buf,' ',FTY_CARDLEN);
      memcpy(buf,"GCOUNT  = ",10);
      memcpy(buf+32-3,"1 /",3);
      str.write(buf, FTY_CARDLEN);
      cnt += FTY_CARDLEN;
    } 
    else if (!strncmp(ptr,"NAXIS3",6)) {
      // skip
      cnt -= FTY_CARDLEN;
    } 
    else if (!strncmp(ptr,"NAXIS4",6)) {
      // skip
      cnt -= FTY_CARDLEN;
    } 
    else if (!strncmp(ptr,"NAXIS5",6)) {
      // skip
      cnt -= FTY_CARDLEN;
    } 
    else if (!strncmp(ptr,"PCOUNT",6)) {
      // skip
      cnt -= FTY_CARDLEN;
    } 
    else if (!strncmp(ptr,"GCOUNT",6)) {
      // skip
      cnt -= FTY_CARDLEN;
    } 
    else if (!strncmp(ptr,"END   ",6)) {
      // skip
      cnt -= FTY_CARDLEN;
    }
    else
      str.write(ptr, FTY_CARDLEN);

    ptr+=FTY_CARDLEN;
    cnt += FTY_CARDLEN;
  }

  // final END
  memset(buf,' ',FTY_CARDLEN);
  memcpy(buf,"END",3);
  str.write(buf, FTY_CARDLEN);
  cnt += FTY_CARDLEN;

  cnt += saveFitsPad(str,cnt,' ');

  return cnt;
}

int FitsFile::saveFits(OutFitsStream& str)
{
  FitsImageHDU* hdu = (FitsImageHDU*)(head()->hdu());

  if (orgFits_)
    str.write((char*)data(), hdu->imgbytes());
  else {
    switch (pArch_) {
    case NATIVE:
      if (!lsb())
	str.write((char*)data(), hdu->imgbytes());
      else
	str.writeSwap((char*)data(), hdu->imgbytes(), head()->bitpix());
      break;
    case BIG:
      str.write((char*)data(), hdu->imgbytes());
      break;
    case LITTLE:
      str.writeSwap((char*)data(), hdu->imgbytes(), head()->bitpix());
      break;
    }
  }
  
  return hdu->imgbytes();
}

int FitsFile::saveFitsPad(OutFitsStream& str, size_t cnt, char fil)
{
  // write any padding
  char buf[FTY_BLOCK];
  memset(buf,fil,FTY_BLOCK);

  int npad = FTY_BLOCK - (cnt % FTY_BLOCK);
  if (npad == FTY_BLOCK)
    npad = 0;
  if( npad > 0 )
    str.write(buf, npad);

  return npad;
}

int FitsFile::saveFitsTable(OutFitsStream& str)
{
  int cnt =0;
  // primary header
  str.write(primary()->cards(), primary()->headbytes());
  cnt += primary()->headbytes();

  // now, ext header
  str.write(head()->cards(), head()->headbytes());
  cnt += head()->headbytes();

  // write valid data
  // our data may be short (mmap or bad fits), so write valid data
  // then write the pad, becareful with arch, if array

  if (orgFits_)
    str.write((char*)data(), head()->allbytes());
  else {
    switch (pArch_) {
    case NATIVE:
      if (!lsb())
	str.write((char*)data(), head()->allbytes());
      else
	str.writeSwap((char*)data(), head()->allbytes(), head()->bitpix());
      break;
    case BIG:
      str.write((char*)data(), head()->allbytes());
      break;
    case LITTLE:
      str.writeSwap((char*)data(), head()->allbytes(), head()->bitpix());
      break;
    }
  }
  cnt += head()->allbytes();

  // we may need to add a buffer to round out to block size
  int diff = head()->padbytes();
  if (diff>0) {
    char* buf = new char[diff];
    memset(buf,'\0',diff);
    str.write(buf, diff);
    delete [] buf;
  }
  cnt += head()->padbytes();

  return cnt;
}

int FitsFile::saveArray(OutFitsStream& str, ArchType endian)
{
  // only save one slice
  size_t size = head_->naxis(0)*head_->naxis(1)*abs(head_->bitpix()/8);
  int bitpix = head_->bitpix();

  switch (endian) {
  case NATIVE:
  case BIG:
    {
      if (orgFits_)
	str.write((char*)data(), size);
      else {
	switch (pArch_) {
	case NATIVE:
	  if (!lsb())
	    str.write((char*)data(), size);
	  else
	    str.writeSwap((char*)data(), size, bitpix);
	  break;
	case BIG:
	  str.write((char*)data(), size);
	  break;
	case LITTLE:
	  str.writeSwap((char*)data(), size, bitpix);
	  break;
	}
      }
    }
    break;
  case LITTLE:
    {
      if (orgFits_)
	str.writeSwap((char*)data(), size, bitpix);
      else {
	switch (pArch_) {
	case NATIVE:
	  if (!lsb())
	    str.writeSwap((char*)data(), size, bitpix);
	  else
	    str.write((char*)data(), size);
	  break;
	case BIG:
	  str.writeSwap((char*)data(), size, bitpix);
	  break;
	case LITTLE:
	  str.write((char*)data(), size);
	  break;
	}
      }
    }
    break;
  }

  return size;
}

