/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file CWE17Module.cxx
    \author Romain BONDUE
    \date 21/08/2005 */
#include <sstream>

#include "CWE17Module.h"
#include "WE17Driver.h"
#include "WE17DriverRoot.h"
#include "WWidgets.h"



namespace
{
    const unsigned TabModeSize (7);
    const nsWireless::Mode_e TabMode [TabModeSize] = {nsWireless::Auto,
                                nsWireless::AdHoc, nsWireless::Managed,
                                nsWireless::Master, nsWireless::Repeater,
                                nsWireless::Secondary,  nsWireless::Monitor};

    const unsigned TabAuthSize (2);
    const nsWireless::AuthType_e TabAuth [TabAuthSize] = {nsWireless::Open,
                                                          nsWireless::Shared};

    const unsigned TabNoneEncryptSize (2);
    const nsWireless::EncryptType_e TabNoneEncrypt [TabNoneEncryptSize] =
                                        {nsWireless::None, nsWireless::WEP};

    const unsigned TabSharedEncryptSize (1);
    const nsWireless::EncryptType_e TabSharedEncrypt [TabSharedEncryptSize] =
                                                            {nsWireless::WEP};

    void ApplyProfile (const nsUserData::CProfile& Profile,
                       nsWireless::CWE17Driver& Driver,
                       const nsWireless::CMacAddress* pAddress = 0)
                                                    throw (nsErrors::CSystemExc)
    {
        Driver.SetSSID (Profile.GetSSID());  
        nsWireless::Mode_e CurrentMode (nsWireless::Unknown);
        try{CurrentMode = Driver.GetMode();}
        catch (...) {} // Ignore everything.
        if (CurrentMode != Profile.GetMode())
            Driver.SetMode (Profile.GetMode());
        try{Driver.SetChannel (Profile.GetChannel());}
        catch (const nsErrors::CSystemExc&)
        {       // May be fine in managed mode.
            if (Profile.GetMode() != nsWireless::Managed)
                throw;
        }
        Driver.SetEncryption (Profile.GetEncryptionD());
        if (pAddress)
            try{Driver.SetAPMacAddr (*pAddress);}
            catch (const nsErrors::CSystemExc&) {} // May fail.
        try{Driver.Commit();}
            // May fail because it's not always a necessity.
        catch (const nsErrors::CSystemExc&) {}

    } // ApplyProfile()

} // anonymous namespace



const char* const nsCore::CWE17Module::NewProfileDefaultName ("New profile");



nsCore::CWE17Module::~CWE17Module () throw() {}


void nsCore::CWE17Module::RecordProfiles () throw (nsErrors::CException)
{
    m_ProfilesFile.Record();

} // RecordProfiles()


void nsCore::CWE17CLIModule::ApplyProfile (unsigned Pos)
                                                    throw (nsErrors::CSystemExc)
{
    ::ApplyProfile (m_ProfilesFile.GetProfile (Pos), m_Driver);

} // ApplyProfile()


nsCore::CWE17Module::CWE17Module (const std::string& DeviceName,
                                  const std::string& Path)
                                 throw (nsErrors::CException, std::bad_alloc)
    : m_pDriver (new nsWireless::CWE17Driver (DeviceName)),
      m_Editor (new nsGUI::CSharedConfigDialog (TabMode, TabModeSize,
                                                m_pDriver->GetSupportedFreq(),
                                                TabAuth, TabAuthSize,
                                                TabNoneEncrypt,
                                                TabNoneEncryptSize,
                                                TabSharedEncrypt,
                                                TabSharedEncryptSize, 0, 0))
{
    m_ProfilesFile.Extract();

} // CWE17Module()


nsUserData::CProfile* nsCore::CWE17Module::EditNewProfile
                                             (::GtkWindow* pMainWindow) throw()
{
    nsUserData::CProfile* const pProfile (&m_ProfilesFile.GetProfile
                                                (m_ProfilesFile.Size() - 1U));
    if (!m_Editor.Edit (pMainWindow, *pProfile))
    {
        m_ProfilesFile.DeleteLastProfile();
        return 0;
    }
    return pProfile;

} // EditNewProfile()


void nsCore::CWE17Module::ApplyProfile (unsigned Pos)
                                                    throw (nsErrors::CSystemExc)
{
    ::ApplyProfile (m_ProfilesFile.GetProfile (Pos), *m_pDriver);

} // ApplyProfile()


void nsCore::CWE17Module::ApplyProfile (unsigned Pos,
            const nsWireless::CMacAddress& Address) throw (nsErrors::CSystemExc)
{
    ::ApplyProfile (m_ProfilesFile.GetProfile (Pos), *m_pDriver, &Address);

} // ApplyProfile()


void nsCore::CWE17Module::BecomeRoot () throw (nsErrors::CException,
                                               std::bad_alloc)
{
    m_pDriver.reset (new nsWireless::CWE17DriverRoot
                                                (m_pDriver->GetDeviceName()));

} // BecomeRoot()
