#ifndef _RHEOLEF_FORM_OPTION_TYPE_H
#define _RHEOLEF_FORM_OPTION_TYPE_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/quadrature_option_type.h"

namespace rheolef {

/*Class:doc
NAME: @code{form_option_type} - send options to the integrate function
@findex integrate
DESCRIPTION:
  This class is used to send options to the integrate function
  when building a form.
  It allows to set the quadrature formulae that is used
  for numerical integration @pxref{quadrature_option_type class}
  and two boolean flags.
FLAGS:
  @table @code
  @item ignore_sys_coord
  @cindex axisymmetric geometry
    This flag has effects only for axisymmetric coordinate systems.
    When set, it omits the @code{r} weight in the @code{r dr dz} measure
    during the numerical integration performed the @code{integrate} function.
    This feature is usefull for computing the stream function
    in the axisymmetric case.
  @item lump
  @cindex mass lumping
    This flag, when set, perfoms a @emph{mass lumping procedure}
    on the matrix at the element level:
    @example
	a(i,i) := sum(j) a(i,j)
    @end example
    The resulting matrix is diagonal.
    This feature is usefull for computing a diagonal approximation
    of the mass matrix for the continuous @code{P1} element.
  @item invert
  @cindex local matrix inversion
    This flag, when set, perfoms a @emph{local inversion}
    on the matrix at the element level:
    @example
	a := inv(a)
    @end example
    This procedure is allowed only when the global matrix
    is block diagonal, e.g. for discontinuous or bubble approximations.
    This property is true when basis functions have a compact support
    inside exactly one element.
  @end table
DEFAULT FLAG VALUES:
  All flags are set to false by default.
End: */

//<doc:
struct form_option_type : quadrature_option_type {
// allocators:
  form_option_type(); 
  form_option_type (const form_option_type& fopt);
  form_option_type (const quadrature_option_type& qopt);
  form_option_type& operator= (const form_option_type& fopt);
  operator quadrature_option_type() const { return *this; }
// data:
  bool ignore_sys_coord, lump, invert; 
};
//>doc:

inline
form_option_type::form_option_type()
 : quadrature_option_type(),
   ignore_sys_coord (false),
   lump   (false),
   invert (false)
{
}
inline
form_option_type::form_option_type (const quadrature_option_type& qopt)
 : quadrature_option_type(qopt), 
   ignore_sys_coord (false),
   lump   (false),
   invert (false)
{
}
inline
form_option_type::form_option_type (const form_option_type& fopt)
 : quadrature_option_type(fopt), 
   ignore_sys_coord (fopt.ignore_sys_coord),
   lump   (fopt.lump),
   invert (fopt.invert)
{
}
inline
form_option_type&
form_option_type::operator= (const form_option_type& fopt)
{
  quadrature_option_type::operator= (fopt);
  ignore_sys_coord = fopt.ignore_sys_coord;
  lump                  = fopt.lump;
  invert                = fopt.invert;
  return *this;
}

}// namespace rheolef
#endif // _RHEOLEF_FORM_OPTION_TYPE_H
