/*
 *	recite - english text speech synthesizer
 *	Copyright (C) 1993 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 1, or (at your option)
 *	any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to use workstation's speaker
 *
 * Portions of this file are derived from a program
 * by  H.F. Silverman (Jan-91) and A. Smith (Feb-91),
 * no copyright notice present.
 */


#include <errno.h>
#include <multimedia/libaudio.h>
#include <multimedia/audio_device.h>
#include <multimedia/ulaw2linear.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>

#include <audio.h>
#include <error.h>
#include <mem.h>


static double volume;


void
ulaw_volume(n)
	double	n;
{
	volume = n;
}


static char *audio_strerror _((int));

static char *
audio_strerror(n)
	int	n;
{
	switch (n)
	{
	case AUDIO_SUCCESS:
		return "operation sucessful";

	case AUDIO_UNIXERROR:
		return strerror(errno);

	case AUDIO_ERR_BADHDR:
		return "bad audio header structure";

	case AUDIO_ERR_BADFILEHDR:
		return "bad file header format";

	case AUDIO_ERR_BADARG:
		return "bad function argument";

	case AUDIO_ERR_NOEFFECT:
		return "device control ignored";

	case AUDIO_ERR_ENCODING:
		return "unknown encoding format";

	case AUDIO_ERR_INTERRUPTED:
		return "operation was interrupted";

	default:
		return "unknown audio error";
	}
}


void
ulaw_read(filename, data_p, datalen_p)
	char		*filename;
	char		**data_p;
	long		*datalen_p;
{
	int		fd;
	char		*data;
	size_t		data_max;
	size_t		datalen;
	Audio_hdr	h;
	int		err;

	/*
	 * open the file
	 */
	if (filename)
	{
		fd = open(filename, O_RDONLY, 0666);
		if (fd < 0)
			nfatal("open \"%s\"", filename);
	}
	else
	{
		fd = 0;
		filename = "(standard input)";
	}

	/*
	 * make sure it is an audio file
	 */
	err = audio_read_filehdr(fd, &h, (char *)0, 0);
	if (err)
		fatal("audio read filehdr \"%s\": %s", filename, audio_strerror(err));

	/*
	 * make sure we like the encoding
	 */
	if
	(
		h.sample_rate != 8000
	||
		h.samples_per_unit != 1
	||
		h.bytes_per_unit != 1
	||
		h.channels != 1
	||
		h.encoding != AUDIO_ENCODING_ULAW
	)
	{
		char buf[AUDIO_MAX_ENCODE_INFO];

		err = audio_enc_to_str(&h, buf);
		if (err)
			strcpy(buf, "in an unknown format");
		fatal
		(
			"the \"%s\" file is %s, which is not handled",
			filename,
			buf
		);
	}

	/*
	 * read the data
	 */
	data_max = (1L << 18);
	data = mem_alloc(data_max);
	datalen = 0;
	for (;;)
	{
		long	rsize;
		long	nbytes;

		if (datalen >= data_max)
		{
			rsize = (1L << 18);
			data_max += rsize;
			mem_change_size(&data, data_max);
		}
		else
			rsize = data_max - datalen;
		nbytes = read(fd, data + datalen, rsize);
		if (nbytes < 0)
			nfatal("read \"%s\"", filename);
		if (!nbytes)
			break;
		datalen += nbytes;
	}
	if (datalen < data_max)
		mem_change_size(&data, datalen);
	*data_p = data;
	*datalen_p = datalen;

	/*
	 * close the file
	 */
	if (close(fd))
		nfatal("close \"%s\"", filename);
}


void
ulaw_write(filename, data, datalen)
	char		*filename;
	char		*data;
	long		datalen;
{
	int		fd;
	Audio_hdr	h;
	int		err;

	if (filename)
	{
		fd = open(filename, O_WRONLY | O_CREAT | O_TRUNC, 0666);
		if (fd < 0)
			nfatal("create \"%s\"", filename);
	}
	else
	{
		fd = 1;
		filename = "(standard output)";
	}

	h.sample_rate = 8000;
	h.samples_per_unit = 1;
	h.bytes_per_unit = 1;
	h.channels = 1;
	h.encoding = AUDIO_ENCODING_ULAW;
	h.data_size = datalen;

	err = audio_write_filehdr(fd, &h, (char *)0, 0);
	if (err)
	{
		fatal
		(
			"audio_write_filehdr \"%s\": %s",
			filename,
			audio_strerror(err)
		);
	}

	if (write(fd, data, datalen) < 0)
		nfatal("write \"%s\"", filename);

	if (close(fd))
		nfatal("close \"%s\"", filename);
}


void
ulaw_play(jack, data, datalen)
	int	jack;
	char	*data;
	long	datalen;
{
        int             err;
        unsigned int    myport;
        double		vol;
	char            *Audio_dev = "/dev/audio";
	int             fd;	/* file descriptor for audio device */
	Audio_hdr       Dev_hdr;	/* audio header for device */

	/*
	 * Try it quickly, first
	 */
        fd = open(Audio_dev, O_WRONLY | O_NDELAY, 0666);
#if 1
        if (fd < 0 && errno == EBUSY)
	{
		error("waiting for %s", Audio_dev);
		fd = open(Audio_dev, O_WRONLY, 0666);
	}
#endif
	if (fd < 0)
		nfatal("open \"%s\"", Audio_dev);

	/*
	 * Get the device output encoding configuration
	 */
        if (audio_get_play_config(fd, &Dev_hdr) != AUDIO_SUCCESS)
		fatal("%s is not an audio device", Audio_dev);

	if (volume > 0)
	{
        	vol = volume;
        	err = audio_set_play_gain(fd, &vol);
        	if (err != AUDIO_SUCCESS)
			fatal("%s: could not set output volume", Audio_dev);
	}

	/*
	 * Set the output port
	 */
	myport = AUDIO_SPEAKER;
        if (jack)
		myport = AUDIO_HEADPHONE;
        err = audio_set_play_port(fd, &myport);
        if (err != AUDIO_SUCCESS)
		fatal("%s: cound not set port", Audio_dev);

	err = write(fd, data, datalen);
	if (err != datalen && datalen > 0)
		nfatal("write \"%s\"", Audio_dev);

	/*
	 * Close audio system
	 */
        audio_drain(fd, 0);
        close(fd);
}
