% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/case-when.R
\name{case_when}
\alias{case_when}
\title{A general vectorised if-else}
\usage{
case_when(..., .default = NULL, .ptype = NULL, .size = NULL)
}
\arguments{
\item{...}{<\code{\link[rlang:dyn-dots]{dynamic-dots}}> A sequence of two-sided
formulas. The left hand side (LHS) determines which values match this case.
The right hand side (RHS) provides the replacement value.

The LHS inputs must evaluate to logical vectors.

The RHS inputs will be coerced to their common type.

All inputs will be recycled to their common size. That said, we encourage
all LHS inputs to be the same size. Recycling is mainly useful for RHS
inputs, where you might supply a size 1 input that will be recycled to the
size of the LHS inputs.

\code{NULL} inputs are ignored.}

\item{.default}{The value used when all of the LHS inputs return either
\code{FALSE} or \code{NA}.

\code{.default} must be size 1 or the same size as the common size computed
from \code{...}.

\code{.default} participates in the computation of the common type with the RHS
inputs.

\code{NA} values in the LHS conditions are treated like \code{FALSE}, meaning that
the result at those locations will be assigned the \code{.default} value. To
handle missing values in the conditions differently, you must explicitly
catch them with another condition before they fall through to the
\code{.default}. This typically involves some variation of \code{is.na(x) ~ value}
tailored to your usage of \code{case_when()}.

If \code{NULL}, the default, a missing value will be used.}

\item{.ptype}{An optional prototype declaring the desired output type. If
supplied, this overrides the common type of the RHS inputs.}

\item{.size}{An optional size declaring the desired output size. If supplied,
this overrides the common size computed from \code{...}.}
}
\value{
A vector with the same size as the common size computed from the
inputs in \code{...} and the same type as the common type of the RHS inputs
in \code{...}.
}
\description{
This function allows you to vectorise multiple \code{\link[=if_else]{if_else()}} statements. Each
case is evaluated sequentially and the first match for each element
determines the corresponding value in the output vector. If no cases match,
the \code{.default} is used as a final "else" statment.

\code{case_when()} is an R equivalent of the SQL "searched" \verb{CASE WHEN} statement.
}
\examples{
x <- 1:70
case_when(
  x \%\% 35 == 0 ~ "fizz buzz",
  x \%\% 5 == 0 ~ "fizz",
  x \%\% 7 == 0 ~ "buzz",
  .default = as.character(x)
)

# Like an if statement, the arguments are evaluated in order, so you must
# proceed from the most specific to the most general. This won't work:
case_when(
  x \%\%  5 == 0 ~ "fizz",
  x \%\%  7 == 0 ~ "buzz",
  x \%\% 35 == 0 ~ "fizz buzz",
  .default = as.character(x)
)

# If none of the cases match and no `.default` is supplied, NA is used:
case_when(
  x \%\% 35 == 0 ~ "fizz buzz",
  x \%\% 5 == 0 ~ "fizz",
  x \%\% 7 == 0 ~ "buzz",
)

# Note that `NA` values on the LHS are treated like `FALSE` and will be
# assigned the `.default` value. You must handle them explicitly if you
# want to use a different value. The exact way to handle missing values is
# dependent on the set of LHS conditions you use.
x[2:4] <- NA_real_
case_when(
  x \%\% 35 == 0 ~ "fizz buzz",
  x \%\% 5 == 0 ~ "fizz",
  x \%\% 7 == 0 ~ "buzz",
  is.na(x) ~ "nope",
  .default = as.character(x)
)

# `case_when()` evaluates all RHS expressions, and then constructs its
# result by extracting the selected (via the LHS expressions) parts.
# In particular `NaN`s are produced in this case:
y <- seq(-2, 2, by = .5)
case_when(
  y >= 0 ~ sqrt(y),
  .default = y
)

# `case_when()` is particularly useful inside `mutate()` when you want to
# create a new variable that relies on a complex combination of existing
# variables
starwars \%>\%
  select(name:mass, gender, species) \%>\%
  mutate(
    type = case_when(
      height > 200 | mass > 200 ~ "large",
      species == "Droid" ~ "robot",
      .default = "other"
    )
  )


# `case_when()` is not a tidy eval function. If you'd like to reuse
# the same patterns, extract the `case_when()` call in a normal
# function:
case_character_type <- function(height, mass, species) {
  case_when(
    height > 200 | mass > 200 ~ "large",
    species == "Droid" ~ "robot",
    .default = "other"
  )
}

case_character_type(150, 250, "Droid")
case_character_type(150, 150, "Droid")

# Such functions can be used inside `mutate()` as well:
starwars \%>\%
  mutate(type = case_character_type(height, mass, species)) \%>\%
  pull(type)

# `case_when()` ignores `NULL` inputs. This is useful when you'd
# like to use a pattern only under certain conditions. Here we'll
# take advantage of the fact that `if` returns `NULL` when there is
# no `else` clause:
case_character_type <- function(height, mass, species, robots = TRUE) {
  case_when(
    height > 200 | mass > 200 ~ "large",
    if (robots) species == "Droid" ~ "robot",
    .default = "other"
  )
}

starwars \%>\%
  mutate(type = case_character_type(height, mass, species, robots = FALSE)) \%>\%
  pull(type)
}
\seealso{
\code{\link[=case_match]{case_match()}}
}
