// This library is part of PLINK 2.00, copyright (C) 2005-2023 Shaun Purcell,
// Christopher Chang.
//
// This library is free software: you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software Foundation; either version 3 of the License, or (at your
// option) any later version.
//
// This library is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with this library.  If not, see <http://www.gnu.org/licenses/>.

#include "plink2_fmath.h"

#ifdef __cplusplus
namespace plink2 {
#endif

// The functions are C ports of Shigeo Mitsunari's fast math library functions
// posted at https://github.com/herumi/fmath .  License is
// https://opensource.org/licenses/BSD-3-Clause .
//
// (I tried porting fmath_log_ps, but it turns out that Firth regression needs
// double-precision log accuracy; logf() actually interferes with convergence.)


// programmatically generated by:
// typedef union {
//   float f4;
//   uint32_t u4;
// } __uni4;
//
// __uni4 u4;
// int32_t ii;
// for (ii = 0; ii < 1024; ii++) {
//   u4.f4 = pow(2.0f, ((float)ii) / 1024.0);
//   printf("0x%08x", u4.u4 & 0x7fffff);
//   if (ii % 4 != 3) {
//     printf(", ");
//   } else {
//     printf(",\n");
//   }
// }

#ifdef __LP64__
const uint32_t kFloatExpLookupInt[]
#  ifdef FVEC_32
  __attribute__((aligned(32)))
#  else
  __attribute__((aligned(16)))
#  endif
  = {
0x00000000, 0x00001630, 0x00002c64, 0x0000429c,
0x000058d8, 0x00006f17, 0x0000855b, 0x00009ba2,
0x0000b1ed, 0x0000c83c, 0x0000de8f, 0x0000f4e6,
0x00010b41, 0x0001219f, 0x00013802, 0x00014e68,
0x000164d2, 0x00017b40, 0x000191b2, 0x0001a828,
0x0001bea1, 0x0001d51f, 0x0001eba1, 0x00020226,
0x000218af, 0x00022f3c, 0x000245ce, 0x00025c63,
0x000272fc, 0x00028998, 0x0002a039, 0x0002b6de,
0x0002cd87, 0x0002e433, 0x0002fae4, 0x00031198,
0x00032850, 0x00033f0d, 0x000355cd, 0x00036c91,
0x00038359, 0x00039a25, 0x0003b0f5, 0x0003c7c9,
0x0003dea1, 0x0003f57d, 0x00040c5d, 0x00042341,
0x00043a29, 0x00045115, 0x00046804, 0x00047ef8,
0x000495f0, 0x0004aceb, 0x0004c3eb, 0x0004daef,
0x0004f1f6, 0x00050902, 0x00052012, 0x00053725,
0x00054e3d, 0x00056558, 0x00057c78, 0x0005939c,
0x0005aac3, 0x0005c1ef, 0x0005d91f, 0x0005f052,
0x0006078a, 0x00061ec6, 0x00063606, 0x00064d4a,
0x00066491, 0x00067bdd, 0x0006932d, 0x0006aa81,
0x0006c1d9, 0x0006d935, 0x0006f095, 0x000707f9,
0x00071f62, 0x000736ce, 0x00074e3e, 0x000765b3,
0x00077d2b, 0x000794a8, 0x0007ac28, 0x0007c3ad,
0x0007db35, 0x0007f2c2, 0x00080a53, 0x000821e8,
0x00083981, 0x0008511e, 0x000868c0, 0x00088065,
0x0008980f, 0x0008afbc, 0x0008c76e, 0x0008df23,
0x0008f6dd, 0x00090e9b, 0x0009265d, 0x00093e24,
0x000955ee, 0x00096dbc, 0x0009858f, 0x00099d66,
0x0009b541, 0x0009cd20, 0x0009e503, 0x0009fcea,
0x000a14d5, 0x000a2cc5, 0x000a44b9, 0x000a5cb1,
0x000a74ad, 0x000a8cad, 0x000aa4b1, 0x000abcba,
0x000ad4c6, 0x000aecd7, 0x000b04ec, 0x000b1d05,
0x000b3523, 0x000b4d44, 0x000b656a, 0x000b7d94,
0x000b95c2, 0x000badf4, 0x000bc62b, 0x000bde65,
0x000bf6a4, 0x000c0ee7, 0x000c272f, 0x000c3f7a,
0x000c57ca, 0x000c701e, 0x000c8876, 0x000ca0d2,
0x000cb933, 0x000cd198, 0x000cea01, 0x000d026e,
0x000d1adf, 0x000d3355, 0x000d4bcf, 0x000d644d,
0x000d7cd0, 0x000d9556, 0x000dade1, 0x000dc671,
0x000ddf04, 0x000df79c, 0x000e1038, 0x000e28d8,
0x000e417d, 0x000e5a25, 0x000e72d3, 0x000e8b84,
0x000ea43a, 0x000ebcf3, 0x000ed5b2, 0x000eee74,
0x000f073b, 0x000f2006, 0x000f38d5, 0x000f51a9,
0x000f6a81, 0x000f835d, 0x000f9c3e, 0x000fb523,
0x000fce0c, 0x000fe6fa, 0x000fffec, 0x001018e2,
0x001031dc, 0x00104adb, 0x001063de, 0x00107ce6,
0x001095f2, 0x0010af02, 0x0010c816, 0x0010e12f,
0x0010fa4d, 0x0011136e, 0x00112c94, 0x001145be,
0x00115eed, 0x00117820, 0x00119158, 0x0011aa93,
0x0011c3d3, 0x0011dd18, 0x0011f661, 0x00120fae,
0x00122900, 0x00124256, 0x00125bb0, 0x0012750f,
0x00128e72, 0x0012a7da, 0x0012c146, 0x0012dab7,
0x0012f42c, 0x00130da5, 0x00132723, 0x001340a5,
0x00135a2b, 0x001373b6, 0x00138d46, 0x0013a6d9,
0x0013c072, 0x0013da0e, 0x0013f3af, 0x00140d55,
0x001426ff, 0x001440ae, 0x00145a60, 0x00147418,
0x00148dd4, 0x0014a794, 0x0014c159, 0x0014db22,
0x0014f4f0, 0x00150ec2, 0x00152898, 0x00154274,
0x00155c53, 0x00157637, 0x00159020, 0x0015aa0d,
0x0015c3ff, 0x0015ddf5, 0x0015f7ef, 0x001611ee,
0x00162bf2, 0x001645fa, 0x00166006, 0x00167a18,
0x0016942d, 0x0016ae47, 0x0016c866, 0x0016e289,
0x0016fcb1, 0x001716dd, 0x0017310e, 0x00174b43,
0x0017657d, 0x00177fbc, 0x001799ff, 0x0017b446,
0x0017ce92, 0x0017e8e3, 0x00180338, 0x00181d92,
0x001837f0, 0x00185253, 0x00186cbb, 0x00188727,
0x0018a197, 0x0018bc0d, 0x0018d686, 0x0018f105,
0x00190b88, 0x0019260f, 0x0019409c, 0x00195b2c,
0x001975c2, 0x0019905c, 0x0019aafa, 0x0019c59e,
0x0019e046, 0x0019faf2, 0x001a15a3, 0x001a3059,
0x001a4b13, 0x001a65d2, 0x001a8096, 0x001a9b5e,
0x001ab62b, 0x001ad0fd, 0x001aebd3, 0x001b06ae,
0x001b218d, 0x001b3c71, 0x001b575a, 0x001b7248,
0x001b8d3a, 0x001ba831, 0x001bc32c, 0x001bde2c,
0x001bf931, 0x001c143b, 0x001c2f49, 0x001c4a5c,
0x001c6573, 0x001c8090, 0x001c9bb1, 0x001cb6d6,
0x001cd201, 0x001ced30, 0x001d0864, 0x001d239c,
0x001d3eda, 0x001d5a1c, 0x001d7562, 0x001d90ae,
0x001dabfe, 0x001dc753, 0x001de2ad, 0x001dfe0b,
0x001e196e, 0x001e34d6, 0x001e5043, 0x001e6bb4,
0x001e872a, 0x001ea2a5, 0x001ebe25, 0x001ed9a9,
0x001ef532, 0x001f10c0, 0x001f2c53, 0x001f47eb,
0x001f6387, 0x001f7f28, 0x001f9ace, 0x001fb679,
0x001fd228, 0x001feddc, 0x00200996, 0x00202553,
0x00204116, 0x00205cde, 0x002078aa, 0x0020947b,
0x0020b051, 0x0020cc2c, 0x0020e80b, 0x002103f0,
0x00211fd9, 0x00213bc7, 0x002157ba, 0x002173b2,
0x00218faf, 0x0021abb0, 0x0021c7b7, 0x0021e3c2,
0x0021ffd2, 0x00221be7, 0x00223801, 0x0022541f,
0x00227043, 0x00228c6b, 0x0022a899, 0x0022c4cb,
0x0022e102, 0x0022fd3e, 0x0023197f, 0x002335c5,
0x0023520f, 0x00236e5f, 0x00238ab3, 0x0023a70d,
0x0023c36b, 0x0023dfce, 0x0023fc37, 0x002418a4,
0x00243516, 0x0024518d, 0x00246e08, 0x00248a89,
0x0024a70f, 0x0024c39a, 0x0024e029, 0x0024fcbe,
0x00251958, 0x002535f6, 0x00255299, 0x00256f42,
0x00258bef, 0x0025a8a2, 0x0025c559, 0x0025e215,
0x0025fed7, 0x00261b9d, 0x00263868, 0x00265538,
0x0026720e, 0x00268ee8, 0x0026abc7, 0x0026c8ac,
0x0026e595, 0x00270283, 0x00271f76, 0x00273c6f,
0x0027596c, 0x0027766e, 0x00279376, 0x0027b082,
0x0027cd94, 0x0027eaaa, 0x002807c6, 0x002824e6,
0x0028420c, 0x00285f37, 0x00287c66, 0x0028999b,
0x0028b6d5, 0x0028d414, 0x0028f158, 0x00290ea1,
0x00292bef, 0x00294942, 0x0029669b, 0x002983f8,
0x0029a15b, 0x0029bec2, 0x0029dc2f, 0x0029f9a1,
0x002a1718, 0x002a3494, 0x002a5215, 0x002a6f9b,
0x002a8d26, 0x002aaab7, 0x002ac84c, 0x002ae5e7,
0x002b0387, 0x002b212c, 0x002b3ed6, 0x002b5c85,
0x002b7a3a, 0x002b97f3, 0x002bb5b2, 0x002bd376,
0x002bf13f, 0x002c0f0d, 0x002c2ce0, 0x002c4ab9,
0x002c6897, 0x002c867a, 0x002ca462, 0x002cc24f,
0x002ce041, 0x002cfe39, 0x002d1c36, 0x002d3a38,
0x002d583f, 0x002d764b, 0x002d945d, 0x002db274,
0x002dd090, 0x002deeb1, 0x002e0cd8, 0x002e2b03,
0x002e4934, 0x002e676b, 0x002e85a6, 0x002ea3e7,
0x002ec22d, 0x002ee078, 0x002efec8, 0x002f1d1e,
0x002f3b79, 0x002f59d9, 0x002f783e, 0x002f96a9,
0x002fb519, 0x002fd38e, 0x002ff209, 0x00301089,
0x00302f0e, 0x00304d98, 0x00306c28, 0x00308abd,
0x0030a957, 0x0030c7f7, 0x0030e69c, 0x00310546,
0x003123f6, 0x003142aa, 0x00316165, 0x00318024,
0x00319ee9, 0x0031bdb3, 0x0031dc83, 0x0031fb57,
0x00321a32, 0x00323911, 0x003257f6, 0x003276e0,
0x003295d0, 0x0032b4c5, 0x0032d3bf, 0x0032f2bf,
0x003311c4, 0x003330cf, 0x00334fde, 0x00336ef4,
0x00338e0e, 0x0033ad2e, 0x0033cc54, 0x0033eb7e,
0x00340aaf, 0x003429e4, 0x0034491f, 0x00346860,
0x003487a6, 0x0034a6f1, 0x0034c642, 0x0034e598,
0x003504f3, 0x00352454, 0x003543bb, 0x00356327,
0x00358298, 0x0035a20f, 0x0035c18b, 0x0035e10d,
0x00360094, 0x00362020, 0x00363fb2, 0x00365f4a,
0x00367ee7, 0x00369e89, 0x0036be31, 0x0036dddf,
0x0036fd92, 0x00371d4a, 0x00373d08, 0x00375ccc,
0x00377c95, 0x00379c63, 0x0037bc37, 0x0037dc11,
0x0037fbf0, 0x00381bd4, 0x00383bbe, 0x00385bae,
0x00387ba3, 0x00389b9e, 0x0038bb9e, 0x0038dba4,
0x0038fbaf, 0x00391bc0, 0x00393bd7, 0x00395bf3,
0x00397c14, 0x00399c3b, 0x0039bc68, 0x0039dc9a,
0x0039fcd2, 0x003a1d10, 0x003a3d53, 0x003a5d9b,
0x003a7dea, 0x003a9e3e, 0x003abe97, 0x003adef6,
0x003aff5b, 0x003b1fc5, 0x003b4035, 0x003b60aa,
0x003b8126, 0x003ba1a6, 0x003bc22d, 0x003be2b9,
0x003c034a, 0x003c23e2, 0x003c447f, 0x003c6521,
0x003c85ca, 0x003ca678, 0x003cc72b, 0x003ce7e5,
0x003d08a4, 0x003d2968, 0x003d4a33, 0x003d6b03,
0x003d8bd8, 0x003dacb4, 0x003dcd95, 0x003dee7c,
0x003e0f68, 0x003e305a, 0x003e5152, 0x003e7250,
0x003e9353, 0x003eb45c, 0x003ed56b, 0x003ef67f,
0x003f179a, 0x003f38ba, 0x003f59df, 0x003f7b0b,
0x003f9c3c, 0x003fbd73, 0x003fdeb0, 0x003ffff2,
0x0040213b, 0x00404289, 0x004063dc, 0x00408536,
0x0040a695, 0x0040c7fb, 0x0040e966, 0x00410ad6,
0x00412c4d, 0x00414dc9, 0x00416f4b, 0x004190d3,
0x0041b261, 0x0041d3f5, 0x0041f58e, 0x0042172d,
0x004238d2, 0x00425a7d, 0x00427c2e, 0x00429de4,
0x0042bfa1, 0x0042e163, 0x0043032b, 0x004324f9,
0x004346cd, 0x004368a7, 0x00438a86, 0x0043ac6b,
0x0043ce57, 0x0043f048, 0x0044123f, 0x0044343c,
0x0044563f, 0x00447848, 0x00449a56, 0x0044bc6b,
0x0044de85, 0x004500a5, 0x004522cc, 0x004544f8,
0x0045672a, 0x00458962, 0x0045aba0, 0x0045cde4,
0x0045f02e, 0x0046127e, 0x004634d3, 0x0046572f,
0x00467991, 0x00469bf8, 0x0046be66, 0x0046e0d9,
0x00470353, 0x004725d2, 0x00474858, 0x00476ae3,
0x00478d75, 0x0047b00c, 0x0047d2aa, 0x0047f54d,
0x004817f7, 0x00483aa6, 0x00485d5b, 0x00488017,
0x0048a2d8, 0x0048c5a0, 0x0048e86d, 0x00490b41,
0x00492e1b, 0x004950fa, 0x004973e0, 0x004996cc,
0x0049b9be, 0x0049dcb5, 0x0049ffb3, 0x004a22b7,
0x004a45c1, 0x004a68d1, 0x004a8be8, 0x004aaf04,
0x004ad226, 0x004af54f, 0x004b187d, 0x004b3bb2,
0x004b5eed, 0x004b822e, 0x004ba575, 0x004bc8c2,
0x004bec15, 0x004c0f6e, 0x004c32ce, 0x004c5633,
0x004c799f, 0x004c9d11, 0x004cc089, 0x004ce407,
0x004d078c, 0x004d2b16, 0x004d4ea7, 0x004d723d,
0x004d95da, 0x004db97e, 0x004ddd27, 0x004e00d6,
0x004e248c, 0x004e4848, 0x004e6c0a, 0x004e8fd2,
0x004eb3a1, 0x004ed775, 0x004efb50, 0x004f1f31,
0x004f4319, 0x004f6706, 0x004f8afa, 0x004faef4,
0x004fd2f4, 0x004ff6fb, 0x00501b08, 0x00503f1b,
0x00506334, 0x00508753, 0x0050ab79, 0x0050cfa5,
0x0050f3d7, 0x00511810, 0x00513c4f, 0x00516094,
0x005184df, 0x0051a931, 0x0051cd89, 0x0051f1e7,
0x0052164c, 0x00523ab7, 0x00525f28, 0x005283a0,
0x0052a81e, 0x0052cca2, 0x0052f12c, 0x005315bd,
0x00533a54, 0x00535ef2, 0x00538396, 0x0053a840,
0x0053ccf1, 0x0053f1a8, 0x00541665, 0x00543b29,
0x00545ff3, 0x005484c3, 0x0054a99a, 0x0054ce77,
0x0054f35b, 0x00551845, 0x00553d35, 0x0055622c,
0x00558729, 0x0055ac2d, 0x0055d137, 0x0055f647,
0x00561b5e, 0x0056407b, 0x0056659f, 0x00568ac9,
0x0056affa, 0x0056d531, 0x0056fa6e, 0x00571fb2,
0x005744fd, 0x00576a4e, 0x00578fa5, 0x0057b503,
0x0057da67, 0x0057ffd2, 0x00582543, 0x00584abb,
0x00587039, 0x005895be, 0x0058bb49, 0x0058e0db,
0x00590673, 0x00592c12, 0x005951b8, 0x00597763,
0x00599d16, 0x0059c2cf, 0x0059e88e, 0x005a0e54,
0x005a3421, 0x005a59f4, 0x005a7fcd, 0x005aa5ae,
0x005acb94, 0x005af182, 0x005b1776, 0x005b3d70,
0x005b6371, 0x005b8979, 0x005baf87, 0x005bd59c,
0x005bfbb8, 0x005c21da, 0x005c4802, 0x005c6e32,
0x005c9468, 0x005cbaa4, 0x005ce0e7, 0x005d0731,
0x005d2d82, 0x005d53d9, 0x005d7a36, 0x005da09b,
0x005dc706, 0x005ded77, 0x005e13f0, 0x005e3a6f,
0x005e60f5, 0x005e8781, 0x005eae14, 0x005ed4ae,
0x005efb4e, 0x005f21f5, 0x005f48a3, 0x005f6f58,
0x005f9613, 0x005fbcd5, 0x005fe39e, 0x00600a6d,
0x00603143, 0x00605820, 0x00607f03, 0x0060a5ee,
0x0060ccdf, 0x0060f3d7, 0x00611ad5, 0x006141db,
0x006168e7, 0x00618ffa, 0x0061b713, 0x0061de34,
0x0062055b, 0x00622c89, 0x006253be, 0x00627af9,
0x0062a23c, 0x0062c985, 0x0062f0d5, 0x0063182c,
0x00633f89, 0x006366ee, 0x00638e59, 0x0063b5cb,
0x0063dd44, 0x006404c4, 0x00642c4b, 0x006453d8,
0x00647b6d, 0x0064a308, 0x0064caaa, 0x0064f253,
0x00651a03, 0x006541b9, 0x00656977, 0x0065913c,
0x0065b907, 0x0065e0d9, 0x006608b2, 0x00663092,
0x00665879, 0x00668067, 0x0066a85c, 0x0066d058,
0x0066f85b, 0x00672064, 0x00674875, 0x0067708c,
0x006798ab, 0x0067c0d0, 0x0067e8fd, 0x00681130,
0x0068396a, 0x006861ac, 0x006889f4, 0x0068b243,
0x0068da99, 0x006902f7, 0x00692b5b, 0x006953c6,
0x00697c38, 0x0069a4b1, 0x0069cd32, 0x0069f5b9,
0x006a1e47, 0x006a46dd, 0x006a6f79, 0x006a981c,
0x006ac0c7, 0x006ae978, 0x006b1231, 0x006b3af1,
0x006b63b7, 0x006b8c85, 0x006bb55a, 0x006bde36,
0x006c0719, 0x006c3003, 0x006c58f4, 0x006c81ec,
0x006caaec, 0x006cd3f2, 0x006cfd00, 0x006d2614,
0x006d4f30, 0x006d7853, 0x006da17d, 0x006dcaae,
0x006df3e7, 0x006e1d26, 0x006e466d, 0x006e6fbb,
0x006e9910, 0x006ec26c, 0x006eebcf, 0x006f1539,
0x006f3eab, 0x006f6824, 0x006f91a4, 0x006fbb2b,
0x006fe4ba, 0x00700e4f, 0x007037ec, 0x00706190,
0x00708b3b, 0x0070b4ee, 0x0070dea8, 0x00710868,
0x00713231, 0x00715c00, 0x007185d7, 0x0071afb5,
0x0071d99a, 0x00720386, 0x00722d7a, 0x00725775,
0x00728177, 0x0072ab81, 0x0072d592, 0x0072ffaa,
0x007329c9, 0x007353f0, 0x00737e1e, 0x0073a853,
0x0073d290, 0x0073fcd4, 0x0074271f, 0x00745172,
0x00747bcc, 0x0074a62d, 0x0074d096, 0x0074fb06,
0x0075257d, 0x00754ffc, 0x00757a82, 0x0075a50f,
0x0075cfa4, 0x0075fa40, 0x007624e4, 0x00764f8f,
0x00767a41, 0x0076a4fb, 0x0076cfbc, 0x0076fa85,
0x00772555, 0x0077502d, 0x00777b0b, 0x0077a5f2,
0x0077d0df, 0x0077fbd5, 0x007826d1, 0x007851d5,
0x00787ce1, 0x0078a7f4, 0x0078d30e, 0x0078fe30,
0x0079295a, 0x0079548b, 0x00797fc3, 0x0079ab03,
0x0079d64a, 0x007a0199, 0x007a2cf0, 0x007a584d,
0x007a83b3, 0x007aaf20, 0x007ada94, 0x007b0610,
0x007b3194, 0x007b5d1f, 0x007b88b2, 0x007bb44c,
0x007bdfed, 0x007c0b97, 0x007c3748, 0x007c6300,
0x007c8ec0, 0x007cba88, 0x007ce657, 0x007d122e,
0x007d3e0c, 0x007d69f2, 0x007d95e0, 0x007dc1d5,
0x007dedd2, 0x007e19d6, 0x007e45e2, 0x007e71f6,
0x007e9e11, 0x007eca34, 0x007ef65f, 0x007f2291,
0x007f4ecb, 0x007f7b0d, 0x007fa756, 0x007fd3a7
};

// programmatically generated by:
// typedef union {
//   double d8;
//   uint64_t u8;
// } __uni8;
//
// __uni8 u8;
// for (int32_t ii = 0; ii < 2048; ++ii) {
//   u8.d8 = pow(2.0, ((double)ii) / 2048.0);
//   printf("0x%016llx", u8.u8 & 0xfffffffffffffLLU);
//   if (ii % 4 != 3) {
//     printf(", ");
//   } else {
//     printf(",\n");
//   }
// }
const uint64_t kExpdLookupInt[]
#  ifdef FVEC_32
  __attribute__((aligned(32)))
#  else
  __attribute__((aligned(16)))
#  endif
  = {
0x0000000000000000LLU, 0x00000162f3904052LLU,
0x000002c605e2e8cfLLU, 0x0000042936faa3d8LLU,
0x0000058c86da1c0aLLU, 0x000006eff583fc3dLLU,
0x0000085382faef83LLU, 0x000009b72f41a12bLLU,
0x00000b1afa5abcbfLLU, 0x00000c7ee448ee02LLU,
0x00000de2ed0ee0f5LLU, 0x00000f4714af41d3LLU,
0x000010ab5b2cbd11LLU, 0x0000120fc089ff63LLU,
0x0000137444c9b5b5LLU, 0x000014d8e7ee8d2fLLU,
0x0000163da9fb3335LLU, 0x000017a28af25567LLU,
0x000019078ad6a19fLLU, 0x00001a6ca9aac5f3LLU,
0x00001bd1e77170b4LLU, 0x00001d37442d5070LLU,
0x00001e9cbfe113efLLU, 0x000020025a8f6a35LLU,
0x00002168143b0281LLU, 0x000022cdece68c4fLLU,
0x00002433e494b755LLU, 0x00002599fb483385LLU,
0x000027003103b10eLLU, 0x0000286685c9e058LLU,
0x000029ccf99d720aLLU, 0x00002b338c811703LLU,
0x00002c9a3e778061LLU, 0x00002e010f835f7bLLU,
0x00002f67ffa765e6LLU, 0x000030cf0ee64571LLU,
0x000032363d42b027LLU, 0x0000339d8abf5851LLU,
0x00003504f75ef071LLU, 0x0000366c83242b47LLU,
0x000037d42e11bbccLLU, 0x0000393bf82a5538LLU,
0x00003aa3e170aafeLLU, 0x00003c0be9e770cbLLU,
0x00003d7411915a8aLLU, 0x00003edc58711c63LLU,
0x00004044be896ab6LLU, 0x000041ad43dcfa24LLU,
0x00004315e86e7f85LLU, 0x0000447eac40aff0LLU,
0x000045e78f5640b9LLU, 0x0000475091b1e76dLLU,
0x000048b9b35659d8LLU, 0x00004a22f4464e00LLU,
0x00004b8c54847a28LLU, 0x00004cf5d41394cfLLU,
0x00004e5f72f654b1LLU, 0x00004fc9312f70c5LLU,
0x000051330ec1a03fLLU, 0x0000529d0baf9a8fLLU,
0x0000540727fc1762LLU, 0x0000557163a9ce9fLLU,
0x000056dbbebb786bLLU, 0x000058463933cd29LLU,
0x000059b0d3158574LLU, 0x00005b1b8c635a28LLU,
0x00005c866520045bLLU, 0x00005df15d4e3d5eLLU,
0x00005f5c74f0bec2LLU, 0x000060c7ac0a4252LLU,
0x00006233029d8216LLU, 0x0000639e78ad3853LLU,
0x0000650a0e3c1f89LLU, 0x00006675c34cf276LLU,
0x000067e197e26c14LLU, 0x0000694d8bff479aLLU,
0x00006ab99fa6407cLLU, 0x00006c25d2da1268LLU,
0x00006d92259d794dLLU, 0x00006efe97f33152LLU,
0x0000706b29ddf6deLLU, 0x000071d7db608693LLU,
0x00007344ac7d9d51LLU, 0x000074b19d37f833LLU,
0x0000761ead925493LLU, 0x0000778bdd8f7005LLU,
0x000078f92d32085dLLU, 0x00007a669c7cdba9LLU,
0x00007bd42b72a836LLU, 0x00007d41da162c8cLLU,
0x00007eafa86a2771LLU, 0x0000801d967157e8LLU,
0x0000818ba42e7d30LLU, 0x000082f9d1a456c5LLU,
0x000084681ed5a462LLU, 0x000085d68bc525fcLLU,
0x0000874518759bc8LLU, 0x000088b3c4e9c635LLU,
0x00008a22912465f2LLU, 0x00008b917d283be8LLU,
0x00008d0088f8093fLLU, 0x00008e6fb4968f5dLLU,
0x00008fdf00068fe2LLU, 0x0000914e6b4accafLLU,
0x000092bdf66607e0LLU, 0x0000942da15b03cdLLU,
0x0000959d6c2c830dLLU, 0x0000970d56dd4875LLU,
0x0000987d61701716LLU, 0x000099ed8be7b23dLLU,
0x00009b5dd646dd77LLU, 0x00009cce40905c8cLLU,
0x00009e3ecac6f383LLU, 0x00009faf74ed66a0LLU,
0x0000a1203f067a63LLU, 0x0000a2912914f38cLLU,
0x0000a402331b9715LLU, 0x0000a5735d1d2a39LLU,
0x0000a6e4a71c726eLLU, 0x0000a856111c3568LLU,
0x0000a9c79b1f3919LLU, 0x0000ab39452843b1LLU,
0x0000acab0f3a1b9cLLU, 0x0000ae1cf9578784LLU,
0x0000af8f03834e52LLU, 0x0000b1012dc0372cLLU,
0x0000b27378110974LLU, 0x0000b3e5e2788ccbLLU,
0x0000b5586cf9890fLLU, 0x0000b6cb1796c65eLLU,
0x0000b83de2530d11LLU, 0x0000b9b0cd3125bfLLU,
0x0000bb23d833d93fLLU, 0x0000bc97035df0a2LLU,
0x0000be0a4eb2353bLLU, 0x0000bf7dba337098LLU,
0x0000c0f145e46c85LLU, 0x0000c264f1c7f30eLLU,
0x0000c3d8bde0ce7aLLU, 0x0000c54caa31c94fLLU,
0x0000c6c0b6bdae53LLU, 0x0000c834e3874886LLU,
0x0000c9a93091632aLLU, 0x0000cb1d9ddec9bbLLU,
0x0000cc922b7247f7LLU, 0x0000ce06d94ea9d7LLU,
0x0000cf7ba776bb94LLU, 0x0000d0f095ed49a3LLU,
0x0000d265a4b520baLLU, 0x0000d3dad3d10dc9LLU,
0x0000d5502343de02LLU, 0x0000d6c593105ed4LLU,
0x0000d83b23395decLLU, 0x0000d9b0d3c1a933LLU,
0x0000db26a4ac0ed5LLU, 0x0000dc9c95fb5d37LLU,
0x0000de12a7b26300LLU, 0x0000df88d9d3ef15LLU,
0x0000e0ff2c62d096LLU, 0x0000e2759f61d6e6LLU,
0x0000e3ec32d3d1a2LLU, 0x0000e562e6bb90a8LLU,
0x0000e6d9bb1be415LLU, 0x0000e850aff79c41LLU,
0x0000e9c7c55189c6LLU, 0x0000eb3efb2c7d7bLLU,
0x0000ecb6518b4874LLU, 0x0000ee2dc870bc07LLU,
0x0000efa55fdfa9c5LLU, 0x0000f11d17dae37fLLU,
0x0000f294f0653b45LLU, 0x0000f40ce9818366LLU,
0x0000f58503328e6dLLU, 0x0000f6fd3d7b2f26LLU,
0x0000f875985e389bLLU, 0x0000f9ee13de7e15LLU,
0x0000fb66affed31bLLU, 0x0000fcdf6cc20b73LLU,
0x0000fe584a2afb21LLU, 0x0000ffd1483c7669LLU,
0x0001014a66f951ceLLU, 0x000102c3a6646210LLU,
0x0001043d06807c2fLLU, 0x000105b68750756aLLU,
0x0001073028d7233eLLU, 0x000108a9eb175b68LLU,
0x00010a23ce13f3e2LLU, 0x00010b9dd1cfc2e8LLU,
0x00010d17f64d9ef1LLU, 0x00010e923b905eb7LLU,
0x0001100ca19ad92fLLU, 0x00011187286fe591LLU,
0x00011301d0125b51LLU, 0x0001147c98851223LLU,
0x000115f781cae1faLLU, 0x000117728be6a309LLU,
0x000118edb6db2dc1LLU, 0x00011a6902ab5ad2LLU,
0x00011be46f5a032cLLU, 0x00011d5ffce9fffeLLU,
0x00011edbab5e2ab6LLU, 0x000120577ab95d00LLU,
0x000121d36afe70c9LLU, 0x0001234f7c30403eLLU,
0x000124cbae51a5c8LLU, 0x0001264801657c12LLU,
0x000127c4756e9e05LLU, 0x000129410a6fe6cbLLU,
0x00012abdc06c31ccLLU, 0x00012c3a97665aafLLU,
0x00012db78f613d5bLLU, 0x00012f34a85fb5f7LLU,
0x000130b1e264a0e9LLU, 0x0001322f3d72dad5LLU,
0x000133acb98d40a2LLU, 0x0001352a56b6af72LLU,
0x000136a814f204abLLU, 0x00013825f4421defLLU,
0x000139a3f4a9d922LLU, 0x00013b22162c1466LLU,
0x00013ca058cbae1eLLU, 0x00013e1ebc8b84ebLLU,
0x00013f9d416e77afLLU, 0x0001411be777658bLLU,
0x0001429aaea92de0LLU, 0x000144199706b04eLLU,
0x00014598a092ccb7LLU, 0x00014717cb50633aLLU,
0x0001489717425438LLU, 0x00014a16846b804fLLU,
0x00014b9612cec861LLU, 0x00014d15c26f0d8bLLU,
0x00014e95934f312eLLU, 0x00015015857214e8LLU,
0x0001519598da9a9aLLU, 0x00015315cd8ba461LLU,
0x000154962388149eLLU, 0x000156169ad2cdefLLU,
0x00015797336eb333LLU, 0x00015917ed5ea789LLU,
0x00015a98c8a58e51LLU, 0x00015c19c5464b2aLLU,
0x00015d9ae343c1f2LLU, 0x00015f1c22a0d6caLLU,
0x0001609d83606e12LLU, 0x0001621f05856c68LLU,
0x000163a0a912b6acLLU, 0x000165226e0b3200LLU,
0x000166a45471c3c2LLU, 0x000168265c495194LLU,
0x000169a88594c157LLU, 0x00016b2ad056f92cLLU,
0x00016cad3c92df73LLU, 0x00016e2fca4b5ad0LLU,
0x00016fb279835224LLU, 0x000171354a3dac91LLU,
0x000172b83c7d517bLLU, 0x0001743b50452885LLU,
0x000175be85981992LLU, 0x00017741dc790cc7LLU,
0x000178c554eaea89LLU, 0x00017a48eef09b7dLLU,
0x00017bccaa8d0888LLU, 0x00017d5087c31ad2LLU,
0x00017ed48695bbc0LLU, 0x00018058a707d4fbLLU,
0x000181dce91c506aLLU, 0x000183614cd61836LLU,
0x000184e5d23816c9LLU, 0x0001866a794536ccLLU,
0x000187ef4200632bLLU, 0x000189742c6c8711LLU,
0x00018af9388c8deaLLU, 0x00018c7e66636363LLU,
0x00018e03b5f3f36bLLU, 0x00018f8927412a2fLLU,
0x0001910eba4df41fLLU, 0x000192946f1d3decLLU,
0x0001941a45b1f487LLU, 0x000195a03e0f0522LLU,
0x0001972658375d2fLLU, 0x000198ac942dea64LLU,
0x00019a32f1f59ab4LLU, 0x00019bb971915c57LLU,
0x00019d4013041dc2LLU, 0x00019ec6d650cdadLLU,
0x0001a04dbb7a5b13LLU, 0x0001a1d4c283b52dLLU,
0x0001a35beb6fcb75LLU, 0x0001a4e336418da9LLU,
0x0001a66aa2fbebc7LLU, 0x0001a7f231a1d60cLLU,
0x0001a979e2363cf8LLU, 0x0001ab01b4bc114dLLU,
0x0001ac89a936440dLLU, 0x0001ae11bfa7c67bLLU,
0x0001af99f8138a1cLLU, 0x0001b122527c80b7LLU,
0x0001b2aacee59c53LLU, 0x0001b4336d51cf38LLU,
0x0001b5bc2dc40bf0LLU, 0x0001b745103f4548LLU,
0x0001b8ce14c66e4cLLU, 0x0001ba573b5c7a4bLLU,
0x0001bbe084045cd4LLU, 0x0001bd69eec109b8LLU,
0x0001bef37b95750bLLU, 0x0001c07d2a849321LLU,
0x0001c206fb91588fLLU, 0x0001c390eebeba2eLLU,
0x0001c51b040fad15LLU, 0x0001c6a53b8726a0LLU,
0x0001c82f95281c6bLLU, 0x0001c9ba10f58454LLU,
0x0001cb44aef2547aLLU, 0x0001cccf6f21833fLLU,
0x0001ce5a51860746LLU, 0x0001cfe55622d773LLU,
0x0001d1707cfaeaedLLU, 0x0001d2fbc611391cLLU,
0x0001d4873168b9aaLLU, 0x0001d612bf046484LLU,
0x0001d79e6ee731d7LLU, 0x0001d92a41141a12LLU,
0x0001dab6358e15e8LLU, 0x0001dc424c581e4bLLU,
0x0001ddce85752c71LLU, 0x0001df5ae0e839d2LLU,
0x0001e0e75eb44027LLU, 0x0001e273fedc396bLLU,
0x0001e400c1631fdbLLU, 0x0001e58da64bedf8LLU,
0x0001e71aad999e82LLU, 0x0001e8a7d74f2c7eLLU,
0x0001ea35236f9330LLU, 0x0001ebc291fdce22LLU,
0x0001ed5022fcd91dLLU, 0x0001eeddd66fb02dLLU,
0x0001f06bac594fa0LLU, 0x0001f1f9a4bcb409LLU,
0x0001f387bf9cda38LLU, 0x0001f515fcfcbf45LLU,
0x0001f6a45cdf6085LLU, 0x0001f832df47bb94LLU,
0x0001f9c18438ce4dLLU, 0x0001fb504bb596ceLLU,
0x0001fcdf35c1137aLLU, 0x0001fe6e425e42f3LLU,
0x0001fffd7190241eLLU, 0x0002018cc359b625LLU,
0x0002031c37bdf872LLU, 0x000204abcebfeab3LLU,
0x0002063b88628cd6LLU, 0x000207cb64a8df0fLLU,
0x0002095b6395e1d2LLU, 0x00020aeb852c95d7LLU,
0x00020c7bc96ffc18LLU, 0x00020e0c306315d1LLU,
0x00020f9cba08e483LLU, 0x0002112d666469efLLU,
0x000212be3578a819LLU, 0x0002144f2748a14aLLU,
0x000215e03bd7580cLLU, 0x000217717327cf2cLLU,
0x00021902cd3d09b9LLU, 0x00021a944a1a0b07LLU,
0x00021c25e9c1d6aaLLU, 0x00021db7ac37707bLLU,
0x00021f49917ddc96LLU, 0x000220db99981f59LLU,
0x0002226dc4893d64LLU, 0x0002240012543b9cLLU,
0x0002259282fc1f27LLU, 0x000227251683ed71LLU,
0x000228b7cceeac25LLU, 0x00022a4aa63f6134LLU,
0x00022bdda27912d1LLU, 0x00022d70c19ec773LLU,
0x00022f0403b385d2LLU, 0x0002309768ba54ecLLU,
0x0002322af0b63bffLLU, 0x000233be9baa428fLLU,
0x0002355269997062LLU, 0x000236e65a86cd81LLU,
0x0002387a6e756238LLU, 0x00023a0ea5683718LLU,
0x00023ba2ff6254f4LLU, 0x00023d377c66c4e1LLU,
0x00023ecc1c78903aLLU, 0x00024060df9ac09bLLU,
0x000241f5c5d05fe6LLU, 0x0002438acf1c783eLLU,
0x0002451ffb82140aLLU, 0x000246b54b043df6LLU,
0x0002484abda600efLLU, 0x000249e0536a6828LLU,
0x00024b760c547f15LLU, 0x00024d0be8675170LLU,
0x00024ea1e7a5eb35LLU, 0x000250380a1358a3LLU,
0x000251ce4fb2a63fLLU, 0x00025364b886e0d0LLU,
0x000254fb44931561LLU, 0x00025691f3da5140LLU,
0x00025828c65fa1ffLLU, 0x000259bfbc261576LLU,
0x00025b56d530b9bcLLU, 0x00025cee11829d31LLU,
0x00025e85711ece75LLU, 0x0002601cf4085c6eLLU,
0x000261b49a425645LLU, 0x0002634c63cfcb65LLU,
0x000264e450b3cb82LLU, 0x0002667c60f1668eLLU,
0x00026814948bacc3LLU, 0x000269aceb85ae9dLLU,
0x00026b4565e27cddLLU, 0x00026cde03a52888LLU,
0x00026e76c4d0c2e5LLU, 0x0002700fa9685d82LLU,
0x000271a8b16f0a30LLU, 0x00027341dce7db03LLU,
0x000274db2bd5e254LLU, 0x000276749e3c32c0LLU,
0x0002780e341ddf29LLU, 0x000279a7ed7dfab5LLU,
0x00027b41ca5f98cbLLU, 0x00027cdbcac5cd1cLLU,
0x00027e75eeb3ab98LLU, 0x00028010362c4877LLU,
0x000281aaa132b832LLU, 0x000283452fca0f8aLLU,
0x000284dfe1f56381LLU, 0x0002867ab7b7c95fLLU,
0x00028815b11456b1LLU, 0x000289b0ce0e2147LLU,
0x00028b4c0ea83f36LLU, 0x00028ce772e5c6d7LLU,
0x00028e82fac9cecaLLU, 0x0002901ea6576df0LLU,
0x000291ba7591bb70LLU, 0x00029356687bceb6LLU,
0x000294f27f18bf72LLU, 0x0002968eb96ba59aLLU,
0x0002982b17779965LLU, 0x000299c7993fb354LLU,
0x00029b643ec70c27LLU, 0x00029d010810bce8LLU,
0x00029e9df51fdee1LLU, 0x0002a03b05f78ba5LLU,
0x0002a1d83a9add08LLU, 0x0002a375930ced25LLU,
0x0002a5130f50d65cLLU, 0x0002a6b0af69b350LLU,
0x0002a84e735a9eecLLU, 0x0002a9ec5b26b45bLLU,
0x0002ab8a66d10f13LLU, 0x0002ad28965ccac9LLU,
0x0002aec6e9cd037bLLU, 0x0002b0656124d56bLLU,
0x0002b203fc675d1fLLU, 0x0002b3a2bb97b763LLU,
0x0002b5419eb90148LLU, 0x0002b6e0a5ce5823LLU,
0x0002b87fd0dad990LLU, 0x0002ba1f1fe1a36eLLU,
0x0002bbbe92e5d3e3LLU, 0x0002bd5e29ea8959LLU,
0x0002befde4f2e280LLU, 0x0002c09dc401fe4cLLU,
0x0002c23dc71afbf7LLU, 0x0002c3ddee40fb02LLU,
0x0002c57e39771b2fLLU, 0x0002c71ea8c07c89LLU,
0x0002c8bf3c203f5fLLU, 0x0002ca5ff3998446LLU,
0x0002cc00cf2f6c18LLU, 0x0002cda1cee517f3LLU,
0x0002cf42f2bda93dLLU, 0x0002d0e43abc41a0LLU,
0x0002d285a6e4030bLLU, 0x0002d42737380fb4LLU,
0x0002d5c8ebbb8a15LLU, 0x0002d76ac47194efLLU,
0x0002d90cc15d5346LLU, 0x0002daaee281e867LLU,
0x0002dc5127e277e3LLU, 0x0002ddf39182258fLLU,
0x0002df961f641589LLU, 0x0002e138d18b6c33LLU,
0x0002e2dba7fb4e33LLU, 0x0002e47ea2b6e077LLU,
0x0002e621c1c14833LLU, 0x0002e7c5051daadfLLU,
0x0002e9686ccf2e3bLLU, 0x0002eb0bf8d8f849LLU,
0x0002ecafa93e2f56LLU, 0x0002ee537e01f9f1LLU,
0x0002eff777277ef0LLU, 0x0002f19b94b1e570LLU,
0x0002f33fd6a454d2LLU, 0x0002f4e43d01f4beLLU,
0x0002f688c7cded23LLU, 0x0002f82d770b6634LLU,
0x0002f9d24abd886bLLU, 0x0002fb7742e77c89LLU,
0x0002fd1c5f8c6b93LLU, 0x0002fec1a0af7ed6LLU,
0x000300670653dfe4LLU, 0x0003020c907cb896LLU,
0x000303b23f2d330bLLU, 0x00030558126879a7LLU,
0x000306fe0a31b715LLU, 0x000308a4268c1648LLU,
0x00030a4a677ac276LLU, 0x00030bf0cd00e71eLLU,
0x00030d975721b004LLU, 0x00030f3e05e04933LLU,
0x000310e4d93fdefbLLU, 0x0003128bd1439df5LLU,
0x00031432edeeb2fdLLU, 0x000315da2f444b39LLU,
0x0003178195479413LLU, 0x000319291ffbbb3cLLU,
0x00031ad0cf63eeacLLU, 0x00031c78a3835ca1LLU,
0x00031e209c5d33a0LLU, 0x00031fc8b9f4a275LLU,
0x00032170fc4cd831LLU, 0x000323196369042eLLU,
0x000324c1ef4c560aLLU, 0x0003266a9ff9fdacLLU,
0x0003281375752b40LLU, 0x000329bc6fc10f3aLLU,
0x00032b658ee0da54LLU, 0x00032d0ed2d7bd8eLLU,
0x00032eb83ba8ea32LLU, 0x00033061c95791ccLLU,
0x0003320b7be6e633LLU, 0x000333b5535a1984LLU,
0x0003355f4fb45e20LLU, 0x0003370970f8e6b3LLU,
0x000338b3b72ae62dLLU, 0x00033a5e224d8fc7LLU,
0x00033c08b26416ffLLU, 0x00033db36771af9cLLU,
0x00033f5e41798daaLLU, 0x00034109407ee57fLLU,
0x000342b46484ebb4LLU, 0x0003445fad8ed52cLLU,
0x0003460b1b9fd712LLU, 0x000347b6aebb26d5LLU,
0x0003496266e3fa2dLLU, 0x00034b0e441d8719LLU,
0x00034cba466b03e1LLU, 0x00034e666dcfa710LLU,
0x00035012ba4ea77dLLU, 0x000351bf2beb3c42LLU,
0x0003536bc2a89cc4LLU, 0x000355187e8a00aeLLU,
0x000356c55f929ff1LLU, 0x0003587265c5b2c7LLU,
0x00035a1f912671b1LLU, 0x00035bcce1b81578LLU,
0x00035d7a577dd72bLLU, 0x00035f27f27af022LLU,
0x000360d5b2b299fcLLU, 0x0003628398280ea0LLU,
0x00036431a2de883bLLU, 0x000365dfd2d94143LLU,
0x0003678e281b7475LLU, 0x0003693ca2a85cd7LLU,
0x00036aeb428335b4LLU, 0x00036c9a07af3aa2LLU,
0x00036e48f22fa77cLLU, 0x00036ff80207b865LLU,
0x000371a7373aa9cbLLU, 0x0003735691cbb85fLLU,
0x0003750611be211cLLU, 0x000376b5b7152147LLU,
0x0003786581d3f669LLU, 0x00037a1571fdde55LLU,
0x00037bc587961726LLU, 0x00037d75c29fdf3fLLU,
0x00037f26231e754aLLU, 0x000380d6a9151839LLU,
0x0003828754870746LLU, 0x00038438257781f6LLU,
0x000385e91be9c811LLU, 0x0003879a37e119abLLU,
0x0003894b7960b71fLLU, 0x00038afce06be10eLLU,
0x00038cae6d05d866LLU, 0x00038e601f31de57LLU,
0x00039011f6f3345fLLU, 0x000391c3f44d1c41LLU,
0x000393761742d808LLU, 0x000395285fd7aa09LLU,
0x000396dace0ed4e1LLU, 0x0003988d61eb9b74LLU,
0x00039a401b7140efLLU, 0x00039bf2faa308c8LLU,
0x00039da5ff8436bcLLU, 0x00039f592a180ed3LLU,
0x0003a10c7a61d55bLLU, 0x0003a2bff064ceecLLU,
0x0003a4738c244064LLU, 0x0003a6274da36eedLLU,
0x0003a7db34e59ff7LLU, 0x0003a98f41ee193cLLU,
0x0003ab4374c020bdLLU, 0x0003acf7cd5efcc6LLU,
0x0003aeac4bcdf3eaLLU, 0x0003b060f0104d04LLU,
0x0003b215ba294f39LLU, 0x0003b3caaa1c41f7LLU,
0x0003b57fbfec6cf4LLU, 0x0003b734fb9d182fLLU,
0x0003b8ea5d318befLLU, 0x0003ba9fe4ad10c4LLU,
0x0003bc559212ef89LLU, 0x0003be0b6566715fLLU,
0x0003bfc15eaadfb1LLU, 0x0003c1777de38434LLU,
0x0003c32dc313a8e5LLU, 0x0003c4e42e3e9808LLU,
0x0003c69abf679c2eLLU, 0x0003c8517692002eLLU,
0x0003ca0853c10f28LLU, 0x0003cbbf56f81488LLU,
0x0003cd76803a5c00LLU, 0x0003cf2dcf8b318dLLU,
0x0003d0e544ede173LLU, 0x0003d29ce065b842LLU,
0x0003d454a1f602d0LLU, 0x0003d60c89a20e40LLU,
0x0003d7c4976d27faLLU, 0x0003d97ccb5a9db3LLU,
0x0003db35256dbd67LLU, 0x0003dceda5a9d55cLLU,
0x0003dea64c123422LLU, 0x0003e05f18aa2891LLU,
0x0003e2180b7501ccLLU, 0x0003e3d124760f3dLLU,
0x0003e58a63b0a09bLLU, 0x0003e743c92805e2LLU,
0x0003e8fd54df8f5cLLU, 0x0003eab706da8d99LLU,
0x0003ec70df1c5175LLU, 0x0003ee2adda82c14LLU,
0x0003efe502816ee3LLU, 0x0003f19f4dab6b9cLLU,
0x0003f359bf29743fLLU, 0x0003f51456fedb16LLU,
0x0003f6cf152ef2b8LLU, 0x0003f889f9bd0e02LLU,
0x0003fa4504ac801cLLU, 0x0003fc0036009c78LLU,
0x0003fdbb8dbcb6d2LLU, 0x0003ff770be4232fLLU,
0x00040132b07a35dfLLU, 0x000402ee7b82437bLLU,
0x000404aa6cffa0e5LLU, 0x0004066684f5a34bLLU,
0x00040822c367a024LLU, 0x000409df2858ed31LLU,
0x00040b9bb3cce07cLLU, 0x00040d5865c6d05aLLU,
0x00040f153e4a136aLLU, 0x000410d23d5a0095LLU,
0x0004128f62f9ef0eLLU, 0x0004144caf2d3653LLU,
0x0004160a21f72e2aLLU, 0x000417c7bb5b2ea5LLU,
0x000419857b5c901fLLU, 0x00041b4361feab3fLLU,
0x00041d016f44d8f5LLU, 0x00041ebfa332727aLLU,
0x0004207dfdcad153LLU, 0x0004223c7f114f50LLU,
0x000423fb2709468aLLU, 0x000425b9f5b61163LLU,
0x00042778eb1b0a8bLLU, 0x00042938073b8cf9LLU,
0x00042af74a1af3f1LLU, 0x00042cb6b3bc9b00LLU,
0x00042e764423ddfdLLU, 0x00043035fb54190bLLU,
0x000431f5d950a897LLU, 0x000433b5de1ce958LLU,
0x0004357609bc3850LLU, 0x000437365c31f2ccLLU,
0x000438f6d5817663LLU, 0x00043ab775ae20f7LLU,
0x00043c783cbb50b4LLU, 0x00043e392aac6413LLU,
0x00043ffa3f84b9d4LLU, 0x000441bb7b47b105LLU,
0x0004437cddf8a8feLLU, 0x0004453e679b0160LLU,
0x0004470018321a1aLLU, 0x000448c1efc15362LLU,
0x00044a83ee4c0dbdLLU, 0x00044c4613d5a9f9LLU,
0x00044e086061892dLLU, 0x00044fcad3f30cbeLLU,
0x0004518d6e8d965bLLU, 0x00045350303487feLLU,
0x0004551318eb43ecLLU, 0x000456d628b52cb4LLU,
0x000458995f95a532LLU, 0x00045a5cbd90108bLLU,
0x00045c2042a7d232LLU, 0x00045de3eee04de2LLU,
0x00045fa7c23ce7a4LLU, 0x0004616bbcc103cbLLU,
0x0004632fde7006f4LLU, 0x000464f4274d5609LLU,
0x000466b8975c563eLLU, 0x0004687d2ea06d15LLU,
0x00046a41ed1d0057LLU, 0x00046c06d2d5761dLLU,
0x00046dcbdfcd34c8LLU, 0x00046f911407a306LLU,
0x000471566f8827d0LLU, 0x0004731bf2522a6aLLU,
0x000474e19c691265LLU, 0x000476a76dd0479cLLU,
0x0004786d668b3237LLU, 0x00047a33869d3aa7LLU,
0x00047bf9ce09c9abLLU, 0x00047dc03cd4484eLLU,
0x00047f86d3001fe5LLU, 0x0004814d9090ba11LLU,
0x00048314758980bfLLU, 0x000484db81edde28LLU,
0x000486a2b5c13cd0LLU, 0x0004886a11070788LLU,
0x00048a3193c2a96cLLU, 0x00048bf93df78de3LLU,
0x00048dc10fa920a1LLU, 0x00048f8908dacda7LLU,
0x000491512990013fLLU, 0x0004931971cc2802LLU,
0x000494e1e192aed2LLU, 0x000496aa78e702dfLLU,
0x0004987337cc91a5LLU, 0x00049a3c1e46c8ebLLU,
0x00049c052c5916c4LLU, 0x00049dce6206e991LLU,
0x00049f97bf53affdLLU, 0x0004a1614442d900LLU,
0x0004a32af0d7d3deLLU, 0x0004a4f4c5161028LLU,
0x0004a6bec100fdbaLLU, 0x0004a888e49c0cbdLLU,
0x0004aa532feaada6LLU, 0x0004ac1da2f05135LLU,
0x0004ade83db0687aLLU, 0x0004afb3002e64ceLLU,
0x0004b17dea6db7d7LLU, 0x0004b348fc71d388LLU,
0x0004b514363e2a20LLU, 0x0004b6df97d62e2cLLU,
0x0004b8ab213d5283LLU, 0x0004ba76d2770a49LLU,
0x0004bc42ab86c8f1LLU, 0x0004be0eac700237LLU,
0x0004bfdad5362a27LLU, 0x0004c1a725dcb517LLU,
0x0004c3739e6717aaLLU, 0x0004c5403ed8c6d2LLU,
0x0004c70d073537caLLU, 0x0004c8d9f77fe01dLLU,
0x0004caa70fbc35a1LLU, 0x0004cc744fedae78LLU,
0x0004ce41b817c114LLU, 0x0004d00f483de431LLU,
0x0004d1dd00638ed8LLU, 0x0004d3aae08c3860LLU,
0x0004d578e8bb586bLLU, 0x0004d74718f466ecLLU,
0x0004d915713adc1eLLU, 0x0004dae3f192308cLLU,
0x0004dcb299fddd0dLLU, 0x0004de816a815ac6LLU,
0x0004e05063202327LLU, 0x0004e21f83ddafefLLU,
0x0004e3eeccbd7b2aLLU, 0x0004e5be3dc2ff30LLU,
0x0004e78dd6f1b6a6LLU, 0x0004e95d984d1c81LLU,
0x0004eb2d81d8abffLLU, 0x0004ecfd9397e0aeLLU,
0x0004eecdcd8e3669LLU, 0x0004f09e2fbf2958LLU,
0x0004f26eba2e35f0LLU, 0x0004f43f6cded8f4LLU,
0x0004f61047d48f73LLU, 0x0004f7e14b12d6cbLLU,
0x0004f9b2769d2ca7LLU, 0x0004fb83ca770efeLLU,
0x0004fd5546a3fc17LLU, 0x0004ff26eb277284LLU,
0x000500f8b804f127LLU, 0x000502caad3ff72dLLU,
0x0005049ccadc0412LLU, 0x0005066f10dc97a0LLU,
0x000508417f4531eeLLU, 0x00050a1416195360LLU,
0x00050be6d55c7ca9LLU, 0x00050db9bd122ec9LLU,
0x00050f8ccd3deb0dLLU, 0x0005116005e33311LLU,
0x00051333670588bfLLU, 0x00051506f0a86e4eLLU,
0x000516daa2cf6642LLU, 0x000518ae7d7df36eLLU,
0x00051a8280b798f4LLU, 0x00051c56ac7fda42LLU,
0x00051e2b00da3b14LLU, 0x00051fff7dca3f75LLU,
0x000521d423536bbeLLU, 0x000523a8f1794495LLU,
0x0005257de83f4eefLLU, 0x0005275307a9100eLLU,
0x000529284fba0d84LLU, 0x00052afdc075cd2fLLU,
0x00052cd359dfd53dLLU, 0x00052ea91bfbac28LLU,
0x0005307f06ccd8baLLU, 0x000532551a56e20cLLU,
0x0005342b569d4f82LLU, 0x00053601bba3a8d1LLU,
0x000537d8496d75fcLLU, 0x000539aefffe3f54LLU,
0x00053b85df598d78LLU, 0x00053d5ce782e955LLU,
0x00053f34187ddc28LLU, 0x0005410b724def7bLLU,
0x000542e2f4f6ad27LLU, 0x000544baa07b9f54LLU,
0x0005469274e05078LLU, 0x0005486a72284b58LLU,
0x00054a4298571b06LLU, 0x00054c1ae7704ae4LLU,
0x00054df35f7766a3LLU, 0x00054fcc006ffa42LLU,
0x000551a4ca5d920fLLU, 0x0005537dbd43baa5LLU,
0x00055556d92600f1LLU, 0x000557301e07f22cLLU,
0x000559098bed1bdfLLU, 0x00055ae322d90be2LLU,
0x00055cbce2cf505bLLU, 0x00055e96cbd377beLLU,
0x00056070dde910d2LLU, 0x0005624b1913aaa8LLU,
0x000564257d56d4a2LLU, 0x000566000ab61e73LLU,
0x000567dac1351819LLU, 0x000569b5a0d751e4LLU,
0x00056b90a9a05c72LLU, 0x00056d6bdb93c8b0LLU,
0x00056f4736b527daLLU, 0x00057122bb080b7dLLU,
0x000572fe68900573LLU, 0x000574da3f50a7e5LLU,
0x000576b63f4d854cLLU, 0x00057892688a3072LLU,
0x00057a6ebb0a3c6dLLU, 0x00057c4b36d13ca5LLU,
0x00057e27dbe2c4cfLLU, 0x00058004aa4268f1LLU,
0x000581e1a1f3bd60LLU, 0x000583bec2fa56c1LLU,
0x0005859c0d59ca07LLU, 0x000587798115ac76LLU,
0x000589571e31939fLLU, 0x00058b34e4b11567LLU,
0x00058d12d497c7fdLLU, 0x00058ef0ede941e4LLU,
0x000590cf30a919edLLU, 0x000592ad9cdae738LLU,
0x0005948c32824135LLU, 0x0005966af1a2bfa5LLU,
0x00059849da3ffa96LLU, 0x00059a28ec5d8a69LLU,
0x00059c0827ff07ccLLU, 0x00059de78d280bbeLLU,
0x00059fc71bdc2f8eLLU, 0x0005a1a6d41f0cdbLLU,
0x0005a386b5f43d92LLU, 0x0005a566c15f5bf3LLU,
0x0005a746f664028bLLU, 0x0005a9275505cc38LLU,
0x0005ab07dd485429LLU, 0x0005ace88f2f35dcLLU,
0x0005aec96abe0d1fLLU, 0x0005b0aa6ff87610LLU,
0x0005b28b9ee20d1eLLU, 0x0005b46cf77e6f06LLU,
0x0005b64e79d138d8LLU, 0x0005b83025de07f2LLU,
0x0005ba11fba87a03LLU, 0x0005bbf3fb342d0aLLU,
0x0005bdd62484bf56LLU, 0x0005bfb8779dcf88LLU,
0x0005c19af482fc8fLLU, 0x0005c37d9b37e5abLLU,
0x0005c5606bc02a6dLLU, 0x0005c743661f6ab6LLU,
0x0005c9268a5946b7LLU, 0x0005cb09d8715ef3LLU,
0x0005cced506b543aLLU, 0x0005ced0f24ac7b2LLU,
0x0005d0b4be135accLLU, 0x0005d298b3c8af4cLLU,
0x0005d47cd36e6747LLU, 0x0005d6611d082522LLU,
0x0005d84590998b93LLU, 0x0005da2a2e263d9fLLU,
0x0005dc0ef5b1de9eLLU, 0x0005ddf3e7401238LLU,
0x0005dfd902d47c65LLU, 0x0005e1be4872c16dLLU,
0x0005e3a3b81e85ecLLU, 0x0005e58951db6eccLLU,
0x0005e76f15ad2148LLU, 0x0005e955039742eeLLU,
0x0005eb3b1b9d799aLLU, 0x0005ed215dc36b7bLLU,
0x0005ef07ca0cbf0fLLU, 0x0005f0ee607d1b29LLU,
0x0005f2d5211826e8LLU, 0x0005f4bc0be189beLLU,
0x0005f6a320dceb71LLU, 0x0005f88a600df413LLU,
0x0005fa71c9784c0bLLU, 0x0005fc595d1f9c0fLLU,
0x0005fe411b078d26LLU, 0x000600290333c8acLLU,
0x0006021115a7f849LLU, 0x000603f95267c5f9LLU,
0x000605e1b976dc09LLU, 0x000607ca4ad8e517LLU,
0x000609b306918c13LLU, 0x00060b9beca47c3eLLU,
0x00060d84fd15612aLLU, 0x00060f6e37e7e6bbLLU,
0x000611579d1fb925LLU, 0x000613412cc084f1LLU,
0x0006152ae6cdf6f4LLU, 0x00061714cb4bbc5bLLU,
0x000618feda3d829fLLU, 0x00061ae913a6f78dLLU,
0x00061cd3778bc944LLU, 0x00061ebe05efa634LLU,
0x000620a8bed63d1fLLU, 0x00062293a2433d18LLU,
0x0006247eb03a5585LLU, 0x00062669e8bf361bLLU,
0x000628554bd58ee5LLU, 0x00062a40d981103cLLU,
0x00062c2c91c56acdLLU, 0x00062e1874a64f97LLU,
0x0006300482276fe8LLU, 0x000631f0ba4c7d64LLU,
0x000633dd1d1929fdLLU, 0x000635c9aa9127fbLLU,
0x000637b662b829f5LLU, 0x000639a34591e2d5LLU,
0x00063b90532205d8LLU, 0x00063d7d8b6c468aLLU,
0x00063f6aee7458cdLLU, 0x000641587c3df0d3LLU,
0x0006434634ccc320LLU, 0x000645341824848bLLU,
0x000647222648ea3dLLU, 0x000649105f3da9b2LLU,
0x00064afec30678b7LLU, 0x00064ced51a70d6bLLU,
0x00064edc0b231e41LLU, 0x000650caef7e61fdLLU,
0x000652b9febc8fb7LLU, 0x000654a938e15ed7LLU,
0x000656989df08719LLU, 0x000658882dedc08cLLU,
0x00065a77e8dcc390LLU, 0x00065c67cec148d8LLU,
0x00065e57df9f096bLLU, 0x000660481b79bea1LLU,
0x0006623882552225LLU, 0x000664291434edf4LLU,
0x00066619d11cdc5fLLU, 0x0006680ab910a809LLU,
0x000669fbcc140be7LLU, 0x00066bed0a2ac343LLU,
0x00066dde735889b8LLU, 0x00066fd007a11b33LLU,
0x000671c1c70833f6LLU, 0x000673b3b1919094LLU,
0x000675a5c740edf5LLU, 0x00067798081a0951LLU,
0x0006798a7420a036LLU, 0x00067b7d0b587083LLU,
0x00067d6fcdc5386aLLU, 0x00067f62bb6ab672LLU,
0x00068155d44ca973LLU, 0x00068349186ed099LLU,
0x0006853c87d4eb62LLU, 0x000687302282b9a1LLU,
0x00068923e87bfb7aLLU, 0x00068b17d9c47168LLU,
0x00068d0bf65fdc34LLU, 0x00068f003e51fcfeLLU,
0x000690f4b19e9538LLU, 0x000692e9504966a8LLU,
0x000694de1a563367LLU, 0x000696d30fc8bde0LLU,
0x000698c830a4c8d4LLU, 0x00069abd7cee1755LLU,
0x00069cb2f4a86ccaLLU, 0x00069ea897d78ceeLLU,
0x0006a09e667f3bcdLLU, 0x0006a29460a33dc8LLU,
0x0006a48a86475795LLU, 0x0006a680d76f4e3cLLU,
0x0006a877541ee718LLU, 0x0006aa6dfc59e7d9LLU,
0x0006ac64d0241683LLU, 0x0006ae5bcf81396cLLU,
0x0006b052fa75173eLLU, 0x0006b24a510376f9LLU,
0x0006b441d3301feeLLU, 0x0006b63980fed9c4LLU,
0x0006b8315a736c75LLU, 0x0006ba295f91a04eLLU,
0x0006bc21905d3df0LLU, 0x0006be19ecda0e53LLU,
0x0006c012750bdabfLLU, 0x0006c20b28f66cd2LLU,
0x0006c404089d8e7dLLU, 0x0006c5fd14050a07LLU,
0x0006c7f64b30aa09LLU, 0x0006c9efae243971LLU,
0x0006cbe93ce38381LLU, 0x0006cde2f77253cfLLU,
0x0006cfdcddd47645LLU, 0x0006d1d6f00db723LLU,
0x0006d3d12e21e2fbLLU, 0x0006d5cb9814c6b5LLU,
0x0006d7c62dea2f8aLLU, 0x0006d9c0efa5eb0dLLU,
0x0006dbbbdd4bc720LLU, 0x0006ddb6f6df91fcLLU,
0x0006dfb23c651a2fLLU, 0x0006e1adade02e99LLU,
0x0006e3a94b549e71LLU, 0x0006e5a514c63941LLU,
0x0006e7a10a38cee8LLU, 0x0006e99d2bb02f99LLU,
0x0006eb9979302bddLLU, 0x0006ed95f2bc9490LLU,
0x0006ef9298593ae5LLU, 0x0006f18f6a09f060LLU,
0x0006f38c67d286ddLLU, 0x0006f58991b6d08bLLU,
0x0006f786e7ba9fefLLU, 0x0006f98469e1c7e1LLU,
0x0006fb8218301b90LLU, 0x0006fd7ff2a96e7eLLU,
0x0006ff7df9519484LLU, 0x0007017c2c2c61cdLLU,
0x0007037a8b3daadbLLU, 0x0007057916894485LLU,
0x00070777ce1303f6LLU, 0x00070976b1debeaeLLU,
0x00070b75c1f04a84LLU, 0x00070d74fe4b7da2LLU,
0x00070f7466f42e87LLU, 0x00071173fbee3409LLU,
0x00071373bd3d6551LLU, 0x00071573aae599dfLLU,
0x00071773c4eaa988LLU, 0x000719740b506c74LLU,
0x00071b747e1abb24LLU, 0x00071d751d4d6e6bLLU,
0x00071f75e8ec5f74LLU, 0x00072176e0fb67bdLLU,
0x00072378057e611aLLU, 0x00072579567925b6LLU,
0x0007277ad3ef9011LLU, 0x0007297c7de57afeLLU,
0x00072b7e545ec1a8LLU, 0x00072d80575f3f90LLU,
0x00072f8286ead08aLLU, 0x00073184e30550c2LLU,
0x000733876bb29cb8LLU, 0x0007358a20f69143LLU,
0x0007378d02d50b8fLLU, 0x000739901151e91eLLU,
0x00073b934c7107c7LLU, 0x00073d96b43645b8LLU,
0x00073f9a48a58174LLU, 0x0007419e09c299d3LLU,
0x000743a1f7916e05LLU, 0x000745a61215dd8eLLU,
0x000747aa5953c849LLU, 0x000749aecd4f0e66LLU,
0x00074bb36e0b906dLLU, 0x00074db83b8d2f39LLU,
0x00074fbd35d7cbfdLLU, 0x000751c25cef4843LLU,
0x000753c7b0d785e8LLU, 0x000755cd31946722LLU,
0x000757d2df29ce7cLLU, 0x000759d8b99b9ed7LLU,
0x00075bdec0edbb6bLLU, 0x00075de4f52407c5LLU,
0x00075feb564267c9LLU, 0x000761f1e44cbfb2LLU,
0x000763f89f46f40fLLU, 0x000765ff8734e9caLLU,
0x000768069c1a861dLLU, 0x00076a0dddfbae9fLLU,
0x00076c154cdc4937LLU, 0x00076e1ce8c03c29LLU,
0x00077024b1ab6e09LLU, 0x0007722ca7a1c5c8LLU,
0x00077434caa72aa7LLU, 0x0007763d1abf8444LLU,
0x0007784597eeba8fLLU, 0x00077a4e4238b5d0LLU,
0x00077c5719a15ea6LLU, 0x00077e601e2c9e07LLU,
0x000780694fde5d3fLLU, 0x00078272aeba85f2LLU,
0x0007847c3ac50219LLU, 0x00078685f401bc05LLU,
0x0007888fda749e5dLLU, 0x00078a99ee219421LLU,
0x00078ca42f0c88a5LLU, 0x00078eae9d396795LLU,
0x000790b938ac1cf6LLU, 0x000792c401689522LLU,
0x000794cef772bcc9LLU, 0x000796da1ace80f4LLU,
0x000798e56b7fcf03LLU, 0x00079af0e98a94adLLU,
0x00079cfc94f2bfffLLU, 0x00079f086dbc3f5eLLU,
0x0007a11473eb0187LLU, 0x0007a320a782f58cLLU,
0x0007a52d08880ad9LLU, 0x0007a73996fe3130LLU,
0x0007a94652e958aaLLU, 0x0007ab533c4d71b7LLU,
0x0007ad60532e6d20LLU, 0x0007af6d97903c04LLU,
0x0007b17b0976cfdbLLU, 0x0007b388a8e61a72LLU,
0x0007b59675e20defLLU, 0x0007b7a4706e9cd1LLU,
0x0007b9b2988fb9ecLLU, 0x0007bbc0ee49586dLLU,
0x0007bdcf719f6bd7LLU, 0x0007bfde2295e808LLU,
0x0007c1ed0130c132LLU, 0x0007c3fc0d73ebe2LLU,
0x0007c60b47635cf9LLU, 0x0007c81aaf0309b3LLU,
0x0007ca2a4456e7a3LLU, 0x0007cc3a0762ecb2LLU,
0x0007ce49f82b0f24LLU, 0x0007d05a16b34593LLU,
0x0007d26a62ff86f0LLU, 0x0007d47add13ca87LLU,
0x0007d68b84f407f8LLU, 0x0007d89c5aa4373eLLU,
0x0007daad5e2850acLLU, 0x0007dcbe8f844cebLLU,
0x0007decfeebc24feLLU, 0x0007e0e17bd3d240LLU,
0x0007e2f336cf4e62LLU, 0x0007e5051fb29370LLU,
0x0007e71736819bcdLLU, 0x0007e9297b406234LLU,
0x0007eb3bedf2e1b9LLU, 0x0007ed4e8e9d15c8LLU,
0x0007ef615d42fa24LLU, 0x0007f17459e88aebLLU,
0x0007f3878491c491LLU, 0x0007f59add42a3e4LLU,
0x0007f7ae63ff260aLLU, 0x0007f9c218cb4881LLU,
0x0007fbd5fbab091fLLU, 0x0007fdea0ca26616LLU,
0x0007fffe4bb55decLLU, 0x00080212b8e7ef82LLU,
0x00080427543e1a12LLU, 0x0008063c1dbbdd2dLLU,
0x00080851156538beLLU, 0x00080a663b3e2d09LLU,
0x00080c7b8f4abaa9LLU, 0x00080e91118ee294LLU,
0x000810a6c20ea617LLU, 0x000812bca0ce06d9LLU,
0x000814d2add106d9LLU, 0x000816e8e91ba871LLU,
0x000818ff52b1ee50LLU, 0x00081b15ea97db82LLU,
0x00081d2cb0d1736aLLU, 0x00081f43a562b9c4LLU,
0x0008215ac84fb2a6LLU, 0x00082372199c627dLLU,
0x00082589994cce13LLU, 0x000827a14764fa86LLU,
0x000829b923e8ed53LLU, 0x00082bd12edcac4aLLU,
0x00082de968443d9aLLU, 0x00083001d023a7c8LLU,
0x0008321a667ef1b2LLU, 0x000834332b5a2291LLU,
0x0008364c1eb941f7LLU, 0x0008386540a057cfLLU,
0x00083a7e91136c5dLLU, 0x00083c9810168840LLU,
0x00083eb1bdadb46dLLU, 0x000840cb99dcfa38LLU,
0x000842e5a4a8634aLLU, 0x000844ffde13f9a7LLU,
0x0008471a4623c7adLLU, 0x00084934dcdbd813LLU,
0x00084b4fa24035eaLLU, 0x00084d6a9654ec9eLLU,
0x00084f85b91e07f1LLU, 0x000851a10a9f9403LLU,
0x000853bc8add9d4cLLU, 0x000855d839dc309cLLU,
0x000857f4179f5b21LLU, 0x00085a10242b2a5fLLU,
0x00085c2c5f83ac36LLU, 0x00085e48c9aceedeLLU,
0x0008606562ab00ecLLU, 0x000862822a81f14dLLU,
0x0008649f2135cf48LLU, 0x000866bc46caaa7fLLU,
0x000868d99b4492edLLU, 0x00086af71ea798e7LLU,
0x00086d14d0f7cd1dLLU, 0x00086f32b239409bLLU,
0x00087150c27004c2LLU, 0x0008736f01a02b53LLU,
0x0008758d6fcdc666LLU, 0x000877ac0cfce86dLLU,
0x000879cad931a436LLU, 0x00087be9d4700ce9LLU,
0x00087e08febc3608LLU, 0x00088028581a3370LLU,
0x00088247e08e1957LLU, 0x00088467981bfc4fLLU,
0x000886877ec7f144LLU, 0x000888a794960d7cLLU,
0x00088ac7d98a6699LLU, 0x00088ce84da91297LLU,
0x00088f08f0f627cbLLU, 0x00089129c375bce8LLU,
0x0008934ac52be8f7LLU, 0x0008956bf61cc361LLU,
0x0008978d564c63e7LLU, 0x000899aee5bee2a4LLU,
0x00089bd0a478580fLLU, 0x00089df2927cdcfbLLU,
0x0008a014afd08a94LLU, 0x0008a236fc777a60LLU,
0x0008a4597875c644LLU, 0x0008a67c23cf887cLLU,
0x0008a89efe88dba1LLU, 0x0008aac208a5daa7LLU,
0x0008ace5422aa0dbLLU, 0x0008af08ab1b49e9LLU,
0x0008b12c437bf1d4LLU, 0x0008b3500b50b4fdLLU,
0x0008b574029db01eLLU, 0x0008b7982967004fLLU,
0x0008b9bc7fb0c302LLU, 0x0008bbe1057f1602LLU,
0x0008be05bad61778LLU, 0x0008c02a9fb9e5e9LLU,
0x0008c24fb42ea033LLU, 0x0008c474f8386591LLU,
0x0008c69a6bdb5598LLU, 0x0008c8c00f1b903aLLU,
0x0008cae5e1fd35c4LLU, 0x0008cd0be48466deLLU,
0x0008cf3216b5448cLLU, 0x0008d1587893f02eLLU,
0x0008d37f0a248b7fLLU, 0x0008d5a5cb6b3896LLU,
0x0008d7ccbc6c19e6LLU, 0x0008d9f3dd2b523dLLU,
0x0008dc1b2dad04c4LLU, 0x0008de42adf55502LLU,
0x0008e06a5e0866d9LLU, 0x0008e2923dea5e85LLU,
0x0008e4ba4d9f60a1LLU, 0x0008e6e28d2b9221LLU,
0x0008e90afc931857LLU, 0x0008eb339bda18f0LLU,
0x0008ed5c6b04b9f6LLU, 0x0008ef856a1721cdLLU,
0x0008f1ae99157736LLU, 0x0008f3d7f803e150LLU,
0x0008f60186e68793LLU, 0x0008f82b45c191d6LLU,
0x0008fa553499284bLLU, 0x0008fc7f5371737eLLU,
0x0008fea9a24e9c5cLLU, 0x000900d42134cc29LLU,
0x000902fed0282c8aLLU, 0x00090529af2ce77eLLU,
0x00090754be472760LLU, 0x0009097ffd7b16e8LLU,
0x00090bab6ccce12cLLU, 0x00090dd70c40b19bLLU,
0x00091002dbdab403LLU, 0x0009122edb9f148fLLU,
0x0009145b0b91ffc6LLU, 0x000916876bb7a289LLU,
0x000918b3fc142a19LLU, 0x00091ae0bcabc413LLU,
0x00091d0dad829e70LLU, 0x00091f3ace9ce785LLU,
0x000921681ffece05LLU, 0x00092395a1ac8100LLU,
0x000925c353aa2fe2LLU, 0x000927f135fc0a74LLU,
0x00092a1f48a640dcLLU, 0x00092c4d8bad039cLLU,
0x00092e7bff148396LLU, 0x000930aaa2e0f204LLU,
0x000932d977168083LLU, 0x000935087bb96107LLU,
0x00093737b0cdc5e5LLU, 0x000939671657e1ceLLU,
0x00093b96ac5be7d1LLU, 0x00093dc672de0b57LLU,
0x00093ff669e2802bLLU, 0x00094226916d7a71LLU,
0x00094456e9832eadLLU, 0x000946877227d1bfLLU,
0x000948b82b5f98e5LLU, 0x00094ae9152eb9b9LLU,
0x00094d1a2f996a33LLU, 0x00094f4b7aa3e0aaLLU,
0x0009517cf65253d1LLU, 0x000953aea2a8fab7LLU,
0x000955e07fac0ccdLLU, 0x000958128d5fc1dcLLU,
0x00095a44cbc8520fLLU, 0x00095c773ae9f5ecLLU,
0x00095ea9dac8e658LLU, 0x000960dcab695c95LLU,
0x0009630faccf9243LLU, 0x00096542deffc160LLU,
0x0009677641fe2446LLU, 0x000969a9d5cef5afLLU,
0x00096bdd9a7670b3LLU, 0x00096e118ff8d0c6LLU,
0x00097045b65a51baLLU, 0x0009727a0d9f2fc0LLU,
0x000974ae95cba768LLU, 0x000976e34ee3f59dLLU,
0x0009791838ec57abLLU, 0x00097b4d53e90b39LLU,
0x00097d829fde4e50LLU, 0x00097fb81cd05f52LLU,
0x000981edcac37d05LLU, 0x00098423a9bbe688LLU,
0x00098659b9bddb5bLLU, 0x0009888ffacd9b5dLLU,
0x00098ac66cef66c8LLU, 0x00098cfd10277e37LLU,
0x00098f33e47a22a2LLU, 0x0009916ae9eb9561LLU,
0x000993a220801829LLU, 0x000995d9883bed0dLLU,
0x0009981121235681LLU, 0x00099a48eb3a9753LLU,
0x00099c80e685f2b5LLU, 0x00099eb91309ac33LLU,
0x0009a0f170ca07baLLU, 0x0009a329ffcb4995LLU,
0x0009a562c011b66dLLU, 0x0009a79bb1a1934cLLU,
0x0009a9d4d47f2598LLU, 0x0009ac0e28aeb316LLU,
0x0009ae47ae3481edLLU, 0x0009b0816514d89eLLU,
0x0009b2bb4d53fe0dLLU, 0x0009b4f566f6397aLLU,
0x0009b72fb1ffd285LLU, 0x0009b96a2e75112eLLU,
0x0009bba4dc5a3dd3LLU, 0x0009bddfbbb3a131LLU,
0x0009c01acc858463LLU, 0x0009c2560ed430e4LLU,
0x0009c49182a3f090LLU, 0x0009c6cd27f90d9fLLU,
0x0009c908fed7d2aaLLU, 0x0009cb4507448aa9LLU,
0x0009cd81414380f2LLU, 0x0009cfbdacd9013dLLU,
0x0009d1fa4a09579dLLU, 0x0009d43718d8d089LLU,
0x0009d674194bb8d5LLU, 0x0009d8b14b665db3LLU,
0x0009daeeaf2d0cb8LLU, 0x0009dd2c44a413d6LLU,
0x0009df6a0bcfc15eLLU, 0x0009e1a804b46403LLU,
0x0009e3e62f564ad5LLU, 0x0009e6248bb9c545LLU,
0x0009e86319e32323LLU, 0x0009eaa1d9d6b4a0LLU,
0x0009ece0cb98ca4bLLU, 0x0009ef1fef2db513LLU,
0x0009f15f4499c647LLU, 0x0009f39ecbe14f97LLU,
0x0009f5de8508a311LLU, 0x0009f81e70141324LLU,
0x0009fa5e8d07f29eLLU, 0x0009fc9edbe894adLLU,
0x0009fedf5cba4ce0LLU, 0x000a01200f816f25LLU,
0x000a0360f4424fcbLLU, 0x000a05a20b01437fLLU,
0x000a07e353c29f50LLU, 0x000a0a24ce8ab8adLLU,
0x000a0c667b5de565LLU, 0x000a0ea85a407ba5LLU,
0x000a10ea6b36d1feLLU, 0x000a132cae453f5eLLU,
0x000a156f23701b15LLU, 0x000a17b1cabbbcd4LLU,
0x000a19f4a42c7ca9LLU, 0x000a1c37afc6b306LLU,
0x000a1e7aed8eb8bbLLU, 0x000a20be5d88e6fbLLU,
0x000a2301ffb99757LLU, 0x000a2545d42523c1LLU,
0x000a2789dacfe68cLLU, 0x000a29ce13be3a6cLLU,
0x000a2c127ef47a74LLU, 0x000a2e571c77021aLLU,
0x000a309bec4a2d33LLU, 0x000a32e0ee7257f5LLU,
0x000a352622f3def6LLU, 0x000a376b89d31f2fLLU,
0x000a39b1231475f7LLU, 0x000a3bf6eebc4108LLU,
0x000a3e3ceccede7cLLU, 0x000a40831d50accdLLU,
0x000a42c980460ad8LLU, 0x000a451015b357d9LLU,
0x000a4756dd9cf36eLLU, 0x000a499dd8073d96LLU,
0x000a4be504f696b1LLU, 0x000a4e2c646f5f7fLLU,
0x000a5073f675f924LLU, 0x000a52bbbb0ec521LLU,
0x000a5503b23e255dLLU, 0x000a574bdc087c1bLLU,
0x000a599438722c03LLU, 0x000a5bdcc77f981eLLU,
0x000a5e25893523d4LLU, 0x000a606e7d9732f1LLU,
0x000a62b7a4aa29a1LLU, 0x000a6500fe726c72LLU,
0x000a674a8af46052LLU, 0x000a69944a346a93LLU,
0x000a6bde3c36f0e6LLU, 0x000a6e286100595fLLU,
0x000a7072b8950a73LLU, 0x000a72bd42f96afaLLU,
0x000a75080031e22bLLU, 0x000a7752f042d7a1LLU,
0x000a799e1330b358LLU, 0x000a7be968ffddaeLLU,
0x000a7e34f1b4bf62LLU, 0x000a8080ad53c195LLU,
0x000a82cc9be14dcaLLU, 0x000a8518bd61cde7LLU,
0x000a876511d9ac32LLU, 0x000a89b1994d5354LLU,
0x000a8bfe53c12e59LLU, 0x000a8e4b4139a8acLLU,
0x000a909861bb2e1dLLU, 0x000a92e5b54a2adcLLU,
0x000a95333beb0b7eLLU, 0x000a9780f5a23cf6LLU,
0x000a99cee2742c9dLLU, 0x000a9c1d0265482cLLU,
0x000a9e6b5579fdbfLLU, 0x000aa0b9dbb6bbd5LLU,
0x000aa308951ff14dLLU, 0x000aa55781ba0d6aLLU,
0x000aa7a6a1897fd2LLU, 0x000aa9f5f492b88cLLU,
0x000aac457ada2803LLU, 0x000aae9534643f03LLU,
0x000ab0e521356ebaLLU, 0x000ab335415228bbLLU,
0x000ab58594bedefaLLU, 0x000ab7d61b8003cdLLU,
0x000aba26d59a09eeLLU, 0x000abc77c311647aLLU,
0x000abec8e3ea86eeLLU, 0x000ac11a3829e52cLLU,
0x000ac36bbfd3f37aLLU, 0x000ac5bd7aed267dLLU,
0x000ac80f6979f340LLU, 0x000aca618b7ecf31LLU,
0x000accb3e100301eLLU, 0x000acf066a028c3aLLU,
0x000ad159268a5a1cLLU, 0x000ad3ac169c10bbLLU,
0x000ad5ff3a3c2774LLU, 0x000ad852916f1606LLU,
0x000adaa61c395493LLU, 0x000adcf9da9f5b9fLLU,
0x000adf4dcca5a413LLU, 0x000ae1a1f250a73bLLU,
0x000ae3f64ba4dec6LLU, 0x000ae64ad8a6c4c5LLU,
0x000ae89f995ad3adLLU, 0x000aeaf48dc58659LLU,
0x000aed49b5eb5803LLU, 0x000aef9f11d0c44bLLU,
0x000af1f4a17a4735LLU, 0x000af44a64ec5d26LLU,
0x000af6a05c2b82e9LLU, 0x000af8f6873c35acLLU,
0x000afb4ce622f2ffLLU, 0x000afda378e438d7LLU,
0x000afffa3f84858cLLU, 0x000b02513a0857dbLLU,
0x000b04a868742ee4LLU, 0x000b06ffcacc8a2aLLU,
0x000b09576115e994LLU, 0x000b0baf2b54cd6dLLU,
0x000b0e07298db666LLU, 0x000b105f5bc5258fLLU,
0x000b12b7c1ff9c61LLU, 0x000b15105c419cb6LLU,
0x000b17692a8fa8cdLLU, 0x000b19c22cee4349LLU,
0x000b1c1b6361ef31LLU, 0x000b1e74cdef2fefLLU,
0x000b20ce6c9a8952LLU, 0x000b23283f687f8fLLU,
0x000b2582465d973cLLU, 0x000b27dc817e5555LLU,
0x000b2a36f0cf3f3aLLU, 0x000b2c919454daafLLU,
0x000b2eec6c13adddLLU, 0x000b314778103f50LLU,
0x000b33a2b84f15fbLLU, 0x000b35fe2cd4b932LLU,
0x000b3859d5a5b0b1LLU, 0x000b3ab5b2c68495LLU,
0x000b3d11c43bbd62LLU, 0x000b3f6e0a09e3ffLLU,
0x000b41ca843581baLLU, 0x000b442732c32044LLU,
0x000b468415b749b1LLU, 0x000b48e12d16887dLLU,
0x000b4b3e78e56786LLU, 0x000b4d9bf9287210LLU,
0x000b4ff9ade433c6LLU, 0x000b5257971d38b2LLU,
0x000b54b5b4d80d4aLLU, 0x000b571407193e63LLU,
0x000b59728de5593aLLU, 0x000b5bd14940eb70LLU,
0x000b5e303930830cLLU, 0x000b608f5db8ae79LLU,
0x000b62eeb6ddfc87LLU, 0x000b654e44a4fc6cLLU,
0x000b67ae07123dc3LLU, 0x000b6a0dfe2a508bLLU,
0x000b6c6e29f1c52aLLU, 0x000b6ece8a6d2c6aLLU,
0x000b712f1fa1177bLLU, 0x000b738fe99217f1LLU,
0x000b75f0e844bfc6LLU, 0x000b78521bbda15cLLU,
0x000b7ab384014f76LLU, 0x000b7d1521145d3fLLU,
0x000b7f76f2fb5e47LLU, 0x000b81d8f9bae684LLU,
0x000b843b35578a51LLU, 0x000b869da5d5de6fLLU,
0x000b89004b3a7804LLU, 0x000b8b632589ec9bLLU,
0x000b8dc634c8d228LLU, 0x000b902978fbbf01LLU,
0x000b928cf22749e4LLU, 0x000b94f0a05009f3LLU,
0x000b9754837a96b7LLU, 0x000b99b89bab881fLLU,
0x000b9c1ce8e77680LLU, 0x000b9e816b32fa95LLU,
0x000ba0e62292ad7dLLU, 0x000ba34b0f0b28c0LLU,
0x000ba5b030a1064aLLU, 0x000ba8158758e06dLLU,
0x000baa7b133751e3LLU, 0x000bace0d440f5caLLU,
0x000baf46ca7a67a7LLU, 0x000bb1acf5e84367LLU,
0x000bb413568f255aLLU, 0x000bb679ec73aa38LLU,
0x000bb8e0b79a6f1fLLU, 0x000bbb47b8081194LLU,
0x000bbdaeedc12f82LLU, 0x000bc01658ca673bLLU,
0x000bc27df9285775LLU, 0x000bc4e5cedf9f50LLU,
0x000bc74dd9f4de4fLLU, 0x000bc9b61a6cb460LLU,
0x000bcc1e904bc1d2LLU, 0x000bce873b96a760LLU,
0x000bd0f01c520628LLU, 0x000bd35932827fb0LLU,
0x000bd5c27e2cb5e5LLU, 0x000bd82bff554b1bLLU,
0x000bda95b600e20bLLU, 0x000bdcffa2341dd7LLU,
0x000bdf69c3f3a207LLU, 0x000be1d41b44128aLLU,
0x000be43ea82a13b5LLU, 0x000be6a96aaa4a46LLU,
0x000be91462c95b60LLU, 0x000beb7f908bec8eLLU,
0x000bedeaf3f6a3c2LLU, 0x000bf0568d0e2756LLU,
0x000bf2c25bd71e09LLU, 0x000bf52e60562f02LLU,
0x000bf79a9a9001d2LLU, 0x000bfa070a893e6cLLU,
0x000bfc73b0468d30LLU, 0x000bfee08bcc96e0LLU,
0x000c014d9d2004aaLLU, 0x000c03bae4458020LLU,
0x000c06286141b33dLLU, 0x000c089614194863LLU,
0x000c0b03fcd0ea5cLLU, 0x000c0d721b6d445aLLU,
0x000c0fe06ff301f4LLU, 0x000c124efa66cf2cLLU,
0x000c14bdbacd586aLLU, 0x000c172cb12b4a7dLLU,
0x000c199bdd85529cLLU, 0x000c1c0b3fe01e67LLU,
0x000c1e7ad8405be6LLU, 0x000c20eaa6aab985LLU,
0x000c235aab23e61eLLU, 0x000c25cae5b090edLLU,
0x000c283b56556999LLU, 0x000c2aabfd172031LLU,
0x000c2d1cd9fa652cLLU, 0x000c2f8ded03e967LLU,
0x000c31ff36385e29LLU, 0x000c3470b59c7521LLU,
0x000c36e26b34e065LLU, 0x000c395457065275LLU,
0x000c3bc679157e38LLU, 0x000c3e38d16716fcLLU,
0x000c40ab5fffd07aLLU, 0x000c431e24e45ed2LLU,
0x000c45912019768cLLU, 0x000c480451a3cc98LLU,
0x000c4a77b9881650LLU, 0x000c4ceb57cb0975LLU,
0x000c4f5f2c715c31LLU, 0x000c51d3377fc517LLU,
0x000c544778fafb22LLU, 0x000c56bbf0e7b5b6LLU,
0x000c59309f4aac9fLLU, 0x000c5ba584289812LLU,
0x000c5e1a9f8630adLLU, 0x000c608ff1682f76LLU,
0x000c630579d34dddLLU, 0x000c657b38cc45b9LLU,
0x000c67f12e57d14bLLU, 0x000c6a675a7aab3eLLU,
0x000c6cddbd398ea4LLU, 0x000c6f54569936f8LLU,
0x000c71cb269e601fLLU, 0x000c74422d4dc667LLU,
0x000c76b96aac2686LLU, 0x000c7930debe3d9cLLU,
0x000c7ba88988c933LLU, 0x000c7e206b10873bLLU,
0x000c8098835a3611LLU, 0x000c8310d26a9479LLU,
0x000c8589584661a1LLU, 0x000c880214f25d1fLLU,
0x000c8a7b087346f4LLU, 0x000c8cf432cddf8bLLU,
0x000c8f6d9406e7b5LLU, 0x000c91e72c2320b0LLU,
0x000c9460fb274c22LLU, 0x000c96db01182c1bLLU,
0x000c99553dfa8313LLU, 0x000c9bcfb1d313eeLLU,
0x000c9e4a5ca6a1f8LLU, 0x000ca0c53e79f0e7LLU,
0x000ca3405751c4dbLLU, 0x000ca5bba732e25dLLU,
0x000ca8372e220e61LLU, 0x000caab2ec240e43LLU,
0x000cad2ee13da7cbLLU, 0x000cafab0d73a12aLLU,
0x000cb22770cac0f9LLU, 0x000cb4a40b47ce3fLLU,
0x000cb720dcef9069LLU, 0x000cb99de5c6cf50LLU,
0x000cbc1b25d25337LLU, 0x000cbe989d16e4cbLLU,
0x000cc1164b994d23LLU, 0x000cc394315e55bfLLU,
0x000cc6124e6ac88bLLU, 0x000cc890a2c36fddLLU,
0x000ccb0f2e6d1675LLU, 0x000ccd8df16c877cLLU,
0x000cd00cebc68e87LLU, 0x000cd28c1d7ff795LLU,
0x000cd50b869d8f0fLLU, 0x000cd78b272421c9LLU,
0x000cda0aff187d02LLU, 0x000cdc8b0e7f6e61LLU,
0x000cdf0b555dc3faLLU, 0x000ce18bd3b84c4bLLU,
0x000ce40c8993d63dLLU, 0x000ce68d76f53122LLU,
0x000ce90e9be12cb9LLU, 0x000ceb8ff85c992aLLU,
0x000cee118c6c4709LLU, 0x000cf09358150754LLU,
0x000cf3155b5bab74LLU, 0x000cf5979645053cLLU,
0x000cf81a08d5e6ecLLU, 0x000cfa9cb313232eLLU,
0x000cfd1f95018d17LLU, 0x000cffa2aea5f825LLU,
0x000d022600053845LLU, 0x000d04a9892421ccLLU,
0x000d072d4a07897cLLU, 0x000d09b142b44480LLU,
0x000d0c35732f2870LLU, 0x000d0eb9db7d0b4fLLU,
0x000d113e7ba2c38cLLU, 0x000d13c353a527ffLLU,
0x000d164863890feeLLU, 0x000d18cdab535307LLU,
0x000d1b532b08c968LLU, 0x000d1dd8e2ae4b97LLU,
0x000d205ed248b287LLU, 0x000d22e4f9dcd795LLU,
0x000d256b596f948cLLU, 0x000d27f1f105c3a0LLU,
0x000d2a78c0a43f72LLU, 0x000d2cffc84fe310LLU,
0x000d2f87080d89f2LLU, 0x000d320e7fe20ffaLLU,
0x000d34962fd2517aLLU, 0x000d371e17e32b2eLLU,
0x000d39a638197a3cLLU, 0x000d3c2e907a1c38LLU,
0x000d3eb72109ef21LLU, 0x000d413fe9cdd164LLU,
0x000d43c8eacaa1d6LLU, 0x000d465224053fbdLLU,
0x000d48db95828ac7LLU, 0x000d4b653f47630fLLU,
0x000d4def2158a91fLLU, 0x000d50793bbb3de9LLU,
0x000d53038e7402ceLLU, 0x000d558e1987d99aLLU,
0x000d5818dcfba487LLU, 0x000d5aa3d8d44639LLU,
0x000d5d2f0d16a1c3LLU, 0x000d5fba79c79aa1LLU,
0x000d62461eec14beLLU, 0x000d64d1fc88f472LLU,
0x000d675e12a31e7fLLU, 0x000d69ea613f7816LLU,
0x000d6c76e862e6d3LLU, 0x000d6f03a81250bfLLU,
0x000d7190a0529c51LLU, 0x000d741dd128b06aLLU,
0x000d76ab3a99745bLLU, 0x000d7938dca9cfdfLLU,
0x000d7bc6b75eab1fLLU, 0x000d7e54cabceeb1LLU,
0x000d80e316c98398LLU, 0x000d83719b895343LLU,
0x000d86005901478fLLU, 0x000d888f4f364ac5LLU,
0x000d8b1e7e2d479dLLU, 0x000d8dade5eb2939LLU,
0x000d903d8674db2bLLU, 0x000d92cd5fcf4971LLU,
0x000d955d71ff6075LLU, 0x000d97edbd0a0d11LLU,
0x000d9a7e40f43c89LLU, 0x000d9d0efdc2dc92LLU,
0x000d9f9ff37adb4aLLU, 0x000da23122212740LLU,
0x000da4c289baaf6eLLU, 0x000da7542a4c633eLLU,
0x000da9e603db3285LLU, 0x000dac78166c0d87LLU,
0x000daf0a6203e4f5LLU, 0x000db19ce6a7a9eeLLU,
0x000db42fa45c4dfdLLU, 0x000db6c29b26c31dLLU,
0x000db955cb0bfbb6LLU, 0x000dbbe93410ea9dLLU,
0x000dbe7cd63a8315LLU, 0x000dc110b18db8cfLLU,
0x000dc3a4c60f7feaLLU, 0x000dc63913c4ccf3LLU,
0x000dc8cd9ab294e4LLU, 0x000dcb625addcd27LLU,
0x000dcdf7544b6b92LLU, 0x000dd08c87006669LLU,
0x000dd321f301b460LLU, 0x000dd5b798544c98LLU,
0x000dd84d76fd269eLLU, 0x000ddae38f013a72LLU,
0x000ddd79e065807dLLU, 0x000de0106b2ef19bLLU,
0x000de2a72f628712LLU, 0x000de53e2d053a9aLLU,
0x000de7d5641c0658LLU, 0x000dea6cd4abe4ddLLU,
0x000ded047eb9d12dLLU, 0x000def9c624ac6b6LLU,
0x000df2347f63c159LLU, 0x000df4ccd609bd61LLU,
0x000df7656641b78cLLU, 0x000df9fe3010ad03LLU,
0x000dfc97337b9b5fLLU, 0x000dff30708780a8LLU,
0x000e01c9e7395b56LLU, 0x000e046397962a4cLLU,
0x000e06fd81a2ece1LLU, 0x000e0997a564a2d6LLU,
0x000e0c3202e04c5dLLU, 0x000e0ecc9a1aea18LLU,
0x000e11676b197d17LLU, 0x000e140275e106d8LLU,
0x000e169dba768949LLU, 0x000e193938df06c8LLU,
0x000e1bd4f11f8220LLU, 0x000e1e70e33cfe8dLLU,
0x000e210d0f3c7fbaLLU, 0x000e23a9752309c0LLU,
0x000e264614f5a129LLU, 0x000e28e2eeb94aecLLU,
0x000e2b8002730c71LLU, 0x000e2e1d5027eb91LLU,
0x000e30bad7dcee90LLU, 0x000e335899971c26LLU,
0x000e35f6955b7b78LLU, 0x000e3894cb2f141aLLU,
0x000e3b333b16ee12LLU, 0x000e3dd1e51811d3LLU,
0x000e4070c9378842LLU, 0x000e430fe77a5ab3LLU,
0x000e45af3fe592e8LLU, 0x000e484ed27e3b15LLU,
0x000e4aee9f495ddcLLU, 0x000e4d8ea64c0651LLU,
0x000e502ee78b3ff6LLU, 0x000e52cf630c16beLLU,
0x000e557018d3970bLLU, 0x000e581108e6cdafLLU,
0x000e5ab2334ac7eeLLU, 0x000e5d539804937aLLU,
0x000e5ff537193e75LLU, 0x000e6297108dd773LLU,
0x000e653924676d76LLU, 0x000e67db72ab0ff2LLU,
0x000e6a7dfb5dcecaLLU, 0x000e6d20be84ba53LLU,
0x000e6fc3bc24e350LLU, 0x000e7266f4435af7LLU,
0x000e750a66e532ebLLU, 0x000e77ae140f7d42LLU,
0x000e7a51fbc74c83LLU, 0x000e7cf61e11b3a4LLU,
0x000e7f9a7af3c60bLLU, 0x000e823f12729791LLU,
0x000e84e3e4933c7eLLU, 0x000e8788f15ac98aLLU,
0x000e8a2e38ce53dfLLU, 0x000e8cd3baf2f118LLU,
0x000e8f7977cdb740LLU, 0x000e921f6f63bcd2LLU,
0x000e94c5a1ba18bdLLU, 0x000e976c0ed5e25dLLU,
0x000e9a12b6bc3181LLU, 0x000e9cb999721e6aLLU,
0x000e9f60b6fcc1c7LLU, 0x000ea2080f6134bcLLU,
0x000ea4afa2a490daLLU, 0x000ea75770cbf025LLU,
0x000ea9ff79dc6d14LLU, 0x000eaca7bddb228cLLU,
0x000eaf503ccd2be5LLU, 0x000eb1f8f6b7a4e9LLU,
0x000eb4a1eb9fa9d1LLU, 0x000eb74b1b8a5749LLU,
0x000eb9f4867cca6eLLU, 0x000ebc9e2c7c20d0LLU,
0x000ebf480d8d786dLLU, 0x000ec1f229b5efb8LLU,
0x000ec49c80faa594LLU, 0x000ec7471360b955LLU,
0x000ec9f1e0ed4ac2LLU, 0x000ecc9ce9a57a12LLU,
0x000ecf482d8e67f1LLU, 0x000ed1f3acad3578LLU,
0x000ed49f67070435LLU, 0x000ed74b5ca0f628LLU,
0x000ed9f78d802dc2LLU, 0x000edca3f9a9cde5LLU,
0x000edf50a122f9e6LLU, 0x000ee1fd83f0d58cLLU,
0x000ee4aaa2188510LLU, 0x000ee757fb9f2d1dLLU,
0x000eea059089f2d0LLU, 0x000eecb360ddfbb8LLU,
0x000eef616ca06dd6LLU, 0x000ef20fb3d66f9eLLU,
0x000ef4be368527f6LLU, 0x000ef76cf4b1be36LLU,
0x000efa1bee615a27LLU, 0x000efccb23992408LLU,
0x000eff7a945e4487LLU, 0x000f022a40b5e4c6LLU,
0x000f04da28a52e59LLU, 0x000f078a4c314b47LLU,
0x000f0a3aab5f6609LLU, 0x000f0ceb4634a98aLLU,
0x000f0f9c1cb6412aLLU, 0x000f124d2ee958b9LLU,
0x000f14fe7cd31c7bLLU, 0x000f17b00678b927LLU,
0x000f1a61cbdf5be7LLU, 0x000f1d13cd0c3256LLU,
0x000f1fc60a046a84LLU, 0x000f227882cd32f3LLU,
0x000f252b376bba97LLU, 0x000f27de27e530daLLU,
0x000f2a91543ec595LLU, 0x000f2d44bc7da917LLU,
0x000f2ff860a70c22LLU, 0x000f32ac40c01fe8LLU,
0x000f35605cce1613LLU, 0x000f3814b4d620bdLLU,
0x000f3ac948dd7274LLU, 0x000f3d7e18e93e39LLU,
0x000f403324feb781LLU, 0x000f42e86d231233LLU,
0x000f459df15b82acLLU, 0x000f4853b1ad3dbbLLU,
0x000f4b09ae1d78a1LLU, 0x000f4dbfe6b16915LLU,
0x000f50765b6e4540LLU, 0x000f532d0c5943c0LLU,
0x000f55e3f9779ba5LLU, 0x000f589b22ce8474LLU,
0x000f5b5288633625LLU, 0x000f5e0a2a3ae925LLU,
0x000f60c2085ad652LLU, 0x000f637a22c83701LLU,
0x000f6632798844f8LLU, 0x000f68eb0ca03a75LLU,
0x000f6ba3dc155226LLU, 0x000f6e5ce7ecc72fLLU,
0x000f7116302bd526LLU, 0x000f73cfb4d7b819LLU,
0x000f768975f5ac86LLU, 0x000f7943738aef61LLU,
0x000f7bfdad9cbe14LLU, 0x000f7eb824305679LLU,
0x000f8172d74af6e1LLU, 0x000f842dc6f1de12LLU,
0x000f86e8f32a4b45LLU, 0x000f89a45bf97e28LLU,
0x000f8c600164b6dcLLU, 0x000f8f1be37135f9LLU,
0x000f91d802243c89LLU, 0x000f94945d830c0cLLU,
0x000f9750f592e677LLU, 0x000f9a0dca590e33LLU,
0x000f9ccadbdac61dLLU, 0x000f9f882a1d5187LLU,
0x000fa245b525f439LLU, 0x000fa5037cf9f26eLLU,
0x000fa7c1819e90d8LLU, 0x000faa7fc319149cLLU,
0x000fad3e416ec354LLU, 0x000faffcfca4e310LLU,
0x000fb2bbf4c0ba54LLU, 0x000fb57b29c7901aLLU,
0x000fb83a9bbeabd1LLU, 0x000fbafa4aab555cLLU,
0x000fbdba3692d514LLU, 0x000fc07a5f7a73c7LLU,
0x000fc33ac5677ab8LLU, 0x000fc5fb685f33a0LLU,
0x000fc8bc4866e8adLLU, 0x000fcb7d6583e482LLU,
0x000fce3ebfbb7237LLU, 0x000fd1005712dd5bLLU,
0x000fd3c22b8f71f1LLU, 0x000fd6843d367c72LLU,
0x000fd9468c0d49ccLLU, 0x000fdc0918192765LLU,
0x000fdecbe15f6314LLU, 0x000fe18ee7e54b2bLLU,
0x000fe4522bb02e6eLLU, 0x000fe715acc55c18LLU,
0x000fe9d96b2a23d9LLU, 0x000fec9d66e3d5d9LLU,
0x000fef619ff7c2b3LLU, 0x000ff226166b3b7aLLU,
0x000ff4eaca4391b6LLU, 0x000ff7afbb861765LLU,
0x000ffa74ea381efcLLU, 0x000ffd3a565efb65LLU
};
#endif

#ifdef __cplusplus
}
#endif
