///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/gui/properties/FloatControllerUI.h>
#include "CircleObject.h"
#include "CircleCreationMode.h"

namespace StdObjects {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(CircleObject, SimpleShapeObject)
DEFINE_REFERENCE_FIELD(CircleObject, FloatController, "Radius", _radius)
SET_PROPERTY_FIELD_LABEL(CircleObject, _radius, "Radius")
SET_PROPERTY_FIELD_UNITS(CircleObject, _radius, WorldParameterUnit)

IMPLEMENT_PLUGIN_CLASS(CircleObjectEditor, PropertiesEditor)
IMPLEMENT_PLUGIN_CLASS(CircleCreationMode, SimpleCreationMode)

/******************************************************************************
* Constructs a new circle object.
******************************************************************************/
CircleObject::CircleObject(bool isLoading) : SimpleShapeObject(isLoading)
{
	INIT_PROPERTY_FIELD(CircleObject, _radius);
	if(!isLoading) {
		_radius = CONTROLLER_MANAGER.createDefaultController<FloatController>();
	}
}

/******************************************************************************
* Asks the object for its validity interval at the given time.
******************************************************************************/
TimeInterval CircleObject::objectValidity(TimeTicks time)
{
	TimeInterval interval = TimeForever;
	_radius->validityInterval(time, interval);
	return interval;
}

/******************************************************************************
* Builds the shape representation of this procedural object.
******************************************************************************/
void CircleObject::buildShape(TimeTicks time, BezierShape& shape, TimeInterval& shapeValidity)
{
	// Reset shape validity interval.
	shapeValidity.setInfinite();

	// Query parameters.
	FloatType radius=0;
	_radius->getValue(time, radius, shapeValidity);

	// Create a new curve.
	shape.curves().resize(1);
	BezierCurve& curve = shape.curves()[0];
	curve.setVertexCount(0);

	// Add 4 vertices to create a circle.
	for(int i=0; i<4; i++) {
		FloatType angle = 0.5 * FLOATTYPE_PI * i;
		FloatType c = cos(angle), s = sin(angle);
		Point3 vp(c*radius, s*radius, 0.0);
		Vector3 tangent(s*radius*0.55178618, -c*radius*0.55178618, 0.0);
		curve.vertices().push_back(BezierVertex(
				BezierVertex::VertexBezier, BezierVertex::SegmentBezier,
				vp, tangent, -tangent));
	}
	curve.setClosed(true);
	curve.invalidate();
}

/******************************************************************************
* Creates the UI controls for the editor.
******************************************************************************/
void CircleObjectEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create the rollout.
	QWidget* rollout = createRollout(tr("Circle"), rolloutParams);

	QGridLayout* layout = new QGridLayout(rollout);
	layout->setContentsMargins(4,4,4,4);
	layout->setSpacing(0);
	layout->setColumnStretch(1, 1);

	// Radius parameter.
	FloatControllerUI* radiusPUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(CircleObject, _radius));
	layout->addWidget(radiusPUI->label(), 0, 0);
	layout->addWidget(radiusPUI->textBox(), 0, 1);
	layout->addWidget(radiusPUI->spinner(), 0, 2);
	radiusPUI->setMinValue(0);
}

};
