///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __FREE_CAMERA_CREATION_MODE_H
#define __FREE_CAMERA_CREATION_MODE_H

#include <core/Core.h>
#include <core/scene/objects/CreationMode.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/snapping/SnappingManager.h>
#include "../StdObjects.h"
#include "CameraObject.h"

namespace StdObjects {

/******************************************************************************
* This input mode lets the user create a free camera object.
******************************************************************************/
class FreeCameraCreationMode : public SimpleCreationMode
{
public:
	/// Default constructor.
	FreeCameraCreationMode() : SimpleCreationMode(tr("Create Free Camera"), tr("Camera")) {}
	
protected:

	/// This creates the actual scene object.
	virtual SceneObject::SmartPtr createObject() {
		return new CameraObject();
	}

	/// Will be called when the user presses the left mouse button.
	virtual void onMousePressed(QMouseEvent* event) {
		SimpleCreationMode::onMousePressed(event);

		if(clickCount() == 1) {
			// Take click point as initial camera position
			Point3 initialPos = ORIGIN;
			if(!viewport()->snapPoint(event->pos(), initialPos))
				onAbort();
				
			// Position scene node.
			AffineTransformation tm = viewport()->grid().gridMatrix() * AffineTransformation::translation(initialPos - ORIGIN);			
			objectNode()->transformationController()->setValue(0, tm);
		}
	}

	/// Will be called when the user releases the left mouse button.
	virtual void onMouseReleased(QMouseEvent* event) {
		SimpleCreationMode::onMouseReleased(event);
		if(clickCount() == 2) {
			onFinish();
		}
	}

	/// Will be called when the user moves the mouse while 
	/// the operation is active.
	virtual void onMouseDrag(QMouseEvent* event) {
		SimpleCreationMode::onMouseDrag(event);
		if(clickCount() == 1) {
			// Take mouse position as new position
			Point3 newPos = ORIGIN;
			if(!viewport()->snapPoint(event->pos(), newPos))
				return;
			
			abortAdjustOperation();
			beginAdjustOperation();

			// Position scene node.
			AffineTransformation tm = viewport()->grid().gridMatrix() * AffineTransformation::translation(newPos - ORIGIN);			
			objectNode()->transformationController()->setValue(0, tm);
		}

		// Update viewports immediately.
		VIEWPORT_MANAGER.processViewportUpdates();
	}

private:

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(FreeCameraCreationMode)
};

};

#endif 	// __FREE_CAMERA_CREATION_MODE_H

