///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/data/ObjectSaveStream.h>
#include <core/reference/PropertyFieldDescriptor.h>
#include <core/plugins/Plugin.h>

namespace Core {

/******************************************************************************
* Opens the stream for writing.
******************************************************************************/
ObjectSaveStream::ObjectSaveStream(QDataStream& destination) : SaveStream(destination)
{
}

/******************************************************************************
* Saves an object with runtime type information to the stream.
******************************************************************************/
void ObjectSaveStream::saveObject(PluginClass* object)
{
	if(object == NULL) *this << (quint32)0;
	else {
		CHECK_OBJECT_POINTER(object);
		OVITO_ASSERT(objects.size() == objectMap.size());
		quint32& id = objectMap[object];
		if(id == 0) {
			objects.push_back(object);
			id = (quint32)objects.size();
		}
		*this << id;
	}
}

/******************************************************************************
* Closes the stream.
******************************************************************************/
void ObjectSaveStream::close()
{
	if(!isOpen())
		return;

	QVector<qint64> objectOffsets;

	// Save all objects.
	beginChunk(0x100);
	for(int i=0; i<objects.size(); i++) {
		PluginClass* obj = objects[i];
		CHECK_OBJECT_POINTER(obj);
		objectOffsets.push_back(filePosition());
		obj->saveToStream(*this);
	}
	endChunk();

	// Save RTTI.
	map<PluginClassDescriptor*, quint32> classes;
	qint64 beginOfRTTI = filePosition();
	beginChunk(0x200);
	Q_FOREACH(PluginClass* obj, objects) {
		PluginClassDescriptor* descriptor = obj->pluginClassDescriptor();
		CHECK_POINTER(descriptor);
		if(classes.find(descriptor) == classes.end()) {
			classes.insert(make_pair(descriptor, (quint32)classes.size()));
			// Write the runtime type information to the stream.
			beginChunk(0x201);
			PluginClassDescriptor::saveRTTI(*this, descriptor);
			endChunk();
			// Write the property fields to the stream.
			beginChunk(0x202);
			for(PluginClassDescriptor* clazz = descriptor; clazz; clazz = clazz->baseClass()) {
				for(const PropertyFieldDescriptor* field = clazz->firstPropertyField(); field; field = field->next()) {
					beginChunk(0x00000001);
					*this << QByteArray(field->identifier());
					OVITO_ASSERT(field->definingClass() == clazz);
					PluginClassDescriptor::saveRTTI(*this, field->definingClass());
					this->writeEnum(field->flags());
					*this << field->isReferenceField();
					if(field->isReferenceField()) {
						PluginClassDescriptor::saveRTTI(*this, field->targetClass());
					}
					endChunk();
				}
			}
			// Write list terminator.
			beginChunk(0x00000000);
			endChunk();

			endChunk();
		}
	}
	endChunk();

	// Save object table.
	qint64 beginOfObjTable = filePosition();
	beginChunk(0x300);
	QVector<qint64>::const_iterator offsetIterator = objectOffsets.begin();
	Q_FOREACH(PluginClass* obj, objects) {
		*this << classes[obj->pluginClassDescriptor()];
		*this << *offsetIterator++;
	}
	endChunk();

	// Write index.
	*this << beginOfRTTI;
	*this << (quint32)classes.size();
	*this << beginOfObjTable;
	*this << (quint32)objects.size();

	SaveStream::close();
}

};
