/*
 *  Copyright (c) 2004,2007,2008,2009,2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ShapeGenerator_p.h"

#include <GTLCore/Debug.h>
#include <GTLCore/Math.h>

using namespace OpenRijn::ImageEngine;

AbstractShapeGenerator::AbstractShapeGenerator(float radius, float ratio, float fh, float fv, int spikes)
        : m_radius(radius), m_ratio(ratio), m_fh(fh), m_fv(fv), m_spikes(spikes)
{
  m_cs = cos(- 2 * M_PI / m_spikes);
  m_ss = sin(- 2 * M_PI / m_spikes);
  m_empty = (m_ratio == 0.0 or m_radius == 0.0);
}

AbstractShapeGenerator::~AbstractShapeGenerator()
{
}

float AbstractShapeGenerator::radius() const
{
  return m_radius;
}

CircleShapeGenerator::CircleShapeGenerator(float radius, float ratio, float fh, float fv, int spikes)
        : AbstractShapeGenerator(radius, ratio, fh, fv, spikes)
{
    m_xcoef = 2.0 / m_radius;
    m_ycoef = 2.0 / (m_ratio * m_radius);
    m_xfadecoef = (m_fh == 0) ? 1 : (2.0 / (m_fh * m_radius));
    m_yfadecoef = (m_fv == 0) ? 1 : (2.0 / (m_fv * m_ratio * m_radius));
    m_cachedSpikesAngle = M_PI / m_spikes;
}

CircleShapeGenerator::~CircleShapeGenerator()
{
}

inline float norme(float a, float b) {
    return a*a + b * b;
}

float CircleShapeGenerator::valueAt(float x, float y) const
{
    if (m_empty) return 1.0;
    double xr = (x /*- m_xcenter*/);
    double yr = fabs(y /*- m_ycenter*/);

    if (m_spikes > 2) {
        double angle = atan2(yr, xr);

        while (angle > m_cachedSpikesAngle ){
            double sx = xr, sy = yr;

            xr = m_cs * sx - m_ss * sy;
            yr = m_ss * sx + m_cs * sy;

            angle -= 2 * m_cachedSpikesAngle;
        }
    }

    double n = norme(xr * m_xcoef, yr * m_ycoef);

    if (n > 1) {
        return 0.0;
    } else {
        double normeFade = norme(xr * m_xfadecoef, yr * m_yfadecoef);
        if (normeFade > 1) {
            double xle, yle;
            // xle stands for x-coordinate limit exterior
            // yle stands for y-coordinate limit exterior
            // we are computing the coordinate on the external ellipse in order to compute
            // the fade value
            if (xr == 0) {
                xle = 0;
                yle = yr > 0 ? 1 / m_ycoef : -1 / m_ycoef;
            } else {
                double c = yr / (double)xr;
                xle = sqrt(1 / norme(m_xcoef, c * m_ycoef));
                xle = xr > 0 ? xle : -xle;
                yle = xle * c;
            }
            // On the internal limit of the fade area, normeFade is equal to 1
            double normeFadeLimitE = norme(xle * m_xfadecoef, yle * m_yfadecoef);
            return 1.0 - (normeFade - 1) / (normeFadeLimitE - 1);
        } else {
            return 1.0;
        }
    }
}

RectangleShapeGenerator::RectangleShapeGenerator(float radius, float ratio, float fh, float fv, int spikes) : AbstractShapeGenerator(radius, ratio, fh, fv, spikes)
{
    if (m_fv == 0 and m_fh == 0) {
        m_c = 0;
    } else {
        m_c = (m_fv / m_fh);
        GTL_ASSERT(not isnan(m_c));
    }
}

RectangleShapeGenerator::~RectangleShapeGenerator()
{
}

float RectangleShapeGenerator::valueAt(float x, float y) const
{

    if (m_empty) return 0.0;
    double xr = std::fabs(x /*- m_xcenter*/) / m_radius;
    double yr = std::fabs(y /*- m_ycenter*/) / m_ratio * m_radius;
    if (xr > m_fh || yr > m_fv) {
        if (yr <= ((xr - m_fh) * m_c + m_fv)) {
            return 1.0 - (xr - 0.5 * m_fh) / (1.0 - 0.5 * m_fh);
        } else {
            return 1.0 - (yr - 0.5 * m_fv) / (1.0 - 0.5 * m_fv);
        }
    } else {
        return 1.0;
    }
}