/*
 *  Copyright (c) 2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _GENERATION_VISITOR_H_
#define _GENERATION_VISITOR_H_

#include <list>
#include <vector>
#include "ExpressionResult.h"
#include "GTLCore/StdTypes.h"

namespace GTLCore {
  class Function;
  class String;
  class Type;
  class CompilationMessage;
  class CompilationMessages;
  namespace AST {
    class Annotation;
    class Expression;
    /**
     * @internal
     * @ingroup GTLCore_AST
     *
     * This class countains the information necesseray to generate code.
     * It is then subclassed by the backends.
     */
    class GenerationVisitor {
      protected:
        GenerationVisitor();
        GenerationVisitor(const GenerationVisitor& );
        GenerationVisitor& operator=(const GenerationVisitor& );
        virtual ~GenerationVisitor();
      public:
        /**
         * Generate a float 32 from @p arg1
         */
        virtual ExpressionResultSP generateFloat32(float arg1, const Annotation& _annotation) = 0;
        /**
         * Generate an integer 32 from @p arg1
         */
        virtual ExpressionResultSP generateInteger32(gtl_int32 arg1, const Annotation& _annotation) = 0;
        /**
         * Generate a boolean from @p arg1
         */
        virtual ExpressionResultSP generateBoolean(bool arg1, const Annotation& _annotation) = 0;
        /**
         * Generate a function call to @p _function with arguments @p _arguments
         */
        virtual ExpressionResultSP generateFunctionCall(GTLCore::Function* _function, const std::list<ExpressionResultSP>& _arguments, const Annotation& _annotation) = 0;
        /**
         * Generate a string from @p arg1.
         */
        virtual ExpressionResultSP generateString(GTLCore::String arg1, const Annotation& _annotation) = 0;
      public: // binary expression
        virtual ExpressionResultSP generateAndExpresion(ExpressionResultSP arg1, ExpressionResultSP arg2, const Annotation& _annotation) = 0;
        virtual ExpressionResultSP generateOrExpresion(ExpressionResultSP arg1, ExpressionResultSP arg2, const Annotation& _annotation) = 0;
        virtual ExpressionResultSP generateBitOrExpresion(ExpressionResultSP arg1, ExpressionResultSP arg2, const Annotation& _annotation) = 0;
        virtual ExpressionResultSP generateBitXorExpresion(ExpressionResultSP arg1, ExpressionResultSP arg2, const Annotation& _annotation) = 0;
        virtual ExpressionResultSP generateBitAndExpresion(ExpressionResultSP arg1, ExpressionResultSP arg2, const Annotation& _annotation) = 0;
        virtual ExpressionResultSP generateEqualExpresion(ExpressionResultSP arg1, ExpressionResultSP arg2, const Annotation& _annotation) = 0;
        virtual ExpressionResultSP generateDifferentExpresion(ExpressionResultSP arg1, ExpressionResultSP arg2, const Annotation& _annotation) = 0;
        virtual ExpressionResultSP generateInferiorExpresion(ExpressionResultSP arg1, ExpressionResultSP arg2, const Annotation& _annotation) = 0;
        virtual ExpressionResultSP generateInferiorEqualExpresion(ExpressionResultSP arg1, ExpressionResultSP arg2, const Annotation& _annotation) = 0;
        virtual ExpressionResultSP generateSupperiorExpresion(ExpressionResultSP arg1, ExpressionResultSP arg2, const Annotation& _annotation) = 0;
        virtual ExpressionResultSP generateSupperiorEqualExpresion(ExpressionResultSP arg1, ExpressionResultSP arg2, const Annotation& _annotation) = 0;
        virtual ExpressionResultSP generateRightShiftExpresion(ExpressionResultSP arg1, ExpressionResultSP arg2, const Annotation& _annotation) = 0;
        virtual ExpressionResultSP generateLeftShiftExpresion(ExpressionResultSP arg1, ExpressionResultSP arg2, const Annotation& _annotation) = 0;
        virtual ExpressionResultSP generateAdditionExpresion(ExpressionResultSP arg1, ExpressionResultSP arg2, const Annotation& _annotation) = 0;
        virtual ExpressionResultSP generateSubtractionExpresion(ExpressionResultSP arg1, ExpressionResultSP arg2, const Annotation& _annotation) = 0;
        virtual ExpressionResultSP generateMultiplicationExpresion(ExpressionResultSP arg1, ExpressionResultSP arg2, const Annotation& _annotation) = 0;
        virtual ExpressionResultSP generateDivisionExpresion(ExpressionResultSP arg1, ExpressionResultSP arg2, const Annotation& _annotation) = 0;
        virtual ExpressionResultSP generateModuloExpresion(ExpressionResultSP arg1, ExpressionResultSP arg2, const Annotation& _annotation) = 0;
      public: // Unary expressions
        virtual ExpressionResultSP generateTildExpression(ExpressionResultSP arg1, const Annotation& _annotation) =0 ;
        virtual ExpressionResultSP generateMinusExpression(ExpressionResultSP arg1, const Annotation& _annotation) = 0;
        virtual ExpressionResultSP generateNotExpression(ExpressionResultSP arg1, const Annotation& _annotation) = 0;
      public: // Convert expression
        virtual ExpressionResultSP convertExpressionTo(ExpressionResultSP arg1, const Type* _type, const Annotation& _annotation) = 0;
        virtual ExpressionResultSP convertArrayToVector(ExpressionResultSP arg1, const Type* type, const Annotation& _annotation) = 0;
        virtual ExpressionResultSP createCompoundConstant(std::vector< ExpressionResultSP> arg1, const Type*, const Annotation& arg2) = 0;
        virtual ExpressionResultSP createCompoundValue(std::vector< ExpressionResultSP> arg1, const Type*, const Annotation& _annotation) = 0;
      public:
        ExpressionResultSP generateFunctionCall(GTLCore::Function* _function, const std::list<AST::Expression*>& _arguments, const Annotation& _annotation);
      public:
        const GTLCore::CompilationMessages& compilationMessages() const;
      protected:
        /**
         * Call this function to report an error occuring at generation time.
         */
        void reportError(const GTLCore::String& _message, const Annotation& _annotation);
      private:
        struct Private;
        Private* const d;
    };
  }
}

#endif
