/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var PINF = require( '@stdlib/constants/float64/pinf' );
var NINF = require( '@stdlib/constants/float64/ninf' );
var isnan = require( '@stdlib/math/base/assert/is-nan' );
var pow = require( '@stdlib/math/base/special/pow' );
var toWords = require( './../../../../float64/base/to-words' );
var fromWords = require( './../lib' );


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.equal( typeof fromWords, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function returns number', function test( t ) {
	var x = fromWords( 5, 4 );
	t.equal( typeof x, 'number', 'returns a number' );
	t.end();
});

tape( 'the function creates a floating-point number from a higher order word and a lower order word', function test( t ) {
	var values;
	var v;
	var x;
	var w;
	var i;

	values = [
		5.0,
		pow( 2.0, 53 ),
		1.0e308,
		-1.0e308,
		-3.14,
		1.0e-324,
		4.94e-324,
		1.234567890123456789,
		-4.94e-324,
		6.333333333333333333e-310,
		-0.0,
		0.0,
		100.0,
		1.0/10.0,
		0.625,
		1.0/3.0,
		5.0e-240,
		-5.0e-240,
		10.0,
		15.0,
		-10.0,
		-15.0,
		pow( 2.0, -42 ),
		-pow( 2.0, 100 ),
		1.0,
		-1.0,
		1.5,
		1111111111111.111111111,
		-1111111111111.111111111,
		pow( 2.0, 54 ),
		pow( 2.0, 53 ) + 1.0,
		pow( 2.0, 53 ) + 2.0,
		pow( 2.0, 55 ),
		pow( 2.0, 56 ) - 1.0,
		-pow( 2.0, 57 ) + 5.0,
		3.0*pow( 2.0, 53 ),
		8.0*pow( 2.0, 54 )
	];

	for ( i = 0; i < values.length; i++ ) {
		v = values[ i ];
		w = toWords( v );
		x = fromWords( w[ 0 ], w[ 1 ] );
		t.equal( x, v, 'returns ' + x + ' from a higher order word (' + w[0] + ') and a lower order word (' + w[1] + ')' );
	}
	t.end();
});

tape( 'the function can return `NaN`', function test( t ) {
	var w;
	var x;

	w = toWords( NaN );
	x = fromWords( w[0], w[1] );

	t.equal( isnan( x ), true, 'returns NaN' );
	t.end();
});

tape( 'the function can return `+infinity`', function test( t ) {
	var w;
	var x;

	w = toWords( PINF );
	x = fromWords( w[0], w[1] );

	t.equal( x, PINF, 'returns positive infinity' );
	t.end();
});

tape( 'the function can return `-infinity`', function test( t ) {
	var w;
	var x;

	w = toWords( NINF );
	x = fromWords( w[0], w[1] );

	t.equal( x, NINF, 'returns negative infinity' );
	t.end();
});
