#include "canvas.h"
#include "graphchart.h"
#include "msc.h"
#include "blockchart.h"
#include "gui.h"
#include "text_editor.h"
#include "ImGuiFileDialog.h"
#include "utf8utils.h"
#include <iostream>
#include <string.h>
#include <atomic>
#include <thread>
#include <concepts>

std::string my_strerror(int errno_) {
    char buff[512] = {'\0'};
    //strerror() is not thread safe and results in all kinds of
    //compiler warnings. There are safe variants, but...
#ifdef _WIN32
    //Well, windows has strerror_s(), which is C11 standard
    strerror_s(buff, sizeof(buff), errno_);
#else 
    //POSIX and GCC each have strerror_r(), but what a mess: https://linux.die.net/man/3/strerror_r
    //The precondition macros listed there do not work on MacOS, so we do this instead.
    using strerror_r_ret_type = decltype(strerror_r(errno_, buff, sizeof(buff)));
    constexpr bool strerror_r_rets_int = std::is_same_v<strerror_r_ret_type, int>;
    constexpr bool strerror_r_rets_charp = std::is_same_v<strerror_r_ret_type, char *>;
    //Note: "Outside a template, a discarded statement is fully checked. if constexpr is not a
    //substitute for the #if preprocessing directive" So both branches must be valid C++.
    if constexpr (strerror_r_rets_int) {
        if (strerror_r(errno_, buff, sizeof(buff)))
            snprintf(buff, sizeof(buff), "Could not get error string for errno=%d", errno_);
    } else if constexpr (strerror_r_rets_charp) {
        char* b = (char*)strerror_r(errno_, buff, sizeof(buff));
        if (b!=buff) {
            strncpy(buff, b, sizeof(buff)-1);
            buff[sizeof(buff)-1] = 0;
        }
    } else
        static_assert(strerror_r_rets_charp || strerror_r_rets_int, "strerror_r() returns neither char* nor int.");
#endif
    return buff;
}

std::map<std::string, std::string> FontLicenses();

std::map<std::string, std::string> RegisterLibrariesGUI() {
    auto ret = FontLicenses();
    ret["Dear ImGui " IMGUI_VERSION];
    ret["ImGui File Dialog " IMGUIFILEDIALOG_VERSION];
    return ret;
}

#define MSC_GEN_INI_FILE_NAME ".msc-generator-ini"

using namespace std::literals;

//normalizes 'a' to fall between 'b' and 'c'.

template <std::totally_ordered T>
inline T between(const T &a, const T &b, const T &c) noexcept { _ASSERT(b<=c); return a<b ? b : c<a ? c : a; }
inline ImVec2 between(ImVec2 a, ImVec2 b, ImVec2 c) noexcept { return {between(a.x, b.x, c.x), between(a.y, b.y, c.y)}; }
inline ImVec2 min(ImVec2 a, ImVec2 b) noexcept { return {std::min(a.x, b.x), std::min(a.y, b.y)}; }
inline ImVec2 max(ImVec2 a, ImVec2 b) noexcept { return {std::max(a.x, b.x), std::max(a.y, b.y)}; }
inline ImVec2 operator+(const ImVec2 &a, const ImVec2 &b) noexcept { return {a.x+b.x, a.y+b.y}; }
inline ImVec2 operator-(const ImVec2 &a, const ImVec2 &b) noexcept { return {a.x-b.x, a.y-b.y}; }
inline ImVec2 operator*(float a, const ImVec2 &b) noexcept { return {a*b.x, a*b.y}; }
inline ImVec2 operator*(const ImVec2 &b, float a) noexcept { return {a*b.x, a*b.y}; }
inline ImVec2 operator/(const ImVec2 &b, float a) noexcept { return {b.x/a, b.y/a}; }
inline bool operator !(const ImVec2 &b) noexcept { return b.x==0.0f && b.y==0.0f; }


std::array<ImFont *, 8> MscGenFonts = {};
float MscGenEditorFontSize;
float MscGenWindowFontSize;

ImFont *GetFont(bool variablespaced, bool bold, bool italics) noexcept {
    const unsigned index = (variablespaced ? 4 : 0) + (bold ? 2 : 0) + (italics ? 1 : 0);
    if (MscGenFonts[index]) return MscGenFonts[index];
    if (italics) return GetFont(variablespaced, bold, false);
    if (bold) return GetFont(variablespaced, false, italics);
    if (variablespaced) return GetFont(false, bold, italics);
    _ASSERT(0); //we must have at least the monospaced, non bold, non italics font set
    return ImGui::GetIO().FontDefault;
}

using my_clock = std::chrono::steady_clock;
static auto last_frame = my_clock::now();
inline float sec_since_last_frame() noexcept { return  std::chrono::nanoseconds(my_clock::now()-last_frame).count() / 1e9f; }

//calculates what is the new value of a moving animation going to 'target'
//Speed=1 is when take the main window size is crossed in 3 seconds.
inline float anim_move_value(float orig, float target, float speed = 1.f) {
    const float pixel_per_sec = std::midpoint(ImGui::GetMainViewport()->Size.x, ImGui::GetMainViewport()->Size.y) * speed / 3;
    const float amount = std::min(float(fabs(target-orig)), pixel_per_sec*sec_since_last_frame());
    return orig < target ? orig + amount : orig - amount;
}

//calculates what is the new value of a [0..1] animation going to 'target'
//Speed=1 is when the 0->1 or 1->0 transition takes 1 sec.
inline float anim_time_value(float orig, float target, float speed = 1.f) {
    const float per_sec = speed;
    const float amount = std::min(float(fabs(target-orig)), per_sec*sec_since_last_frame());
    return orig < target ? orig + amount : orig - amount;
}

//This should be a completely thread-safe structure
class CompileData {
    using clock = std::chrono::steady_clock;
    enum class State { Idle, Compiling, Ready, Stopped };
    std::mutex my_mutex;
    std::atomic<State> state;
    std::atomic_bool stop = false;
    std::thread compile_thread;
    clock::time_point compiling_started_ended = clock::now();
    float progress;
    struct Result {
        std::unique_ptr<Chart> pChart;
        int main_file_no; //The number of the currently edited file in pChart->Errors.Files
        TextEditor::UndoBufferIndex  compiled_at;
        clock::duration compile_time;
    };
    Result result;
    static bool progressbar(const ProgressBase *progress, void *p,
                            ProgressBase::EPreferredAbortMethod a) {
        CompileData *me = (CompileData *)p;
        me->progress = (float)progress->GetPercentage();
        if (!me->stop.load(std::memory_order_acquire)) return false;
        switch (a) {
        default:
        case ProgressBase::NONE: return false;
        case ProgressBase::EXCEPTION: throw AbortCompilingException();
        case ProgressBase::RETVAL: return true;
        }
    }

public:
    //Stops ongoing compilation, blocks until it finishes and starts a new one
    void Start(std::unique_ptr<Chart> &&chart, std::string &&text, const std::string &fn, TextEditor::UndoBufferIndex editorbuffer);
    //Signals the stop of a compilation. State will move to Stopped, when compile thread detected the signal
    void Stop() noexcept { stop.store(true, std::memory_order::release); } 
    //true when compiling, false when ready, stopped or not started 
    bool is_compiling() const noexcept { return state.load(std::memory_order_acquire)==State::Compiling; }
    //true when stopped (will be cleared by next start)
    bool is_stopped() const noexcept { return state.load(std::memory_order_acquire)==State::Stopped; }
    //returns time elapsed since compilation started, stopped or ended
    std::chrono::milliseconds time_elapsed() const noexcept { return std::chrono::duration_cast<std::chrono::milliseconds>(clock::now()-compiling_started_ended); }
    //return the progress during compilation, -1 else
    float get_progress() const noexcept {
        std::mutex my_mutex;
        return is_compiling() ? progress : -1.f;
    }
    //Atomically returns result if ready and moves to idle
    //If compilation is ongoing, we wait fi 'block' is set, else return empty
    //If there is no result (Stopped, Idle), we set Idle and return empty.
    std::optional<Result> GetResult(bool block) noexcept;
};

struct HintData {
    bool active = false;     ///<True if the hint window needs to be activated
    bool is_user_requested;  ///<True if the the current hints session was invoked via Ctrl+Space
    bool till_cursor_only;   ///<True if the the current hints session started at the beginning of a word.
    bool restart_hint;       ///<True for one frame after substitution if we need to keep doing it.
    std::unique_ptr<Csh> csh;///<Coloring moves the resulting csh here.
    bool partial_match;      ///<If the hints displayed begin with the word under the cursor. Set by ProcessHints().
    struct ItemData {
        std::optional<ImTextureID> graphics; ///<The hint graphics, empty if none
        bool bold, italics, underline;       ///<Text formatting for the hint
        ImU32 color;                         ///<Text formatting for the hint
    };
    std::vector<ItemData> items; ///<The hint visuals for the hints in csh->Hints (in that order)
    std::optional<ImTextureID> empty_graphics;
};

/** Functions (potentially) opening modal popups will return this.*/
enum class StatusRet { 
    Ongoing,   ///<The modal popup is still open, come back in the next frame
    Cancelled, ///<The operation has been cancelled, the popup closed, dont come back next frame
    Completed  ///<The operation has been completed, the popup closed (or never opened), dont come back next frame
};

struct ConfigData {
    using clock = std::chrono::steady_clock;
    const GUIInit *G = nullptr;
    float dpi_mul = 1.;
    TextEditor editor;
    std::string open_filter_groups;

    std::string file_name;
    std::string window_title; // shown in the host window. Filename plus asterisk on dirty.
    std::string full_path;
    TextEditor::UndoBufferIndex saved_at;    // Saved at this undo buffer index
    TextEditor::UndoBufferIndex compiled_at; // Compiled at this undo buffer index

    struct RecentFile {
        std::string full_path;
        std::string file_name;
        bool operator ==(const RecentFile &o) const noexcept { return full_path==o.full_path; }
        explicit operator bool() const noexcept { return !full_path.empty(); }
        void clear() noexcept { full_path.clear(); file_name.clear(); }
        void set(std::string_view path) { full_path = path; file_name = FileNameFromPath(path); }
    };
    std::vector<RecentFile> recent_files;

    const LanguageData *lang = nullptr; //The current language
    LoadDataSet load_data;              //Load information per language to estimate progress
    std::chrono::milliseconds last_csh_time = std::chrono::milliseconds::zero();      //How long does a re-coloring take
    std::chrono::milliseconds last_compile_time = std::chrono::milliseconds::zero();  //How long does a recompilation last
    CompileData compile_data;
    std::unique_ptr<Chart> pChart; //the currently compiled chart
    int main_file_no = 0;          //The number of the currently edited file in pChart->Errors.Files
    int selected_error = -1;       //The currently selected error
    HintData hint;                 //States related to the hint window
    std::string control_state;     //The collapse/expand status coming from the GUI

    struct Texture {
        ImTextureID texture; ///<The backend specific texture ID
        ImVec2 size;         ///<The size of the texture bitmap in pixels (already zoomed to the current zoom)
    };
    struct Cache {
        std::optional<Texture> chart;                            //The chart drawn at the current zoom
        std::array<std::optional<Texture>, int(EGUIControlType::MAX)> controls; //The user controls drawn at the current zoom
    };
    Cache compile_cache;
    /** A tracking outline/rectangle */
    struct Overlay {
        const Element *element;// The element this overlay represents
        float fade;            // animation status: the alpha channel to use. 
        bool do_fade_out;      // if true, we will slowly fade out
        ImVec2 offset;         // The offset from which this texture is to be shown (in the current zoom)
        std::optional<Texture> texture; // The texture to show (in the current zoom)
        Overlay(const Element *e, float f, bool d) : element(e), fade(between(f, 0.f, 1.f)), do_fade_out(d) {};
    };
    std::vector<Overlay> overlays;
    /** An element having user controls */
    struct Control {
        Element* element; // The element the controls belong to
        float size;       //[0.01..1] animation status. 1=fully shown
    };
    std::vector<Control> controls;
    std::unordered_set<ImTextureID> textures_to_destroy; //destroy on the next frame, when it is certain not to be used.

    bool pedantic = false;
    bool technical_info = true; //show technical info among errors
    bool warnings = true;      //show warnings among errors
    bool instant_compile = true;
    std::string forced_layout;
    float zoom = 1;
    bool do_fit_window = false;   //After compilation fit to window
    bool do_fit_width = false;    //After compilation fit to window width
    unsigned page = 0;            //page shown (0 is all)
    bool tracking_mode = false; 
    bool show_controls = true;
    bool auto_heading = true;
    bool presentation_mode = false;
    bool auto_hint = true;
    float editor_font_scale = 1.0f;    
    bool error_squiggles = true;
    bool smart_indent = true;
    bool auto_save = true;
    static constexpr std::chrono::milliseconds autosave_interval = 1s;

    bool exit_requested = false;  //someone wanted us to exit
    clock::time_point ini_last_saved = clock::now();
    clock::time_point last_autosave = clock::now();
    bool show_search = false;  //Is the search window (intended to be) open

    void Init(const GUIInit &_G);
    void LoadSettings();
    void SaveSettings(bool force);

    std::optional<RecentFile> RecentFilesAsMenu() const;
    void SetFileDialogColors() const;
    StatusRet New(bool select_language);
    StatusRet Load(const RecentFile *load_this = nullptr);
    StatusRet CheckOverwrite(const char *message);
    enum class SaveMode { 
        As,           ///<always open a dialog to ask filename
        Interactive,  ///<Open a dialog to ask name, if no name. Display dialog on error.
        Quiet         ///<Do not save if no filename. Do not show dialog on error.
    };
    StatusRet Save(SaveMode mode = SaveMode::Interactive);
    static int SmartIndentStatic(void *data, TextEditor::Lines &lines, char character, bool shift,
                            const TextEditor::Coordinates & from, const TextEditor::Coordinates &till) 
        { return ((ConfigData *)data)->SmartIndent(lines, character, shift, from, till); }
    int SmartIndent(TextEditor::Lines &lines, char character, bool shift,
                     const TextEditor::Coordinates &from, const TextEditor::Coordinates &till);
    static void ColorizeStatic(TextEditor::Lines &lines, void *data) { return ((ConfigData *)data)->Colorize(lines); }
    void Colorize(TextEditor::Lines &lines);
    bool ProcessHints(Csh &csh);

    void ResetText(const LanguageData *l);
    void StartCompile(); //Stops current compilation (blocks until done) & starts a new one. Returns true if we could start.
    bool CompileTakeResult(bool block); //return true if a new chart has been successfully compiled
    void DrawCache(); //(re)draw the cache parts that are invalid - leave valid ones intact
    void SetZoom(float z) { if (z==zoom) return; zoom = z; ResetCompileCache(); };
    float fit_to_window_zoom(const ImVec2 &canvas) const noexcept { return (float)std::min(canvas.x/pChart->GetTotal().x.Spans(), canvas.y/(pChart->GetTotal().y.Spans() + pChart->GetCopyrightTextHeight())); }
    float fit_to_width_zoom(const ImVec2 &canvas) const noexcept { return float(canvas.x/pChart->GetTotal().x.Spans()); }
    void FitToWindow(const ImVec2 &canvas) noexcept { if (pChart && pChart->GetTotal().x.Spans() && pChart->GetTotal().y.Spans()) SetZoom(fit_to_window_zoom(canvas)); }
    void FitToWidth(const ImVec2 &canvas) noexcept { if (pChart && pChart->GetTotal().x.Spans() && pChart->GetTotal().y.Spans()) SetZoom(fit_to_width_zoom(canvas)); }
    bool IsFittedToWindow(const ImVec2 &canvas) const noexcept { return !pChart || !pChart->GetTotal().x.Spans() || !pChart->GetTotal().y.Spans() || fit_to_window_zoom(canvas)==zoom; } //returns true if chart not compiled
    bool IsFittedToWidth(const ImVec2 &canvas) const noexcept { return !pChart || !pChart->GetTotal().x.Spans() || !pChart->GetTotal().y.Spans() || fit_to_width_zoom(canvas)==zoom; } //returns true if chart not compiled
    void Animate(const Element *element, float initial_fade = 1, bool do_fade_out = true);
    void StartFadeOut(const Element *element) noexcept { if (ConfigData::Overlay *o = GetAnimation(element)) o->do_fade_out = true; }
    const Overlay *GetAnimation(const Element *element) const { auto i = std::ranges::find_if(overlays, [element](const Overlay &o) {return o.element==element; }); return i !=overlays.end() ? &*i : nullptr; }
    Overlay *GetAnimation(const Element *element) { auto i = std::ranges::find_if(overlays, [element](const Overlay &o) {return o.element==element; }); return i !=overlays.end() ? &*i : nullptr; }
    void PruneAnimations();
    void ShowControls(Element* element);
    void ControlClicked(Element* element, EGUIControlType t);

    void ResetCompileCache();
    void DestroyTexture(ImTextureID texture) { textures_to_destroy.insert(texture); }
    void CleanupTextures() { for (ImTextureID t : textures_to_destroy) ::DestroyTexture(t); textures_to_destroy.clear(); }

    void AddRecentFile(RecentFile &&f) {
        std::erase(recent_files, f);
        recent_files.push_back(std::move(f));
    }
    void DelRecentFile(const std::string &fn) {
        std::erase_if(recent_files, [&fn](const RecentFile &r) { return r.full_path==fn; });
    }
    bool IsDirty() const noexcept { return lang && saved_at != editor.GetUndoBufferIndex(); }
    bool IsCompiled() const noexcept { return compiled_at == editor.GetUndoBufferIndex(); }
    void PostEditorRender();
    void GotoError() {
        if (!pChart || selected_error<0) return;
        const ErrorElement *E = pChart->Error.GetError((unsigned)selected_error, warnings, technical_info);
        if (E->relevant_line.file>=0 && E->relevant_line.line>=1 && E->relevant_line.col>=1)
            editor.SetCursorPosition(TextEditor::Coordinates(E->relevant_line.line-1, E->relevant_line.col-1));
    }
    void NavigateErrors(bool up, bool detailed) noexcept;
    float GetHeadingSize() noexcept;
    //returns a mouse position in chart space
    std::optional<XY> MouseToChart(const ImVec2& mouse) const noexcept;
    //returns what chart element is under the mouse. 
    //mouse shall be the coordinate on the cache canvas (not yet normalized to chart space)
    const Area *GetArcByMouse(const ImVec2 &mouse) const noexcept;
    Element *GetArcByEditorPos(const TextEditor::Coordinates &pos) const noexcept;
    bool SetCursorTo(const Element *E); //returns true if we have changed the cursor
    
    void HintStart() {
        editor.ReColor();
        hint.restart_hint = false;
        hint.is_user_requested = true;
        hint.active = ProcessHints(*hint.csh);
        if (hint.active) {
            const auto cursor = editor.GetCursorPosition();
            hint.till_cursor_only = editor.FindWordStart(cursor)==cursor;
        } else
            HintCancel(true);
    }
    void HintCancel(bool force);
    void HintSubstitute(const CshHint *hint);

    GUIReturn Ret();
};


//Extract up until the first zero
//display error and return empty if text is empty.
//Move 'text' to the char after the zero.
std::optional<std::string_view> extract(std::string_view &text, const GUIInit *G) {
    if (text.empty()) {
        std::cerr << "File '" << G->settings_dir << MSC_GEN_INI_FILE_NAME << "' is too short. Using what I can."<< std::endl;
        return {};
    }
    const size_t pos = text.find('\0');
    std::string_view ret;
    if (pos == text.npos) { ret = text; text.remove_prefix(text.size()); }
    else { ret = text.substr(0, pos); text.remove_prefix(pos+1); }
    return ret;
}


constexpr float min_editor_scale = 0.2f, max_editor_scale = 5;
#define MSC_GENERATOR_INI_INTRO "Msc-generator ini file\n"

void ConfigData::LoadSettings() {
    FILE *in = G->file_open_proc((G->settings_dir + MSC_GEN_INI_FILE_NAME).c_str(), false, true); //true=binary Keep \n on Windows, too.
    if (!in) return;
    std::string stext = ReadFile(in);
    fclose(in);
    std::string_view text = stext;
    if (!text.starts_with(MSC_GENERATOR_INI_INTRO)) {
        std::cerr << "File '" << G->settings_dir << MSC_GEN_INI_FILE_NAME << "' does not seem to be an Msc-generator ini file. Ignoring it."<< std::endl;
        return;
    }
    text.remove_prefix(strlen(MSC_GENERATOR_INI_INTRO));
    //Read in until first zero
    size_t pos = text.find('\0');
    if (pos==text.npos) pos = text.length();
    ImGui::LoadIniSettingsFromMemory(text.data(), pos);
    text.remove_prefix(std::min(pos+1, text.length()));
    //First byte is version
    if (text.empty() || text.front()!='0') {
        std::cerr << "File '" << G->settings_dir << MSC_GEN_INI_FILE_NAME << "' is of a newer version that I support. Ignoring some of it."<< std::endl;
        return;
    }
    text.remove_prefix(1);
    //Now come the newline terminated recent file list, terminated by a zero
    while (text.size() && text.front()!='\0') {
        RecentFile r;
        size_t pos = text.find('\n');
        if (pos == text.npos) pos = text.size();
        r.full_path.assign(text.substr(0, pos));
        text.remove_prefix(std::min(pos+1, text.length()));
        pos = text.find('\n');
        if (pos == text.npos) pos = text.size();
        r.file_name.assign(text.substr(0, pos));
        text.remove_prefix(std::min(pos+1, text.length()));
        recent_files.push_back(std::move(r));
    }
    if (text.size()) text.remove_prefix(1); //swallow terminating zero
    if (auto e = extract(text, G); !e) return;
    else pedantic = *e=="1";
    if (auto e = extract(text, G); !e) return;
    else technical_info = *e=="1";
    if (auto e = extract(text, G); !e) return;
    else warnings = *e=="1";
    if (auto e = extract(text, G); !e) return;
    else instant_compile = *e=="1";
    if (auto e = extract(text, G); !e) return;
    else do_fit_width = *e=="1", do_fit_window = *e=="2";
    if (auto e = extract(text, G); !e) return;
    else auto_heading = *e=="1";
    if (auto e = extract(text, G); !e) return;
    else auto_hint = *e=="1";
    if (auto e = extract(text, G); !e) return;
    else if (double s; from_chars(*e, s)) std::cerr<<"Could not parse editor scale from '"<<*e<<"'. Ignoring this setting."<<std::endl;
    else editor_font_scale = between(float(s), min_editor_scale, max_editor_scale);
    if (auto e = extract(text, G); !e) return;
    else error_squiggles = *e=="1";
    if (auto e = extract(text, G); !e) return;
    else smart_indent = *e=="1";
    if (auto e = extract(text, G); !e) return;
    else if (from_chars(*e, G->languages.cshParams->m_II)) std::cerr<<"Could not parse instruction indent from '"<<*e<<"'. Ignoring this setting."<<std::endl;
    if (auto e = extract(text, G); !e) return;
    else if (from_chars(*e, G->languages.cshParams->m_IB)) std::cerr<<"Could not parse block indent from '"<<*e<<"'. Ignoring this setting."<<std::endl;
    if (auto e = extract(text, G); !e) return;
    else if (from_chars(*e, G->languages.cshParams->m_IM)) std::cerr<<"Could not parse middle indent from '"<<*e<<"'. Ignoring this setting."<<std::endl;
    if (auto e = extract(text, G); !e) return;
    else if (from_chars(*e, G->languages.cshParams->m_IA)) std::cerr<<"Could not parse attribute indent from '"<<*e<<"'. Ignoring this setting."<<std::endl;
    if (auto e = extract(text, G); !e) return;
    else G->languages.cshParams->m_bSIText = *e=="1"; //special indent for labels
    if (auto e = extract(text, G); !e) return;
    else G->languages.cshParams->m_bSIAttr = *e=="1"; //special indent for attributes
    if (auto e = extract(text, G); !e) return;
    else if (int s;  from_chars(*e, s)) std::cerr<<"Could not parse color_scheme from '"<<*e<<"'. Ignoring this setting."<<std::endl;
    else G->languages.cshParams->color_scheme = between(s, 0, 3);
    //Settings introduced in v7.1
    if (auto e = extract(text, G); !e) return;
    else show_controls = *e=="1";
    if (auto e = extract(text, G); !e) return;
    else auto_save = *e=="1";
    //Leave remainder of settings file unloaded (forward compatibility)
}

void ConfigData::SaveSettings(bool force) {
    ImGuiIO &io = ImGui::GetIO();
    if (!io.WantSaveIniSettings && !force && clock::now()-ini_last_saved<10s)
        return;
    io.WantSaveIniSettings = false;
    ini_last_saved = clock::now();

    FILE *out = G->file_open_proc((G->settings_dir + MSC_GEN_INI_FILE_NAME).c_str(), true, true); //true=binary Keep \n on Windows, too.
    static int error_reported = 0;
    if (!out) {
        constexpr int max_error_report = 3;
        if (error_reported<max_error_report) {
            std::cerr<< "Unable to write to settings file '" + G->settings_dir << MSC_GEN_INI_FILE_NAME << "': " << my_strerror(errno) << std::endl;
            if (++error_reported == max_error_report)
                std::cerr<< "I will not report further settings file save problems." << std::endl;
        }
        return;
    } else
        error_reported = 0;
    fprintf(out, MSC_GENERATOR_INI_INTRO); //intro 
    size_t imgui_ini_len;
    const char *const imgui_ini_text = ImGui::SaveIniSettingsToMemory(&imgui_ini_len);
    fwrite(imgui_ini_text, 1, imgui_ini_len, out);
    fputc('\0', out);
    fputc('0', out); //version
    std::string config;
    for (const RecentFile &r : recent_files) {
        config.append(r.full_path).push_back('\n');
        config.append(r.file_name).push_back('\n');
    }
    config.push_back('\0');
    fwrite(config.data(), 1, config.size(), out);

    if (pedantic) fputc('1', out);
    fputc('\0', out);
    if (technical_info) fputc('1', out);
    fputc('\0', out);
    if (warnings) fputc('1', out);
    fputc('\0', out);
    if (instant_compile) fputc('1', out);
    fputc('\0', out);
    if (do_fit_width) fputc('1', out);
    else if (do_fit_window) fputc('2', out);
    fputc('\0', out);
    if (auto_heading) fputc('1', out);
    fputc('\0', out);
    if (auto_hint) fputc('1', out);
    fputc('\0', out);
    fprintf(out, "%f", editor_font_scale);
    fputc('\0', out);
    if (error_squiggles) fputc('1', out);
    fputc('\0', out);
    if (smart_indent) fputc('1', out);
    fputc('\0', out);
    fprintf(out, "%d", G->languages.cshParams->m_II);
    fputc('\0', out);
    fprintf(out, "%d", G->languages.cshParams->m_IB);
    fputc('\0', out);
    fprintf(out, "%d", G->languages.cshParams->m_IM);
    fputc('\0', out);
    fprintf(out, "%d", G->languages.cshParams->m_IA);
    fputc('\0', out);
    if (G->languages.cshParams->m_bSIText) fputc('1', out);
    fputc('\0', out);
    if (G->languages.cshParams->m_bSIAttr) fputc('1', out);
    fputc('\0', out);
    fprintf(out, "%d", G->languages.cshParams->color_scheme);
    fputc('\0', out);
    //Settings introduced in v7.1
    if (show_controls) fputc('1', out);
    fputc('\0', out);
    if (auto_save) fputc('1', out);
    fputc('\0', out);

    fclose(out);
}

void ConfigData::Init(const GUIInit &_G) {
    if (G) return;
    G = &_G;
    MscGenEditorFontSize = 13 * dpi_mul;
    MscGenWindowFontSize = 16 * dpi_mul;
    LoadFonts();
    LoadSettings();

    std::string all_exts_machine;
    for (const std::string &lname : G->languages.GetLanguages()) {
        auto l = G->languages.GetLanguage(lname);
        std::string exts_human, exts_machine;
        for (const std::string &e : l->extensions) {
            if (exts_human.empty()) exts_human = "*."+e;
            else exts_human.append(" *.").append(e);
            if (exts_machine.empty()) exts_machine = "."+e;
            else exts_machine.append(",.").append(e);
        }
        open_filter_groups.append(l->description).append("s (").append(exts_human).append("){")
                          .append(exts_machine).append("},");
        if (all_exts_machine.empty()) all_exts_machine = std::move(exts_machine);
        else all_exts_machine.append(",").append(exts_machine);
    }
    open_filter_groups.append(",Embedded charts (*.png){.png}");
    all_exts_machine.append(",.png");
    open_filter_groups = "All chart types{"+all_exts_machine+"},"+open_filter_groups;

    MscError Error;
    FileLineCol opt_pos;
    load_data = SplitLoadData(G->load_data, &Error, &opt_pos);
    std::cerr << Error.Print(true, false); //nothing if no error

    editor.mIndent = smart_indent ? &ConfigData::SmartIndentStatic : nullptr;
    editor.mIndentData = this;
    editor.mIndentChars = "\n\t\x08{}[]";
    editor.mColorize = &ColorizeStatic;
    editor.mColorizeData = this;
    editor.SetShowWhitespaces(false);
    editor.SetFonts(GetFont(false, false, false), GetFont(false, true, false),
                    GetFont(false, false, true), GetFont(false, true, true), MscGenEditorFontSize*editor_font_scale,
                    GetFont(true, false, false));
    editor.mScale = editor_font_scale;
    editor.SetPalette(MscCshAppearanceList[G->languages.cshParams->color_scheme]);

    SetFileDialogColors();
}

void ConfigData::ResetText(const LanguageData *l) {
    lang = l;
    editor.SetText(lang->default_text);
    file_name = "Untitled."+lang->extensions.front();
    full_path.clear();
    saved_at = editor.GetUndoBufferIndex();
    StartCompile();
    window_title = file_name;
}

void Help(const char *desc) {
    if (ImGui::IsItemHovered()) {
        ImGui::SetNextWindowPos(ImGui::GetIO().MousePos + ImVec2(16, 16)); //default pos would be +16,+8
        ImGui::PushStyleVar(ImGuiStyleVar_WindowRounding, 4);
        ImGui::BeginTooltip();
        ImGui::PushTextWrapPos(ImGui::GetFontSize() * 35.0f);
        ImGui::TextUnformatted(desc);
        ImGui::PopTextWrapPos();
        ImGui::EndTooltip();
        ImGui::PopStyleVar();
    }
}
std::optional<ConfigData::RecentFile> ConfigData::RecentFilesAsMenu() const {
    std::optional<ConfigData::RecentFile> ret;
    if (recent_files.empty())
        ImGui::MenuItem("...No recently opened files...", "", nullptr, false);
    else
        for (auto i = recent_files.rbegin(); i!=recent_files.rend(); i++) {
            if (ImGui::MenuItem(i->file_name.c_str(), "", nullptr))
                ret = *i; //Do not Load here as it will screw Config.recent_files;
            Help(i->full_path.c_str());
        }
    return ret;
}

void ConfigData::SetFileDialogColors() const {
    static constexpr std::array colors = {
        IM_COL32(92,0,0,255),
        IM_COL32(0,92,0,255),
        IM_COL32(0,0,92,255),
        IM_COL32(0,60,60,255),
        IM_COL32(60,0,60,255),
        IM_COL32(60,60,0,255),
    };
    if (!G) return;
    const auto langs = G->languages.GetLanguages();
    for (const std::string& lang : langs) {
        const ImU32 col = colors[(&lang-langs.data())%colors.size()];
        for (const std::string& ext : G->languages.GetLanguage(lang)->extensions) {
            const std::string dot_ext = "."+ext;
            ImGuiFileDialog::Instance()->SetFileStyle(IGFD_FileStyleByExtention, dot_ext.c_str(), ImColor(col).Value);
        }
    }
    ImGuiFileDialog::Instance()->SetFileStyle(IGFD_FileStyleByExtention, ".png", ImColor(colors[langs.size()%colors.size()]).Value);
    ImGuiFileDialog::Instance()->SetFileStyle(IGFD_FileStyleByTypeDir, "", ImColor(0, 0, 0), "", GetFont(true, true, false));
}

bool escape_pressed = false;

StatusRet ConfigData::New(bool select_language) {
    enum Status {
        INITIAL,
        CHECK_OVERWRITE,
        SELECT_LANGUAGE,
        SELECT_FILE,
        LOAD,
        DO_NEW
    };
    static Status status = INITIAL;
    static std::optional<ConfigData::RecentFile> load_this; //When status is LOAD, this is the file we attempt
    if (status == INITIAL) {
        load_this.reset(); //dont keep old selections hanging around, just because.
        if (IsDirty()) status = CHECK_OVERWRITE;
        else status = SELECT_LANGUAGE;
    }
    if (status==CHECK_OVERWRITE) switch (CheckOverwrite("%s is changed. Do you want to save it?")) {
    case StatusRet::Ongoing: return StatusRet::Ongoing;
    case StatusRet::Cancelled: return StatusRet::Cancelled; //CheckOverwrite reports Cancelled if user selected save and cancelled
    case StatusRet::Completed: status = SELECT_LANGUAGE;
    }
    if (status==SELECT_LANGUAGE)
        if (!select_language && (lang || G->type.size()))
            status = DO_NEW;

    StatusRet ret = StatusRet::Ongoing;
    if (status==SELECT_LANGUAGE) {
        ImGui::OpenPopup(lang ? "Select Language" : "Welcome to Msc-generator");
        // Always center this window when appearing
        ImVec2 center = ImGui::GetMainViewport()->GetCenter();
        ImGui::SetNextWindowPos(center, ImGuiCond_Always, ImVec2(0.5f, 0.5f));
        if (ImGui::BeginPopupModal(lang ? "Select Language" : "Welcome to Msc-generator", NULL,
                                   ImGuiWindowFlags_AlwaysAutoResize | ImGuiWindowFlags_NoScrollbar
                                   | ImGuiWindowFlags_NoScrollWithMouse | ImGuiWindowFlags_NoCollapse
                                   | ImGuiWindowFlags_NoSavedSettings)) {
            const bool do_welcome = lang==nullptr; //lang will change in ResetText() below, so we save this
            const ImVec2 button_size{150 * dpi_mul, 0};
            ImGui::SetItemDefaultFocus(); //Whatever button is first
            if (do_welcome) {
                ImGui::BeginTable("Welcome", 2, ImGuiTableFlags_SizingStretchProp
                                  | ImGuiTableFlags_BordersInnerV);
                ImGui::TableNextRow();
                ImGui::TableNextColumn();
                ImGui::Text("Open Existing\n");
                if (ImGui::Button("Open", button_size)) {
                    ImGui::CloseCurrentPopup();
                    status = SELECT_FILE;
                }
                ImGui::Spacing();
                ImGui::Spacing();
                ImGui::Text("Create new\n");
            } else
                ImGui::Text("Available languages:\n");
            for (auto &lname : G->languages.GetLanguages())
                if (auto *l = G->languages.GetLanguage(lname);  ImGui::Button(l->description.c_str(), button_size)) {
                    ImGui::CloseCurrentPopup();
                    ResetText(l);
                    ret = StatusRet::Completed;
                }
            ImGui::Spacing();
            ImGui::Spacing();
            if (ImGui::Button(lang ? "Cancel" : "Exit", button_size) || escape_pressed) {
                ImGui::CloseCurrentPopup();
                exit_requested = lang==nullptr; //User has cancelled initial language selection - we exit
                ret = StatusRet::Cancelled;
            }
            if (do_welcome) {
                ImGui::TableNextColumn();
                ImGui::Text("Open Recent\n");
                ImGui::Indent(10.f * dpi_mul);
                if (ImGui::BeginListBox("##Recent", {200 * dpi_mul, 250 * dpi_mul})) {
                    load_this = RecentFilesAsMenu();
                    if (load_this) {
                        status = LOAD;
                        ImGui::CloseCurrentPopup();
                    }
                    ImGui::EndListBox();
                }
                ImGui::EndTable();
            }
            ImGui::EndPopup();
        }
    } 
    if (status == SELECT_FILE)
        ret = Load();
    if (status == LOAD) {
        _ASSERT(load_this);
        ret = Load(load_this ? &*load_this : nullptr);  //load_this should have a value here
    }
    if (status==DO_NEW) {
        const LanguageData *l = G->languages.GetLanguage(lang ? lang->GetName() : G->type);
        if (!l) l = G->languages.GetLanguage(G->languages.GetLanguages().front());
        ResetText(l);
        ret = StatusRet::Completed;
    }
    if (ret != StatusRet::Ongoing) status = INITIAL;

    return ret;
}

bool GUIError(const char *msg, bool append_errno = false) {
    bool proceed = false;
    ImGui::OpenPopup("Msc-generator");
    // Always center this window when appearing
    ImVec2 center = ImGui::GetMainViewport()->GetCenter();
    ImGui::SetNextWindowPos(center, ImGuiCond_Appearing, ImVec2(0.5f, 0.5f));
    if (ImGui::BeginPopupModal("Msc-generator", nullptr, ImGuiWindowFlags_AlwaysAutoResize)) {
        std::string storage;
        if (append_errno && errno) {
            storage = std::string(msg) + " (" + my_strerror(errno) + ")";
            msg = storage.c_str();
        }
        ImGui::Text("%s", msg);
        if (ImGui::Button("OK") || escape_pressed) {
            ImGui::CloseCurrentPopup();
            proceed = true;
        }
        ImGui::EndPopup();
    }
    return proceed;
}

StatusRet ConfigData::Load(const RecentFile *load_this) {
    enum Status {
        INITIAL,
        SELECT,
        CHECK_OVERWRITE,
        LOAD,
        SHOW_ERROR,
        CANCELLED,
    };
    static Status status = INITIAL;
    static std::string errorText;
    static bool errorIncludeErrno = false;
    static std::string path, name;
    if (status == INITIAL) {
        if (load_this) {
            path = load_this->full_path;
            name = load_this->file_name;
            status = CHECK_OVERWRITE;
        } else {
            errorText.clear();
            path.clear();
            name.clear();
            errorIncludeErrno = false;
            status = SELECT;
        }
    }
    if (status == SELECT) {
        ImGuiFileDialog::Instance()->OpenModal("FileDialog", "Open file", open_filter_groups.c_str(), "");
        if (ImGuiFileDialog::Instance()->Display("FileDialog", ImGuiWindowFlags_NoCollapse, ImVec2(0, 150))) {
            if (ImGuiFileDialog::Instance()->IsOk()) {
                path = ImGuiFileDialog::Instance()->GetFilePathName();
                name = ImGuiFileDialog::Instance()->GetCurrentFileName();
                status = CHECK_OVERWRITE;
            } else
                status = INITIAL;
            ImGuiFileDialog::Instance()->Close();
        }
        if (escape_pressed) {
            status = INITIAL;
            ImGuiFileDialog::Instance()->Close();
        }
    }
    if (status == CHECK_OVERWRITE) switch (CheckOverwrite("%s is changed. Do you want to save it?")) {
    case StatusRet::Ongoing: break;
    case StatusRet::Cancelled: status = CANCELLED; break;
    case StatusRet::Completed: status = LOAD; break;
    }
    if (status == LOAD) { 
        const size_t dot_pos = path.find_last_of('.');
        if (dot_pos == path.npos) {
            DelRecentFile(path);
            errorText = "Could not recognize file type.";
        } else {
            std::string type(path.substr(dot_pos+1));
            std::string text;
            if (type=="png") {
                if (FILE* in = G->file_open_proc(path.c_str(), false, true)) {
                    MscError Error;
                    std::tie(type, text) = pngutil::Select(in, G->languages, Error);
                    fclose(in);
                    if (Error.hasErrors()) {
                        DelRecentFile(path);
                        errorText = Error.GetError(0, false, false)->message;
                    }
                } else {
                    DelRecentFile(path);
                    errorText = "Could not open '"+path+"'. ";
                    errorIncludeErrno = true;
                }
            } else if (G->languages.GetLanguage(type) == nullptr) {
                DelRecentFile(path);
                errorText = "Extension '"+type+"' is not a language I support.";
            } else if (FILE *in = G->file_open_proc(path.c_str(), false, false)) {
                text = ReadFile(in);
                fclose(in);
            } else {
                DelRecentFile(path);
                errorText = "Could not open '"+path+"'. ";
                errorIncludeErrno = true;
            }
            if (errorText.empty()) {
                if (auto l = G->languages.GetLanguage(type)) { //re-lookup as for PNG we did not
                    lang = l;
                    editor.SetText(text);
                    full_path = path;
                    file_name = name;
                    saved_at = editor.GetUndoBufferIndex();
                    StartCompile();
                    AddRecentFile({path, name});
                    window_title = name;
                } else {
                    DelRecentFile(path);
                    errorText = "Extension '"+type+"' is not a language I support.";
                }
            }
        }
        status = errorText.empty() ? INITIAL : SHOW_ERROR;
    }
    if (status == SHOW_ERROR)
        if (GUIError(errorText.c_str(), errorIncludeErrno))
            status = INITIAL;
    switch (status) {
    case INITIAL: return StatusRet::Completed;
    case CANCELLED: status = INITIAL;  return StatusRet::Cancelled;
    default: return StatusRet::Ongoing;
    }    
}

//We return Cancelled if the user pressed Cancel or selected Save and cancelled the selection of the filename.
StatusRet ConfigData::CheckOverwrite(const char *message) {
    static bool ask = true; //this is our status
    if (!IsDirty()) {
        ask = true;
        return StatusRet::Completed;
    }
    StatusRet ret = StatusRet::Ongoing;
    if (ask) {
        ImGui::OpenPopup("Msc-generator");
        // Always center this window when appearing
        ImVec2 center = ImGui::GetMainViewport()->GetCenter();
        ImGui::SetNextWindowPos(center, ImGuiCond_Appearing, ImVec2(0.5f, 0.5f));
        if (ImGui::BeginPopupModal("Msc-generator", NULL, ImGuiWindowFlags_AlwaysAutoResize)) {
            ImGui::Text(message, file_name.c_str());
            if (ImGui::Button("Yes")) {
                ImGui::CloseCurrentPopup();
                ask = false;
            }
            ImGui::SameLine();
            ImGui::SetItemDefaultFocus();
            if (ImGui::Button("No")) {
                ImGui::CloseCurrentPopup();
                ret = StatusRet::Completed;
            }
            ImGui::SameLine();
            if (ImGui::Button("Cancel") || escape_pressed) {
                ImGui::CloseCurrentPopup();
                ret = StatusRet::Cancelled;
            }
            ImGui::EndPopup();
        }
    } else
        ret = Save(SaveMode::Interactive);
    if (ret != StatusRet::Ongoing) ask = true;
    return ret;
}

//If mode==Quiet, we will not return Ongoing, but Completed on success and Cancelled on failure.
StatusRet ConfigData::Save(SaveMode mode) {
    if (!lang) return StatusRet::Completed;
    static std::string _full_path, _file_name, _ext; //names selected by the user - or the old ones (then _ext is empty)
    enum Status {
        INITIAL,          //This function has never been called or the previous workflow has ended
        SELECT,           //The user is currently selecting a filename, the file dialog is open
        CONFIRM_PROBLEM,  //The user has selected a file and we save - or a problematic file and we wait for confirming the choice
        SHOW_ERROR,        //The save had an error and we are waiting for the user to ack it.
        CANCELLED
    };
    static Status status = INITIAL;

    if (status==INITIAL) {
        if (mode==SaveMode::As || (mode==SaveMode::Interactive && full_path.empty())) {
            _ext.clear();
            status = SELECT;
        } else if (mode!=SaveMode::Quiet || full_path.size()) {
            //Re-use existing file name
            _full_path = full_path;
            _file_name = file_name;
            const size_t dot_pos = _file_name.find_last_of('.');
            _ext = dot_pos!=_file_name.npos
                ? _file_name.substr(dot_pos+1) //extension of the current filename
                : lang->extensions.front();    //or the canonical extension
            status = CONFIRM_PROBLEM;
        } //else if quiet and no filename, keep INITIAL status - do nothing
    }
    static TextEditor::UndoBufferIndex  editor_undo_buffer_when_selected_file;
    if (status==SELECT) {
        std::string filter = lang->description + " (";
        for (const std::string &e : lang->extensions)
            filter.append("*.").append(e).push_back(' ');
        filter.back() = ')';
        filter.append("{");
            for (const std::string &e : lang->extensions)
                filter.append(".").append(e).push_back(',');
        filter.back() = '}';
        std::string filter2 = "Embedded chart (*.png){.png}";
        if (file_name.ends_with(".png")) std::swap(filter, filter2); //'Save as' on a png shall offer PNG as primary extension
        filter.append(",").append(filter2);
        ImGuiFileDialog::Instance()->OpenModal("FileDialog", SaveMode::As == mode ? "Save file as" : "Save file",
                                               filter.c_str(), full_path, 1, nullptr, ImGuiFileDialogFlags_ConfirmOverwrite);
        if (ImGuiFileDialog::Instance()->Display("FileDialog", ImGuiWindowFlags_NoCollapse, ImVec2(0, 150))) {
            if (ImGuiFileDialog::Instance()->IsOk()) {
                _full_path = ImGuiFileDialog::Instance()->GetFilePathName();
                _file_name = ImGuiFileDialog::Instance()->GetCurrentFileName();
                const bool filter_is_png = ImGuiFileDialog::Instance()->GetCurrentFilter().find("*.png")!=std::string::npos;
                const size_t dot_pos = _file_name.find_last_of('.');
                _ext = dot_pos!=_file_name.npos
                    ? _file_name.substr(dot_pos+1)                     //user supplied extension
                    : filter_is_png ? "png" : lang->extensions.front(); //png or the canonical extension
                if (dot_pos==_file_name.npos) {
                    _file_name.append(".").append(_ext);
                    _full_path.append(".").append(_ext);
                }
                status = CONFIRM_PROBLEM;
                editor_undo_buffer_when_selected_file = editor.GetUndoBufferIndex();
            } else
                status = CANCELLED;
            ImGuiFileDialog::Instance()->Close();
        }
        if (escape_pressed) {
            status = INITIAL;
            ImGuiFileDialog::Instance()->Close();
        }
    }
    if (status==CONFIRM_PROBLEM) {
        if (_ext=="png" && mode != SaveMode::Quiet) { //do not recompile on quiet 
            if (!pChart || editor_undo_buffer_when_selected_file != compiled_at) {
                if (!compile_data.is_compiling()) {
                    editor_undo_buffer_when_selected_file = compiled_at;
                    StartCompile();
                } //result will be taken in the main loop
            }
            if (pChart && editor_undo_buffer_when_selected_file == compiled_at) {
                if (pChart->Error.hasErrors()) {
                    ImGui::OpenPopup("Warning");
                    // Always center this window when appearing
                    ImVec2 center = ImGui::GetMainViewport()->GetCenter();
                    ImGui::SetNextWindowPos(center, ImGuiCond_Appearing, ImVec2(0.5f, 0.5f));
                    if (ImGui::BeginPopupModal("Warning")) {
                        ImGui::Text("The chart had errors, do you want to save it into a PNG nevertheless ?");
                        ImGui::SetItemDefaultFocus();
                        if (ImGui::Button("Save")) {
                            ImGui::CloseCurrentPopup();
                            status = SHOW_ERROR;
                        }
                        ImGui::SameLine();
                        if (ImGui::Button("Cancel") || escape_pressed) {
                            ImGui::CloseCurrentPopup();
                            status = CANCELLED;
                        }
                        ImGui::EndPopup();
                    }
                } else
                    status = SHOW_ERROR;
            }
        } else {
            if (std::ranges::find(lang->extensions, _ext)==lang->extensions.end() 
                && !CaseInsensitiveEqual(_ext, "png")) {
                if (mode == SaveMode::Quiet)
                    status = CANCELLED;
                else {
                    ImGui::OpenPopup("Warning");
                    // Always center this window when appearing
                    ImVec2 center = ImGui::GetMainViewport()->GetCenter();
                    ImGui::SetNextWindowPos(center, ImGuiCond_Appearing, ImVec2(0.5f, 0.5f));
                    if (ImGui::BeginPopupModal("Warning")) {
                        ImGui::Text("Changing the file extension to one not supported by this language.");
                        ImGui::Text("Consider using one of %s.", PrintStrings(lang->extensions).c_str());
                        ImGui::SetItemDefaultFocus();
                        if (ImGui::Button(("Use "+_ext).c_str())) {
                            ImGui::CloseCurrentPopup();
                            status = SHOW_ERROR;
                        }
                        ImGui::SameLine();
                        if (ImGui::Button("Cancel") || escape_pressed) {
                            ImGui::CloseCurrentPopup();
                            status = CANCELLED;
                        }
                        ImGui::EndPopup();
                    }
                }
            } else 
                status = SHOW_ERROR;
        }
        if (status == SHOW_ERROR) { 
            bool write_ok = false;
            if (_ext=="png") {
                if (pChart) //maybe false on quiet save
                    write_ok = pChart->DrawToFile(Canvas::PNG, XY(1, 1), _full_path, true, true, editor.GetText().c_str());
            } else {
                FILE *out = G->file_open_proc(_full_path.c_str(), true, false);
                std::string text = editor.GetText();
                write_ok = text.size() == fwrite(text.data(), 1, text.size(), out);
                fclose(out);
            }
            if (write_ok) {
                full_path = std::move(_full_path);
                file_name = std::move(_file_name);
                saved_at = editor.GetUndoBufferIndex();
                AddRecentFile({full_path, file_name});
                window_title = file_name;
                status = INITIAL;
            }
        }
    }
    if (status==SHOW_ERROR)
        if (mode==SaveMode::Quiet || GUIError("Error writing file.", true)) //do not show error on quiet
            status = INITIAL;
    switch (status) {
    case INITIAL: return StatusRet::Completed; //if we are set back to initial, we are done.
    case CANCELLED: status = INITIAL; return StatusRet::Cancelled;
    default: return StatusRet::Ongoing;
    }
}

int get_indent(const TextEditor::Line &L) noexcept {
    int spaces = 0;
    while (spaces<(int)L.size() && (L[spaces].mChar==' ' || L[spaces].mChar=='\t'))
        spaces++;
    return spaces;
}


int ConfigData::SmartIndent(TextEditor::Lines &lines, char character, bool shift,
                             const TextEditor::Coordinates &from, const TextEditor::Coordinates &till) {
    int inserted = 0;
    int char_index = 0;
    for (int l = 0; l<from.mLine; l++)
        char_index += lines[l].num_chars()+1; //+1 for newline
    switch (character) {
    case '\t':
        for (int l = from.mLine; l<=till.mLine; l++) {
            const int spaces = hint.csh->FindProperLineIndent(char_index, false) - get_indent(lines[l]);
            char_index += spaces + lines[l].num_chars() + 1;
            if (spaces==0) continue;
            if (spaces>0) lines[l].insert(lines[l].begin(), spaces, TextEditor::Glyph{' '});
            else lines[l].erase(lines[l].begin(), lines[l].begin()-spaces);
            lines[l].invalidate_num_chars();
            inserted = 1; //dont return actual number, just a bool
            if (l<till.mLine)
                Colorize(lines);
        }
        break;
    case '\x08':
        if (from.mColumn || get_indent(lines[from.mLine])<from.mColumn) //if at beginning of line or not in leading whitespace
            return 0;
        else if (const int needed = hint.csh->FindProperLineIndent(char_index, false)
                    , current = get_indent(lines[from.mLine])
                ; needed >= current) return 0;
        else 
            return current-needed;
    case '{':
    case '}':
    case '[':
    case ']':
        if (get_indent(lines[from.mLine])<from.mColumn-1) //newly inserted char is not the first in line
            break;
        FALLTHROUGH;
    case '\n':
        inserted = hint.csh->FindProperLineIndent(char_index, character=='\n') - get_indent(lines[from.mLine]);
        char_index += inserted + lines[from.mLine].num_chars() + 1;
        if (inserted==0) break;
        if (inserted>0) lines[from.mLine].insert(lines[from.mLine].begin(), inserted, TextEditor::Glyph{' '});
        else lines[from.mLine].erase(lines[from.mLine].begin(), lines[from.mLine].begin()-inserted);
        lines[from.mLine].invalidate_num_chars();
    }
    return inserted;
}

/* Moves line/col and pos one character ahead.
 * When encountering a new line
 * - if stop_after_newline is clear: we jump to the beginning of the next non-empty line or file end.
 * - If stop_after_newline is set: we only jump to the beginning of the next line and return true. */
inline bool Advance(const TextEditor::Lines &lines, size_t &line, size_t &col, size_t &pos, bool stop_after_newline=false) {
    _ASSERT(line<lines.size());
    if (col < lines[line].size()) { //If at a line-end, just skip the newline(s) below
        ++pos;
        col += UTF8TrailingBytes(lines[line][col].mChar)+1;
    }
    while (line<lines.size() && lines[line].size() <= col) { //cycle to process end-of line and also empty lines
        ++line; col = 0; //jump to new line
        pos++; //step over the newline in the input
        if (stop_after_newline) return true;
    } 
    return false;
}

void ConfigData::Colorize(TextEditor::Lines &lines) {
    if (!lang || !lang->pCsh) return;
    const auto started = clock::now();
    hint.csh = lang->pCsh->Clone();
    Csh *csh = hint.csh.get();
    csh->ParseText(editor.GetText(), editor.GetCharPos(editor.GetCursorPosition()), pedantic);
    csh->CshList.SortByPos();
    size_t line = 0, col = 0, pos = 0, mpos = 0;
    for (const CshEntry &c : csh->CshList) {
        while (int(pos+1)<c.first_pos) {
            if (mpos<=pos && col<lines[line].size())
                lines[line][col].mColorIndex = COLOR_NORMAL;
            Advance(lines, line, col, pos);
        }
        size_t mline = line, mcol = col;
        mpos = pos;
        while (int(mpos)<c.last_pos && mline<lines.size()) {
            if (mcol<lines[mline].size())
                lines[mline][mcol].mColorIndex = c.color;
            Advance(lines, mline, mcol, mpos);
        }
    }
    TextEditor::ErrorMarkers errors;
    if (error_squiggles) {
        line = 0, col = 0, pos = 0;
        for (const CshError &c : csh->CshErrors.error_ranges) {
            while (int(pos)<c.first_pos) Advance(lines, line, col, pos);
            TextEditor::Coordinates start(line, col);
            size_t start_pos = pos;
            while (int(pos)<c.last_pos+1 && line<lines.size())
                if (Advance(lines, line, col, pos, true)) //Newline encountered
                    if (start_pos<pos) { //non-empty erroneous part
                        errors[start] = std::pair(pos-start_pos, c.text);
                        start = TextEditor::Coordinates(line, col);
                        start_pos = pos;
                    }
            if (start_pos+1<pos) //non-empty erroneous part remained
                errors[start] = std::pair(pos-start_pos-1, c.text);
        }
    }
    editor.SetErrorMarkers(std::move(errors));
    last_csh_time = std::chrono::duration_cast<std::chrono::milliseconds>(clock::now()-started);
    if (auto_hint                                                           //turn hinting automatically on if
           && csh->hintedStringPos.first_pos<csh->hintedStringPos.last_pos) //there is a word under the cursor.
        hint.active = true;                                                 //This is to avoid popping the hint after an enter.
    if (hint.active && !ProcessHints(*csh))
        HintCancel(true);
}

void ConfigData::StartCompile() {
    selected_error = -1;
    if (!lang) return;
    std::unique_ptr<Chart> chart = lang->create_chart(G->file_read_proc, G->file_read_proc_param);
    if (!chart) return;
    chart->prepare_for_tracking = true;
    chart->prepare_element_controls = true;
    //add shapes
    chart->Shapes = lang->shapes;
    //add errors
    chart->Error = lang->designlib_errors;
    //add designs
    chart->ImportDesigns(lang->designs);
    if (G->languages.cshParams->ForcedDesign.size())
        chart->ApplyForcedDesign(G->languages.cshParams->ForcedDesign);
    //set copyright text
    chart->copyrightText = G->copyright_text;

    //MscChart specific settings
    if (msc::MscChart* msc = dynamic_cast<MscChart*>(chart.get())) {
        //copy pedantic flag from app settings
        msc->mscgen_compat = EMscgenCompat::AUTODETECT;
        msc->Error.warn_mscgen = true;
    }
    //Graph specific settings
    if (graph::GraphChart* graph = dynamic_cast<GraphChart*>(chart.get())) {
        graph->do_orig_lang = false;
        //set layout after setting the default style - this overrides that
        if (forced_layout.size())
            graph->ApplyForcedLayout(forced_layout);
    }
    chart->SetPedantic(pedantic);
    chart->GetProgress()->ReadLoadData(load_data[lang->GetName()]);
    //copy forced collapse/expand entities/boxes/graphs, etc.
    chart->DeserializeGUIState(control_state.c_str());

    compile_data.Start(std::move(chart), editor.GetText(), file_name.empty() ? full_path : file_name, editor.GetUndoBufferIndex());
}


void CompileData::Start(std::unique_ptr<Chart> &&chart, std::string &&text, const std::string &fn,
                        TextEditor::UndoBufferIndex editorbuffer) {
    Stop();
    GetResult(true);
    std::scoped_lock lock(my_mutex);
    if (state.load(std::memory_order_acquire)!=State::Idle)
        return;
    state.store(State::Compiling, std::memory_order_release);
    compiling_started_ended = clock::now();

    result.pChart = std::move(chart);
    result.compiled_at = editorbuffer;

    compile_thread = std::thread([this, text = std::move(text), fn]{
        auto started = clock::now();
        //parse chart text
        result.pChart->GetProgress()->callback = progressbar;
        result.pChart->GetProgress()->data = this;
        try {
            result.pChart->GetProgress()->StartParse();
            result.main_file_no = result.pChart->ParseText(text, fn);
            if (!result.pChart->Error.hasFatal())
                //Do postparse, compile, calculate sizes and sort errors by line
                result.pChart->CompleteParse(Canvas::EMF, false, /* autopaginate =*/ false,
                                             false, XY(0, 0), false, true);
            result.pChart->Error.RemovePathFromErrorMessages();
            result.pChart->GetProgress()->Done();
            state.store(State::Ready, std::memory_order_release);
        } catch (const AbortCompilingException &) {
            state.store(State::Stopped, std::memory_order_release);

        }
        result.compile_time = clock::now()-started;
    });
}

std::optional<CompileData::Result> CompileData::GetResult(bool block) noexcept {
    switch (state.load(std::memory_order_acquire)) {
    case State::Stopped:
    {
        if (compile_thread.joinable())
            compile_thread.join();
        std::scoped_lock lock(my_mutex);
        state.store(State::Idle, std::memory_order_release);
        stop.store(false, std::memory_order_release);
        compiling_started_ended = clock::now();
        result.pChart.release();
        return {};
    }
    default:
    case State::Idle:
        stop.store(false, std::memory_order_release);
        return {};
    case State::Compiling:
        if (!block) return {};
        FALLTHROUGH;
    case State::Ready:
        if (compile_thread.joinable())
            compile_thread.join();
        compile_thread = {};
        std::scoped_lock lock(my_mutex);
        stop.store(false, std::memory_order_release);
        compiling_started_ended = clock::now();
        if (state.exchange(State::Idle, std::memory_order_acq_rel) == State::Ready)
            return std::move(result);
        result.pChart.release();
        return {};
    }
}

bool ConfigData::CompileTakeResult(bool block) {
    auto result = compile_data.GetResult(block);
    if (!result) return false;
    load_data[lang->GetName()] = result->pChart->GetProgress()->WriteLoadData();
    if (G->load_data) *G->load_data = PackLoadData(load_data);
    control_state = result->pChart->SerializeGUIState();
    last_compile_time = std::chrono::duration_cast<std::chrono::milliseconds>(result->compile_time);
    result->pChart->Error.TechnicalInfo({}, StrCat("Compilation time was: ", std::to_string(last_compile_time.count()), "ms"));
    const auto res = result->pChart->GetProgress()->GetUnifiedSectionPercentage();
    //If we did not end on 100%, scale up, so that printed results sum up to 100%
    const double ratio = 100./result->pChart->GetProgress()->GetPercentage();
    for (unsigned u = 0; u<res.size(); u++)
        if (int(round(res[u]*ratio)))
            result->pChart->Error.TechnicalInfo(
                {}, StrCat(result->pChart->GetProgress()->GetUnifiedSectionName(u), ":",
                           std::to_string(int(round(res[u]*ratio))), "% ",
                           std::to_string(int(round(res[u]*ratio*last_compile_time.count()/100.))), "ms"));
    pChart = std::move(result->pChart);
    main_file_no = result->main_file_no;
    compiled_at = result->compiled_at;
    ResetCompileCache();
    overlays.clear();
    if (page>pChart->GetPageNum()) page = pChart->GetPageNum();
    return true;
}


void ConfigData::DrawCache() {
    if (!pChart) return;
    if (!compile_cache.chart) {
        const PBDataVector pageBreakData = pChart->GetPageVector();
        Block b = page==0 ? pChart->GetTotal() : Block(pageBreakData[page-1]->xy, pageBreakData[page-1]->xy + pageBreakData[page-1]->wh);
        if (page && pageBreakData[page-1]->headingLeftingSize.y>0) b.y.from -= pageBreakData[page-1]->headingLeftingSize.y;
        if (page && pageBreakData[page-1]->headingLeftingSize.x>0) b.x.from -= pageBreakData[page-1]->headingLeftingSize.x;
        if (b.IsInvalid() || (int)b.x.Spans()<2 || (int)b.y.Spans()<2) return;
        b.y.till += pChart->GetCopyrightTextHeight();
        b.Scale(zoom);

        cairo_surface_t *surf = cairo_image_surface_create(CAIRO_FORMAT_ARGB32, (int)b.x.Spans(), (int)b.y.Spans());
        cairo_status_t status = cairo_surface_status(surf);
        if (status != CAIRO_STATUS_SUCCESS) {
            cairo_surface_destroy(surf);
            return;
        }
        Canvas canvas(Canvas::PNG, surf, pChart->GetTotal(), pChart->GetCopyrightTextHeight(), XY(zoom, zoom), &pageBreakData, page);
        if (canvas.ErrorAfterCreation(&pChart->Error, &pageBreakData, true)) {
            cairo_surface_destroy(surf);
            return;
        }
        pChart->DrawComplete(canvas, true, page);
        if (auto oTexture = TextureFromSurface(surf))
            compile_cache.chart.emplace(*oTexture, ImVec2((float)b.x.Spans(), (float)b.y.Spans()));
        //destroying canvas, will destroy the cairo surface
    }
    const ImVec2 ctrl_size = ImVec2(float(Element::control_size.x), float(Element::control_size.x))*zoom;
    for (unsigned u = 1; u<int(EGUIControlType::MAX); u++) {
        const EGUIControlType type = EGUIControlType(u);
        std::optional<Texture>& ctrl = compile_cache.controls[u];
        if (ctrl && ctrl->size!=ctrl_size) ctrl.reset();
        if (!ctrl) {
            cairo_surface_t* surf = cairo_image_surface_create(CAIRO_FORMAT_ARGB32, (int)ctrl_size.x, (int)ctrl_size.y);
            cairo_status_t status = cairo_surface_status(surf);
            if (status != CAIRO_STATUS_SUCCESS) {
                cairo_surface_destroy(surf);
                continue;
            }
            cairo_t* cr = cairo_create(surf);
            cairo_scale(cr, zoom, zoom);
            Element::DrawControl(cr, type, 1.0);
            cairo_destroy(cr);
            if (auto oTexture = TextureFromSurface(surf))
                ctrl.emplace(*oTexture, ctrl_size);
            cairo_surface_destroy(surf);
        }
    }
    //Draw overlays to texture
    for (Overlay &o: overlays) if (!o.texture) {
        Block b = o.element->GetAreaToDraw().GetBoundingBox();
        if (b.IsInvalid() || (int)b.x.Spans()<2 || (int)b.y.Spans()<2) continue;
        b.Expand(5); //If the outline draw ends up going a bit outside the bounding box (e.g., miters, etc.)
        b.Scale(zoom);
        cairo_surface_t *surf = cairo_image_surface_create(CAIRO_FORMAT_ARGB32, (int)b.x.Spans(), (int)b.y.Spans());
        cairo_status_t status = cairo_surface_status(surf);
        if (status != CAIRO_STATUS_SUCCESS) {
            cairo_surface_destroy(surf);
            continue;
        }
        cairo_t *cr = cairo_create(surf);
        cairo_translate(cr, -b.x.from, -b.y.from);
        cairo_scale(cr, zoom, zoom);
        o.element->GetAreaToDraw().CairoPath(cr, false);
        cairo_set_source_rgba(cr, 1, 0.4, 0.4, 0.5);
        cairo_fill_preserve(cr);
        cairo_set_source_rgba(cr, 0.7, 0.2, 0.2, 1);
        cairo_stroke(cr);
        cairo_destroy(cr);
        cairo_surface_flush(surf);
        if (auto oTexture = TextureFromSurface(surf)) {
            o.texture.emplace(*oTexture, (ImVec2)b.Spans());
            o.offset = (ImVec2)(b.UpperLeft()-pChart->GetTotal().UpperLeft()*zoom);
        }
        cairo_surface_destroy(surf);
    }
}
void ConfigData::Animate(const Element *element, float initial_fade, bool do_fade_out) {
    auto i = std::ranges::find_if(overlays, [element](const Overlay &o) {return o.element==element; });
    if (i==overlays.end()) {
        overlays.emplace_back(element, initial_fade, do_fade_out);
        DrawCache(); //will only touch the last we just added
    } else {
        i->fade = initial_fade;
        i->do_fade_out |= do_fade_out;
    }
}

void ConfigData::PruneAnimations() {
    //Prune animation that has finished.
    std::erase_if(overlays, [this](const Overlay &o) {
        if (o.fade) 
            return false; 
        if (o.texture) DestroyTexture(o.texture->texture); 
        return true; 
    });
}

void ConfigData::ShowControls(Element* element) {
    auto i = std::ranges::find_if(controls, [element](const Control& c) {return c.element==element; });
    if (i==controls.end()) {
        controls.emplace_back(element, 0.1f);
    } else {
        i->size = anim_time_value(i->size, 1, 10);
    }
}

void ConfigData::ControlClicked(Element* element, EGUIControlType t) {
    if (element==nullptr) return;
    if (!pChart) return;
    if (t==EGUIControlType::INVALID) return;
    if (pChart->ControlClicked(element, t)) {
        //OK, something changed. Take new GUI state...
        control_state = pChart->SerializeGUIState();
        StartCompile();
    }
}

//We dont destroy the controls - on redraw we check if their size has changed
void ConfigData::ResetCompileCache() { 
    if (compile_cache.chart) { 
        DestroyTexture(compile_cache.chart->texture); 
        compile_cache.chart.reset(); 
    }
    for (auto &o : overlays)
        if (o.texture) {
            DestroyTexture(o.texture->texture);
            o.texture.reset();
        }
    PruneAnimations();
}


void ConfigData::PostEditorRender() {
    if (int index = editor.UndoBufferClearedFrom(); index>=0) {
        if (index<=saved_at.mIndex) saved_at = {};
        if (index<=compiled_at.mIndex) compiled_at = {};
    }
    if (IsDirty() && !window_title.starts_with('*'))
        window_title.insert(window_title.begin(), '*');
    else if (!IsDirty() && window_title.starts_with('*'))
        window_title.erase(window_title.begin());
    if (auto &io = ImGui::GetIO();  ImGui::IsItemHovered() && io.MouseWheel && io.KeyCtrl) {
        editor_font_scale = between(float(editor_font_scale*pow(1.1f, between(io.MouseWheel, -5.f, +5.f))),
                                    min_editor_scale, max_editor_scale);
        editor.mFontSize = MscGenEditorFontSize * editor_font_scale;
        editor.mScale = editor_font_scale;
    }
}
 
void ConfigData::NavigateErrors(bool up, bool detailed) noexcept {
    if (!pChart) return;
    const int errnum = pChart->Error.GetErrorNum(warnings, technical_info);
    if (errnum==0) return;
    if (selected_error<0) selected_error = up ? errnum : -1;
    const int started_at = selected_error;
    do
        selected_error = (selected_error + (up ? -1 : +1) + errnum)%errnum;
    while (selected_error!=started_at 
            && ((!detailed 
                && pChart->Error.GetError(selected_error, warnings, technical_info)->line_type != ErrorElement::ELineType::Main)
                || pChart->Error.GetError(selected_error, warnings, technical_info)->relevant_line.file != main_file_no));
    GotoError();
}
float ConfigData::GetHeadingSize() noexcept {
    if (!auto_heading) return 0;
    //This is MscChart specific
    MscChart *msc = dynamic_cast<MscChart *>(pChart.get());
    if (msc==nullptr) return 0;
    if (page<=1 || page > msc->GetPageNum())
        return zoom*float(msc->GetHeadingSize() - msc->GetTotal().x.from);
    return zoom*float(msc->GetPageData(page-1)->headingLeftingSize.y);
}

std::optional<XY> ConfigData::MouseToChart(const ImVec2& mouse) const noexcept {
    if (!pChart) return {};
    if (page>pChart->GetPageNum()) return {};
    XY PageOffset{0,0};
    if (page)
        if (const auto* pData = pChart->GetPageData(page))
            PageOffset = pData->xy;
    return (XY)mouse/zoom + PageOffset + pChart->GetTotal().UpperLeft();
}

const Area *ConfigData::GetArcByMouse(const ImVec2 &mouse) const noexcept {
    const auto M = MouseToChart(mouse);
    if (!M) return nullptr;
    if (!pChart->GetTotal().IsWithinBool(*M)) return nullptr;
    return pChart->AllCovers.InWhichFromBack(*M);
}
Element *ConfigData::GetArcByEditorPos(const TextEditor::Coordinates &pos) const noexcept {
    if (tracking_mode) return nullptr;
    if (!pChart || !IsCompiled()) return nullptr;
    FileLineCol linenum(int(pChart->Error.Files.size()-1), pos.mLine+1, pos.mColumn+1);
    MscChart::LineToArcMapType::const_iterator entity;
    //in the map linenum_ranges are sorted by increasing length, we search the shortest first
    auto i = std::ranges::find_if(pChart->AllElements, [&linenum](auto &p) { return p.first.start <= linenum && linenum <= p.first.end; });
    if (i==pChart->AllElements.end())
        return nullptr;
    return i->second;
}


bool ConfigData::SetCursorTo(const Element *E) {
    if (!E) return false;
    const auto &P = E->file_pos;
    if (P.start.file != main_file_no || P.end.file != main_file_no) return false;
    const TextEditor::Coordinates s1{(int)P.start.line-1, (int)P.start.col-1};
    const TextEditor::Coordinates s2{(int)P.end.line-1, (int)P.end.col}; //notice we do not substract from P.end.col - that is the ID of the last char, not that of beyond one
    editor.SetCursorPosition(s1);
    editor.SetSelection(s1, s2);
    return true;
}


//Process hints after a re-coloring 
//returns false if we can cancel the hints (or was not active)
bool ConfigData::ProcessHints(Csh &csh) {
    //destroy the old graphics
    for (auto &i : hint.items)
        if (i.graphics) textures_to_destroy.insert(*i.graphics);
    hint.items.clear();
    if (hint.empty_graphics) {
        textures_to_destroy.insert(*hint.empty_graphics);
        hint.empty_graphics.reset();
    }
    //cancel hints if
    if (editor.HasSelection()              //1. multiple characters selected; or
           || csh.Hints.size()==0) {       //2. no hints collected; or
    cancel:
        csh.Hints.clear();
        return false;
    }
    const CshPos &p = csh.hintedStringPos;
    const int start = editor.GetCharPos(editor.GetCursorPosition());
    if (start<0) goto cancel;             //3. caret is outside the text (how can that be)?; or
    if (p.first_pos != p.last_pos         //4. the current hint is the previous autocompletion (initiated from this function below)...
            && p.first_pos == start       //   ... and the cursor stands at the beginning of a real nonzero len hinted word ...
            && !hint.is_user_requested)   //   ... and the used did not press Ctrl+Space.
        goto cancel;
    
    if (hint.till_cursor_only)
        if (start < csh.hintedStringPos.last_pos)
            csh.hintedStringPos.last_pos = start;
    std::string under_cursor;
    if (csh.hintedStringPos.first_pos>=0)
        under_cursor = editor.GetText(editor.GetCoordinate(csh.hintedStringPos.first_pos),
                                      editor.GetCoordinate(csh.hintedStringPos.last_pos));

    for (bool filter_by_uc : { true, false }) {
        //Save hints
        std::set<CshHint> saved_hints = csh.Hints;
        //Now process the list of hints: fill extra fields, compute size, filter by under_cursor and compact
        if (csh.Hints.size()) {
            cairo_surface_t *surf = cairo_image_surface_create(CAIRO_FORMAT_ARGB32, HINT_GRAPHIC_SIZE_X, HINT_GRAPHIC_SIZE_Y);
            Canvas canvas(Canvas::PNG, surf, Block(0, HINT_GRAPHIC_SIZE_X, 0, HINT_GRAPHIC_SIZE_Y));
            csh.ProcessHints(canvas, nullptr, under_cursor, filter_by_uc, /*compact=*/true);
        }

        //See how many of the remaining hits fit the word under the cursor 
        unsigned no_selectable = 0; //count how many selectable hints we have fow which 'text' is a prefix
        unsigned no_unselectable = 0; //count how many unselectable hints we have 
        auto hit = csh.Hints.end(); //The last selectable hint
        for (auto i = csh.Hints.begin(); i!=csh.Hints.end(); i++) {
            //find a selectable item or one that the text under cursor fits 
            if (!i->selectable) {
                no_unselectable++;
            } else if (i->plain.starts_with(under_cursor)) {
                no_selectable++;
                hit = i;
            }
        }
        //we will not consider unselectable hints if we have something under the cursor
        if (under_cursor.length()) no_unselectable = 0;
        //Check if we have only one hint (group hints count as multiple)
        const bool onlyOne = no_unselectable==0 && no_selectable==1
                          && hit->decorated[hit->decorated.size()-1]!='*';
        if (onlyOne && hint.is_user_requested && hit->GetReplacementString() != under_cursor) {
            //If we are about to start hint mode due to a Ctrl+Space and there is only one selectable hit 
            //then auto complete without popping up the hint list. 
            HintSubstitute(&*hit);
            goto cancel;
        } else if (onlyOne && hit->GetReplacementString()== under_cursor) {
            //if there is only one hit and it is equal to the word under cursor, cancel hit mode,
            //but not if it was a Ctrl+Space - in that case show the only choice as a feedback
            //to Ctrl+Space
            if (hint.is_user_requested) {
                //...if the user requested the thing, preprocess again, but without
                //filtering.
                if (filter_by_uc) {
                    csh.Hints = saved_hints; //restore hints;
                    continue; //repeat processing with filter_by_uc=false
                }
                //if filtering was already off, just keep what we have now
            } else
                goto cancel;
        }
        hint.partial_match = bool(no_selectable);
        break;
    }
    FillAttr white(ColorType(255, 255, 255));
    white.MakeComplete();
    constexpr double X = HINT_GRAPHIC_SIZE_X*5, Y = HINT_GRAPHIC_SIZE_Y*5;
    cairo_surface_t *surf = cairo_image_surface_create(CAIRO_FORMAT_ARGB32, int(X), int(Y));
    Canvas canvas(Canvas::PNG, surf, Block(0, 0, X, Y));
    cairo_scale(canvas.GetContext(), X/HINT_GRAPHIC_SIZE_X, Y/HINT_GRAPHIC_SIZE_Y);
    StringFormat sf;
    for (const auto &h : csh.Hints) {
        HintData::ItemData &item = hint.items.emplace_back();
        if (h.callback) {
            canvas.Fill({0,0}, {HINT_GRAPHIC_SIZE_X, HINT_GRAPHIC_SIZE_Y}, white);
            h.callback(&canvas, h.param, csh);
            cairo_surface_flush(surf); //We still have a context for this surface in canvas. I hope it is OK.
            item.graphics = TextureFromSurface(surf);
        } 
        sf.Default();
        sf.Apply(h.decorated); //h is const, so this matches the Apply(string_view) variant and will leave 'decorated' unchanged.
        item.bold = sf.bold.GetValue(ETriState::no)==ETriState::yes;
        item.italics = sf.italics.GetValue(ETriState::no)==ETriState::yes;
        item.underline= sf.underline.GetValue(ETriState::no)==ETriState::yes;
        ColorType color = sf.color.GetValue(ColorType(0, 0, 0)); //black
        item.color = IM_COL32(color.r, color.g, color.b, color.a);
        //Add * at the end if it ends in a dot.
        if (h.plain.ends_with('.')) {
            if (h.replaceto.empty()) h.replaceto = h.plain;
            h.plain.push_back('*');
        }
    }
    canvas.Fill({0,0}, {HINT_GRAPHIC_SIZE_X, HINT_GRAPHIC_SIZE_Y}, white);
    cairo_surface_flush(surf); //We still have a context for this surface in canvas. I hope it is OK.
    hint.empty_graphics = TextureFromSurface(surf);
    return csh.Hints.size()>0; //if we have no hints, cancel hint mode
}

void ConfigData::HintCancel(bool force) {
    hint.active = hint.restart_hint && !force;
    hint.restart_hint = false;
    hint.is_user_requested = false;
    hint.till_cursor_only = false;
}

void ConfigData::HintSubstitute(const CshHint *h) {
    TextEditor::Coordinates from = editor.GetCoordinate(hint.csh->hintedStringPos.first_pos);
    TextEditor::Coordinates till = editor.GetCoordinate(hint.csh->hintedStringPos.last_pos);
    editor.SetSelection(from, till);
    editor.Insert((h->replaceto.empty() ? h->plain : h->replaceto).c_str());
    hint.restart_hint = h->keep;
    hint.is_user_requested = false;
    hint.till_cursor_only = false;
    if (!h->keep) hint.active = false;
}


GUIReturn ConfigData::Ret() {
    const bool do_exit = [this] {
        if (exit_requested) switch (CheckOverwrite("Unsaved changes to %s. Do you want to save now?")) {
        case StatusRet::Completed: return true;
        case StatusRet::Cancelled: exit_requested = false; return false;
        case StatusRet::Ongoing: return false; 
        }
        return false;
    }();
    return {do_exit, window_title, presentation_mode};
}

ConfigData Config;

//returns true if we added page selector
bool PageSelector() {
    if (!Config.pChart || Config.pChart->GetPageNum()<=1) return false;
    static int page_selected = 0;
    ImGui::Text("Page:"); ImGui::SameLine();
    Help("Select which page of the chart to view. 0 shows all.");
    ImGui::SetNextItemWidth(80);
    if (ImGui::InputInt("##Page", &page_selected, 1, 1, ImGuiInputTextFlags_CharsDecimal)) {
        page_selected = page_selected<0 ? 0 : (unsigned)page_selected > Config.pChart->GetPageNum() ? (int)Config.pChart->GetPageNum() : page_selected;
        if ((int)Config.page!=page_selected) {
            Config.page = page_selected;
            Config.ResetCompileCache();
        }
    }
    Help("Select which page of the chart to view. 0 shows all.");
    return true;
}

void AddUnderLine() {
    ImVec2 min = ImGui::GetItemRectMin();
    ImVec2 max = ImGui::GetItemRectMax();
    ImColor col = ImGui::GetStyle().Colors[ImGuiCol_Text];
    min.y = max.y;
    ImGui::GetWindowDrawList()->AddLine(min, max, col, 1.0f);
}

void InitGUI(const GUIInit &_G) {
    Config.Init(_G);
}

bool PreFrameGUI(float dpi_mul) {
    if (Config.dpi_mul == dpi_mul) return false;
    auto &io = ImGui::GetIO();
    io.Fonts->Clear();
    Config.dpi_mul = dpi_mul;
    MscGenEditorFontSize = 13 * dpi_mul;
    MscGenWindowFontSize = 16 * dpi_mul;
    LoadFonts();
    Config.editor.SetFonts(GetFont(false, false, false), GetFont(false, true, false),
                           GetFont(false, false, true), GetFont(false, true, true), MscGenEditorFontSize*Config.editor_font_scale,
                           GetFont(true, false, false));
    return true;
}

GUIReturn DoGUI(const GUIEvents &events)
{
    bool shift_focus_to_editor = false;
    Config.exit_requested |= events.request_exit;
    constexpr float min_zoom = 0.01f, max_zoom = 10.f;

    //calcualte CPU usage. Smooth value
    static double smoothed_cpu_usage = 0;
    static std::chrono::microseconds last_total_cpu = std::chrono::microseconds::zero();
    static auto last_cpu_calc = std::chrono::steady_clock::now();
    if (events.total_cpu) {
        const auto now = std::chrono::steady_clock::now();
        const std::chrono::microseconds elapsed_us = std::chrono::duration_cast<std::chrono::microseconds>(now-last_cpu_calc);
        if (events.total_cpu>last_total_cpu && elapsed_us>0us) {
            const double cpu = (double)(*events.total_cpu-last_total_cpu).count()/(double)elapsed_us.count();
            last_cpu_calc = now;
            last_total_cpu = *events.total_cpu;
            smoothed_cpu_usage = cpu - (cpu - smoothed_cpu_usage) * pow(0.7, elapsed_us.count()/1e6); //smooth
        }
    }
    //Save files dropped on us locally
    static GUIEvents::Drop dropped_on_us;
    if (events.dropped) {
        if (dropped_on_us) GUIBeep();
        dropped_on_us = std::move(events.dropped);
    }

    ImGui::PushFont(GetFont(true, false, false));
    if (Config.file_name.empty() && !dropped_on_us) {
        static ConfigData::RecentFile open_on_startup{Config.G->file_to_open_path, Config.G->file_to_open_name};
        if (open_on_startup.full_path.size()) 
            switch (Config.Load(&open_on_startup)) {
            case StatusRet::Completed: shift_focus_to_editor = true; break;
            case StatusRet::Cancelled: open_on_startup.clear(); break;
            case StatusRet::Ongoing: break;
        }
        if (open_on_startup.full_path.empty()) //re-test: cleared on failure above
            shift_focus_to_editor = Config.New(false)==StatusRet::Completed;
    }
    Config.CleanupTextures();

    static enum class State { Idle, FileNew, FileOpen, FileSave, FileSaveAs } main_state = State::Idle;
    static bool help_is_open = false;
    static bool licence_is_open = false;
    static bool rename_is_open = false;
    bool do_open_rename = false;
    static bool refocus_search_window = false;

    ImGuiIO &io = ImGui::GetIO();
    auto shift = io.KeyShift;
    auto ctrl = io.ConfigMacOSXBehaviors ? io.KeySuper : io.KeyCtrl;
    auto alt = io.ConfigMacOSXBehaviors ? io.KeyCtrl : io.KeyAlt;
    escape_pressed = !ctrl && !alt && !shift && ImGui::IsKeyPressed(ImGui::GetKeyIndex(ImGuiKey_Escape)); //make it globally available
    Config.editor.SetHandleKeyboardInputs(true); //In a previous frame we may have disabled it (when user presses Ctrl+Space)
    if (main_state == State::Idle && dropped_on_us) { //If we are idle and something dropped on us
        help_is_open = false;
        licence_is_open = false; 
        if (dropped_on_us.full_names.size()>1) {
            if (GUIError("Drop only a single file on me."))
                dropped_on_us.clear();
        } else {
            ConfigData::RecentFile load_this; load_this.set(dropped_on_us.full_names.front());
            switch (Config.Load(&load_this)) {
            case StatusRet::Completed: shift_focus_to_editor = true; FALLTHROUGH;
            case StatusRet::Cancelled: dropped_on_us.clear();
            case StatusRet::Ongoing: break;
            }
        }
    } else if (main_state == State::Idle    //Handle keyboard shortcuts only if we do not Save/Load anything
            && !help_is_open                //and the help dialog is not open
            && !licence_is_open             //and the licence dialog is also not open
            && Config.file_name.size()) {   //and we are not in the Welcome screen
        if (io.KeyCtrl && !io.KeyAlt && !shift && ImGui::IsKeyPressed(ImGui::GetKeyIndex(ImGuiKey_Space))) { //use the Ctrl key also on Apple
            Config.HintStart();
            Config.editor.SetHandleKeyboardInputs(false);
        } else if (escape_pressed) {
            if (Config.hint.active) {
                Config.HintCancel(true);
            } else if (Config.show_search) { //Drop search window
                Config.show_search = false;
            } else { //Kill both full screen and tracking mode          
                Config.presentation_mode = false;
                Config.tracking_mode = false;
                for (auto &o : Config.overlays) o.do_fade_out = true;
            }
        } else if (!ctrl && !alt && !shift 
                   && (ImGui::IsKeyPressed(MscGenKeyMap[MscGenKey_F2])
                       || ImGui::IsKeyPressed(MscGenKeyMap[MscGenKey_F5]))) {
            Config.StartCompile();
        } else if (!ctrl && !alt && !shift && ImGui::IsKeyPressed(MscGenKeyMap[MscGenKey_F4])) {
            Element *const element = Config.GetArcByEditorPos(Config.editor.GetCursorPosition());
            if (element) Config.Animate(element);
            // else beep
        } else if (ctrl && !alt && shift && ImGui::IsKeyPressed(MscGenKeyMap[MscGenKey_S])) {
            main_state = State::FileSaveAs;
        } else if (ctrl && !alt && !shift && ImGui::IsKeyPressed(MscGenKeyMap[MscGenKey_S])) {
            main_state = State::FileSave;
        } else if (ctrl && !shift && ImGui::IsKeyPressed(MscGenKeyMap[MscGenKey_N])) {
            main_state = State::FileNew;
        } else if (ctrl && !shift && ImGui::IsKeyPressed(MscGenKeyMap[MscGenKey_O])) {
            main_state = State::FileOpen;
        } else if (!alt && ImGui::IsKeyPressed(MscGenKeyMap[MscGenKey_F8])) {
            Config.NavigateErrors(shift, ctrl);
        } else if (ctrl && !alt && !shift && ImGui::IsKeyPressed(MscGenKeyMap[MscGenKey_R])) {
            do_open_rename = true;
        } else if (ctrl && !alt && !shift && ImGui::IsKeyPressed(MscGenKeyMap[MscGenKey_F])) {
            Config.show_search = true;
            refocus_search_window = true;
        }
    } else if (dropped_on_us) {
        //something dropped on us while not idle.
        dropped_on_us.clear();
        GUIBeep();
    }

    static bool fitted_to_window = false; //set below so that we can enable/disable Fit to Window button.
    static bool fitted_to_width = false; //set below so that we can enable/disable Fit to Width button.
    bool fit_to_window = false; //signal that this is needed
    bool fit_to_width = false; //signal that this is needed

    ImGui::SetNextWindowPos(ImVec2(0, 0));
    ImGui::SetNextWindowSize(io.DisplaySize);
    ImGui::PushStyleVar(ImGuiStyleVar_WindowPadding, {4, 0});
    ImGui::Begin("Main Window", nullptr, 
                 ImGuiWindowFlags_NoDecoration                                        //to occupy the entire drawing area
                 | ImGuiWindowFlags_NoScrollbar | ImGuiWindowFlags_NoScrollWithMouse  //to prevent 1-bit accidental scrolling
                 | ImGuiWindowFlags_NoBringToFrontOnFocus);                           //to allow the toolbox window in presentation mode to be on top
    ImGui::PopStyleVar();

    static std::optional<ConfigData::RecentFile> load_this;
    if (!Config.presentation_mode) {
        ImGui::PushStyleVar(ImGuiStyleVar_CellPadding, {4, 4});
        ImGui::PushStyleVar(ImGuiStyleVar_ItemSpacing, {8, 0});
        const bool show_compile = !Config.instant_compile;
        const bool show_pages = Config.pChart && Config.pChart->GetPageNum() > 1;
        const int columns = 4 + int(show_compile) + int(show_pages);
        if (ImGui::BeginTable("ribbon", columns, ImGuiTableFlags_BordersInnerV | ImGuiTableFlags_SizingFixedFit)) {
            ImGui::TableSetupColumn("File...", ImGuiTableColumnFlags_WidthFixed);
            if (show_compile) ImGui::TableSetupColumn("Compile", ImGuiTableColumnFlags_WidthFixed);
            ImGui::TableSetupColumn("Zoom", ImGuiTableColumnFlags_WidthFixed);
            if (show_pages) ImGui::TableSetupColumn("Pages", ImGuiTableColumnFlags_WidthFixed);
            ImGui::TableSetupColumn("FPS", ImGuiTableColumnFlags_WidthStretch);
            ImGui::TableSetupColumn("Help", ImGuiTableColumnFlags_WidthFixed);
            ImGui::TableNextRow();

            ImGui::TableNextColumn();
            ImGui::PushStyleColor(ImGuiCol_Button, IM_COL32_BLACK_TRANS);
            if (ImGui::Button("File..."))
                ImGui::OpenPopup("FileMenu");
            ImGui::SameLine();
            if (ImGui::Button("Settings..."))
                ImGui::OpenPopup("Settings");

            ImGui::PopStyleVar(2);
            if (ImGui::BeginPopup("FileMenu")) {
                if (ImGui::MenuItem("New...", "Ctrl+N", nullptr)) main_state = State::FileNew;
                if (ImGui::MenuItem("Open...", "Ctrl+O", nullptr)) main_state = State::FileOpen;
                if (ImGui::MenuItem("Save...", "Ctrl+S", nullptr)) main_state = State::FileSave;
                if (ImGui::MenuItem("Save As...", "Shift+Ctrl+S", nullptr)) main_state = State::FileSaveAs;
                ImGui::Separator();
                load_this = Config.RecentFilesAsMenu();
                ImGui::EndPopup(); //FileMenu
            }
            ImGui::PushStyleVar(ImGuiStyleVar_CellPadding, {4, 4});
            ImGui::PushStyleVar(ImGuiStyleVar_ItemSpacing, {8, 0});

            if (ImGui::BeginPopup("Settings", ImGuiWindowFlags_NoDecoration | ImGuiWindowFlags_NoScrollbar
                                  | ImGuiWindowFlags_NoScrollWithMouse | ImGuiWindowFlags_AlwaysAutoResize)) {
                ImGui::PopStyleColor(); //show button background
                ImGui::PushStyleVar(ImGuiStyleVar_ItemSpacing, {8, 4});
                ImGui::PushStyleVar(ImGuiStyleVar_FrameBorderSize, 1);
                if (ImGui::BeginTable("Compiling", 3, ImGuiTableFlags_NoSavedSettings | ImGuiTableFlags_SizingFixedFit
                                      | ImGuiTableFlags_BordersInnerV)) {
                    ImGui::TableSetupColumn("Compile", ImGuiTableColumnFlags_WidthFixed, 200);
                    ImGui::TableSetupColumn("View", ImGuiTableColumnFlags_WidthFixed, 200);
                    ImGui::TableSetupColumn("Editor", ImGuiTableColumnFlags_WidthFixed, 200);
                    ImGui::TableNextRow();
                    ImGui::PushFont(GetFont(true, true, false));
                    ImGui::TableSetColumnIndex(0);
                    ImGui::Text("Compilation");
                    ImGui::TableSetColumnIndex(1);
                    ImGui::Text("View");
                    ImGui::TableSetColumnIndex(2);
                    ImGui::Text("Editor");
                    ImGui::PopFont();

                    ImGui::TableNextRow(); //of the Settings, row containing the checkboxes
                    ImGui::TableSetColumnIndex(0);
                    ImGui::Checkbox("Instant compilation", &Config.instant_compile);
                    Help("When set, the chart will automatically be recompiled every time a change is made. "
                         "When unset, the 'Compile' button or F2 or F5 needs to be pressed to update the chart view. "
                         "The latter is useful for complex charts that take longer to compile.");
                    if (ImGui::Checkbox("Pedantic", &Config.pedantic))
                        Config.StartCompile();
                    Help("When pedantic is set Msc-generator enforces stricter language interpretation. "
                         "For signalling charts, it means generating a warning if an entity is not declared explicitly before use. "
                         "For graphs, it means generating a warning for each use of directed edges in undirected graphs and vice versa "
                         "(graphviz does not allow such mixing), but when turned off, mixing directed and undirected edges in a graph becomes possible. "
                         "For Block Diagrams, it prevents auto-generating an block when mentioned without "
                         "a 'box' or similar keyword. This allows forward referencing blocks in arrow "
                         "definitions(by avoiding their creation instead of referencing a later definition.)");
                    if (ImGui::Checkbox("Technical info", &Config.technical_info))
                        Config.StartCompile();
                    Help("When set a few compilation statistics are added to the Error/Warning list.");

                    static int current_forced_design = 0;
                    std::string items_separated_by_zero = "(chart defined)\0"s;
                    const bool has_designs = Config.lang && Config.lang->designs.size();
                    if (has_designs)
                        for (const auto &[name, _] : Config.lang->designs)
                            items_separated_by_zero.append(name).push_back('\0');
                    ImGui::Text("Design:"); ImGui::SameLine();
                    ImGui::BeginDisabled(!has_designs);
                    if (ImGui::Combo("##Design", &current_forced_design, items_separated_by_zero.c_str(), 5)) {
                        if (current_forced_design)
                            Config.G->languages.cshParams->ForcedDesign = std::next(Config.lang->designs.begin(), current_forced_design-1)->first;
                        else
                            Config.G->languages.cshParams->ForcedDesign.clear();
                        Config.StartCompile();
                    }
                    ImGui::EndDisabled();
                    Help("The design selected here will be applied to the chart (overriding its selection).");

                    static int current_forced_layout = 0;
                    static const auto layout_algos = [] { auto set = GetLayoutMethods(); return std::vector<std::string>(set.begin(), set.end()); }();
                    items_separated_by_zero = "(graph defined)\0"s;
                    const bool has_layouts = Config.lang && Config.lang->GetName()=="graph";
                    if (has_layouts)
                        for (const std::string &name: layout_algos)
                            items_separated_by_zero.append(name).push_back('\0');
                    ImGui::Text("Layout:"); ImGui::SameLine();
                    ImGui::BeginDisabled(!has_layouts);
                    if (ImGui::Combo("##Layout", &current_forced_layout, items_separated_by_zero.c_str(), 5)) {
                        if (current_forced_layout)
                            Config.forced_layout = layout_algos[current_forced_layout-1];
                        else
                            Config.forced_layout.clear();
                        Config.StartCompile();
                    }
                    ImGui::EndDisabled();
                    Help("The layout algorithm selected here will be applied to the graph (overriding its selection).");

                    ImGui::TableSetColumnIndex(1); //of the Settings, row containing the checkboxes
                    if (ImGui::Checkbox("Fit to Window", &Config.do_fit_window))
                        if (Config.do_fit_window) Config.do_fit_width = false;
                    Help("Make the entire chart visible after each compilation.");
                    if (ImGui::Checkbox("Fit to Width", &Config.do_fit_width))
                        if (Config.do_fit_width) Config.do_fit_window = false;
                    Help("Fit the chart to the width of the window after each compilation.");
                    static float slider_zoom = 100;
                    ImGui::SetNextItemWidth(200);
                    if (ImGui::SliderFloat("##Zoom", &slider_zoom, min_zoom*100, max_zoom*100, "Zoom: %.0f%%", ImGuiSliderFlags_AlwaysClamp | ImGuiSliderFlags_Logarithmic)) {
                        Config.SetZoom(slider_zoom/100);
                        fitted_to_width = fitted_to_window = false; //To prevent re-fitting when Config.do_fit_window/width is set.
                    } else
                        slider_zoom = Config.zoom * 100;
                    Help("You can also use Ctrl+mouse wheel while hoovering over the chart area to zoom the chart.");
                    ImGui::Checkbox("Tracking mode", &Config.tracking_mode);
                    Help("In tracking mode each chart element hoovered above will be highlighted "
                         "and its corresponding text will be selected in the text editor. Likewise "
                         "if you move around in the editor, the chart element the cursor is currently "
                         "in will be highlighted.\n"
                         "In addition, you can mark chart elements by clicking on them. This allows "
                         "easy explanation over a screen sharing session.\n"
                         "Tracking mode can also be turned on by double-clicking the chart area and "
                         "can be turned off by pressing Esc.");
                    ImGui::BeginDisabled(!Config.lang->has_autoheading);
                    ImGui::Checkbox("Auto Heading", &Config.auto_heading);
                    ImGui::EndDisabled();
                    Help("For signalling charts only: always show the entities at the top irrespective "
                         "of scroll and zoom status.");
                    ImGui::BeginDisabled(!Config.lang->has_element_controls);
                    ImGui::Checkbox("Collapse/Expand Controls", &Config.show_controls);
                    Help("Show collapse/expand controls for boxes/groups for languages that support "
                         "them. (Currently signalling charts and graphs.)");
                    ImGui::EndDisabled();
                    ImGui::PopStyleVar(); //end of framesize=1
                    if (ImGui::Button("Presentation mode"))
                        Config.presentation_mode = true;
                    Help("Show the chart only in full screen. Good for presenting about a chart. "
                         "Press Esc to exit full screen mode.");

                    ImGui::TableSetColumnIndex(2); //of the Settings, row containing the checkboxes
                    ImGui::PushStyleVar(ImGuiStyleVar_FrameBorderSize, 1);
                    ImGui::Checkbox("Auto Save", &Config.auto_save);
                    Help("Automatically save the document every second. "
                         "Charts embedded in PNG files, are saved with the picture of last compilation, "
                         "but are not recompiled on save. (You can set 'Instant Compilation' to "
                         "recompile after each change.");
                    ImGui::Text("Color Scheme:"); ImGui::SameLine();
                    ImGui::SetNextItemWidth(100);
                    if (ImGui::Combo("##Color Scheme", &Config.G->languages.cshParams->color_scheme,
                                     "Minimal\0Standard\0Colorful\0Error-oriented\0", 4)) {
                        Config.editor.SetPalette(MscCshAppearanceList[Config.G->languages.cshParams->color_scheme]);
                    }
                    ImGui::Checkbox("Automatic Hints", &Config.auto_hint);
                    Help("Automatically brings up hints as you type. If turned off hints are only "
                         "given if you press Ctrl+Space.");
                    if (ImGui::Checkbox("Error squiggles", &Config.error_squiggles))
                        Config.editor.ReColor();
                    Help("Underlines problematic areas of chart text and provides explanation in a tooltip. "
                         "This works even if the chart is not compiled, so this is useful for larger charts "
                         "with instant compilation turned off. Note that compilation results in more "
                         "comprehensive error diagnostics.");
                    if (ImGui::Checkbox("Smart Indent", &Config.smart_indent))
                        Config.editor.mIndent = Config.smart_indent ? &ConfigData::SmartIndentStatic : nullptr;
                    Help("Automatically adjust indentation when pressing Tab, Enter, {, [ or typing a label.");
                    static float editor_slider_zoom = 100;
                    ImGui::SetNextItemWidth(200);
                    if (ImGui::SliderFloat("##FontScale", &editor_slider_zoom, min_editor_scale*100, max_editor_scale*100,
                                           "Font scale: %.0f%%", ImGuiSliderFlags_AlwaysClamp | ImGuiSliderFlags_Logarithmic)) {
                        Config.editor_font_scale = editor_slider_zoom/100;
                        Config.editor.mFontSize = MscGenEditorFontSize * Config.editor_font_scale;
                        Config.editor.mScale = Config.editor_font_scale;
                    } else
                        editor_slider_zoom = Config.editor_font_scale * 100;
                    Help("You can also use Ctrl+mouse wheel while hoovering over the text editor to scale the editor font.");

                    ImGui::PopStyleVar(); //end of framesize=1
                    ImGui::EndTable(); //Settings
                }
                ImGui::PopStyleVar(); //item spacing
                ImGui::PushStyleColor(ImGuiCol_Button, IM_COL32_BLACK_TRANS);
                ImGui::EndPopup(); //Settings
            }

            if (show_compile) {
                ImGui::TableNextColumn(); //of the ribbon
                ImGui::BeginDisabled(Config.instant_compile || Config.IsCompiled());
                if (ImGui::Button("Compile"))
                    Config.StartCompile();
                ImGui::EndDisabled();
            }
            ImGui::TableNextColumn(); //of the ribbon
            ImGui::PushStyleVar(ImGuiStyleVar_ItemSpacing, {0, 0});
            ImGui::PushStyleVar(ImGuiStyleVar_FrameBorderSize, 1);
            if (ImGui::Checkbox("##Fit to window", &Config.do_fit_window)) {
                if (Config.do_fit_window) Config.do_fit_width = false;
            }
            Help("Make the entire chart visible after each compilation.");
            ImGui::PopStyleVar();
            ImGui::SameLine();
            ImGui::BeginDisabled(fitted_to_window);
            if (ImGui::Button("Fit to Window")) {
                fit_to_window = true;
                fitted_to_width = false; //to prevent automatically re-fit to width if Config.do_fit_width
            }
            Help("Make the entire chart visible now.");
            ImGui::EndDisabled();
            ImGui::PopStyleVar();
            ImGui::SameLine();
            ImGui::PushStyleVar(ImGuiStyleVar_FrameBorderSize, 1);
            if (ImGui::Checkbox("##Fit to width", &Config.do_fit_width)) {
                if (Config.do_fit_width) Config.do_fit_window = false;
            }
            Help("Fit the chart to the width of the window after each compilation.");
            ImGui::PopStyleVar();
            ImGui::PushStyleVar(ImGuiStyleVar_ItemSpacing, {0, 0});
            ImGui::SameLine();
            ImGui::BeginDisabled(fitted_to_width);
            if (ImGui::Button("Fit to Width")) {
                fit_to_width = true;
                fitted_to_window = false; //to prevent automatically re-fit to window if Config.do_fit_window
            }
            Help("Fit the chart to the width of the window now.");
            ImGui::EndDisabled();
            ImGui::PopStyleVar();
            ImGui::PopStyleColor();
            if (show_pages) {
                ImGui::TableNextColumn();  //of the ribbon
                PageSelector();
            }

            ImGui::TableNextColumn(); //of the ribbon
            if (events.total_cpu)
                ImGui::TextDisabled("%02.0f FPS, %03.0f%% CPU, coloring: %dms, compile: %dms", io.Framerate, smoothed_cpu_usage*100,
                                    (int)Config.last_csh_time.count(), (int)Config.last_compile_time.count());
            else
                ImGui::TextDisabled("%02.0f FPS, coloring: %dms, compile: %dms", io.Framerate,
                                    (int)Config.last_csh_time.count(), (int)Config.last_compile_time.count());

            ImGui::TableNextColumn(); //of the ribbon
            ImGui::PushStyleColor(ImGuiCol_Button, IM_COL32_BLACK_TRANS);
            if (ImGui::Button("Help")) {
                ImGui::OpenPopup("Help");
                help_is_open = true;
            }
            ImGui::PopStyleColor();
            if (ImGui::BeginPopupModal("Help", &help_is_open, ImGuiWindowFlags_NoResize | ImGuiWindowFlags_NoScrollbar |
                                       ImGuiWindowFlags_NoCollapse | ImGuiWindowFlags_NoSavedSettings)) {
                ImGui::PushTextWrapPos(ImGui::GetFontSize()*35);
                ImGui::Text("These are the things you should try to discover all the features of this GUI.");
                ImGui::BulletText("Pressing F2 or F5 for compiling the chart. (If you turn instant compilation on, then this is not necessary.)");
                ImGui::BulletText("Dragging the chart to pan it. Ctrl+mouse wheel while hoovering in the chart to zoom it.");
                ImGui::BulletText("Clicking an element in the chart to jump to it in the chart text.");
                ImGui::BulletText("Pressing F4 in the chart text to highlight the element under the cursor on the chart.");
                ImGui::BulletText("Double clicking the chart to enter tracking mode.");
                ImGui::BulletText("Pressing Ctrl+Space to get hints on chart syntax and language. (Also try automatic hints.)");
                ImGui::BulletText("Pressing Tab to auto indent a line (or lines in a selection).");
                ImGui::BulletText("Pressing Ctrl+F for search and replace.");
                ImGui::BulletText("Right click the editor for a context menu.");
                ImGui::TextUnformatted("");
                std::string v = version(Config.G->languages);
                std::ranges::replace(v, '\n', '\0');
                for (size_t pos = 0; pos<v.length(); /*nope*/) {
                    ImGui::TextUnformatted(v.c_str()+pos);
                    pos = v.find('\0', pos);
                    if (pos == v.npos) break;
                    else pos++;
                }
                ImGui::SameLine(ImGui::GetWindowContentRegionWidth()-ImGui::CalcTextSize("Show License").x);
                if (ImGui::Button("Show License")) {
                    ImGui::OpenPopup("License");
                    licence_is_open = true;
                }
                const std::string lic = licence(Config.G->languages);
                ImGui::SetNextWindowPos(ImGui::GetMainViewport()->GetCenter(), ImGuiCond_Always, ImVec2(0.5f, 0.5f));
                ImGui::SetNextWindowSizeConstraints({ImGui::CalcTextSize(lic.c_str()).x, 0}, ImGui::GetMainViewport()->WorkSize*0.8f);
                if (ImGui::BeginPopupModal("License", &licence_is_open, ImGuiWindowFlags_NoResize | ImGuiWindowFlags_NoScrollbar |
                                           ImGuiWindowFlags_NoCollapse| ImGuiWindowFlags_NoSavedSettings)) {
                    ImGui::TextUnformatted(lic.c_str());
                    ImGui::EndPopup();
                }
                ImGui::PopTextWrapPos();
                ImGui::EndPopup();
            }
            ImGui::EndTable();
        }
        ImGui::PopStyleVar(2);
    } //!presentation_mode

    switch (main_state) {
    case State::FileNew:
        if (Config.New(true)!=StatusRet::Ongoing) main_state = State::Idle;
        break;

    case State::FileOpen:
        if (Config.Load()!=StatusRet::Ongoing) main_state = State::Idle;
        break;

    case State::FileSave:
    case State::FileSaveAs:
        if (const ConfigData::SaveMode mode = main_state==State::FileSaveAs ? ConfigData::SaveMode::As : ConfigData::SaveMode::Interactive
                 ; Config.Save(mode)!=StatusRet::Ongoing)
            main_state = State::Idle;
        break;
   case State::Idle:
       if (load_this && Config.Load(&*load_this)!=StatusRet::Ongoing)
           load_this.reset();
       else if (Config.auto_save
                && Config.IsDirty()
                && Config.autosave_interval < ConfigData::clock::now() - Config.last_autosave) {
           const StatusRet ret = Config.Save(ConfigData::SaveMode::Quiet);
           _ASSERT(ret!=StatusRet::Ongoing);
           if (ret!=StatusRet::Ongoing)
               Config.last_autosave = ConfigData::clock::now(); //Update timestamp even on failure - to limit the rate of retries.
       }
       break;
    }
    if (Config.CompileTakeResult(false)) {
        fit_to_window |= Config.do_fit_window;
        fit_to_width |= Config.do_fit_width;
    } else {
        //if previously we were fitted, keep being so
        fit_to_window |= fitted_to_window && Config.do_fit_window;
        fit_to_width |= fitted_to_width && Config.do_fit_width;
    }
    if (Config.instant_compile && Config.compiled_at != Config.editor.GetUndoBufferIndex() && !Config.compile_data.is_compiling())
        Config.StartCompile();

    if (!Config.presentation_mode) {
        ImGuiTableFlags flags = ImGuiTableFlags_Resizable | ImGuiTableFlags_Reorderable
            | ImGuiTableFlags_BordersInnerV | ImGuiTableFlags_BordersOuterH | ImGuiTableFlags_SizingFixedSame;
        ImGui::BeginTable("chart", 2, flags, ImGui::GetContentRegionAvail());
        ImGui::TableSetupColumn(Config.file_name.c_str(), ImGuiTableColumnFlags_WidthStretch);
        ImGui::TableSetupColumn("Chart", ImGuiTableColumnFlags_WidthStretch | ImGuiTableColumnFlags_NoHeaderLabel);
        ImGui::TableNextRow();
        ImGui::TableSetColumnIndex(0);
        if (shift_focus_to_editor) {
            ImGui::SetNextWindowFocus();
            shift_focus_to_editor = false;
        }
        ImGui::PushFont(GetFont(false, false, false));
        Config.editor.Render("TextEditor");
        Config.PostEditorRender();
            //Context menu
        if (ImGui::IsMouseReleased(ImGuiMouseButton_Right))
            ImGui::OpenPopupOnItemClick("context");
        static std::string ask_entity_name_replace;
        static std::pair<std::string, size_t> entity_name_to_replace;
        static ImVec2 rename_dialog_location;
        ImGui::PushFont(GetFont(true, false, false));
        if (ImGui::BeginPopup("context")) {
            const TextEditor::Coordinates cursor_now = Config.editor.ScreenPosToCoordinates(ImGui::GetWindowPos());// ImGui::GetIO().MousePos);
            Element *const element = Config.GetArcByEditorPos(cursor_now);
            if (ImGui::MenuItem("Highlight this element", "F4", false, bool(element)))
                Config.Animate(element);
            const int pos = Config.editor.GetCharPos(cursor_now);
            std::pair<std::string, size_t> entity_name;
            if (Config.hint.csh && pos>=0)
                entity_name = Config.hint.csh->EntityNameUnder(pos+1); //we need a 1-indexed char pos
            const std::string menuname = entity_name.first.size()
                ? "Rename " + Config.lang->entityname + " '" + entity_name.first + "' (in selection)"
                : "Rename " + Config.lang->entityname + " (in selection)";
            if (ImGui::MenuItem(menuname.c_str(), "Ctrl+R", false, !entity_name.first.empty()) 
                     && Config.hint.csh && Config.lang) {
                entity_name_to_replace = entity_name; //this triggers the replace dialog
                rename_dialog_location = {ImGui::GetItemRectMin().x, ImGui::GetItemRectMin().y};
            }
            if (ImGui::MenuItem("Search and Replace...", "Ctrl+F")) {
                Config.show_search = true;
                refocus_search_window = true;
            }
            ImGui::EndPopup(); //ctx menu
        }
        if (Config.lang) {
            if (Config.hint.csh && do_open_rename) {
                const int pos = Config.editor.GetCharPos(Config.editor.GetCursorPosition());
                if (pos>=0) {
                    entity_name_to_replace = Config.hint.csh->EntityNameUnder(pos+1); //we need a 1-indexed char pos
                    auto xywh = Config.editor.CoordinatesToScreenPos(Config.editor.GetCursorPosition());
                    if (xywh)
                        rename_dialog_location = xywh->first + xywh->second;
                    else
                        rename_dialog_location = {0,0};
                } //else beep or flash
            }
            if (entity_name_to_replace.first.size() && !rename_is_open) {
                ImGui::OpenPopup(("Rename "+Config.lang->entityname).c_str());
                ask_entity_name_replace = Config.hint.csh->AskReplace(entity_name_to_replace.first, entity_name_to_replace.second);
                ask_entity_name_replace.reserve(100);
                rename_is_open = true;
            }
            if (rename_dialog_location.x && rename_dialog_location.y)
                ImGui::SetNextWindowPos(rename_dialog_location);
            //This is to avoid clipping the TextInput for one frame on appearing (prevents focus from working)
            //See https://github.com/ocornut/imgui/issues/4079
            ImGui::SetNextWindowSize({100,70}, ImGuiCond_Appearing); 
            if (ImGui::BeginPopupModal(("Rename "+Config.lang->entityname).c_str(), &rename_is_open,
                                       ImGuiWindowFlags_AlwaysAutoResize | ImGuiWindowFlags_NoSavedSettings)) {
                ImGui::TextUnformatted(("Enter new name for "+Config.lang->entityname+" '"+entity_name_to_replace.first+"':").c_str());
                ImGui::PushFont(GetFont(false, false, false));
                ImGui::PushStyleVar(ImGuiStyleVar_FrameBorderSize, 1);
                if (ImGui::IsWindowAppearing())
                    ImGui::SetKeyboardFocusHere();
                const bool text_trigger
                    = ImGui::InputText("##renameinput", ask_entity_name_replace.data(), ask_entity_name_replace.capacity(),
                                       ImGuiInputTextFlags_AutoSelectAll |ImGuiInputTextFlags_EnterReturnsTrue);
                ImGui::PopStyleVar();
                ImGui::PopFont();
                const bool ok_trigger = ImGui::Button("OK");
                if (text_trigger || ok_trigger) {
                    if (Config.hint.csh && ask_entity_name_replace.size()) {
                        const auto [start, end] = Config.editor.GetSelectionCharPos();
                        if (start>=0 && end>=0) {
                            std::string new_text
                                = Config.hint.csh->ReplaceEntityName(
                                    entity_name_to_replace.first, entity_name_to_replace.second,
                                    ask_entity_name_replace.c_str(), //the string length still contains the original length, not what the user typed in.
                                    start+1, end+1); //1-based index expected by CSH
                            if (new_text.size() && new_text != Config.hint.csh->input_text) {
                                Config.editor.SelectAll();
                                Config.editor.Insert(new_text.c_str());
                            }
                        }
                    }
                    ImGui::CloseCurrentPopup();
                    rename_is_open = false;
                    entity_name_to_replace.first = {};
                }
                ImGui::SameLine();
                if (ImGui::Button("Cancel") || escape_pressed) {
                    ImGui::CloseCurrentPopup();
                    rename_is_open = false;
                    entity_name_to_replace.first = {};
                }
                ImGui::EndPopup(); //ask rename dialog
            } else
                entity_name_to_replace.first.clear(); //when we closed the dialog with the X button clear this to prevent re-open
        }
        ImGui::PopFont();

        static bool hint_window_shown = false;
        static int selected_hint = -1;     //Hint selected
        static int highlighted_hint = -1; //Hint hoovered by the mouse
        static int described_hint = -1;   //hint we show the description for
        constexpr float ratio = float(HINT_GRAPHIC_SIZE_X)/float(HINT_GRAPHIC_SIZE_Y);
        const float HintGraphicHeight = ImGui::GetFontSize()+2;
        const float HintGraphicWidth = HintGraphicHeight*ratio;
        static float max_height;
        if (Config.hint.active) {
            if (!hint_window_shown) {
                TextEditor::Coordinates hint_at = Config.editor.GetCursorPosition();
                const int hint_word_off = Config.hint.csh->cursor_pos - Config.hint.csh->hintedStringPos.first_pos; //Which character in the hinted string (word under cursor) are the cursor at
                _ASSERT(hint_word_off>=0);
                _ASSERT(hint_at.mColumn>=hint_word_off);
                hint_at.mColumn = std::max(0, hint_at.mColumn-std::max(0, hint_word_off));
                if (auto xywh = Config.editor.CoordinatesToScreenPos(hint_at)) {
                    //Check how much space is below the line for the hint window
                    ImVec2 at(xywh->first.x - HintGraphicWidth*Config.editor_font_scale - 2, //-2 is window border and whatnot
                              xywh->first.y + xywh->second.y); 
                    const float max_y = ImGui::GetMainViewport()->WorkPos.x + ImGui::GetMainViewport()->WorkSize.y;
                    const float min_height = Config.editor.mFontSize*3;
                    if (at.y+min_height < max_y) {
                        max_height = std::min(200*Config.editor_font_scale, max_y - at.y);
                    } else {
                        max_height = std::min(200*Config.editor_font_scale, 
                                                           Config.hint.csh->Hints.size()*Config.editor.mFontSize);
                        at.y = xywh->first.y - max_height;
                    }
                    ImGui::SetNextWindowPos(at, ImGuiCond_Appearing);
                    ImGui::OpenPopup("Hint");
                    hint_window_shown = true;
                    selected_hint = -1;
                    highlighted_hint = -1;
                    described_hint = -1;
                }
            } else {
                if (ImGui::IsKeyPressed(ImGui::GetKeyIndex(ImGuiKey_Enter))
                    || ImGui::IsKeyPressed(ImGui::GetKeyIndex(ImGuiKey_Tab))) {
                    if (0<=selected_hint && selected_hint<(int)Config.hint.csh->Hints.size()) {
                        auto h = std::next(Config.hint.csh->Hints.begin(), selected_hint);
                        if (h->selectable)
                            if (h!=Config.hint.csh->Hints.end())
                                Config.HintSubstitute(&*h);
                    }
                } else if (ImGui::IsKeyPressed(ImGui::GetKeyIndex(ImGuiKey_Delete))) {
                    Config.editor.Delete(io.KeyCtrl);
                    hint_window_shown = false; //Changing the text will re-focus the editor and kill the popup - and we want to keep it
                    Config.hint.is_user_requested = false; //From this on, if a single match remains, we do not automatically substitute it.
                } else if (!io.InputQueueCharacters.empty()) {
                    bool need_recolor = false;
                    for (int i = 0; i < io.InputQueueCharacters.Size; i++) {
                        auto c = io.InputQueueCharacters[i];
                        if (!isalnum(c) && c != '_' && c>=32 && c<127) { //non-alphanumeric: substitute hint first
                            if (need_recolor) {
                                Config.editor.ReColor();
                                selected_hint = -1;
                                need_recolor = false;
                            }
                            if (0<=selected_hint && selected_hint<(int)Config.hint.csh->Hints.size()) {
                                auto h = std::next(Config.hint.csh->Hints.begin(), selected_hint);
                                if (Config.hint.active && h!=Config.hint.csh->Hints.end() && h->selectable) {
                                    if (h!=Config.hint.csh->Hints.end()) {
                                        Config.HintSubstitute(&*h);
                                        if (h->GetReplacementString().back()=='.'&& c=='.')
                                            continue; //hint ends with dot, user pressed dot - swallow the dot
                                    }
                                } else
                                    Config.HintCancel(true);
                            }
                        }
                        if (c == 8)
                            Config.editor.Backspace(false);
                        else if (c == 127)
                            Config.editor.Backspace(true); //for some reason ctrl+backspace generates this char
                        else if (c == '\n' || c >= 32)
                            Config.editor.EnterCharacter(c, io.KeyShift);
                        else
                            continue;
                        //we have changed the text
                        hint_window_shown = false; //Changing the text will re-focus the editor and kill the popup - and we want to keep it
                        Config.hint.is_user_requested = false; //From this on, if a single match remains, we do not automatically substitute it.
                        need_recolor = true;
                    }
                    io.InputQueueCharacters.resize(0);
                } else if (ImGui::IsKeyPressed(ImGui::GetKeyIndex(ImGuiKey_Backspace))) { //For some reason Backspace does not appear in the input queue on Linux
                    Config.editor.Backspace(io.KeyCtrl);
                    hint_window_shown = false; //Changing the text will re-focus the editor and kill the popup - and we want to keep it
                    Config.hint.is_user_requested = false; //From this on, if a single match remains, we do not automatically substitute it.
                }
            }
        } else
            hint_window_shown = false;
        ImGui::PushStyleVar(ImGuiStyleVar_WindowPadding, {2, 2});
        ImGui::SetNextWindowSizeConstraints(ImVec2{0, 0}, ImVec2(200*Config.editor_font_scale, max_height));
        if (ImGui::BeginPopup("Hint", ImGuiWindowFlags_NoTitleBar | ImGuiWindowFlags_NoResize | ImGuiWindowFlags_NoCollapse
                              | ImGuiWindowFlags_NoSavedSettings)) {
            ImGui::SetWindowFontScale(Config.editor_font_scale);
            int i = 0;
            int sel = -1, hoo = -1; //the index of the item selected & hoovered
            ImVec2 sxy, hxy;        //the orign of the above items.
            ImGui::PushStyleVar(ImGuiStyleVar_ItemSpacing, {0, 2});
            if (!Config.hint.partial_match) {
                ImGui::SetItemDefaultFocus();
                ImGui::Dummy(ImVec2(0, 0)); //No automatic selection of first item when no hint begins with the word under cursor
            }
            for (const CshHint &h : Config.hint.csh->Hints) {
                const ImVec2 origin = ImGui::GetCursorScreenPos();
                if (Config.hint.empty_graphics) {
                    ImGui::Image(Config.hint.items[i].graphics.value_or(*Config.hint.empty_graphics), 
                                 ImVec2{HintGraphicWidth, HintGraphicHeight}*Config.editor_font_scale);
                    ImGui::SameLine();
                }
                ImGui::PushStyleColor(ImGuiCol_Text, Config.hint.items[i].color);
                ImGui::PushFont(GetFont(false, Config.hint.items[i].bold, Config.hint.items[i].italics));
                if (ImGui::Selectable(h.plain.c_str(), i==selected_hint, ImGuiSelectableFlags_DontClosePopups)
                        && h.selectable && Config.hint.active) 
                    Config.HintSubstitute(&h);
                if (Config.hint.items[i].underline)
                    AddUnderLine();
                ImGui::PopFont();
                ImGui::PopStyleColor();
                if (ImGui::IsItemFocused())
                    sel = i, sxy=origin;
                if (ImGui::IsItemHovered()) 
                    hoo = i, hxy=origin;
                i++;
            }
            ImGui::PopStyleVar();
            if (sel>=0 && sel!=selected_hint) 
                selected_hint = described_hint = sel;
            else if (highlighted_hint!=hoo) {
                highlighted_hint = hoo;
                if (hoo>=0) described_hint = hoo;
                else described_hint = sel; //no item hoovered over
            }
            if (0<=described_hint && described_hint<(int)Config.hint.csh->Hints.size()) {
                const std::string &descr = std::next(Config.hint.csh->Hints.begin(), described_hint)->description;
                const HintData::ItemData &item = Config.hint.items[described_hint];
                if (descr.size()) {
                    if (described_hint==selected_hint)
                        ImGui::SetNextWindowPos({sxy.x + ImGui::GetWindowWidth(), sxy.y});
                    else if (described_hint==highlighted_hint)
                        ImGui::SetNextWindowPos({hxy.x + ImGui::GetWindowWidth(), hxy.y});
                    else {
                        _ASSERT(0);
                    }
                    ImGui::PushFont(GetFont(true, false, false));
                    ImGui::PushStyleVar(ImGuiStyleVar_WindowRounding, 4);
                    ImGui::BeginTooltip();
                    const float width = ImGui::GetFontSize() * 35.0f;
                    const float height = ImGui::CalcTextSize(descr.c_str(), nullptr, false, width).y;
                    float Y = 0;
                    if (item.graphics) {
                        constexpr float ratio = float(HINT_GRAPHIC_SIZE_X)/float(HINT_GRAPHIC_SIZE_Y);
                        Y = ImGui::GetFontSize()*2;
                        ImGui::Image(*item.graphics, {Y*ratio, Y});
                        ImGui::SameLine();
                    }
                    if (Y && Y>height) {
                        ImGui::BeginGroup();
                        ImGui::PushStyleVar(ImGuiStyleVar_ItemSpacing, {0, 0});
                        ImGui::Dummy({1, (Y-height)/2});
                    }
                    ImGui::PushTextWrapPos(ImGui::GetCursorPosX()+width);
                    ImGui::TextUnformatted(descr.c_str());
                    ImGui::PopTextWrapPos();
                    if (Y && Y>height) {
                        ImGui::PopStyleVar();
                        ImGui::EndGroup();
                    }
                    ImGui::EndTooltip();
                    ImGui::PopStyleVar();
                    ImGui::PopFont();
                }
            }
            ImGui::EndPopup();
        } else {
            Config.HintCancel(false);
            hint_window_shown = false;
        }
        ImGui::PopStyleVar();
        ImGui::PopFont();
        ImGui::TableSetColumnIndex(1);
    } //!presentation_mode
    ImVec2 canvas_sz = ImGui::GetContentRegionAvail();   // Resize chart graphics to what's available
    if (!Config.compile_cache.chart)
        Config.DrawCache();

    //Calculate how much space for the error window calculate the animation status
    const size_t errnum = Config.pChart ? Config.pChart->Error.GetErrorNum(Config.warnings, Config.technical_info) : 0;
    const ImVec2 save_cursor = ImGui::GetCursorPos();
    const float line_height = ImGui::GetTextLineHeight()*1.1f;
    const float extra_height = 25; //Add this to the window to cater for vertical scrollbar, padding and border
    const float target_space_for_errors = errnum
        //Occupy at most 3/4 of the drawing space for errors
        ? between(line_height*errnum+extra_height, canvas_sz.y*0.1f, canvas_sz.y*0.75f)
        : 0;
    static float space_for_errors = 0;
    space_for_errors = anim_move_value(space_for_errors, target_space_for_errors);
        
    //show error window
    if (space_for_errors && !Config.presentation_mode) {
        ImGui::SetCursorPosY(save_cursor.y + canvas_sz.y - space_for_errors);
        canvas_sz.y -= space_for_errors + 1; //+1 to prevent overwriting the edge of the error window
        ImGui::PushStyleColor(ImGuiCol_ChildBg, IM_COL32_WHITE);
        ImGui::PushStyleVar(ImGuiStyleVar_ItemSpacing, {0,0});
        ImGui::BeginChild("errors", ImVec2(0, target_space_for_errors), true, ImGuiWindowFlags_HorizontalScrollbar);
        int now_selected = Config.selected_error;
        for (size_t i = 0; i<errnum; i++) {
            const ErrorElement *E = Config.pChart->Error.GetError(i, Config.warnings, Config.technical_info);
            const bool was_selected = Config.selected_error==int(i);
            const ImU32 color = [E] {
                switch (E->line_type) {
                case ErrorElement::ELineType::Context:       return IM_COL32(128, 128, 128, 255);
                case ErrorElement::ELineType::Substitutuion: return IM_COL32(192, 192, 192, 255);
                case ErrorElement::ELineType::Main: switch (E->type) {
                case ErrorElement::EType::Error:             return IM_COL32(128, 0, 0, 255); 
                case ErrorElement::EType::Warning:           return IM_COL32(0, 128, 0, 255); 
                case ErrorElement::EType::TechnicalInfo:     return IM_COL32(0, 64, 128, 255);
                }}
                return IM_COL32_BLACK;
            }();
            ImGui::PushStyleColor(ImGuiCol_Text, was_selected ? IM_COL32_WHITE : color);
            ImGui::PushStyleColor(ImGuiCol_HeaderActive, color);
            ImGui::PushStyleColor(ImGuiCol_Header, color);
            ImColor c(color);
            if (!was_selected) //dim the float color if not selected
                for (float *f :{&c.Value.x, &c.Value.y, &c.Value.z, &c.Value.w})
                    *f = 0.7f * 0.3f**f;
            ImGui::PushStyleColor(ImGuiCol_HeaderHovered, c.Value);
            constexpr size_t ImGuiSelectableFlags_SelectOnClick = 1 << 22;  // COPIED FROM IMGUI INTERNAL!!! Override button behavior to react on Click (default is Click+Release)
            //remove filename of the currently edited file for shorter error lines
            const char *show_text = E->relevant_line.file != Config.main_file_no
                ? E->text.c_str()
                : E->text.c_str() + (E->text.find(':')+1); //rely on npos == -1
            static_assert(std::string::npos+1 == 0);
            if (ImGui::Selectable(show_text, was_selected, ImGuiSelectableFlags_SelectOnClick, ImVec2{0, line_height}))
                now_selected = int(i);
            ImGui::PopStyleColor(4);
        }
        if (ImGui::IsWindowFocused(ImGuiFocusedFlags_ChildWindows))
            shift_focus_to_editor = true;
        ImGui::EndChild();
        ImGui::PopStyleVar();
        ImGui::PopStyleColor();

        //If selection changed, jump to position in file
        if (now_selected!=Config.selected_error) {
            Config.selected_error = now_selected;
            Config.GotoError();
        }
        ImGui::SetCursorPos(save_cursor); //Draw the picture above the errors
    }

    const ImVec2 full_canvas_sz = canvas_sz;
    if (fit_to_window) Config.FitToWindow(full_canvas_sz);
    if (fit_to_width) Config.FitToWidth(full_canvas_sz);

    //draw the chart
    static float search_fade = 0;
    static bool search_hoovered = false; //if the search window was hoovered in the previous frame (avoid tracking)
    const ImVec2 image_viewport_origin = ImGui::GetCursorPos();
    if (Config.compile_cache.chart) {
        ImDrawList *draw_list = ImGui::GetWindowDrawList();
        static ImVec2 scrolling(0.0f, 0.0f);
        const ImVec2 &total = Config.compile_cache.chart->size;
        const ImVec2 offset = max(ImVec2(0, 0), canvas_sz - total)/2; //nonzero, if the top-left corner of the image is not at (0,0) (center chart when small)
        canvas_sz = min(canvas_sz, total);
        scrolling = between(scrolling, {0,0}, max({0,0}, total-canvas_sz));
        ImVec2 uv1{scrolling.x/total.x, scrolling.y/total.y}, uv2{uv1.x+canvas_sz.x/total.x, uv1.y+canvas_sz.y/total.y};
        ImGui::SetCursorPos(image_viewport_origin+offset);
        ImGui::Image(Config.compile_cache.chart->texture, canvas_sz, uv1, uv2);
        for (auto &o : Config.overlays) 
            if (o.texture && o.fade) {
                ImGui::SetCursorPos(image_viewport_origin+offset+o.offset-scrolling);
                ImGui::Image(o.texture->texture, o.texture->size, {0,0}, {1,1}, {1, 1, 1, o.fade});
                if (o.do_fade_out) o.fade = anim_time_value(o.fade, 0);
            }
        //Overlays that faded out are handled in PruneAnimations();
        for (auto& ctrl : Config.controls) 
            if (ctrl.element && ctrl.size) {
                ImVec2 off(ctrl.element->GetControlLocation().UpperLeft()*Config.zoom);
                for (EGUIControlType t : ctrl.element->GetControls()) {
                    const auto& texture = Config.compile_cache.controls[int(t)];
                    if (texture) {
                        const ImVec2 size_off(Element::control_size * (Config.zoom * (1-ctrl.size) / 2));
                        ImGui::SetCursorPos(image_viewport_origin+offset+off-scrolling+size_off);
                        ImGui::Image(texture->texture, texture->size*ctrl.size, {0,0}, {1,1});
                    }
                    off.y += float(Element::control_size.y)*Config.zoom;
                }
                ctrl.size = anim_time_value(ctrl.size, 0, 5);
            }
        //remove controls that have disappeared
        std::erase_if(Config.controls, [](const ConfigData::Control& c) {return c.size==0.; });
        static float heading_fade = 0;
        const float heading = scrolling.y || heading_fade ? Config.GetHeadingSize() : 0;
        if (heading) {
            ImGui::SetCursorPos(image_viewport_origin+offset);
            if (scrolling.y) {
                ImVec2 uv1{scrolling.x/total.x, 0}, uv2{uv1.x+canvas_sz.x/total.x, heading/total.y};
                ImGui::Image(Config.compile_cache.chart->texture, ImVec2{canvas_sz.x, heading}, uv1, uv2);
            }
            draw_list->AddRectFilled(image_viewport_origin+offset,
                                     image_viewport_origin+offset+ImVec2(canvas_sz.x, heading),
                                     IM_COL32(0, 0, 0, 30*heading_fade));
            if (scrolling.y)
                draw_list->AddLine(image_viewport_origin+ImVec2(offset.x, heading),
                                    image_viewport_origin+ImVec2(offset.x+canvas_sz.x, heading),
                                    IM_COL32(0, 0, 0, 255*heading_fade), 1);
            heading_fade = anim_time_value(heading_fade, scrolling.y ? 1.f : 0.f, 3.f);
        }

        constexpr float fade_ms = 500;
        static float fade = 0;
        fade = anim_time_value(fade, float(Config.compile_data.is_compiling()), 1000.f/fade_ms);
        if (const unsigned char fade_int = (unsigned char)(round(fade*192))) {
            //We are compiling - draw compile indicators
            if (Config.instant_compile) {
                //smaller indication for instant compile
                static float R1 = 0;
                constexpr float rad_speed_per_sec = 30 * float(M_PI/180);
                const float radius = 20*Config.dpi_mul;
                R1 += rad_speed_per_sec * sec_since_last_frame(); if (R1>2*M_PI) R1 -= float(2*M_PI);
                const ImVec2 C{image_viewport_origin.x + full_canvas_sz.x - radius*1.1f, image_viewport_origin.y + radius*1.1f}; //center of the circle
                constexpr int num_segments = 5;
                const float a_max = float(M_PI * 2.0f) * ((float)num_segments - 1.0f) / (float)num_segments;
                draw_list->PathArcTo(C, radius, R1, R1+a_max, num_segments - 1);
                draw_list->PathFillConvex(IM_COL32(192, 192, 192, fade_int));                    
                const ImFont *font = GetFont(true, false, false);
                constexpr const char *text = "Work";
                const float fontsize = 12 * Config.dpi_mul;
                const ImVec2 fsize = font->CalcTextSizeA(fontsize, 1e10, 0, text);
                draw_list->AddText(font, fontsize, {C.x-fsize.x/2, C.y-fsize.y/2}, IM_COL32_WHITE, text);
            } else {
                //gray out and progress bar, when instant compilation is off
                draw_list->AddRectFilled(image_viewport_origin, image_viewport_origin+full_canvas_sz, IM_COL32(192, 192, 192, fade_int));
                if (float f = Config.compile_data.get_progress(); f>=0 && fade==1) {
                    ImGui::SetNextWindowPos(image_viewport_origin+full_canvas_sz/2, ImGuiCond_Always, ImVec2(0.5f, 0.5f));
                    ImGui::PushStyleVar(ImGuiStyleVar_WindowRounding, 5);
                    ImGui::Begin("ProgressBar", nullptr, ImGuiWindowFlags_NoDecoration | ImGuiWindowFlags_NoSavedSettings);
                    ImGui::ProgressBar(f/100, {-1, -1}, "");
                    ImGui::SetWindowSize({full_canvas_sz.x*0.8f, 20.f}, ImGuiCond_Always);
                    ImGui::End();
                    ImGui::PopStyleVar();
                }
            }
        } else {
            //We are not compiling - draw tracking mode indicator if applicable
            static float tracking_mode_fade = 0;
            tracking_mode_fade = anim_time_value(tracking_mode_fade, float(Config.tracking_mode), 3);
            if (tracking_mode_fade) {
                constexpr const char* text = "Tracking mode";
                const float fontsize = 12 * Config.dpi_mul;
                const ImFont* font = GetFont(true, true, false);
                const ImVec2 fsize = font->CalcTextSizeA(fontsize, 1e10, 0, text);
                const ImVec2 off = { fontsize/2, fontsize / 4 };
                const float margin = fontsize;
                const ImVec2 O{ image_viewport_origin.x + full_canvas_sz.x - off.x * 2 - fsize.x - margin, image_viewport_origin.y + margin };
                draw_list->AddRectFilled(O, O + fsize + off * 2, IM_COL32(255, 0, 0, (unsigned char)128*tracking_mode_fade), fontsize / 3);
                draw_list->AddText(font, fontsize, O+off, IM_COL32(255, 255, 255, (unsigned char)255 * tracking_mode_fade), text);
            }
            //Get mouse input
            ImGui::SetCursorPos(image_viewport_origin);
            ImVec2 canvas_p0 = ImGui::GetCursorScreenPos();      // ImDrawList API uses screen coordinates!
            // Using InvisibleButton() as a convenience 1) it will advance the layout cursor and 2) allows us to use IsItemHovered()/IsItemActive()
            ImGui::InvisibleButton("canvas", full_canvas_sz, ImGuiButtonFlags_MouseButtonLeft | ImGuiButtonFlags_MouseButtonRight);
            // Shift focus back to the editor
            if (ImGui::IsItemFocused()) shift_focus_to_editor = true;
            // Pan (we use a zero mouse threshold as the context menu is on the other button)
            // You may decide to make that threshold dynamic based on whether the mouse is hovering something etc.
            if (ImGui::IsItemActive() && ImGui::IsMouseDragging(ImGuiMouseButton_Left, 0.f))
                scrolling = scrolling - io.MouseDelta; //'scrolling' will be normalized at the next frame above
            // Zoom & Mouse scroll
            if (ImGui::IsItemHovered() && io.MouseWheel) {
                if (io.KeyCtrl) {
                    const float new_zoom = between(float(Config.zoom * pow(1.1f, between(io.MouseWheel, -5.f, +5.f))), min_zoom, max_zoom);
                    if (new_zoom != Config.zoom) {
                        const ImVec2 P = io.MousePos - canvas_p0 + scrolling - offset; //mouse pos on the surface
                        const ImVec2 pivot{between(P.x/total.x, 0.f, 1.f), between(P.y/total.y, 0.f, 1.f)}; //mouse pos mapped to [0,1];[0,1] rectangle
                        const ImVec2 new_total = total/Config.zoom*new_zoom;
                        const ImVec2 Q{new_total.x*pivot.x, new_total.y*pivot.y}; //mouse pos on new surface
                        scrolling = Q - io.MousePos + canvas_p0 + offset; //'scrolling' will be normalized at the next frame
                        Config.SetZoom(new_zoom);
                    }
                } else {
                    scrolling.y -= between(io.MouseWheel, -5.f, +5.f)*40;
                }
            }
            //Tracking
            const bool chart_hoovered = ImGui::IsItemHovered() && !search_hoovered;
            const bool clicked = ImGui::IsMouseReleased(ImGuiMouseButton_Left) && !ImGui::GetMouseDragDelta(ImGuiMouseButton_Left) && chart_hoovered;
            const bool to_track = (Config.tracking_mode || Config.show_controls) && chart_hoovered;
            //Switch to tracking mode only if no Load/Save dialogs present and the mouse is over the chart
            if (chart_hoovered && ImGui::IsMouseDoubleClicked(ImGuiMouseButton_Left) && main_state == State::Idle) {
                Config.tracking_mode = !Config.tracking_mode;
                if (!Config.tracking_mode)
                    for (auto& o : Config.overlays) o.do_fade_out = true;
            }
            //determine where the mouse is in chart space
            ImVec2 M = io.MousePos - canvas_p0; //where mouse is relative to chart display area begin
            if (heading==0 || scrolling.y==0 ||  heading < M.y) //if we are not in the heading...
                M = M + scrolling - offset;                     //... adjust this to be relative to chart origin (still multiplied by zoom)
            else
                M.x += scrolling.x - offset.x;                  //... else adjust only the X dir - as the heading is shown
            const Area *const area_hoovered = clicked || to_track ? Config.GetArcByMouse(M) : nullptr; //save the search
            Element *const element_hoovered = area_hoovered ? area_hoovered->arc : nullptr;
            static TextEditor::Coordinates saved_cursor_pos; //TODO: save full selection
            static const Element *cursor_saved_for = nullptr; //set if we have saved the editor cursor pos and highlight the text of an Element
            if (element_hoovered) {
                if (Config.tracking_mode) {
                    if (ConfigData::Overlay *o = Config.GetAnimation(element_hoovered); o && !o->do_fade_out)
                        o->do_fade_out = clicked;
                    else {
                        if (o) {
                            o->fade = clicked ? 1.f : std::max(0.4f, o->fade);
                            o->do_fade_out = !clicked;
                        } else
                            Config.Animate(element_hoovered, clicked ? 1.f : 0.4f, !clicked);
                        if (cursor_saved_for != element_hoovered 
                                && element_hoovered->file_pos.start.file == Config.main_file_no
                                && element_hoovered->file_pos.end.file == Config.main_file_no) {
                            if (cursor_saved_for == nullptr)
                                saved_cursor_pos = Config.editor.GetCursorPosition();
                            cursor_saved_for = element_hoovered;
                            Config.SetCursorTo(element_hoovered);
                        }
                    } 
                } else {
                    if (Config.show_controls && element_hoovered->GetControls().size())
                        Config.ShowControls(element_hoovered);
                    if (clicked) {
                        Config.Animate(element_hoovered);
                        Config.SetCursorTo(element_hoovered);
                    }
                }
            } else if (cursor_saved_for) {
                cursor_saved_for = nullptr;
                Config.editor.SetCursorPosition(saved_cursor_pos);
                Config.editor.SetSelection(saved_cursor_pos, saved_cursor_pos);
            }
            //if we are hoovering over some element controls, keep them live & check for clicks
            if (auto M = Config.MouseToChart(io.MousePos - canvas_p0 + scrolling - offset))
                for (const ConfigData::Control& ctrl : Config.controls)
                    if (ctrl.element)
                        if (const EGUIControlType t = ctrl.element->WhichControl(*M); t!=EGUIControlType::INVALID) {
                            Config.ShowControls(ctrl.element);
                            if (clicked)
                                Config.ControlClicked(ctrl.element, t);
                        }
            const Element *highlight_now = nullptr;
            if (Config.tracking_mode && Config.editor.IsCursorPositionChanged() && Config.pChart) { //shall we disable it on dirty?
                TextEditor::Coordinates cursor_now = Config.editor.GetCursorPosition();
                FileLineCol linenum(int(Config.pChart->Error.Files.size()-1), cursor_now.mLine+1, cursor_now.mColumn+1);
                MscChart::LineToArcMapType::const_iterator entity;
                //in the map linenum_ranges are sorted by increasing length, we search the shortest first
                auto i = std::ranges::find_if(Config.pChart->AllElements, [&linenum](auto &p) 
                                                { return p.first.start <= linenum && linenum <= p.first.end; });
                if (i!=Config.pChart->AllElements.end())
                    highlight_now = i->second;
            }
            if (highlight_now) Config.Animate(highlight_now, 0.4f, false);
        }
    }
    if (!Config.presentation_mode) {
        ImGui::EndTable();
        search_fade = anim_time_value(search_fade, (float)Config.show_search, 3.f);
        static bool search_open = false;
        search_open = search_fade;
        if (search_fade) {
            ImGui::PushStyleVar(ImGuiStyleVar_Alpha, search_fade);
            ImGui::PushStyleVar(ImGuiStyleVar_FrameBorderSize, 1);
            ImVec2 pos = image_viewport_origin;
            pos.y += full_canvas_sz.y/2;
            ImGui::SetNextWindowPos(pos, ImGuiCond_Once, ImVec2{0, 0.5f});
            ImGui::SetNextWindowSize({100,70}, ImGuiCond_Appearing);
            if (refocus_search_window)
                ImGui::SetNextWindowFocus();
            refocus_search_window = false;
            ImGui::Begin("Search and Replace", &search_open, 
                         ImGuiWindowFlags_NoScrollbar | ImGuiWindowFlags_NoScrollWithMouse 
                         | ImGuiWindowFlags_NoCollapse | ImGuiWindowFlags_NoSavedSettings
                         | ImGuiWindowFlags_AlwaysAutoResize |ImGuiWindowFlags_NoResize);
            search_hoovered = ImGui::IsWindowHovered();
            static bool previous_frame_focus = false;
            static std::optional<std::pair<TextEditor::Coordinates, TextEditor::Coordinates>> saved_selection, last_found_pos;
            static std::string last_found_str;
            if (ImGui::IsWindowFocused() != previous_frame_focus && !previous_frame_focus) {
                //newly opened/focused serarch/replace 
                saved_selection = Config.editor.GetSelection();
                if (saved_selection->first>=saved_selection->second)
                    saved_selection.reset();
                last_found_pos.reset();
            }
            previous_frame_focus = ImGui::IsWindowFocused();
            const bool enter_pressed = ImGui::IsWindowFocused() && !shift && !ctrl && !alt && ImGui::IsKeyPressed(ImGui::GetKeyIndex(ImGuiKey_Enter));
            const bool F3_pressed = ImGui::IsWindowFocused() && !shift && !ctrl && !alt && ImGui::IsKeyPressed(MscGenKeyMap[MscGenKey_F3]);
            const bool Shift_F3_pressed = ImGui::IsWindowFocused() && shift && !ctrl && !alt && ImGui::IsKeyPressed(MscGenKeyMap[MscGenKey_F3]);
            const bool Alt_A_pressed = ImGui::IsWindowFocused() && !shift && !ctrl && alt && ImGui::IsKeyPressed(ImGui::GetKeyIndex(ImGuiKey_A));
            ImGui::Text("Search");
            static char search_buff[512] = "\0";
            if (ImGui::IsWindowAppearing())
                ImGui::SetKeyboardFocusHere();
            ImGui::InputText("##Search", search_buff, sizeof(search_buff), ImGuiInputTextFlags_AutoSelectAll);
            const bool search_pressed = ImGui::IsItemFocused() && enter_pressed;
            static bool case_sensitive = false;
            static bool whole_words_only = false;
            static bool in_selection_only = true;
            ImGui::Checkbox("Case Sensitive", &case_sensitive); 
            ImGui::SameLine();
            ImGui::Checkbox("Whole Words Only", &whole_words_only);
            ImGui::PopStyleVar();
            const bool search_fw_pressed = ImGui::Button("Find Next") || F3_pressed;
            Help("Find the next occurence (F3)");
            ImGui::SameLine();
            const bool search_bw_pressed = ImGui::Button("Find Prev") || Shift_F3_pressed;
            Help("Find the previous occurence (Shift+F3)");
            ImGui::Dummy({10,10});
            ImGui::PushStyleVar(ImGuiStyleVar_FrameBorderSize, 1);
            ImGui::Text("Replace to");
            static char replace_buff[512] = "\0";
            ImGui::InputText("##Replace", replace_buff, sizeof(replace_buff), ImGuiInputTextFlags_AutoSelectAll);
            const bool replace_pressed = ImGui::IsItemFocused() && enter_pressed;
            ImGui::BeginDisabled(!saved_selection);
            ImGui::Checkbox("In Selection only", &in_selection_only);
            ImGui::EndDisabled();
            ImGui::PopStyleVar();
            const bool replace_once_pressed = ImGui::Button("Replace Next");
            Help("Replace one occurence of the search string and moves to the next. If a selection is made and "
                 "'In Selection only' is set, we limit the replaces to the selection");
            ImGui::SameLine();
            const bool replace_all_pressed = ImGui::Button("Replace All") || Alt_A_pressed;
            Help("Replace of all occurence of the search string. If a selection is made and "
                 "'In Selection only' is set, we limit the replaces to the selection. (Alt+A)");
            ImGui::End();
            ImGui::PopStyleVar();

            if (search_buff != last_found_str) {
                last_found_str.clear();
                last_found_pos.reset();
            }
            const bool last_found_selected = last_found_pos == Config.editor.GetSelection();
            //OK, now for the search-replace logic
            if (search_pressed || search_fw_pressed || search_bw_pressed) {
            find_next:
                if (Config.editor.Find(search_buff, case_sensitive, whole_words_only,
                                       !search_bw_pressed, //will be true when doing this after a replace
                                       {}, last_found_selected)) {
                    last_found_pos = Config.editor.GetSelection();
                    last_found_str = search_buff;
                }
            } else if (replace_once_pressed || replace_pressed) {
                if (last_found_selected) {
                    if (replace_buff[0])
                        Config.editor.Insert(replace_buff);
                    else
                        Config.editor.Delete(false);
                    refocus_search_window = true;
                }
                goto find_next;
            } else if (replace_all_pressed) {
                Config.editor.SetCursorPosition(saved_selection ? saved_selection->first : TextEditor::Coordinates(0, 0));
                bool next = false;
                while (Config.editor.Find(search_buff, case_sensitive, whole_words_only,
                                          true, saved_selection, next))
                    Config.editor.Insert(replace_buff);
                refocus_search_window = true;
            }
        } else
            search_hoovered = false; //no search window visible, clear hoover flag
        if (!search_open)
            Config.show_search = false;
    } else if (Config.pChart && Config.pChart->GetPageNum()>1) { //presentation mode
        const ImVec2 LR = ImGui::GetMainViewport()->WorkPos + ImGui::GetMainViewport()->WorkSize;
        static ImVec2 page_selector_size(0, 0);
        ImGui::SetNextWindowPos(LR - page_selector_size - ImVec2(20, 20));
        ImGui::SetNextWindowBgAlpha(0.5f);
        ImGui::Begin("Presentation mode", &Config.presentation_mode,
                     ImGuiWindowFlags_NoDecoration | ImGuiWindowFlags_NoScrollWithMouse
                     | ImGuiWindowFlags_AlwaysAutoResize);
        PageSelector();
        page_selector_size = ImGui::GetWindowSize();
        ImGui::End();
    }
    fitted_to_window = Config.IsFittedToWindow(full_canvas_sz);
    fitted_to_width = Config.IsFittedToWidth(full_canvas_sz);

    //ImGui::ShowDemoWindow();

    ImGui::End(); //Main Window
    ImGui::PopFont();

    Config.PruneAnimations();
    Config.SaveSettings(false);
    last_frame = my_clock::now();

    return Config.Ret();
}


void ShutdownGUI() {
    Config.SaveSettings(true);
}
