/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */

// GLib based libraries
#include <grilo.h>
#include <gio/gio.h>
#include <gst/pbutils/pbutils.h>

// Boost C++
#include <boost/algorithm/string.hpp>
#include <boost/filesystem.hpp>
#include <boost/locale/format.hpp>
#include <boost/scoped_ptr.hpp>

// C++ Standard Library
#include <list>
#include <set>
#include <string>
#include <vector>
#include <memory>

// Media Scanner Library
#include "mediascanner/commitpolicy.h"
#include "mediascanner/dbusservice.h"
#include "mediascanner/filesystemscanner.h"
#include "mediascanner/filter.h"
#include "mediascanner/glibutils.h"
#include "mediascanner/locale.h"
#include "mediascanner/logging.h"
#include "mediascanner/mediaartdownloader.h"
#include "mediascanner/mediaroot.h"
#include "mediascanner/metadataresolver.h"
#include "mediascanner/property.h"
#include "mediascanner/propertyschema.h"
#include "mediascanner/settings.h"
#include "mediascanner/taskfacades.h"
#include "mediascanner/taskmanager.h"
#include "mediascanner/utilities.h"
#include "mediascanner/writablemediaindex.h"

// TODO(M3): L10N: properly pull gettext()
#define _(MsgId) (MsgId)
#define N_(MsgId) (MsgId)

namespace boost {

template<>
inline void checked_delete(GOptionContext *p) {
    g_option_context_free(p);
}

} // namespace boost

namespace mediascanner {

// Boost C++
using boost::algorithm::ends_with;
using std::shared_ptr;
using boost::locale::format;

// C++ Standard Library
using std::list;
using std::string;
using std::vector;
using std::wstring;

// Context specific logging domains
static const logging::Domain kError("error/main", logging::error());
static const logging::Domain kWarning("warning/main", logging::warning());
static const logging::Domain kInfo("info/main", logging::info());
static const logging::Domain kTraceGSt("trace/gstreamer", logging::trace());
static const logging::Domain kTraceDBus("trace/dbus", logging::trace());

static const std::string kErrorSetupFailed =
        "com.canonical.MediaScanner.MediaIndexError.SetupFailed";
static const std::string kErrorQueryFailed =
        "com.canonical.MediaScanner.MediaIndexError.QueryFailed";
static const std::string kErrorStoreFailed =
        "com.canonical.MediaScanner.MediaIndexError.StoreFailed";
static const std::string kErrorRemoveFailed =
        "com.canonical.MediaScanner.MediaIndexError.RemoveFailed";

/**
 * @brief The class implements the media scanner service.
 * It manages the file system scanner and it hosts the D-Bus service giving
 * access to the media index.
 */
class MediaScannerService : public dbus::Service {
public:
    int Run(int argc, char *argv[]);

protected:
    void Connected();

    typedef std::function<void(WritableMediaIndex *index)> TaskFunction;

    void setup_media_index(WritableMediaIndex *index) const;

    void push_task(const TaskFunction &task,
                  dbus::MethodInvocationPtr invocation);
    void RunTask(const TaskFunction &task);

    bool ParseCommandLine(int argc, char *argv[]);
    bool LoadMetadataSources();

    void OnMediaRootsChanged();
    void OnSourcesChanged();

private:
    class Interface;
    class CommitNotifier;

    typedef MediaIndexFacade<WritableMediaIndex> TaskFacade;

    std::shared_ptr<Interface> interface_;
    std::shared_ptr<CommitNotifier> notifier_;
    std::shared_ptr<MetadataResolver> metadata_resolver_;
    std::shared_ptr<MediaRootManager> root_manager_;
    std::shared_ptr<TaskManager> task_manager_;
    std::shared_ptr<TaskFacade> task_facade_;
    std::shared_ptr<FileSystemScanner> scanner_;
    Settings settings_;
    Settings::MetadataSourceList metadata_sources_;
};

/**
 * @brief This class describes the the media scanner's D-Bus service.
 * @ingroup dbus
 */
class MediaScannerService::Interface : public dbus::MediaScannerSkeleton {
public:
    explicit Interface(MediaScannerService *service);

    class MediaInfoExistsMethod;
    class LookupMediaInfoMethod;
    class QueryMediaInfoMethod;
    class StoreMediaInfoMethod;
    class RemoveMediaInfoMethod;
    class IndexPathProperty;
    class MediaRootsProperty;

    shared_ptr<MediaInfoAvailableSignal> media_info_available() const {
        return media_info_available_;
    }

    shared_ptr<MediaInfoChangedSignal> media_info_changed() const {
        return media_info_changed_;
    }

private:
    std::shared_ptr<MediaInfoAvailableSignal> media_info_available_;
    std::shared_ptr<MediaInfoChangedSignal> media_info_changed_;
};

static Property::Set ResolveFields(const vector<string> &field_names,
                                   dbus::MethodInvocation *invocation) {
    Property::Set properties;

    for (const string &name: field_names) {
        const Property &p = Property::FromFieldName(ToUnicode(name));

        if (p.is_null()) {
            invocation->return_error
                    (G_DBUS_ERROR, G_DBUS_ERROR_INVALID_ARGS,
                     (format("Unknown field name: \"{1}\"") % name).str());
            return Property::Set();
        }

        properties.insert(p);
    }

    return properties;
}

class MediaScannerService::Interface::MediaInfoExistsMethod
        : public dbus::MediaScannerSkeleton::MediaInfoExistsMethod {
public:
    explicit MediaInfoExistsMethod(MediaScannerService *service)
        : service_(service) {
    }

protected:
    void Invoke(InvocationPtr invocation) const {
        const string url = invocation->arg<0>();
        kTraceDBus("{1}: url=<{2}>") % name() % url;

        service_->push_task(std::bind(&MediaInfoExistsMethod::RunQuery,
                                       this, std::placeholders::_1, ToUnicode(url), invocation),
                           invocation);
    }

    void RunQuery(WritableMediaIndex *media_index,
                  const wstring &url, InvocationPtr invocation) const {
        const bool media_exists = media_index->Exists(url);

        Idle::AddOnce(std::bind(&MediaInfoExistsMethod::return_value,
                                  this, media_exists, invocation));
    }

    void return_value(bool media_exists, InvocationPtr invocation) const {
        invocation->return_value(output_value_type(media_exists));
    }

private:
    MediaScannerService *const service_;
};

class MediaScannerService::Interface::LookupMediaInfoMethod
        : public dbus::MediaScannerSkeleton::LookupMediaInfoMethod {
public:
    explicit LookupMediaInfoMethod(MediaScannerService *service)
        : service_(service) {
    }

protected:
    void Invoke(InvocationPtr invocation) const {
        const string url = invocation->arg<0>();
        const vector<string> field_names = invocation->arg<1>();
        kTraceDBus("{1}: url=<{2}>") % name() % url;

        service_->push_task(std::bind(&LookupMediaInfoMethod::RunQuery,
                                       this, std::placeholders::_1, ToUnicode(url),
                                       ResolveFields(field_names,
                                                     invocation.get()),
                                       invocation),
                           invocation);
    }

    void RunQuery(WritableMediaIndex *media_index,
                  const wstring &url, const Property::Set &properties,
                  InvocationPtr invocation) const {
        const MediaInfo media =
                not properties.empty() ? media_index->Lookup(url, properties)
                                       : media_index->Lookup(url);

        Idle::AddOnce(std::bind(&LookupMediaInfoMethod::return_value,
                                  media, invocation));
    }

    static void return_value(const MediaInfo &media, InvocationPtr invocation) {
        invocation->return_value(output_value_type(media));
    }

private:
    MediaScannerService *const service_;
};

class MediaScannerService::Interface::QueryMediaInfoMethod
        : public dbus::MediaScannerSkeleton::QueryMediaInfoMethod {
public:
    explicit QueryMediaInfoMethod(MediaScannerService *service)
        : service_(service) {
    }

protected:
    void Invoke(InvocationPtr invocation) const {
        const string query = invocation->arg<0>();
        const vector<string> field_names = invocation->arg<1>();
        const int32_t offset = invocation->arg<2>();
        const int32_t limit = invocation->arg<3>();

        kTraceDBus("{1}: query=\"{2}\", offset={3}, limit={4}")
                % name() % query % offset % limit;

        service_->push_task(std::bind(&QueryMediaInfoMethod::RunQuery,
                                       this, std::placeholders::_1, ToUnicode(query),
                                       ResolveFields(field_names,
                                                     invocation.get()),
                                       offset, limit, invocation),
                           invocation);
    }

    void RunQuery(WritableMediaIndex *media_index, const wstring &query,
                  const Property::Set &/*properties*/, int32_t offset,
                  int32_t limit, InvocationPtr invocation) const {
        // FIXME(M4): Pass property set to Query()
        Filter filter;

        if (not query.empty())
            filter = QueryStringFilter(query);

        vector<MediaInfo> items;
        items.reserve(batch_size());
        const uint32_t serial = invocation->serial();

        InvocationPtr origin(new Invocation(invocation->dup()));

        if (media_index->Query(std::bind(&QueryMediaInfoMethod::OnItem,
                                           this, std::placeholders::_1, serial, &items, origin),
                               filter, limit, offset)) {
            if (not items.empty()) {
                // OK, if not empty, notify
                emit_media_info_available(serial, &items, origin);
            }
            // TODO(someone): Dubious. Why notify again?
            emit_media_info_available(serial, &items, origin);

            Idle::AddOnce(std::bind(&QueryMediaInfoMethod::return_value,
                                      invocation));
        } else {
            Idle::AddOnce(std::bind(&Invocation::return_dbus_error,
                                      invocation, kErrorQueryFailed,
                                      media_index->error_message()));
        }
    }

    void OnItem(const MediaInfo &media, uint32_t serial,
                vector<MediaInfo> *items, InvocationPtr origin) const {
        items->push_back(media);

        if (items->size() > batch_size())
            emit_media_info_available(serial, items, origin);
    }

    void emit_media_info_available(uint32_t serial, vector<MediaInfo> *items,
                                   InvocationPtr origin) const {
        MediaInfoAvailableSignal::args_type::value_type args(serial, *items);
        items->clear();

        service_->interface_->
                media_info_available()->
                emit_result_on_idle(args, origin);
    }

    size_t batch_size() const {
        // TOOD(M5): Read batch size from settings.
        return 32;
    }

    static void return_value(InvocationPtr invocation) {
        invocation->return_value(output_value_type());
    }

private:
    MediaScannerService *const service_;
};

class MediaScannerService::Interface::StoreMediaInfoMethod
        : public dbus::MediaScannerSkeleton::StoreMediaInfoMethod {
public:
    explicit StoreMediaInfoMethod(MediaScannerService *service)
        : service_(service) {
    }

protected:
    void Invoke(InvocationPtr invocation) const {
        std::set<std::string> failed_keys;
        const MediaInfo item = invocation->arg<0>(&failed_keys);
        const wstring url = item.first(schema::kUrl);
        kTraceDBus("{1}: url=<{2}>") % name() % url;

        if (url.empty()) {
            invocation->return_error(G_DBUS_ERROR, G_DBUS_ERROR_INVALID_ARGS,
                                     "Valid URL required for storing media "
                                     "information.");
            return;
        }

        service_->push_task(std::bind(&StoreMediaInfoMethod::RunQuery,
                                       this, std::placeholders::_1, url, item, failed_keys,
                                       invocation),
                           invocation);
    }

    void RunQuery(WritableMediaIndex *media_index,
                  const wstring &url, const MediaInfo &item,
                  const std::set<std::string> &failed_keys,
                  InvocationPtr invocation) const {
        if (media_index->Insert(url, item)) {
            Idle::AddOnce(std::bind(&StoreMediaInfoMethod::return_value,
                                      failed_keys, invocation));
        } else {
            Idle::AddOnce(std::bind(&Invocation::return_dbus_error,
                                      invocation, kErrorStoreFailed,
                                      media_index->error_message()));
        }

        // TODO(M3): Figure out if we should trigger metadata discovery here.
    }

    static void return_value(const std::set<std::string> &failed_keys,
                             InvocationPtr invocation) {
        invocation->return_value(output_value_type(failed_keys));
    }

private:
    MediaScannerService *const service_;
};

class MediaScannerService::Interface::RemoveMediaInfoMethod
        : public dbus::MediaScannerSkeleton::RemoveMediaInfoMethod {
public:
    explicit RemoveMediaInfoMethod(MediaScannerService *service)
        : service_(service) {
    }

protected:
    void Invoke(InvocationPtr invocation) const {
        const string url = invocation->arg<0>();
        kTraceDBus("{1}: url=<{2}>") % name() % url;

        service_->push_task(std::bind(&RemoveMediaInfoMethod::RunQuery,
                                       this, std::placeholders::_1, ToUnicode(url), invocation),
                           invocation);
    }

    void RunQuery(WritableMediaIndex *media_index,
                  const wstring &url, InvocationPtr invocation) const {
        if (media_index->Delete(url)) {
            Idle::AddOnce(std::bind(&RemoveMediaInfoMethod::return_value,
                                      invocation));
        } else {
            Idle::AddOnce(std::bind(&Invocation::return_dbus_error,
                                      invocation, kErrorRemoveFailed,
                                      media_index->error_message()));
        }
    }

    static void return_value(InvocationPtr invocation) {
        invocation->return_value(output_value_type());
    }

private:
    MediaScannerService *const service_;
};

class MediaScannerService::Interface::IndexPathProperty
        : public dbus::MediaScannerSkeleton::IndexPathProperty {
public:
    explicit IndexPathProperty(MediaScannerService *service)
        : service_(service) {
    }

protected:
    GVariant* GetValue(const string &, const string &, GError **) const {
        return dbus_type::make_variant(service_->task_facade_->
                                       media_index_path().string());
    }

private:
    MediaScannerService *const service_;
};

class MediaScannerService::Interface::MediaRootsProperty
        : public dbus::MediaScannerSkeleton::MediaRootsProperty {
public:
    explicit MediaRootsProperty(MediaScannerService *service)
        : service_(service) {
    }

protected:
    GVariant* GetValue(const string &, const string &, GError **) const {
        return dbus_type::make_variant(service_->scanner_->directories());
    }

private:
    MediaScannerService *const service_;
};

MediaScannerService::Interface::Interface(MediaScannerService *service)
    : media_info_available_(register_signal<MediaInfoAvailableSignal>())
    , media_info_changed_(register_signal<MediaInfoChangedSignal>()) {
    // FIXME(M5): Do this in MediaScannerInterface()
    register_method<MediaInfoExistsMethod>(service);
    register_method<LookupMediaInfoMethod>(service);
    register_method<QueryMediaInfoMethod>(service);
    register_method<StoreMediaInfoMethod>(service);
    register_method<RemoveMediaInfoMethod>(service);

    register_property<IndexPathProperty>(service);
    register_property<MediaRootsProperty>(service);
}

class MediaScannerService::CommitNotifier : public CommitPolicy {
public:
    explicit CommitNotifier(MediaScannerService *service,
                            CommitPolicyPtr parent = default_policy())
        : service_(service)
        , parent_(parent) {
    }

    bool OnCreate(const std::vector<std::wstring> &media_urls,
                  WritableMediaIndex *media_index) {
        queue_notification(dbus::MEDIA_INFO_CREATED, media_urls);

        if (not parent_->OnCreate(media_urls, media_index))
            return false;

        SendNotifications();
        return true;
    }

    bool OnUpdate(const std::vector<std::wstring> &media_urls,
                  WritableMediaIndex *media_index) {
        queue_notification(dbus::MEDIA_INFO_UPDATED, media_urls);

        if (not parent_->OnUpdate(media_urls, media_index))
            return false;

        SendNotifications();
        return true;
    }

    bool OnRemove(const std::vector<std::wstring> &media_urls,
                  WritableMediaIndex *media_index) {
        queue_notification(dbus::MEDIA_INFO_REMOVED, media_urls);

        if (not parent_->OnRemove(media_urls, media_index))
            return false;

        SendNotifications();
        return true;
    }

protected:
    void queue_notification(dbus::MediaChangeType change_type,
                            const std::vector<std::wstring> &media_urls) {
        kTraceDBus("Queuing {1} notification...") % to_string(change_type);
        pending_notifications_.push_back(Notification(change_type, media_urls));
    }

    void SendNotifications() {
        if (not service_->interface_)
            return;

        while (not pending_notifications_.empty()) {
            const Notification &notify = pending_notifications_.front();

            kTraceDBus("Emitting {1} notification for {2}.")
                    % to_string(notify.change_type())
                    % notify.media_urls();

            const Interface::MediaInfoChangedSignal::args_type::value_type
                    args(notify.change_type(), notify.media_urls());

            service_->interface_->media_info_changed()->emit_signal_on_idle
                    (args, service_->interface_->object_path(),
                     service_->interface_->name(),
                     service_->connection());

            pending_notifications_.pop_front();
        }
    }

private:
    class Notification {
    public:
        Notification(dbus::MediaChangeType change_type,
                     const std::vector<std::wstring> &media_urls)
            : change_type_(change_type) {
            media_urls_.reserve(media_urls.size());

            for (const std::wstring &url: media_urls)
                media_urls_.push_back(FromUnicode(url));
        }

        dbus::MediaChangeType change_type() const {
            return change_type_;
        }

        const std::vector<std::string>& media_urls() const {
            return media_urls_;
        }

    private:
        dbus::MediaChangeType change_type_;
        std::vector<std::string> media_urls_;
    };

    std::list<Notification> pending_notifications_;
    MediaScannerService *const service_;
    const CommitPolicyPtr parent_;
};

void MediaScannerService::Connected() {
    Service::Connected();

    if (not interface_)
        interface_.reset(new Interface(this));

    RegisterObject(interface_->object_path(), interface_);
}

static void report_error(dbus::MethodInvocationPtr invocation,
                         const string &error_message) {
    invocation->return_dbus_error(kErrorSetupFailed, error_message);
}

static void report_error_on_idle(dbus::MethodInvocationPtr invocation,
                                 const string &error_message) {
    // Cannot directly bind MethodInvocation::return_dbus_message because
    // we must keep the MethodInvocation instance alive until the idle handler
    // is called.
    Idle::AddOnce(std::bind(&report_error, invocation, error_message));
}

void MediaScannerService::setup_media_index(WritableMediaIndex *index) const {
    index->set_commit_policy(notifier_);
}

void MediaScannerService::push_task(const TaskFunction &task,
                                    dbus::MethodInvocationPtr invocation) {
    const TaskFacade::ErrorFunction on_error =
            std::bind(&report_error_on_idle, invocation, std::placeholders::_1);
    task_manager_->AppendTask(task_facade_->bind(task, on_error));
}

static void print_error(const string &error_message) {
    kError("{1}") % error_message;
}

void MediaScannerService::RunTask(const TaskFunction &task) {
    const TaskFacade::ErrorFunction on_error = std::bind(&print_error, std::placeholders::_1);
    task_manager_->RunTask(task_facade_->bind(task, on_error));
}

static std::shared_ptr<GList> list_gst_decoders() {
    GList *decoders = gst_element_factory_list_get_elements
            (GST_ELEMENT_FACTORY_TYPE_DECODER, GST_RANK_NONE);

    return std::shared_ptr<GList>(decoders, gst_plugin_feature_list_free);
}

static std::shared_ptr<GList> list_gst_demuxers() {
    GList *decoders = gst_element_factory_list_get_elements
            (GST_ELEMENT_FACTORY_TYPE_DEMUXER, GST_RANK_NONE);

    return std::shared_ptr<GList>(decoders, gst_plugin_feature_list_free);
}

static bool VerifyMediaFeatures(const std::shared_ptr<GList> available,
                                const Settings::MediaFormatList &mandatory) {
    Wrapper<GstCaps> available_caps = take(gst_caps_new_empty());
    bool success = true;

    for (const GList *p = available.get(); p; p = p->next) {
        GstElementFactory *const factory =
                static_cast<GstElementFactory *>(p->data);

        kTraceGSt("scanning plugin feature \"{1}\"")
                % gst_plugin_feature_get_name(p->data);

        const GList *const templates =
                gst_element_factory_get_static_pad_templates(factory);

        for (const GList *l = templates; l; l = l->next) {
            GstStaticPadTemplate *const t =
                    static_cast<GstStaticPadTemplate *>(l->data);

            if (t->direction != GST_PAD_SINK)
                continue;

            Wrapper<GstCaps> caps = take(gst_static_caps_get(&t->static_caps));

            if (gst_caps_is_any(caps))
                continue;

            caps.take(gst_caps_make_writable(caps.release()));

            while (Wrapper<GstStructure> st =
                   take(gst_caps_steal_structure(caps.get(), 0))) {
                if (ends_with(gst_structure_get_name(st.get()), "/x-raw"))
                    continue;

                gst_caps_append_structure(available_caps.get(), st.release());
            }
        }
    }

    available_caps.take(gst_caps_simplify(available_caps.get()));
    kTraceGSt("available: {1}") % to_string(available_caps);

    for (const Settings::MediaFormat &feature: mandatory) {
        const Wrapper<GstCaps> mandatory_caps =
                take(gst_caps_from_string(feature.caps.c_str()));

        if (not mandatory_caps) {
            kWarning("Invalid capability string for {1}: \"{2}\"")
                    % feature.name % feature.caps;
            continue;
        }

        const Wrapper<GstCaps> common_caps =
                take(gst_caps_intersect_full(mandatory_caps.get(),
                                             available_caps.get(),
                                             GST_CAPS_INTERSECT_FIRST));

        std::string mand_caps_str = to_string(mandatory_caps);
        std::string common_caps_str = to_string(common_caps);
        kTraceGSt("mandatory: {1}") % mand_caps_str;
        kTraceGSt("common: {1}") % common_caps_str;

        if (gst_caps_is_empty(common_caps)) {
            // TODO(M5): L10N or maybe trigger plugin installer
            kWarning("No GStreamer support found for {1} ({2})")
                    % feature.name % feature.caps;
            success = false;
            continue;
        }

        if (gst_caps_get_size(common_caps)
                < gst_caps_get_size(mandatory_caps)) {
            // TODO(M5): L10N or maybe trigger plugin installer
            kWarning("Only partial GStreamer support found for {1} ({2})")
                    % feature.name % feature.caps;
            success = false;
            continue;
        }
    }

    return success;
}

/**
 * @brief Parses the command line arguments passed to the program.
 * @param argc The number of command line arguments.
 * @param argv The arguments passed to this program.
 * @param @c true if parsing succeeded
 */
bool MediaScannerService::ParseCommandLine(int argc, char *argv[]) {
    gboolean enable_scanning = true;
    gboolean enable_file_monitor = true;
    gboolean enable_volume_monitor = true;

    Wrapper<char> media_index_path;
    Wrapper<char> metadata_sources;

    const GOptionEntry entries[] = {
        {
            "disable-scanning", 0, G_OPTION_FLAG_REVERSE,
            G_OPTION_ARG_NONE, &enable_scanning,
            N_("Disables filesystem scanning"),
            nullptr
        }, {
            "enable-scanning", 0, G_OPTION_FLAG_HIDDEN,
            G_OPTION_ARG_NONE, &enable_scanning, nullptr, nullptr
        }, {
            "disable-file-monitor", 0, G_OPTION_FLAG_REVERSE,
            G_OPTION_ARG_NONE, &enable_file_monitor,
            N_("Disables monitoring of directory changes"),
            nullptr
        }, {
            "enable-file-monitor", 0, G_OPTION_FLAG_HIDDEN,
            G_OPTION_ARG_NONE, &enable_file_monitor, nullptr, nullptr
        }, {
            "disable-volume-monitor", 0, G_OPTION_FLAG_REVERSE,
            G_OPTION_ARG_NONE, &enable_volume_monitor,
            N_("Disables monitoring of removable media"),
            nullptr
        }, {
            "enable-volume-monitor", 0, G_OPTION_FLAG_HIDDEN,
            G_OPTION_ARG_NONE, &enable_volume_monitor, nullptr, nullptr
        }, {
            "media-index-path", 0, 0,
            G_OPTION_ARG_FILENAME, media_index_path.out_param(),
            N_("Path of the media index database"),
            N_("PATH")
        }, {
            "metadata-sources", 0, 0,
            G_OPTION_ARG_STRING, metadata_sources.out_param(),
            N_("Names of the Grilo metadata sources to use"),
            N_("PLUGIN1:SOURCE1[,PLUGIN2:SOURCE2[,...]]")
        }, {
            nullptr, 0, 0,
            static_cast<GOptionArg>(0), nullptr, nullptr, nullptr
        }
    };

    boost::scoped_ptr<GOptionContext> context
            (g_option_context_new(_("[MEDIA-ROOTS...]")));

    g_option_context_add_group(context.get(), gst_init_get_option_group());
    g_option_context_add_group(context.get(), grl_init_get_option_group());
    g_option_context_add_main_entries(context.get(), entries, GETTEXT_DOMAIN);

    Wrapper<GError> error;

    if (not g_option_context_parse(context.get(), &argc, &argv,
                                   error.out_param())) {
        kError("{1}") % error->message;
        return false;
    }

    // Now really initialize underlaying frameworks. If we'd do early we'd
    // get trouble with command line parsing. Well, _init() functions are evil.
    // Especially if they parse command line options.
    grl_init(nullptr, nullptr);

    if (not gst_init_check(nullptr, nullptr, error.out_param())) {
        const string error_message = to_string(error);
        kError("Could not initialize GStreamer: {1}") % error_message;
        return false;
    }

    // Make sure the network monitor is initialised early, since it is
    // not reliable if initialised from a thread.
    GNetworkMonitor *network_monitor = g_network_monitor_get_default();
    if (not network_monitor) {
        kError("Could not initialize GNetworkMonitor");
        return false;
    }

    // Really check all plugins. Avoid lazy evaluation.
    // Don't abort on missing codecs. We already print a warning.
    VerifyMediaFeatures(list_gst_demuxers(), settings_.mandatory_containers());
    VerifyMediaFeatures(list_gst_decoders(), settings_.mandatory_decoders());

    root_manager_.reset(new MediaRootManager);

    // Setup media roots.
    std::vector<std::string> media_roots;

    if (argc == 1) {
        // No media roots specified. Consult settings and listen for changes.
        settings_.connect
                (Settings::kMediaRoots,
                 std::bind(&MediaScannerService::OnMediaRootsChanged, this));

        media_roots = settings_.media_root_paths();
    } else {
        // Setup media roots according to the command line.
        for (int i = 1; i < argc; ++i) {
            std::string path = argv[i];
            boost::algorithm::trim(path);

            if (path.empty())
                continue;

            const Wrapper<GFile> file =
                    take(g_file_new_for_commandline_arg(path.c_str()));

            if (g_file_query_file_type(file.get(), G_FILE_QUERY_INFO_NONE,
                                       nullptr) != G_FILE_TYPE_DIRECTORY)
                continue;

            media_roots.push_back(take(g_file_get_path(file.get())).get());
        }
    }

    for (const std::string &p: media_roots)
        root_manager_->AddManualRoot(p);

    // Setup metadata sources.
    if (not metadata_sources) {
        // No sources specified. Consult settings and listen for changes.
        settings_.connect
                (Settings::kMetadataSources,
                 std::bind(&MediaScannerService::OnSourcesChanged, this));

        metadata_sources_ = settings_.metadata_sources();
    } else {
        metadata_sources_.clear();
        std::istringstream iss(metadata_sources.get());

        for (std::string id; std::getline(iss, id, ','); ) {
            const std::string::size_type i = id.find(':');
            Settings::MetadataSource info;

            if (i != std::string::npos) {
                info.plugin_id = id.substr(0, i);
                info.source_id = id.substr(i + 1);
            } else {
                info.plugin_id = id;
                info.source_id = id;
            }

            boost::algorithm::trim(info.plugin_id);
            boost::algorithm::trim(info.source_id);
            metadata_sources_.push_back(info);
        }
    }

    // Setup the metadata resolver
    metadata_resolver_.reset(new MetadataResolver);

    if (not LoadMetadataSources())
        return false;

    // Set up the media art downloader
    MediaArtDownloader::LoadGriloPlugin();

    // Setup the scanner itself.
    notifier_.reset(new CommitNotifier(this));
    task_manager_.reset(new TaskManager("media scanner service"));
    task_facade_.reset(new TaskFacade(root_manager_));
    scanner_.reset(new FileSystemScanner(metadata_resolver_, task_manager_,
                                         task_facade_));
    scanner_->set_directories(media_roots);

    if (media_index_path)
        task_facade_->set_media_index_path(media_index_path.get());

    // Start scanning
    scanner_->set_file_monitor_enabled(enable_file_monitor);
    root_manager_->set_enabled(enable_volume_monitor);

    if (enable_scanning) {
        kInfo("Starting the file system scanner");
        scanner_->start_scanning();
    } else {
        kInfo("Not starting the file system scanner");
    }

    return true;
}

void MediaScannerService::OnMediaRootsChanged() {
    // FIXME(M5): Reconfigure scanner
    kInfo("media roots changed...");
}

void MediaScannerService::OnSourcesChanged() {
    // FIXME(M5): Reconfigure scanner
    kInfo("metadata sources changed...");
}

bool MediaScannerService::LoadMetadataSources() {
    const std::vector<std::string> available_sources =
            settings_.LoadMetadataSources(metadata_sources_);

    if (available_sources.size() != metadata_sources_.size())
        return false;

    return metadata_resolver_->SetupSources(available_sources);
}

/**
 * @brief Entry point of the program.
 * @param argc The number of command line arguments.
 * @param argv The arguments passed to this program.
 * @return The exit code of the program.
 */
int MediaScannerService::Run(int argc, char *argv[]) {
    if (not ParseCommandLine(argc, argv))
        return EXIT_FAILURE;

    RunTask(std::bind(&MediaScannerService::setup_media_index, this, std::placeholders::_1));
    Service::Connect(dbus::MediaScannerSkeleton::service_name());
    g_main_loop_run(take(g_main_loop_new(0, false)).get());

    return EXIT_SUCCESS;
}

} // namespace mediascanner

int main(int argc, char *argv[]) {
    std::set_terminate(mediascanner::abort_with_backtrace);
    mediascanner::logging::capture_glib_messages();

    mediascanner::SetupLocale();

    return mediascanner::MediaScannerService().Run(argc, argv);
}
