
/*******************************************************************************
 * Copyright (c) 2023 Olivier Langella <Olivier.Langella@u-psud.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/

// cmake .. -DCMAKE_BUILD_TYPE=Debug  -DMAKE_TESTS=1  -DUSEPAPPSOTREE=1

//./tests/catch2-only-tests [MsRunReader] -s
//./tests/catch2-only-tests [MsRunReaderPerf] -s
//./tests/catch2-only-tests [MsRunReaderPerf] -s --benchmark-samples 5

#define CATCH_CONFIG_ENABLE_BENCHMARKING

#ifdef CATCH2_MAJOR_VERSION_2
#include <catch2/catch.hpp>
#elif CATCH2_MAJOR_VERSION_3
#include <catch2/catch_all.hpp>
using namespace Catch;
#endif


#include <iostream>

#include <pappsomspp/pappsoexception.h>
#include <pappsomspp/exception/exceptionnotpossible.h>
#include <pappsomspp/msfile/msfileaccessor.h>
#include <pappsomspp/msrun/output/mzxmloutput.h>
#include <pappsomspp/processing/uimonitor/uimonitorvoid.h>
#include <pappsomspp/msrun/private/timsframesmsrunreader.h>
#include <pappsomspp/processing/filters/filterpass.h>
#include <QDebug>
#include <QtCore>
#include <QFile>
#include <QtConcurrent>
#include "config.h"
// #include "common.h"

using namespace std;


TEST_CASE("Test MsRunReader new API", "[MsRunReader]")
{

  // Set the debugging message formatting pattern.
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));

#if USEPAPPSOTREE == 1

  SECTION("..::  test mz merge ::..")

  {
    pappso::MsFileAccessor file_access_bruker(
      "/gorgone/pappso/versions_logiciels_pappso/bruker/"
      "200ngHeLaPASEF_2min_compressed.d",
      "");
    file_access_bruker.setPreferredFileReaderType(
      pappso::MzFormat::brukerTims, pappso::FileReaderType::tims_frames);
    pappso::MsRunReaderSPtr msrunA01 =
      file_access_bruker.getMsRunReaderSPtrByRunId("", "runa01");

    pappso::MsRunReaderQualifiedSpectrumList spectrum_list_reader;

    pappso::MsRunReadConfig config;
    config.setNeedPeakList(true);
    config.setMsLevels({1});
    msrunA01.get()->readSpectrumCollection2(config, spectrum_list_reader);
    REQUIRE(spectrum_list_reader.getQualifiedMassSpectrumList().size() == 131);
    auto spectrum = spectrum_list_reader.getQualifiedMassSpectrumList().at(100);
    auto peak_list = pappso::FilterGreatestY(10).filter(
      *(spectrum.getMassSpectrumSPtr().get()));


    std::vector<double> expected_mass_list({706.3999766729,
                                            706.9020200197,
                                            706.9062396402,
                                            707.4042417417,
                                            714.3395398357,
                                            714.3437815601,
                                            715.3494308409,
                                            929.5019128764,
                                            1045.4947003008,
                                            1045.9976567258});

    REQUIRE_THAT(peak_list.xValues(),
                 Catch::Approx(expected_mass_list).margin(0.00001));

    config.setParameterValue(pappso::MsRunReadConfigParameter::
                               TimsFramesMsRunReader_mz_index_merge_window,
                             15);
    spectrum_list_reader.clear();
    msrunA01.get()->readSpectrumCollection2(config, spectrum_list_reader);
    REQUIRE(spectrum_list_reader.getQualifiedMassSpectrumList().size() == 131);

    auto peak_list_merged = pappso::FilterGreatestY(10).filter(
      *(spectrum_list_reader.getQualifiedMassSpectrumList()
          .at(100)
          .getMassSpectrumSPtr()
          .get()));

    std::vector<double> expected_mass_list_merged({575.3077201826,
                                                   679.3848524324,
                                                   706.3915404671,
                                                   706.8978004475,
                                                   707.4042417417,
                                                   714.3225729564,
                                                   715.3409413994,
                                                   929.5261058305,
                                                   1045.5049635122,
                                                   1045.9668599888});
    REQUIRE_THAT(peak_list_merged.xValues(),
                 Catch::Approx(expected_mass_list_merged).margin(0.00001));
  }

  SECTION(
    "..::  MsRunReader uses FileReaderType::tims_frames (explicit reader "
    "config, first "
    "data file) ::..")
  {
    qDebug();
    pappso::MsFileAccessor file_access_A01(
      "/gorgone/pappso/versions_logiciels_pappso/bruker/"
      "200ngHeLaPASEF_2min_compressed.d",
      "");
    std::cout << "number of runIds = " << file_access_A01.getMsRunIds().size()
              << std::endl;

    file_access_A01.setPreferredFileReaderType(
      pappso::MzFormat::brukerTims, pappso::FileReaderType::tims_frames);
    pappso::MsRunReaderSPtr msrunA01 =
      file_access_A01.getMsRunReaderSPtrByRunId("", "runa01");

    pappso::MsRunSimpleStatistics simple_stats;

    pappso::MsRunReadConfig config;
    // config.setRetentionTimeStartInSeconds(300);
    // config.setRetentionTimeEndInSeconds(500);
    config.setNeedPeakList(false);
    config.setMsLevels({1, 2});

    REQUIRE(config.getMsLevels()[0] == false);
    REQUIRE(config.getMsLevels()[1] == true);
    REQUIRE(config.getMsLevels()[2] == true);
    REQUIRE(config.getMsLevels()[3] == false);

    REQUIRE(config.acceptRetentionTimeInSeconds(300) == true);
    msrunA01.get()->readSpectrumCollection2(config, simple_stats);

    REQUIRE(simple_stats.getMsLevelCount(1) == 131);

    REQUIRE(config.needPeakList() == false);

    INFO("selecting 0.5s retention time range and only MS level==1")
    // 2499
    config.setMsLevels({1});

    REQUIRE(config.getMsLevels()[0] == false);
    REQUIRE(config.getMsLevels()[1] == true);
    REQUIRE(config.getMsLevels()[2] == false);
    REQUIRE(config.getMsLevels()[3] == false);

    config.setRetentionTimeStartInSeconds(2499);
    config.setRetentionTimeEndInSeconds(2499.5);
    config.setNeedPeakList(true);

    pappso::MsRunReaderQualifiedSpectrumList spectrum_list_reader;
    msrunA01.get()->readSpectrumCollection2(config, spectrum_list_reader);

    REQUIRE(spectrum_list_reader.getQualifiedMassSpectrumList().size() == 1);
    REQUIRE(spectrum_list_reader.getQualifiedMassSpectrumList()
              .at(0)
              .getMassSpectrumId()
              .getNativeId()
              .toStdString() ==
            "frame id=1184 scan index=794464 im_begin=0 im_end=670");
    REQUIRE(spectrum_list_reader.getQualifiedMassSpectrumList()
              .at(0)
              .getMassSpectrumSPtr()
              ->size() == 138397);


    INFO(
      "selecting 0.5s retention time range and only MS level==1, downgrading "
      "mz precision")
    spectrum_list_reader.clear();
    config.setParameterValue(pappso::MsRunReadConfigParameter::
                               TimsFramesMsRunReader_mz_index_merge_window,
                             30);

    msrunA01.get()->readSpectrumCollection2(config, spectrum_list_reader);

    REQUIRE(spectrum_list_reader.getQualifiedMassSpectrumList().size() == 1);
    REQUIRE(spectrum_list_reader.getQualifiedMassSpectrumList()
              .at(0)
              .getMassSpectrumId()
              .getNativeId()
              .toStdString() ==
            "frame id=1184 scan index=794464 im_begin=0 im_end=670");
    REQUIRE(spectrum_list_reader.getQualifiedMassSpectrumList()
              .at(0)
              .getMassSpectrumSPtr()
              ->size() == 10588);

    INFO(
      "selecting 0.5s and only MS level==1, downgrading mz precision and "
      "selecting ion mobility range")
    spectrum_list_reader.clear();
    config.setParameterValue(pappso::MsRunReadConfigParameter::
                               TimsFramesMsRunReader_mobility_index_begin,
                             150);
    config.setParameterValue(pappso::MsRunReadConfigParameter::
                               TimsFramesMsRunReader_mobility_index_end,
                             200);

    msrunA01.get()->readSpectrumCollection2(config, spectrum_list_reader);
    REQUIRE(spectrum_list_reader.getQualifiedMassSpectrumList().size() == 1);
    REQUIRE(spectrum_list_reader.getQualifiedMassSpectrumList()
              .at(0)
              .getMassSpectrumId()
              .getNativeId()
              .toStdString() ==
            "frame id=1184 scan index=794464 im_begin=150 im_end=200");
    REQUIRE(spectrum_list_reader.getQualifiedMassSpectrumList()
              .at(0)
              .getMassSpectrumSPtr()
              ->size() == 5071);
  }

  SECTION(
    "..::  MsRunReader uses FileReaderType::tims_frames (explicit reader "
    "config, "
    "second data file) ::..")
  {
    qDebug();
    pappso::MsFileAccessor file_access_A01(
      "/gorgone/pappso/data_extraction_pappso/mzXML/"
      // "/data/mzXML/"
      //"/home/langella/data1/mzxml/"
      "20120906_balliau_extract_1_A01_urnb-1.mzXML",
      "");
    std::cout << "number of runIds = " << file_access_A01.getMsRunIds().size()
              << std::endl;
    pappso::MsRunReaderSPtr msrunA01 =
      file_access_A01.getMsRunReaderSPtrByRunId("", "runa01");

    pappso::MsRunReadConfig config;
    config.setRetentionTimeStartInSeconds(300);
    config.setRetentionTimeEndInSeconds(500);
    config.setNeedPeakList(false);
    config.setMsLevels({1, 2});

    pappso::MsRunSimpleStatistics simple_stats;

    msrunA01.get()->readSpectrumCollection2(config, simple_stats);

    REQUIRE(simple_stats.getMsLevelCount(1) == 572);
  }
#endif
}
