//
//  Lynkeos 
//  $Id: MyDocument.m,v 1.24 2005/02/06 15:04:09 j-etienne Exp $
//
//  Created by Jean-Etienne LAMIAUD on Thu Jul 29 2004.
//  Copyright (c) 2003-2005. Jean-Etienne LAMIAUD
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//


#ifdef GNUSTEP
#else
#include <IOKit/pwr_mgt/IOPMLib.h>
#include <IOKit/IOMessage.h>
#endif


#include "fourier.h"

#include "MyDocument.h"

#include "MyDocumentData.h"
#include "MyUserPrefsController.h"

#define K_DOCUMENT_TYPE		@"Lynkeos project"

#define K_SAVE_IMAGE_TAG	100
#define K_EXPORT_MOVIE_TAG	101

//==============================================================================
// Private part of MyDocument
//==============================================================================

@interface ThreadControl : NSObject
{
@public
   NSConnection*	_cnx;
   id                   _threaded;
}
@end


@interface MyDocument(Private)
- (MyImageList*) currentList ;

#ifdef GNUSTEP
#else
- (void) allowSleep :(natural_t) messageType arg:(void*)messageArgument ;
#endif

- (void) stopAllThreads ;
@end

#ifdef GNUSTEP
#else
static void MySleepCallBack(void * x, io_service_t y, natural_t messageType, 
                            void * messageArgument)
{
   [(MyDocument*)x allowSleep:messageType arg:messageArgument];
}
#endif


@implementation ThreadControl

- (id) init
{
   if ( (self = [super init]) != nil )
   {
      _cnx = nil;
      _threaded = nil;
   }
   return self;
}

- (void) dealloc
{
   [_threaded release];
   [_cnx release];
}

@end

@implementation MyDocument(Private)

- (MyImageList*) currentList
{
   MyImageList* list = nil;
   switch( [_myWindow windowMode] )
   {
      case ImageMode :
         list = _imageList;
         break;
      case FlatFieldMode :
         list = _flatFieldList;
         break;
      case DarkFrameMode :
         list =  _darkFrameList;
         break;
   }

   return( list );
}

#ifdef GNUSTEP
#else
- (void) allowSleep :(natural_t) messageType arg:(void*)messageArgument
{
   switch ( messageType )
   {
      case kIOMessageSystemWillSleep:
         IOAllowPowerChange(_rootPort, (long)messageArgument);
         break;
      case kIOMessageCanSystemSleep:
         switch( [_myWindow windowState] )
         {
            case Aligning :
            case Analyzing :
            case Stacking :
               IOCancelPowerChange(_rootPort, (long)messageArgument);
               break;
            default :
               IOAllowPowerChange(_rootPort, (long)messageArgument);
               break;
         }
         break;
      case kIOMessageSystemHasPoweredOn:
         break;
   }
}
#endif

- (void) stopAllThreads
{
   NSEnumerator *threadList;
   ThreadControl *thr;

   threadList = [_threads objectEnumerator];
   while( (thr = [threadList nextObject]) != NULL )
      [thr->_threaded stopProcessing];
}
@end

@implementation MyDocument

//==============================================================================
// Initializers, creators and destructors
//==============================================================================
- (id)init
{
#ifdef GNUSTEP
#else
   IONotificationPortRef  notify;
   io_object_t            anIterator;
#endif

   self = [super init];

   if (self)
   {
      _imageList = [[MyObjectImageList imageListWithArray:nil] retain];
      _darkFrameList = [[MyImageList imageListWithArray:nil] retain];
      _flatFieldList = [[MyImageList imageListWithArray:nil] retain];
      _analysisMethod = EntropyAnalysis;
      _sizeLock = [[MySizeLock sizeLock] retain];
      _stackedImageNb = 0;
      _stackedImage = nil;
      _stackedRep = nil;
      _stack_sequence = 0;
      _minLevel = HUGE;
      _maxLevel = -HUGE;
      _monochromeFlat = NO;
      _windowFrame = nil;

      _postProcess = [[MyPostProcessing alloc] init];

      _threads = [[NSMutableArray array] retain];
      _threadImageList = nil;
      _alignLock = [[NSLock alloc] init];
      FFT_DATA_INIT(&_alignSpectrum);

#ifdef GNUSTEP
#else
      _rootPort = IORegisterForSystemPower(self, &notify, MySleepCallBack, 
                                           &anIterator);
      if ( _rootPort == NULL )
      {
         printf("IORegisterForSystemPower failed\n");
      }
      else
         CFRunLoopAddSource([[NSRunLoop currentRunLoop] getCFRunLoop],
                            IONotificationPortGetRunLoopSource(notify),
                            kCFRunLoopCommonModes);
#endif

   }

   return self;
}

- (void) dealloc
{
   [_imageList release];
   [_darkFrameList release];
   [_flatFieldList release];
   [_sizeLock release];

   if ( _stackedImage != nil )
      [_stackedImage release];
   if ( _stackedRep != nil )
      [_stackedRep release];

   [_windowFrame release];

   [_postProcess release];

   [_threads release];
   [_threadImageList release];
   [_alignLock release];

   [super dealloc];
}

//==============================================================================
// Document load and save
//==============================================================================
- (NSData *)dataRepresentationOfType:(NSString *)aType
{
   MyDocumentDataV1 *myData;

#ifdef GNUSTEP
   // Bug on gnustep ?? aType seem's always equal to '.'
#else
   NSAssert( [aType isEqual:K_DOCUMENT_TYPE], @"Unknown type to export" );
#endif

   myData = [[[MyDocumentDataV1 alloc] init] autorelease];
   myData->_imageList = _imageList;
   myData->_darkFrameList = _darkFrameList;
   myData->_flatFieldList = _flatFieldList;
   myData->_monochromeFlat = _monochromeFlat;
   myData->_analysisMethod = (int)_analysisMethod;
   myData->_windowFrame = [[_myWindow window] stringWithSavedFrame];

   return [NSKeyedArchiver archivedDataWithRootObject:myData];
}

- (BOOL)loadDataRepresentation:(NSData *)data ofType:(NSString *)aType
{
   id myData;
   NSEnumerator* files;
   NSMutableArray* lostFiles;
   MyImageListItem* item;
   MyImageList *list;

#ifdef GNUSTEP
   // Bug on gnustep ?? aType seem's always equal to '.'
#else
   NSAssert( [aType isEqual:K_DOCUMENT_TYPE], @"Unknown type to import" );
#endif

   // Get document data from the file data
   myData = [NSKeyedUnarchiver unarchiveObjectWithData:data];

   if ( [myData isMemberOfClass:[MyDocumentDataV1 class]] )
   {
      // OK, Post V1 file format : store it in our attributes
      [_imageList release];
      _imageList = ((MyDocumentDataV1*)myData)->_imageList;
      [_darkFrameList release];
      _darkFrameList = ((MyDocumentDataV1*)myData)->_darkFrameList;
      [_flatFieldList release];
      _flatFieldList = ((MyDocumentDataV1*)myData)->_flatFieldList;
      _monochromeFlat = ((MyDocumentDataV1*)myData)->_monochromeFlat;
      _analysisMethod = ((MyDocumentDataV1*)myData)->_analysisMethod;
      _windowFrame = ((MyDocumentDataV1*)myData)->_windowFrame;
   }
   else if ( [myData isMemberOfClass:[MyImageListData class]] )
   {
      // V0 file format : store in imageList
      [_imageList initWithImageListData:myData];
   }

   // Check file references
   lostFiles = [NSMutableArray array];
   for ( list = _imageList; list != nil ; )
   {
      files = [[list imageArray] objectEnumerator];
      while ( (item = [files nextObject]) != nil )
      {
         if ( ! [[NSFileManager defaultManager] isReadableFileAtPath:
                                                   [[item getURL] path]] )
            [lostFiles addObject:item];
      }

      if ( list == _imageList )
         list = _darkFrameList;
      else if ( list == _darkFrameList )
         list = _flatFieldList;
      else
         list = nil;
   }

   // If some referenced files don't exist any more, remove the items 
   // from the document
   if ( [lostFiles count] != 0 )
   {
      NSMutableString* message = [NSMutableString stringWithString:
                                                     NSLocalizedString(
                                                        @"FilesNotFound",
                               @"First line of lost file alert panel message")];
      files = [lostFiles objectEnumerator];
      while ( (item = [files nextObject]) != nil )
      {
         [_imageList deleteItem:item];
         [message appendFormat:@"\n%@", [[item getURL] absoluteString]];
      }
      NSRunAlertPanel(NSLocalizedString(@"LostFile",
                                        @"Lost file alert panel title"),
                      message, nil, nil, nil );
   }

   // Update the lock and reject non compliant files
   for ( list = _imageList; list != nil ; )
   {
      files = [[_imageList imageArray] objectEnumerator];
      while ( (item = [files nextObject]) != nil )
      {
         if ( ![_sizeLock addSize: MyIntegerSizeFromNSSize([item imageSize])] )
         {
            // This should not happen (whatever the user do), so log it
            NSLog( @"SizeLock addSize failed for %@ , file discarded", 
                   [[item getURL] absoluteString] );
            [_imageList deleteItem: item];
         }
      }

      if ( list == _imageList )
      {
         if ( ( [[_darkFrameList imageArray] count] != 0 
                || [[_flatFieldList imageArray] count] != 0 )
              && (! [_sizeLock lockSize]) )
         {
            // This should not happen (whatever the user do), so log it
            NSLog( @"SizeLock lock failed, discarding dark frames and flat"
                   "fields" );
            [_darkFrameList release];
            _darkFrameList = nil;
            [_flatFieldList release];
            _flatFieldList = nil;
            
            list = nil;
         }
         else
            list = _darkFrameList;
      }
      else if ( list == _darkFrameList )
         list = _flatFieldList;
      else
         list = nil;
   }
   
   return ( [[_imageList imageArray] count] != 0 
            || [[_darkFrameList imageArray] count] != 0 
            || [[_flatFieldList imageArray] count] != 0 );
}

//==============================================================================
// GUI control
//==============================================================================
- (void) makeWindowControllers
{
   _myWindow = [[MyImageListWindow alloc] init];
   [self addWindowController:_myWindow];
}

- (BOOL)validateMenuItem:(id <NSMenuItem>)menuItem
{
   switch ( [menuItem tag] )
   {
      case K_SAVE_IMAGE_TAG:
         return( _stackedRep != nil && [_myWindow windowState] == ProcessMode );
      case K_EXPORT_MOVIE_TAG:
         return( [[[self currentList] imageArray] count] != 0 );
   }
   return( [super validateMenuItem:(NSMenuItem*)menuItem] );
}

//==============================================================================
// Read accessors
//==============================================================================
- (MyObjectImageList*) imageList { return( _imageList ); }
- (MyImageList*) darkFrameList { return( _darkFrameList ); }
- (MyImageList*) flatFieldList { return( _flatFieldList ); }
- (BOOL) monochromeFlat { return( _monochromeFlat ); }

- (MyAnalysisMethod) analysisMethod { return( _analysisMethod ); }
- (MyIntegerSize) lockedSize { return( [_sizeLock size] ); }
- (NSImage*) stackedImage { return( _stackedImage ); }
- (REAL) minLevel { return( _minLevel ); }
- (REAL) maxLevel{ return( _maxLevel ); }
- (NSString*) windowFrame { return( _windowFrame ); }


//==============================================================================
// Actions
//==============================================================================
- (void) addEntry :(MyImageListItem*)item
{
   NSUndoManager *undo = [self undoManager];

   if ( [item isMemberOfClass: [MyMovieImage class]] )
      // This add is actually an undo inside a movie
      [[(MyMovieImage*)item getParent] addMovieImage :(MyMovieImage*)item];

   else
   {
      MyImageList *list = [self currentList];

      if ( list != _imageList )
      {
         // Try to lock the size because of dark frame or flat field use
         if ( ! [_sizeLock lockSize] )
         {
            NSRunAlertPanel(NSLocalizedString(@"CannotLockTitle",
                                              @"Cannot lock alert panel title"),
                            NSLocalizedString(@"CannotLockText",
                                              @"Cannot lock alert panel text"), 
                            nil, nil, nil );
            return;
         }
      }

      // Check if size is compatible with lock
      if ( [_sizeLock addSize : MyIntegerSizeFromNSSize([item imageSize])] )
      {
         // OK add it
         [list addItem: item];

         // Initialize all alignment to no offset if dark frame or flat field
         if ( list != _imageList )
         {
            NSPoint still = {0.0,0.0};
         
            if ( [item isMemberOfClass:[MyMovie class]] )
            {
               u_long n = [(MyMovie*)item numberOfImages];
               u_long i;
            
               for( i = 0; i < n; i++ )
                  [[(MyMovie*)item getMovieImageAtIndex:i] 
                                                         setAlignOffset:still];
            }
            else
               [item setAlignOffset:still];
         }
         // And for image list, update the side popups
         else
            [_myWindow updateSidePopup];
      }
      else
      {
         NSRunAlertPanel(NSLocalizedString(@"CannotAddTitle",
                                           @"Cannot add alert panel title"),
                         NSLocalizedString(@"CannotAddText",
                                           @"Cannot add alert panel text"), 
                         nil, nil, nil );
         if ( [[_darkFrameList imageArray] count] == 0 &&
              [[_flatFieldList imageArray] count] == 0 )
            [_sizeLock unlockSize];
         return;
      }
   }

   [_myWindow refreshOutline];

   [undo registerUndoWithTarget:self
                       selector:@selector(deleteEntry:)
                         object:item];
   if ( [undo isUndoing] )
      [undo setActionName: 
         NSLocalizedString(@"Delete image",@"Delete image action")];
   else
      [undo setActionName:NSLocalizedString(@"Add image",@"Add image action")];
}

- (void) deleteEntry :(MyImageListItem*)item
{
   NSUndoManager *undo = [self undoManager];
   MyImageListItem *ref = nil;
   MyImageList *list = [self currentList];

   if ( [_myWindow windowMode] == ImageMode )
      ref = [_imageList referenceItem];

   [undo registerUndoWithTarget:self
                       selector:@selector(addEntry:)
                         object:item]; 
   if ( [undo isUndoing] )
      [undo setActionName:NSLocalizedString(@"Add image",@"Add image action")];
   else
      [undo setActionName:
         NSLocalizedString(@"Delete image",@"Delete image action")];

   if ( item ==  ref ||
        ( [ref isMemberOfClass: [MyMovieImage class]] &&
          item == [(MyMovieImage*)ref getParent]) )
      ref = nil;

   [list deleteItem:item];
   if ( ref == nil && [_myWindow windowMode] == ImageMode )
      [self changeReferenceEntry : [_imageList firstItem]];

   // Remove its size from the list if it's a movie or an image
   if ( ![item isMemberOfClass: [MyMovieImage class]] )
      [_sizeLock removeSize:MyIntegerSizeFromNSSize([item imageSize])];

   if ( [[_darkFrameList imageArray] count] == 0 &&
        [[_flatFieldList imageArray] count] == 0 )
      [_sizeLock unlockSize];

   // Free the stack when there is no image left
   if ( [[list imageArray] count] == 0 )
   {
      [list setStack:nil size:0];
      [list invalidateLevels];
      if ( _stackedImage != nil )
         [_stackedImage release];
      _stackedImage = nil;
      if ( _stackedRep != nil )
         [_stackedRep release];
      _stackedRep = nil;
   }

   [_myWindow refreshOutline];
   [_myWindow outlineViewSelectionDidChange:nil];
   if ( [_myWindow windowMode] == ImageMode )
      [_myWindow updateSidePopup];
}

- (void) changeEntrySelection :(MyImageListItem*)item value:(BOOL)v
{
   if ( [[self currentList] changeItemSelection:item value:v] )
      [self updateChangeCount:NSChangeDone];
}

- (void) changeReferenceEntry :(MyImageListItem*)entry
{
   MyImageListItem *oldRef = [_imageList referenceItem];

   NSAssert( [_myWindow windowMode] == ImageMode, 
             @"Change of reference not in image list" );

   if ( [_imageList setReferenceItem: entry] )
   {
      NSUndoManager *undo = [self undoManager];

      [undo registerUndoWithTarget:self
                          selector:@selector(changeReferenceEntry:)
                            object:oldRef];
//      [undo setActionName:NSLocalizedString(@"Change reference image",@"Reference image action")];

      [_myWindow updateAlignControls];
   }
}

- (void) changeSearchSquareOrigin :(MyIntegerPoint)o
{
   NSAssert( [_myWindow windowMode] == ImageMode, 
             @"Change of search square not in image list" );

   if ( [_imageList setSearchSquareOrigin: o] )
      [self updateChangeCount:NSChangeDone];
}

- (void) changeSearchSquareSide :(u_short)side
{
   NSAssert( [_myWindow windowMode] == ImageMode, 
             @"Change of search square not in image list" );

   if ( [_imageList setSearchSquareSide:side] )
      [self updateChangeCount:NSChangeDone];
}

- (void) changeAnalyzeSquareOrigin :(MyIntegerPoint)o
{
   NSAssert( [_myWindow windowMode] == ImageMode, 
             @"Change of analyze square not in image list" );

   if ( [_imageList setAnalyzeSquareOrigin:o] )
      [self updateChangeCount:NSChangeDone];
}

- (void) changeAnalyzeSquareSide :(u_short)side
{
   NSAssert( [_myWindow windowMode] == ImageMode, 
             @"Change of analyze square not in image list" );

   if ( [_imageList setAnalyzeSquareSide:side] )
      [self updateChangeCount:NSChangeDone];
}

- (void) changeAnalysisMethod :(MyAnalysisMethod )method
{
   if ( method != _analysisMethod )
   {
      _analysisMethod = method;
      [self updateChangeCount:NSChangeDone];
   }
}

- (void) autoChangeSelection :(double)selectThreshold
{
   NSAssert( [_myWindow windowMode] == ImageMode, 
             @"Autoselect not in image list" );

   if ( [_imageList autoSelect:selectThreshold] )
      [self updateChangeCount:NSChangeDone];
}

- (void) changeCropRectangle :(MyIntegerRect)rect
{
   NSAssert( [_myWindow windowMode] == ImageMode, 
             @"Change crop rectangle not in image list" );

   if ( [_imageList setCropRectangle:rect] )
      [self updateChangeCount:NSChangeDone];
}

- (void) setDoubleSize :(BOOL)ds
{
   NSAssert( [_myWindow windowMode] == ImageMode, 
             @"Set double size not in image list" );

   if ( [_imageList setSizeFactor:(ds ? 2 : 1)] )
      [self updateChangeCount:NSChangeDone];
}

- (void) setMonochromeFlat :(BOOL)mono
{
   NSAssert( [_myWindow windowMode] == FlatFieldMode, 
             @"Set monochrome flat not in flat fields list" );

   if ( _monochromeFlat != mono )
   {
      [self updateChangeCount:NSChangeDone];
      _monochromeFlat = mono;
   }
}

//==============================================================================
// Image processing
//==============================================================================
- (void) align
{
   ParalelOptimization_t optim;
   u_char i, nListThreads, nFourierThreads;

   NSAssert( [_myWindow windowMode] == ImageMode, 
             @"Align not in image list" );

   switch( [_myWindow windowState] )
   {
      case AlignMode :
         // Start aligning according to user preferences
         optim = [MyUserPrefsController alignMultiProc];
         nListThreads = 1;
         nFourierThreads = 1;

         if ( optim == FFTW3ThreadsOptimization )
            // Paralel FFT
            nFourierThreads = numberOfCpus;

         if ( optim == ListThreadsOptimizations )
            // Parallel list processing
            nListThreads = numberOfCpus;

#ifdef GNUSTEP
         //in gnustep, due to no-thread support, we must inverted two instructions.

         // First memorize the state
         [_myWindow setWindowState : Aligning];

         // Tell FFTW to use the required number of threads
         fftw_plan_with_nthreads(nFourierThreads);
         // And then Create the required number of processing threaded object
         for( i = 0; i < nListThreads; i++ )
            [self createConnectedThread:[MyImageAligner class]];
#else
         // Tell FFTW to use the required number of threads
         fftw_plan_with_nthreads(nFourierThreads);
         // Create the required number of processing threaded object
         for( i = 0; i < nListThreads; i++ )
            [self createConnectedThread:[MyImageAligner class]];

         // And memorize the state
         [_myWindow setWindowState : Aligning];
#endif
         break;
      case Aligning :
         [self stopAllThreads];
         break;
      default :
         NSAssert1( NO, @"Align action in invalid state %d", 
                    [_myWindow windowState] );
   }
}

- (void) analyzeQuality
{
   ParalelOptimization_t optim;
   u_char i, nListThreads, nFourierThreads;

   NSAssert( [_myWindow windowMode] == ImageMode, 
             @"Analyze not in image list" );

   switch( [_myWindow windowState] )
   {
      case AnalyzeMode :
         // Start analyzing according to user preferences
         optim = [MyUserPrefsController analysisMultiProc];
         nListThreads = 1;
         nFourierThreads = 1;
         
         if ( optim == FFTW3ThreadsOptimization )
            // Paralel FFT
            nFourierThreads = numberOfCpus;
            
         if ( optim == ListThreadsOptimizations )
            // Parallel list processing
            nListThreads = numberOfCpus;

#ifdef GNUSTEP
         // in gnustep, due to no-thread support, we must inverte two instructions.

         // First we just memorize the state
         [_myWindow setWindowState : Analyzing];

         // Tell FFTW to use the required number of threads
         fftw_plan_with_nthreads(nFourierThreads);
         // And next, we Create the required number of processing threaded object
         for( i = 0; i < nListThreads; i++ )
            [self createConnectedThread:[MyImageAnalyzer class]];
#else
         // Tell FFTW to use the required number of threads
         fftw_plan_with_nthreads(nFourierThreads);
         // Create the required number of processing threaded object
         for( i = 0; i < nListThreads; i++ )
            [self createConnectedThread:[MyImageAnalyzer class]];

         // And just memorize the state
         [_myWindow setWindowState : Analyzing];
#endif
         break;
      case Analyzing :
         [self stopAllThreads];
         break;
      default :
         NSAssert1( NO, @"Analyze action in invalid state %d", 
                    [_myWindow windowState] );
   }
}

- (void) stack
{
   u_char i, nListThreads;

   switch( [_myWindow windowState] )
   {
      case StackMode :
         // Start stacking according to user prefrence
         if ( [MyUserPrefsController stackMultiProc] )
            nListThreads = numberOfCpus;
         else
            nListThreads = 1;

#ifdef GNUSTEP
         //gnustep is no-threaded, so we need inverted 2 instructions.

         // First we just memorize the state
         [_myWindow setWindowState : Stacking];

         // And next wue create the "thread"
         for( i = 0; i < nListThreads; i++ )
            [self createConnectedThread:[MyImageStack class]];
#else
         for( i = 0; i < nListThreads; i++ )
            [self createConnectedThread:[MyImageStack class]];

         // And just memorize the state
         [_myWindow setWindowState : Stacking];
#endif
         break;
      case Stacking :
         [self stopAllThreads];
         break;
      default :
         NSAssert1( NO, @"Stack action in invalid state %d", 
                    [_myWindow windowState] );
   }
}

- (void) invalidateStackedImage 
{
   if ( _stackedImage != nil )
      [_stackedImage release];
   _stackedImage = nil;
   if ( _stackedRep != nil )
      [_stackedRep release];
   _stackedRep = nil;
}

- (void) postProcess  :(double)dRadius :(double)dThreshold 
                      :(double) uRadius :(double) uGain 
{
   MyImageList *list = [self currentList];
   RGB *stack = [list stack];
   MyIntegerRect crop;
   u_short w, h, factor = 1;
   BOOL paramChange = YES;

   switch ( [_myWindow windowMode] )
   {
      case ImageMode :
         crop = [_imageList cropRectangle];
         factor = [_imageList sizeFactor];
         break;
      case FlatFieldMode :
      case DarkFrameMode :
         crop.origin = MyMakeIntegerPoint(0,0);
         crop.size = [_sizeLock size];
         factor = 1;
         dThreshold = 1.0;  // Copy processing
         uGain = 0.0;
         break;
   }

   // Do something only if there is 
   //  1/ something to process
   //  2/ not yet a processed image
   //     or some change in the process parameters
   if ( list == _imageList )
      paramChange = [_imageList setProcessParameters :dRadius :dThreshold 
                                                     :uRadius :uGain];
   if ( stack != NULL && 
        ( _stackedImage == nil || ( list == _imageList && paramChange ) ) )
   {
      // Process it (according to user preferences)
      if ( [MyUserPrefsController processMultiProc] )
         fftw_plan_with_nthreads(numberOfCpus);
      else
         fftw_plan_with_nthreads(1);

      w = crop.size.width*factor;
      h = crop.size.height*factor;
      [_postProcess process :stack seqnb:_stack_sequence width:w height:h
                    deconvRadius:dRadius
                    deconvThreshold:dThreshold
                    unsharpRadius:uRadius
                    unsharpGain:uGain];

      // Update levels extremes
      _minLevel = [_postProcess minValue];
      _maxLevel = [_postProcess maxValue];

      // On the first processing, update the levels
      if ( ! [list validLevels] )
      {
         if ( list == _flatFieldList )
            // Trick to display a percentage and use a float 0..1
            [list setBlackLevel:0 whiteLevel:_maxLevel*100];
         else
            [list setBlackLevel:0 whiteLevel:_maxLevel];
      }      

      [self updateChangeCount:NSChangeDone];
   }
}

- (void) adjustLevels :(REAL)minV :(REAL)maxV
{
   MyImageList *list = [self currentList];
   NSSize image_size;

   if ( [_myWindow windowMode] == ImageMode )
   {
      int factor = [_imageList sizeFactor];

      image_size = NSSizeFromIntegerSize([_imageList cropRectangle].size);
      image_size.width *= factor;
      image_size.height *= factor;
   }
   else
      image_size = NSSizeFromIntegerSize([_sizeLock size]);

   // Save the parameters
   [list setBlackLevel:minV whiteLevel:maxV];

   // Get rid of the now useless representation
   if ( _stackedImage != nil && _stackedRep != nil )
      [_stackedImage removeRepresentation:_stackedRep];
   if ( _stackedRep != nil )
      [_stackedRep release];
   _stackedRep = nil;

   // Trick to display flat field levels as percentage 
   // and use internally a float 0..1
   if ( [_myWindow windowMode] == FlatFieldMode )
   {
      minV /= 100.0;
      maxV /= 100.0;
   }

   // Create the image with these levels
   if ( [[self currentList] stack] )
      _stackedRep = [[_postProcess makeImageWithMin:minV Max:maxV] retain];

   // Abort if no image returned
   if ( _stackedRep == nil )
      return;

   // Attach the result to an image (or THE image)
   if ( _stackedImage == nil )
      _stackedImage = [[NSImage alloc] initWithSize:image_size];

   [_stackedImage addRepresentation:_stackedRep];

   [self updateChangeCount:NSChangeDone];
}

- (void) saveStackedImage :(id)sender
{
   NSSavePanel* panel = [NSSavePanel savePanel];

   NSAssert( [_myWindow windowMode] == ImageMode, 
             @"Save image not in image list" );

   if ( _stackedRep == nil )
      return;

   [panel setTitle:NSLocalizedString(@"Save image",@"Save image window title")];
   [panel setCanSelectHiddenExtension:YES];
   [panel setRequiredFileType:@"tif"];

   if ( [panel runModal] == NSFileHandlingPanelOKButton )
   {
#ifdef GNUSTEP
     u_char *plane[5];
     NSSize size;
     unsigned int w,h;
     
     [_stackedRep getBitmapDataPlanes:plane];
     size = [_stackedRep size];
     w = size.width;
     h = size.height;

     saveTIFFPictureToFile([[panel filename]  cString],plane[0],w,h);
#else
      // The user returned a filename, save the image in it
      NSData *image = [_stackedRep TIFFRepresentationUsingCompression:
                                               NSTIFFCompressionLZW factor:0.0];
      NSFileWrapper* file = [[[NSFileWrapper alloc] 
                                initRegularFileWithContents:image] autorelease];
      [file writeToFile:[panel filename] atomically:YES updateFilenames:YES];
#endif
   }
}

- (void) exportMovie :(id)sender
{
}

// Thread management
- (void)createConnectedThread :(Class)c
{
   NSMutableDictionary *attrib = [NSMutableDictionary dictionaryWithCapacity:2];
   ThreadControl *thr = [[ThreadControl alloc] init];

   [_threads addObject:thr];

   thr->_cnx = [[NSConnection alloc] initWithReceivePort:[NSPort port]
                                                   sendPort:[NSPort port]];
   [thr->_cnx setRootObject:self];

   // Ports switched here.
   [attrib setObject:[thr->_cnx sendPort] forKey:K_RX_PORT_KEY];
   [attrib setObject:[thr->_cnx receivePort] forKey:K_TX_PORT_KEY];

#ifdef GNUSTEP
   //in gnustep, not thread support for now.
   [c threadWithAttributes: attrib];
#else
   [NSThread detachNewThreadSelector:@selector(threadWithAttributes:) toTarget:c withObject:attrib];
#endif
}

- (void)processDidCreate :(id)obj
{
   NSEnumerator *threadList;
   ThreadControl *thr;
   NSConnection *threadCnx;
   MyIntegerRect r;
   u_short factor = 1;
   MyImageList *list = [self currentList];
   MyImageListItem *ref;
   RGB *dark = [_darkFrameList stack];
   RGB *flat = [_flatFieldList stack];
   FFT_DATA *fftData = &_alignSpectrum;

   // Find which thread it is
   threadList = [_threads objectEnumerator];
   threadCnx = [obj connectionForProxy];
   while( (thr = [threadList nextObject]) != NULL && thr->_cnx != threadCnx )
      ;
   NSAssert( thr != nil, @"Unknown created thread" );
   
   // Keep a reference on the new threaded object proxy
   thr->_threaded = [obj retain];

   // Create an image list enumerator if needed
   if ( _threadImageList == nil )
      _threadImageList = [[list imageEnumerator] retain];

   switch( [_myWindow windowState] )
   {
      case Aligning :
         // Tell the threaded object to align
         r.origin = [_imageList searchSquareOrigin];
         r.size.width = [_imageList searchSquareSide];
         r.size.height = r.size.width;
         ref = [_imageList referenceItem];

         if ( _alignSpectrum.spectrum == NULL )
            allocate_spectrum( &_alignSpectrum, r.size.width, r.size.height, 
                               1, FOR_DIRECT );

         [obj alignWithList:
               [NSData dataWithBytes: &_threadImageList
                              length: sizeof(MyImageListEnumerator*)]
            reference:
               [NSData dataWithBytes: &ref
                              length: sizeof(MyImageListItem*)]
            refBuffer:[NSData dataWithBytes: &fftData
                                     length:sizeof(FFT_DATA*)]
            lock: [NSData dataWithBytes: &_alignLock
                                 length: sizeof(NSLock*)]
            darkFrame: [NSData dataWithBytes: &dark length: sizeof(RGB*)]
            flatField: [NSData dataWithBytes: &flat length: sizeof(RGB*)]
            rectangle: r
               cutOff: [MyUserPrefsController alignFrequencyCutoff]
            precisionThreshold: [MyUserPrefsController alignThreshold]];
         break;

      case Analyzing :
         [_imageList setMinQuality : HUGE];
         [_imageList setMaxQuality : -1];
         r.origin = [_imageList analyzeSquareOrigin];
         r.size.width = [_imageList analyzeSquareSide];
         r.size.height = r.size.width;

         [obj analyzeWithList:
               [NSData dataWithBytes:&_threadImageList
                              length: sizeof(MyImageListEnumerator*)] 
            darkFrame: [NSData dataWithBytes: &dark length: sizeof(RGB*)]
            flatField: [NSData dataWithBytes: &flat length: sizeof(RGB*)]
                          rectangle: r
                             method:_analysisMethod
            lowCutoff: [MyUserPrefsController analysisLowerCutoff]
            highCutoff: [MyUserPrefsController analysisUpperCutoff]];
         break;

      case Stacking :
         switch ( [_myWindow windowMode] )
         {
            case ImageMode :
               factor = [_imageList sizeFactor];
               r = [_imageList cropRectangle];
               break;
            case FlatFieldMode :
            case DarkFrameMode :
               dark = NULL;
               flat = NULL;
               factor = 1;
               r.origin = MyMakeIntegerPoint(0,0);
               r.size = [_sizeLock size];
               break;
         }

         // Free all the process related stuff since it will be rebuilt
         [list setStack:nil size:0];
         [list invalidateLevels];
         if ( _stackedImage != nil )
            [_stackedImage release];
         _stackedImage = nil;
         if ( _stackedRep != nil )
            [_stackedRep release];
         _stackedRep = nil;

         _stackedImageNb = 0;

         // Tell the threaded object to stack
         [obj stackWithList: 
                  [NSData dataWithBytes:&_threadImageList
                                 length: sizeof(MyImageListEnumerator*)] 
               darkFrame: [NSData dataWithBytes: &dark length: sizeof(RGB*)]
               flatField: [NSData dataWithBytes: &flat length: sizeof(RGB*)]
               rectangle: r sizeFactor: factor];

         break;

      default :
         NSAssert1( NO, @"Thread creation in state %d", 
                    [_myWindow windowState] );
   }
}

- (void) processDidProgress :(NSData*)item data:(NSData*)what
{
   MyImageListWindowState state = [_myWindow windowState];
   MyImageListItem* imageItem;
   NSPoint offset;
   double q;

   [item getBytes:&imageItem];

   switch( state )
   {
      case Aligning :
         if ( what != nil )
         {
            [what getBytes:&offset];
            [imageItem setAlignOffset: offset];
            [self updateChangeCount:NSChangeDone];
         }
         break;
      case Analyzing :
         if ( what != nil )
         {
            [what getBytes:&q];
            if ( q < [_imageList minQuality] )
               [_imageList setMinQuality : q];
            if ( q > [_imageList maxQuality] )
	       [_imageList setMaxQuality : q];
            [imageItem setQuality: q];
            [self updateChangeCount:NSChangeDone];
         }
         break;
      case Stacking :
         _stackedImageNb++;
         break;
      default :
         NSAssert1( NO, @"Thread progress in state %d", state );
   }

   [_myWindow highlightItem:imageItem];
}

- (void) processDidFinish :(id)sender data:(NSData*)result
{
   MyImageList *list = [self currentList];
   NSEnumerator *threadList;
   ThreadControl *thr;
   NSConnection *threadCnx;
   RGB *rawStack;
   MyIntegerSize s;
   u_short f;
   u_long length = 0;
   
   // Find which thread it is
   threadList = [_threads objectEnumerator];
   threadCnx = [sender connectionForProxy];
   while( (thr = [threadList nextObject]) != NULL && thr->_cnx != threadCnx )
      ;
   NSAssert( thr != nil, @"Unknown finishing thread" );

   // Release the threaded object proxy
   [_threads removeObject:thr];

   // If stacking, get the partial stack
   if ( [_myWindow windowState] == Stacking )
   {
      RGB *currentStack = [list stack];
   
      // Get the result (that's what we're waiting for)
      // The result address is stored in the NSDAta
      //(because we don't want object translation in thread communication).
      [result getBytes:&rawStack];

      // Store the first partial stack received
      if ( currentStack == NULL )
      {
         if ( [_myWindow windowMode] == ImageMode )
         {
            f = [_imageList sizeFactor];
            s = [_imageList cropRectangle].size;
         }
         else
         {
            f = 1.0;
            s = [_sizeLock size];
         }
         length = sizeof(RGB)*s.width*f*s.height*f;
         [list setStack:rawStack size:length];
         _stack_sequence++;
      }
      // Accumulate the other partial stacks in the first one
      else
      {
         u_long i;

         length = [list stackSize];
         for( i = 0; i < length/sizeof(REAL); i++ )
            ((REAL*)currentStack)[i] += ((REAL*)rawStack)[i];

         free( rawStack );

         rawStack = currentStack;
      }
   }

   // Check if all threads have finished working
   if ( [_threads count] == 0 )
   {
      NSSound *snd = [NSSound soundNamed:@"Glass"];

      switch( [_myWindow windowState] )
      {
         case Aligning :
            free_spectrum( &_alignSpectrum );
            [_myWindow setWindowState : AlignMode];
            break;

         case Analyzing :
            // Update "select threshold" slider
            [_imageList setSelectThreshold: [_imageList minQuality]];
            [_myWindow setWindowState: AnalyzeMode];
            break;

         case Stacking :
            switch( [_myWindow windowMode] )
            {
               case ImageMode :
                  break;

               case FlatFieldMode :
                  // Normalize the stack to max = 1.0
                  normalize_rgb( rawStack, length/sizeof(RGB), 0.0, 
                                 _monochromeFlat );
                  break;

               case DarkFrameMode :
                  // Normalize the stack to a mean value
                  normalize_rgb( rawStack, length/sizeof(RGB), 
                                 1.0/(double)_stackedImageNb, NO );
                  break;
            }

            // Apply final processing
            [_myWindow postProcessAction :nil];
            
            // Restore the view as it was on entry
            [_myWindow highlightItem:[list firstItem]];

            // Exit the state
            [_myWindow setWindowState: StackMode];
            break;

         default :
            NSAssert1( NO, @"Thread termination in state %d", 
                       [_myWindow windowState] );
      }

      // Clean up
      [_threadImageList release];
      _threadImageList = nil;

      // Announce the great news
      [snd play];

#ifdef GNUSTEP
#else
      // Wake up the screen if needed
      UpdateSystemActivity(HDActivity);
#endif
   }
}

@end
