package latexDraw.ui.listeners;

import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.util.Vector;

import latexDraw.figures.Draw;
import latexDraw.figures.Figure;
import latexDraw.ui.LaTeXDrawFrame;
import latexDraw.util.UndoRedoManager;


/** 
 * This class allows the program to listen the shortcuts the user want to do.<br>
 * <br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE.  See the GNU General Public License for more details.<br>
 * <br>
 * 09/15/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 * @since 1.9<br>
 */
public class ShortcutsListener extends AbstractListener implements KeyListener
{
	public static final int INTERVAL_SCROLLBAR = 5;
	
	/** This set of figures is used when the selection is moved for the undo/redo manager. @since 1.9.1 */
	protected Draw clone;
	
	private long oldTime;
	
	
	/**
	 * The constructor.
	 * @param frame The main frame of the program; cannot be null.
	 */
	public ShortcutsListener(LaTeXDrawFrame frame)
	{
		super(frame);
		
		clone = null;
		oldTime = 0;
	}

	
	
	
	public void keyPressed(KeyEvent e)
	{
		int key = e.getKeyCode();

		if(key==KeyEvent.VK_SPACE)
		{// Back space select the first or the next figure (if a single figure is already selected).
			if(mainFrame.isOnSelection())
				mainFrame.getDrawPanel().getDraw().selectNextFigure();
			return ;
		}

		if(key==KeyEvent.VK_A)// ctrl+A = select of the figures.
		{
			if(e.getModifiers()==InputEvent.CTRL_MASK)
			{
				mainFrame.selectAllFigures();
				return ;
			}
			return ;
		}
		
		if(key==KeyEvent.VK_U)// ctrl+U = update the selected figures to the magnetic grid.
		{
			if(e.getModifiers()==InputEvent.CTRL_MASK)
			{
				mainFrame.getDrawPanel().updateSelectionToGrid();
				return ;
			}
			return ;
		}
		
		if(key==KeyEvent.VK_RIGHT)
		{
			if(e.getModifiers()==0)// Right arrow = move the horizontal scrollbar to the right.
			{
				mainFrame.getDrawPanel().moveHorizontalScrollbar(INTERVAL_SCROLLBAR);
				return ;
			}
			
			if(e.getModifiers()==InputEvent.CTRL_MASK)
			{// Right arrow + ctrl = move strongly the horizontal scrollbar to the right.
				mainFrame.getDrawPanel().moveHorizontalScrollbar(mainFrame.getDrawPanel().getWidth());
				return ;
			}
			
			
			if(e.getModifiers()==(InputEvent.CTRL_MASK+InputEvent.SHIFT_MASK))
			{// Right arrow + ctrl + shift = move at the end of the horizontal scrollbar.
				mainFrame.getDrawPanel().getHorizontalScrollBar().setValue(
									mainFrame.getDrawPanel().getHorizontalScrollBar().getMaximum());
				return ;
			}
			
			if(e.getModifiers()==InputEvent.ALT_MASK)
			{// alt + right arrow = move the selected figures to the right.
				if(clone==null)
					try 
					{ 
						oldTime = e.getWhen();
						clone = (Draw)mainFrame.getDrawPanel().getDraw().getSelected().clone(); 
					}
					catch(CloneNotSupportedException e1) { clone = null; }
				
				mainFrame.getDrawPanel().moveSelectedFiguresHorizontally(1);
				return ;
			}
			return ;
		}
		
		
		if(key==KeyEvent.VK_LEFT)
		{
			if(e.getModifiers()==0)// left arrow = move the horizontal scrollbar to the left.
			{
				mainFrame.getDrawPanel().moveHorizontalScrollbar(-INTERVAL_SCROLLBAR);
				return ;
			}
			
			if(e.getModifiers()==InputEvent.CTRL_MASK)
			{// left arrow + ctrl = move strongly the horizontal scrollbar to the left.
				mainFrame.getDrawPanel().moveHorizontalScrollbar(-mainFrame.getDrawPanel().getWidth());
				return ;
			}
			
			if(e.getModifiers()==(InputEvent.CTRL_MASK+InputEvent.SHIFT_MASK))
			{// left arrow + ctrl + shift = move at the beginning of the horizontal scrollbar.
				mainFrame.getDrawPanel().getHorizontalScrollBar().setValue(
								mainFrame.getDrawPanel().getHorizontalScrollBar().getMinimum());
				return ;
			}
			
			if(e.getModifiers()==InputEvent.ALT_MASK)
			{// alt + right arrow = move the selected figures to the left.
				try 
				{ 
					oldTime = e.getWhen();
					clone = (Draw)mainFrame.getDrawPanel().getDraw().getSelected().clone(); 
				}
				catch(CloneNotSupportedException e1) { clone = null; }
					
				mainFrame.getDrawPanel().moveSelectedFiguresHorizontally(-1);
				return ;
			}
			return ;
		}
		
		if(key==KeyEvent.VK_UP)
		{
			if(e.getModifiers()==0)// up arrow = move the vertical scrollbar to the top.
			{
				mainFrame.getDrawPanel().moveVerticalScrollbar(-INTERVAL_SCROLLBAR);
				return ;
			}
			
			if(e.getModifiers()==InputEvent.CTRL_MASK)
			{// up arrow + ctrl = move strongly the vertical scrollbar to the top.
				mainFrame.getDrawPanel().moveVerticalScrollbar(-mainFrame.getDrawPanel().getWidth());
				return ;
			}
			
			if(e.getModifiers()==(InputEvent.CTRL_MASK+InputEvent.SHIFT_MASK))
			{// up arrow + ctrl + shift = move at the beginning of the vertical scrollbar.
				mainFrame.getDrawPanel().getVerticalScrollBar().setValue(
									mainFrame.getDrawPanel().getHorizontalScrollBar().getMinimum());
				return ;
			}
			
			if(e.getModifiers()==InputEvent.ALT_MASK)
			{// alt + right arrow = move the selected figures to the top.
				try 
				{ 
					oldTime = e.getWhen();
					clone = (Draw)mainFrame.getDrawPanel().getDraw().getSelected().clone(); 
				}
				catch(CloneNotSupportedException e1) { clone = null; }
					
				mainFrame.getDrawPanel().moveSelectedFiguresVertically(-1);
				return ;
			}
			return ;
		}
		
		
		if(key==KeyEvent.VK_DOWN)
		{
			if(e.getModifiers()==0)// down arrow = move the vertical scrollbar to the bottom.
			{
				mainFrame.getDrawPanel().moveVerticalScrollbar(INTERVAL_SCROLLBAR);
				return ;
			}
			
			if(e.getModifiers()==InputEvent.CTRL_MASK)
			{// down arrow + ctrl = move strongly the vertical scrollbar to the bottom.
				mainFrame.getDrawPanel().moveVerticalScrollbar(mainFrame.getDrawPanel().getWidth());
				return ;
			}
			
			if(e.getModifiers()==(InputEvent.CTRL_MASK+InputEvent.SHIFT_MASK))
			{// down arrow + ctrl + shift = move at the end of the horizontal scrollbar.
				mainFrame.getDrawPanel().getVerticalScrollBar().setValue(
									mainFrame.getDrawPanel().getHorizontalScrollBar().getMaximum());
				return ;
			}
			
			if(e.getModifiers()==InputEvent.ALT_MASK)
			{// alt + right arrow = move the selected figures to the bottom.
				try 
				{ 
					oldTime = e.getWhen();
					clone = (Draw)mainFrame.getDrawPanel().getDraw().getSelected().clone(); 
				}
				catch(CloneNotSupportedException e1) { clone = null; }
					
				mainFrame.getDrawPanel().moveSelectedFiguresVertically(1);
				return ;
			}
			return ;
		}
	}

	
	
	
	public void keyReleased(KeyEvent e)
	{
		if(clone!=null && e.getWhen()-oldTime>100)
		{
			Draw selection = mainFrame.getDrawPanel().getDraw().getSelected();
			
			if(selection!=null && !selection.isEmpty())
			{
				Vector<Figure> figures = mainFrame.getDrawPanel().getDraw().getFigures();
				int[] id = new int[selection.size()];
				
				for(int i=0, size=selection.size(); i<size; i++)
				{
					id[i] = figures.indexOf(selection.getFigureAt(i));
					
					if(id[i]<0)
						throw new IndexOutOfBoundsException();
				}
				
				mainFrame.getUndoManager().add(UndoRedoManager.LABEL_CHANGE_MOVE, id, clone, false);
				mainFrame.updateUndoRedo();
				clone = null;
				mainFrame.setIsModified(true);
			}
		}
		
		oldTime = e.getWhen();
	}

	
	
	public void keyTyped(KeyEvent e)
	{
		/*
		 * Nothing to do.
		 */
	}
}
