package latexDraw.parsers.svg;

import latexDraw.figures.*;
import latexDraw.generators.svg.*;
import latexDraw.parsers.svg.elements.*;
import latexDraw.util.LaTeXDrawNamespace;

/**
 * Creates latexdraw figures according to the given SVG element.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 09/21/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public class LaTeXDrawFiguresFactory
{
	public static Figure createFigure(SVGElement e)
	{
		return createFigure(e, true);
	}
	
	
	public static Figure createFigure(SVGElement e, boolean withTransformations)
	{
		if(e==null || !e.enableRendering())
			return null;
		
		try
		{
			if(e instanceof SVGRectElement)
				return new LRectangleSVGGenerator((SVGRectElement)e).getShape();
			else if(e instanceof SVGEllipseElement)
				return new LEllipseSVGGenerator((SVGEllipseElement)e).getShape();
			else if(e instanceof SVGCircleElement)
				return new LCircleSVGGenerator((SVGCircleElement)e).getShape();
			else if(e instanceof SVGPolygonElement)
				return new LPolygonSVGGenerator((SVGPolygonElement)e).getShape();
			else if(e instanceof SVGPolyLineElement)
				return new LLinesSVGGenerator((SVGPolyLineElement)e).getShape();
			else if(e instanceof SVGImageElement)
				return new LPictureSVGGenerator((SVGImageElement)e).getShape();
			else if(e instanceof SVGLineElement)
				return new LLineSVGGenerator((SVGLineElement)e).getShape();
			else if(e instanceof SVGTextElement)
				return new LTextSVGGenerator((SVGTextElement)e).getShape();
			else if(e instanceof SVGPathElement)
			{
				SVGPathElement p = (SVGPathElement)e;
				
				if(p.isPolygon())
					return new LPolygonSVGGenerator((SVGPathElement)e).getShape();
				else
					if(p.isLines())
						return new LLinesSVGGenerator((SVGPathElement)e).getShape();
					else
						if(p.isLine())
							return new LLineSVGGenerator((SVGPathElement)e).getShape();
			}
			else
				if(e instanceof SVGGElement)
				{
					String ltdPref = e.lookupPrefixUsable(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE_URI);
					String type = e.getAttribute(ltdPref+LaTeXDrawNamespace.XML_TYPE);
					
					if(type==null || type.length()==0 || LaTeXDrawNamespace.XML_TYPE_DRAWING.equals(type))
					{
						switch(e.getChildNodes().getLength())
						{
							case 0:
								return null;
							case 1:
								return createFigure((SVGElement)e.getChildNodes().item(0));
							default:
								return new LDrawingSVGGenerator((SVGGElement)e, withTransformations).getShape();
						}
					}
					
					if(type.equals(LaTeXDrawNamespace.XML_TYPE_RECT))
						return new LRectangleSVGGenerator((SVGGElement)e, withTransformations).getShape();
					else if(type.equals(LaTeXDrawNamespace.XML_TYPE_ELLIPSE))
						return new LEllipseSVGGenerator((SVGGElement)e, withTransformations).getShape();
					else if(type.equals(LaTeXDrawNamespace.XML_TYPE_CIRCLE))
						return new LCircleSVGGenerator((SVGGElement)e, withTransformations).getShape();
					else if(type.equals(LaTeXDrawNamespace.XML_TYPE_POLYGON))
						return new LPolygonSVGGenerator((SVGGElement)e, withTransformations).getShape();
					else if(type.equals(LaTeXDrawNamespace.XML_TYPE_SQUARE))
						return new LSquareSVGGenerator((SVGGElement)e, withTransformations).getShape();
					else if(type.equals(LaTeXDrawNamespace.XML_TYPE_TRIANGLE))
						return new LTriangleSVGGenerator((SVGGElement)e, withTransformations).getShape();
					else if(type.equals(LaTeXDrawNamespace.XML_TYPE_RHOMBUS))
						return new LRhombusSVGGenerator((SVGGElement)e, withTransformations).getShape();
					else if(type.equals(LaTeXDrawNamespace.XML_TYPE_JOINED_LINES))
						return new LLinesSVGGenerator((SVGGElement)e, withTransformations).getShape();
					else if(type.equals(LaTeXDrawNamespace.XML_TYPE_AKIN_PTS))
						return new LFreeHandSVGGenerator((SVGGElement)e, withTransformations).getShape();
					else if(type.equals(LaTeXDrawNamespace.XML_TYPE_LINE))
						return new LLineSVGGenerator((SVGGElement)e, withTransformations).getShape();
					else if(type.equals(LaTeXDrawNamespace.XML_TYPE_BEZIER_CURVE))
						return new LBezierCurveSVGGenerator((SVGGElement)e, withTransformations).getShape();
					else if(type.equals(LaTeXDrawNamespace.XML_TYPE_GRID))
						return new LGridSVGGenerator((SVGGElement)e, withTransformations).getShape();
					else if(type.equals(LaTeXDrawNamespace.XML_TYPE_AXE))
						return new LAxeSVGGenerator((SVGGElement)e, withTransformations).getShape();
					else if(type.equals(LaTeXDrawNamespace.XML_TYPE_TEXT))
						return new LTextSVGGenerator((SVGGElement)e, withTransformations).getShape();
					else if(type.equals(LaTeXDrawNamespace.XML_TYPE_DRAWING))
						return new LDrawingSVGGenerator((SVGGElement)e, withTransformations).getShape();
					else if(type.equals(LaTeXDrawNamespace.XML_TYPE_DOT))
						return new LDotSVGGenerator((SVGGElement)e, withTransformations).getShape();
					else if(type.equals(LaTeXDrawNamespace.XML_TYPE_ARC))
						return new LArcSVGGenerator((SVGGElement)e, withTransformations).getShape();
					else if(type.equals(LaTeXDrawNamespace.XML_TYPE_PICTURE))
						return new LPictureSVGGenerator((SVGGElement)e, withTransformations).getShape();
				}
		}
		catch(Exception ex) { ex.printStackTrace(); }
		
		return null;
	}
}
