/***************************************************************************
                          currentaccount.h  -  description
                             -------------------
    begin                : Sun Jan 5 2003
    copyright            : (C) 2003 by Mike K. Bennett
    email                : mkb137b@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CURRENTACCOUNT_H
#define CURRENTACCOUNT_H

#include "account.h"

#include <QStringList>


// Forward declarations
class ContactBase;
class InvitedContact;
class KMessTest;
class ContactList;
class MsnSwitchboardConnection;



/**
 * @brief Data class for session information of the logged in user.
 *
 * When the user logs into MSN Messenger, the data of the Account class
 * is copied to this class with the copyAccount() method.
 * During the session, all modifications are stored here.
 * When the user disconnects, the data is copied back and eventually saved to disk.
 *
 * This class also stores session-related data While the user is connected.
 * This includes the contact list state.
 *
 * @author Mike K. Bennett
 * @ingroup Root
 */
class CurrentAccount : public Account
{
  Q_OBJECT

  friend class KMessTest;

  public:
    // Add an unknown contact to the invited contact list
    InvitedContact *         addInvitedContact( const QString &handle, const QString &friendlyName = QString(), const uint capabilities = 0 );
    // Receive notice that some emails were deleted
    void                     changeNoEmails( int change );
    // Copy an account
    virtual void             copyAccount( const Account *account );
    // Delete the instance of the contact list
    static void              destroy();
    // Return whether or not to autoreply to messages
    bool                     getAutoreply() const;
    // Return a contact by handle
    ContactBase              *getContactByHandle(const QString &handle) const;
    // Return a contact's name by handle
    QString                  getContactFriendlyNameByHandle(const QString &handle, FormattingMode mode = STRING_CLEANED ) const;
    // Return the contact list as read-only
    const ContactList *      getContactList() const;
    // Return the contact which was last dragged.
    const ContactBase*       getContactLastDragged() const;
    // Return the email URL
    const QString&           getEmailUrl() const;
    // Return the external IP, as seen from the MSN server.
    const QString&           getExternalIp() const;
    // Return the external port, as seen from the MSN server.
    uint                     getExternalPort() const;
    // Return the language code setted for current account
    const QString&           getLanguageCode() const;
    // Return the local IP.
    const QString&           getLocalIp() const;
    // Return the number of email in the inbox
    int                      getNoEmails() const;
    // Return the key for offline im messages
    const QString&           getOfflineImKey() const;
    // Return the user's staus
    Status                   getStatus() const;
    // Return the requested token
    const QString            getToken( const QString &domain ) const;
    // Return the qhash that contain url informations
    QHash<QString,QString>   &getUrlInformation();
    // Return whether the given contact exists in the contact list.
    bool                     hasContactInList(const QString &handle) const;
    // Return a singleton instance of the current account
    static CurrentAccount*   instance();
    // Open email with the given command, url, and folder
    void                     openMail( const QString &folder );
    // Open email at the compose page with the given contact handle
    void                     openMailAtCompose(QString contactHandle);
    // Open email at the user's inbox
    void                     openMailAtInbox();
    // Set some account information
    void                     setAccountInformation( QString authorization, QString preferredEmail,
                                                    QString sid, QString languageCode, bool emailSupported,
                                                    QString externalIp, uint externalPort,
                                                    QString internalIp );
    // Set whether or not to autoreply to a contact's chat message.
    void                     setAutoreply(bool autoreply);
    // Store the contact list, maintained by the MsnNotificationConnection class
    void                     setContactList(const ContactList *contactList);
    // Set initial email information
    void                     setInitialEmailInformation( int noEmailsInInbox, int noEmailsInOtherFolders );
    // Set the email post url
    void                     setEmailUrl( QString postUrl );
    // Set the number of emails
    void                     setNoEmails( int noEmails );
    // Change the key for offline im messages
    void                     setOfflineImKey( const QString &key );
    // Set the user's status
    void                     setStatus( Status status );
    // Set the tokens
    void                     setTokens( const QHash<QString,QString> tokens );

  private: // Private methods
    // The constructor
                             CurrentAccount();
    // The destructor
    virtual                 ~CurrentAccount();

  private slots: // Private slots
    // An invited contact left all chats.
    void                     slotInvitedContactLeftAllChats( ContactBase *contactBase );

  private:  // private methods
    // Find an invited contact by handle
    InvitedContact *         getInvitedContactByHandle(const QString &handle) const;

  private: // Private attributes
    // Whether or not the program should autoreply to chat messages.
    bool                     autoreply_;
    // The contact list of the current account
    const ContactList       *contactList_;
    // The user's preferred email URL (i.e. www.hotmail.com)
    QString                  emailUrl_;
    // Our (external) IP address seen from the MSN server.
    QString                  externalIp_;
    // Our (external) Port seen from the MSN server.
    uint                     externalPort_;
    // The instance of the singleton account
    static CurrentAccount   *instance_;
    // The list of invited contacts, not part of the MSN ContactList of the user.
    QList<InvitedContact*>   invitedContacts_;
    // Key for offline im messages
    QString                  offlineImKey_;
    // Current language code setted for current account
    QString                  languageCode_;
    // The local IP address
    QString                  localIp_;
    // The number of emails in the user's inbox.
    int                      noEmails_;
    // The email address considered by the server to be the user's preferred address.  I don't know when it differs from the handle.
    QString                  preferredEmail_;
    // A value used to access the user's hotmail account.
    QString                  sid_;
    // The online status of the user.
    Status                   status_;
    // All tokens for each domain
    QHash<QString,QString>   tokens_;
    // All url received by URL command sended by ns
    QHash<QString,QString>   urls_;

  signals: // Public signals
    // Signal that the account went invisible
    void                     accountInvisible();
    // Signal that the account went offline
    void                     accountOffline();
    // Signal that the account went online
    void                     accountOnline();
    // Signal that the number of emails has changed
    void                     changedNoEmails();
    // Signal that the user's status has changed
    void                     changedStatus();
};

#endif
