/* -*- mode: C; tab-width: 2; indent-tabs-mode: nil; fill-column: 79; coding: iso-latin-1-unix -*- */
/* bcnrand.c

This is a version of bcnrand.f written using 64-bit integers (on 64-bit systems it's 'long',
on 32-bit systems it's 'long long').

 */
/*
!   This routine generates a sequence of IEEE 64-bit floating-point pseudorandom
!   numbers in the range (0, 1), based on the recently discovered class of
!   normal numbers described in the paper "Random Generators and Normal Numbers"
!   by DHB and Richard Crandall, Experimental Mathematics, vol 11, no 4 (2002),
!   available at http://crd.lbl.gov/~dhbailey/dhbpapers/bcnormal-em.pdf.

!   The sequence generated consists of consecutive 53-bit sections of the binary
!   expansion of alpha_{2,3} = sum_k 1/((3^k)*2^(3^k))), which is known to be
!   2-normal, normalized by 2^53 so as to be uniform in (0,1).  The parameter a
!   is the starting index, or in other words the seed of the pseudorandom
!   sequence.  To obtain the maximum period, a should be set to at least
!   3^33 + 100 and must not exceed 2^53.  When a is set in this range, the period
!   of the sequence generated by this code is 2x3^32 = 3.706e15.  Also, the
!   sequence will not necessarily match the binary digits of alpha_{2,3} if more
!   than this many elements are generated.

!   The bcnrand routine facilitates simple parallelization.  For example, in
!   an MPI program, suppose that kpr is the processor number and npr is the
!   number of processors.  Then the line

!   call bcnrand (n/npr, a + 53*n/npr*kpr, x)

!   generates on each processor a section of length n/npr.  In this way, the
!   npr processors collectively have the same n-long sequence (provided that
!   n is divisible by npr) as is generated on a single processor system by
!   means of the line

!   call bcnrand (n, a, x)

!   This code is designed for systems with IEEE 64-bit floating-point arithmetic.
!   On IBM systems and others with a fused multiply-add instruction, see the
!   !> comments below for changes that will improve performance. On IBM systems,
!   use the compiler flag -qstrict.

!   David H. Bailey    2004-05-12
*/

#include "hpccfft.h"

typedef u64Int_t Big[2];

/* r = a * b */
static void
ddmuldd(u64Int_t a, u64Int_t b, Big r) {
  u64Int_t a0, a1, b0, b1, hb, acc, acc1;

  /* 'hb' should be 0xFFFFFFFF (first 32-bits set to one) */
  hb = 65535L;
  hb = (hb << 16) | hb;

  /* split 'a' and 'b' into two 32-bit quantities */
  a0 = a & hb;
  a1 = (a >> 32) & hb;
  b0 = b & hb;
  b1 = (b >> 32) & hb;

  acc = a0 * b0;
  r[0] = acc & hb;
  acc >>= 32;
  acc += a1 * b0;
  acc1 = acc >> 32;
  acc &= hb;
  acc += a0 * b1;
  r[0] += (acc & hb) << 32;
  acc >>= 32;
  acc += acc1;
  acc += a1 * b1;
  r[1] = acc;
}

/* r = a - b */
static void
ddsub(Big a, Big b, Big r) {
  u64Int_t mx = 0;
  mx = ~mx;

  r[1] = a[1] - b[1];
  if (a[0] >= b[0])
    r[0] = a[0] - b[0];
  else {
    r[1] -= 1;
    r[0] = mx - b[0] + 1 + a[0];
  }
}

/* q = d / v; reminder Ur */
static void
dddiv(Big d, u64Int_t v, Big q, u64Int_t *Ur) {
  u64Int_t r1, r0, v1, v0, msb = 1, mx = 0, one = 1;
  int i;

  msb <<= 63;
  mx = ~mx;
  q[0] = q[1] = 0;

  if (v <= d[1]) {
    q[1] = d[1] / v;
    r1 = d[1] % v;
  } else {
    r1 = d[1];
    q[1] = 0;
  }
  r0 = d[0];

  while (r1) {
    v1 = 0;
    v0 = v;

    for (i = 0; v1 <= r1; i++) {
      v1 <<= 1;
      if (msb & v0) v1 |= 1;
      v0 <<= 1;
    }
    do {
    i--;
    v0 >>= 1;
    v0 &= mx;
    if (1 & v1) v0 |= msb;
    v1 >>= 1;
    } while (v1 == r1 && v0 > r0); /* make sure (v1,v0) is not too big */

    q[0] += one << i;
    r1 -= v1;
    if (r0 >= v0)
      r0 -= v0;
    else {
      r0 += mx - v0 + 1;
      r1 -= 1;
    }

  }

  q[0] += r0 / v;
  r0 %= v;

  if (Ur) *Ur = r0;
}

/*
!   expm2 = 2^p mod am.  p2 is a table with powers of 2, i.e., p2(i) = 2^i.
!   This routine uses a left-to-right binary exponentiation scheme.
*/
static u64Int_t
expm2(u64Int_t p, u64Int_t am) {
  u64Int_t p2, p1, pt1, r;
  Big ddm, dd1, dd2;
  int i;

  for (p2 = i = 1; i < 54; i++) {
    p2 <<= 1;
    if (p2 > p) break;
  }

  p1 = p;
  pt1 = p2 >> 1;
  r = 1;
  ddm[0] = am;
  ddm[1] = 0;

  while (1) {
    if (p1 >= pt1) {
      /* r = mod(2.0 * r, am) */
      ddmuldd( 2, r, dd1 );
      if (dd1[0] > am) {
	ddsub(dd1, ddm, dd2);
	dd1[0] = dd2[0];
	dd1[1] = dd2[1];
      }
      r = dd1[0];
      p1 = p1 - pt1;
    }

    pt1 /= 2;
    if (pt1 >= 1) {
      /* r = mod(r * r, am) */
      ddmuldd( r, r, dd1 );
      dddiv( dd1, am, dd2, &r );
      continue;
    }
    break;
  }

  return r;
}

/*
  Let minA = 3^33 + 100
  If `a' is smaller than `minA' then `a' is incremented by `minA' this value.
  In this way, you can seed the generator with small integers and the requirements
  will be fullfilled internally.
 */
int
HPCC_bcnrand(u64Int_t n, u64Int_t a, void *vx) {
  u64Int_t d1, d2, d3, t53, p3i, ui, minA;
  s64Int_t sd1, sp3i;
  Big dd1, dd2;
  int i;
  double rcp, two64, v, *x = (double *)vx;

  /* minA = 3.d0 ** 33 + 100.d0 */
  minA = 20709114;
  minA <<= 28;
  minA += 106609639;

  /* make sure `a' is big enough */
  if (a < minA) a += minA;

  t53 = 1;
  t53 <<= 53;

  d1 = 1;
  for (i = 0; i < 53; i++) {
    d1 *= 3;
    if (d1 > a) break;
  }

  /* two64 = 2 ** 64 */
  two64 = 2.0;
  for (i = 0; i < 6; i++)
    two64 *= two64;

  p3i = d1 / 3;
  sp3i = (s64Int_t)p3i;
  rcp = 1.0 / p3i;

  /*
!   Calculate starting element.  This code performs the following:
!   d1 = [int[p3i/2] * 2^(a-p3i)] mod p3i.
  */
  /* d1 = (p3i/2 * (2 ** (a-p3i))) % p3i */
  d2 = expm2( a - p3i, p3i );
  d3 = p3i / 2;
  ddmuldd( d2, d3, dd1 );
  dddiv( dd1, p3i, dd2, &d1 );

  x[0] = d1 * rcp;
  for (ui = 1; ui < n; ui++) {
    /* dd1 = d1 * t53 */
    dd1[1] = (d1 >> 11);
    dd1[0] = (d1 << 53);

    /* Approximate `dd1/p3i' (the result should be off by 1) */
    v = ((two64 * (double)dd1[1]) + (double)dd1[0]) * rcp;

    ddmuldd( (u64Int_t)v, p3i, dd2 );

    /* The value of `dd1-dd2' should between `-p3i' and 'p3i',
     hence upper halves of `dd1' and `dd2' can be ignored */
    sd1 = (s64Int_t)(dd1[0] - dd2[0]);

    /* Check the above approximation */
    if (sd1 < 0) sd1 += sp3i;
    if (sd1 > sp3i) sd1 -= sp3i;

    /* d1 = (d1 * t53) % p3i */
    d1 = (u64Int_t)sd1;
    x[ui] = d1 * rcp;
  }

  return 0;
}
