/* 
   device.cpp - Implementation of the Device class

   Copyright (c) 2003 Tim Stadelmann

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
*/



/** HEADERS **/

/* standard C++ headers */
#include <cstdio>
#include <cerrno>
using namespace std;

/* KDE headers */
#include <klocale.h>
#include <kmessagebox.h>

/* QT headers */
#include <qstring.h>

/* program headers */
#include "device.h"



/** FUNCTIONS **/

/* The constructor for the Device class.  Apart from the parent widget
   (for dialog boxes) his function takes the number of the IDE
   controller and the path to the hotswap backend as arguments.
   Defaults for both are defined in device.h.  */
Device::Device (QWidget *parent_arg,
		const unsigned int ide_controller_arg,
		QString hotswap_path_arg)
{
  parent = parent_arg;
  ide_controller = ide_controller_arg;
  hotswap_path = hotswap_path_arg;

  /* Probe the currently configured device and update the internal
     state.  */
  probe ();
}

/* The destructor for the Device class.  Currently does nothing.  */
Device::~Device ()
{
}

/* This method returns true if a device is present.  */
bool
Device::present ()
{
  /* Return the internal flag indicating the presence of a device.  */
  return present_flag;
}

/* This method returns the IDE model string or the configured device,
   or i18n ("absent") if no device is configured.  */
QString
Device::name ()
{
  return model_string;
}

/* This method inserts a device by calling the backend with the
   correct arguments for rescanning the IDE bus.  Returns true if a
   device was configured successfully.  */
bool
Device::insert ()
{
  for (;;)
    {
      /* Call the backend to rescan the IDE bus.  */
      if (!call_backend (" -n rescan-ide"))
	return false;

      probe ();

      /* Return successfully if a device has been found.  */
      if (present_flag)
	return true;
      
      if (KMessageBox::warningYesNo
	  (parent,
	   i18n ("No IDE device has been found.  "
		 "If you are sure that the device "
		 "connects to the IDE bus, verify that "
		 "the module is inserted correctly.\n"
		 "Do you want to try again?"),
	   i18n ("No Device Found"))
	  == KMessageBox::No)
	return false;
    }
}


/* Call the backend to remove the device from the system
   configuration.  Returns false if an error has occurred, true
   otherwise.  */
bool
Device::remove ()
{
  if (mounted ())
    {
      KMessageBox::sorry 
	(parent,
	 i18n ("At least one filesystem on the current "
	       "device is still mounted.  Unmount the "
	       "mounted filesystems and try again."),
	 i18n ("Mounted File Systems"));

      return false;
    }

  /* Call the backend to remove the device.  */
  call_backend (" -n unregister-ide");

  probe ();

  if (!present_flag)
    return true;
  else
    return false;
}

/* Calls the backend to find out whether there are any mounted
   filesystems left on the device.  Returns true if there are mounted
   filesystems, false otherwise.  */

bool
Device::mounted ()
{
  bool success;			/* return value from call_backend */

  /* Return immediately if no device should be present.  */
  if (!present_flag)
    return false;

  /* Run the backend.  */
  success = call_backend (" -n mounted-ide");

  /* Return if the command did not execute successfully.  */
  if (!success)
    return false;

  /* Check the result and return the appropriate value.  */
  if (strncmp ("yes", buffer, strlen ("yes")) == 0)
    return true;
  else
    return false;
}

void
Device::probe ()
{
  bool success;

  /* Call the backend with the appropriate options. */
  success = call_backend (" -n probe-ide");

  if (success)
    {
      /* Only the side effect of the strtok function is used.  */
      strtok (buffer, "\n");

      /* Save the result in the model_string variable.  */
      strncpy (model_string, buffer, 40);
      
      /* Check whether the result indicates that no device is present,
	 and set the flag accordingly.  */
      if (strcoll ("absent", model_string) == 0)
	  present_flag = false;
      else
	  present_flag = true;
    }
  else
    {
      strncpy (model_string, "[error]", 40);
      present_flag = false;
    }
}

/* Call the backend with arbitrary arguments and deal with errors.
   Read both stdout and stderr of the backend into buffer.  Return
   false if an error has occured, true otherwise.  */

bool
Device::call_backend (QString arguments)
{
  /* Construct the command string passed to the shell.  */

  QString command;		/* the command string */

  command += hotswap_path;
  command += " --ide-controller " + QString::number (ide_controller);
  command += " " + arguments;
  command += " 2>&1";


  /* Run the backend.  */

  FILE *backend_out;		/* the output stream from the backend */

  backend_out = popen (command.latin1 (), "r");
  if (backend_out == NULL)
    {      
      QString message;

      message = i18n ("An error occurred while calling the backend:\n");
      message += strerror (errno);

      KMessageBox::error (parent, message, i18n ("Backend Error"));

      return false;
    }

  /* Read the output of the backend.  Surprisingly enough, this cannot
     be done in a standard way using the C++ stream classes.  */

  while (!feof (backend_out))
  {
    fgets (buffer, BUFFER_LENGTH, backend_out);
  }

  /* Close the connection to the backend.  pclose does return the exit
     code of the program if it has succeeded.  The fact that at least
     one version of the Linux man page fails to mention this is a bug
     in the documentation, not in the library.  */
  
  int error;			/* return value from pclose */

  error = pclose (backend_out);
  if (error == -1)
    {
      /* In this case, the pclose call itself failed.  */
      
      QString message;

      message = i18n ("An error occurred while calling the backend:\n");
      message += strerror (errno);

      KMessageBox::error (parent, message, i18n ("Backend Error"));

      return false;
    }
  else if (error)
    {
      /* Any other return value corresponds to the exit status of the
	 backend.  */

      QString message;

      KMessageBox::error
	(parent,
	 i18n ("An error occurred while running the backend."),
	 i18n ("Backend Error"));

      return false;
    }

  return true;
}
