/* 
 * File: trackinfo.c
 *
 * Copyright (c) 2005-2008 Johannes Heimansberg
 *
 * Released under the GNU General Public License v2
 */

#include <string.h>
#include <stdlib.h>
#include <glib.h>
#include <xmmsclient/xmmsclient.h>
#include <xmmsclient/xmmsclient-glib.h>
#include "trackinfo.h"
#include "xmms2ctrl/xmms2ctrl.h"
#include "sdecode/sdecode.h"

gchar *decode_url(const gchar *url)
{
	xmmsv_t *url_v;
	gchar *res = NULL;
	url_v = xmmsv_new_string(url);
	if (url_v) {
		xmmsv_t *decoded_url_v;
		decoded_url_v = xmmsv_decode_url(url_v);
		xmmsv_unref(url_v);
		if (decoded_url_v) {
			const unsigned char *burl;
			unsigned int blen;
			if (xmmsv_get_bin(decoded_url_v, &burl, &blen)) {
				res = g_malloc(blen + 1);
				memcpy(res, burl, blen);
				res[blen] = '\0';
			}
			xmmsv_unref(decoded_url_v);
		}
	}
	return res;
}

void trackinfo_update(xmmsv_t *val,
                      trackinfo      *track)
{
	gint32       bitrate = 0, samplerate = 0, id, playtime = 0, vbr = 0, filesize = 0;
	gchar       *album, *artist, *channel, *comment, *date;
	gchar       *genre, *mime,   *title,   *url,     *url_utf8;

	track->stream        = FALSE;
	track->vbr           = FALSE;
	track->no_artist     = FALSE;
	track->no_title      = FALSE;
	track->no_album      = FALSE;
	track->no_playtime   = FALSE;
	track->no_bitrate    = FALSE;
	track->no_samplerate = FALSE;
	track->no_filesize   = FALSE;

	val = xmmsv_propdict_to_dict(val, NULL);

	if (!xmmsv_dict_entry_get_string(val, "artist", (const gchar **)&artist)) {
		artist = "Unknown";
		track->no_artist = TRUE;
	}
	if (!xmmsv_dict_entry_get_string(val, "title",  (const gchar **)&title)) {
		title = "Unknown";
		track->no_title = TRUE;
	}
	if (!xmmsv_dict_entry_get_string(val, "album", (const gchar **)&album)) {
		album = "Unknown";
		track->no_album = TRUE;
	}
	if (!xmmsv_dict_entry_get_string(val, "comment", (const gchar **)&comment))
		comment = "";
	if (!xmmsv_dict_entry_get_int(val, "duration", &playtime))
		track->no_playtime = TRUE;
	if (!xmmsv_dict_entry_get_int(val, "id", &id))
		id = 0;
	if (!xmmsv_dict_entry_get_string(val, "mime", (const gchar **)&mime))
		mime = "Unknown";
	if (!xmmsv_dict_entry_get_string(val, "genre", (const gchar **)&genre))
		genre = "";
	if (!xmmsv_dict_entry_get_int(val, "bitrate", &bitrate))
		track->no_bitrate = TRUE;
	if (!xmmsv_dict_entry_get_int(val, "samplerate", &samplerate))
		track->no_samplerate = TRUE;
	if (!xmmsv_dict_entry_get_int(val, "size", &filesize))
		track->no_filesize = TRUE;
	if (!xmmsv_dict_entry_get_string(val, "url", (const gchar **)&url))
		url = "";
	if (!xmmsv_dict_entry_get_string(val, "date", (const gchar **)&date))
		date = "";
	if (!xmmsv_dict_entry_get_string(val, "channel", (const gchar **)&channel))
		channel = "";
	else
		track->stream = TRUE;
	xmmsv_dict_entry_get_int(val, "isvbr", &vbr);
	if (vbr == 1) track->vbr = TRUE;

	url_utf8 = decode_url(url);

	g_utf8_strncpy(track->artist,   artist,   sizeof(track->artist));
	g_utf8_strncpy(track->title,    title,    sizeof(track->title));
	g_utf8_strncpy(track->album,    album,    sizeof(track->album));
	g_utf8_strncpy(track->comment,  comment,  sizeof(track->comment));
	g_utf8_strncpy(track->genre,    genre,    sizeof(track->genre));
	g_utf8_strncpy(track->date,     date,     sizeof(track->date));
	g_utf8_strncpy(track->filetype, mime,     sizeof(track->filetype));
	g_utf8_strncpy(track->url,      url_utf8, sizeof(track->url));
	g_utf8_strncpy(track->channel,  channel,  sizeof(track->channel));
	track->id = id;
	track->bitrate = bitrate / 1000;
	track->samplerate = samplerate;
	/* playtime/1000=track length in seconds */
	track->playtime = playtime;
	track->min = playtime / 60000;
	track->sec = playtime % 60000 / 1000;
	track->filesize = filesize / 1024;

	g_free(url_utf8);

	if (strcmp(track->artist, "Unknown Artist") == 0 && 
	    strcmp(track->title, "Unknown Track") == 0) {

		gchar tmp[1024] = "";
		gint  left = 0, right = 0, counter = 0;

		while(counter < strlen(track->url)) {
			if (track->url[counter] == '/' && counter+1 < strlen(track->url))
				left = counter+1;
			if (track->url[counter] == '.' && counter-1 > 0)
				right = counter-1;
			counter++;
		}
		for (counter = left; 
			  counter <= right && counter - left < 1023; 
			  counter++) {
			tmp[counter - left] = track->url[counter];
		}
		tmp[counter] = '\0';
		g_snprintf(track->full_title, sizeof(track->full_title),
		           "%s (%d:%02d)", tmp, track->min, track->sec);
	} else {
		g_snprintf(track->full_title, sizeof(track->full_title),
		           "%s - %s (%d:%02d)", 
		           track->artist, track->title, track->min, track->sec);
	}

	xmmsv_unref(val);
}

gchar *trackinfo_get_album_str(trackinfo *track)
{
	return track->album;
}

gchar *trackinfo_get_artist_str(trackinfo *track)
{
	return track->artist;
}

gchar *trackinfo_get_bitrate_str_alloc(trackinfo *track)
{
	gchar *result = g_malloc(sizeof(gchar) * 14);
	
	if (track->no_bitrate == FALSE)
		g_snprintf(result, 14, "%d kbit/s", track->bitrate);
	else
		g_snprintf(result, 14, "N/A");
	return result;
}

gint32 trackinfo_get_file_size(trackinfo *track)
{
	return track->filesize;
}

void trackinfo_get_full_title(gchar *target, gint target_length,
                              trackinfo *track, gboolean show_time)
{
	gchar *prefix = "", *separator = "";
	gchar *filename = NULL;
	gchar *artist = trackinfo_get_artist_str(track);
	gchar *title  = trackinfo_get_title_str(track);
	gchar  time[16] = "";

	if (show_time) {
		g_snprintf(time, 15, " (%d:%02d)", trackinfo_get_minutes(track),
		                                   trackinfo_get_seconds(track));
	}

	filename = g_path_get_basename(trackinfo_get_url_str(track));
	if (trackinfo_is_stream(track))
		prefix = "[Stream] ";
	if (!trackinfo_has_artist(track))
		artist = "";
	else
		separator = " - ";
	if (!trackinfo_has_artist(track) && !trackinfo_has_title(track))
		title = filename;

	g_snprintf(target, target_length,
	           "%s%s%s%s%s", prefix, artist, separator, title, time);

	g_free(filename);
}

gchar *trackinfo_get_samplerate_str_alloc(trackinfo *track)
{
	gchar *result = g_malloc(sizeof(gchar) * 14);
	
	if (track->no_samplerate == FALSE)
		g_snprintf(result, 14, "%d Hz", track->samplerate);
	else
		g_snprintf(result, 14, "N/A");
	return result;
}

gchar *trackinfo_get_channel_str(trackinfo *track)
{
	return track->channel;
}

gchar *trackinfo_get_comment_str(trackinfo *track)
{
	return track->comment;
}

gchar *trackinfo_get_date_str(trackinfo *track)
{
	return track->date;
}

gchar *trackinfo_get_genre_str(trackinfo *track)
{
	return track->genre;
}

guint trackinfo_get_id(trackinfo *track)
{
	return track->id;
}

gchar *trackinfo_get_mimetype_str(trackinfo *track)
{
	return track->filetype;
}

guint trackinfo_get_minutes(trackinfo *track)
{
	return track->min;
}

gint32 trackinfo_get_playtime(trackinfo *track)
{
	return track->playtime;
}

gchar *trackinfo_get_playtime_str_alloc(trackinfo *track)
{
	gchar *result = g_malloc(sizeof(gchar) * 10);

	if (track->no_playtime == FALSE)
		g_snprintf(result, 10, "%d:%02d", track->min, track->sec);
	else
		g_snprintf(result, 10, "N/A");
	return result;
}

guint trackinfo_get_seconds(trackinfo *track)
{
	return track->sec;
}

gchar *trackinfo_get_title_str(trackinfo *track)
{
	return track->title;
}

gchar *trackinfo_get_url_str(trackinfo *track)
{
	return track->url;
}

gboolean trackinfo_has_album(trackinfo *track)
{
	return !track->no_album;
}

gboolean trackinfo_has_artist(trackinfo *track)
{
	return !track->no_artist;
}

gboolean trackinfo_has_bitrate(trackinfo *track)
{
	return !track->no_bitrate;
}

gboolean trackinfo_has_file_size(trackinfo *track)
{
	return !track->no_filesize;
}

gboolean trackinfo_has_playtime(trackinfo *track)
{
	return !track->no_playtime;
}

gboolean trackinfo_has_samplerate(trackinfo *track)
{
	return !track->no_samplerate;
}

gboolean trackinfo_has_title(trackinfo *track)
{
	return !track->no_title;
}

gboolean trackinfo_is_stream(trackinfo *track)
{
	return track->stream;
}

gboolean trackinfo_is_vbr(trackinfo *track)
{
	return track->vbr;
}
