#include "kifuAnalyzer.h"
#include "searchMonitor.h"
#include "osl/sennichite.h"
#include "osl/search/simpleHashRecord.h"
#include "osl/search/fixedEval.h"
#include "osl/search/simpleHashTable.h"
#include "osl/state/numEffectState.h"
#include "osl/eval/ml/openMidEndingEval.h"
#include "gpsshogi/gui/util.h"

#include <QThread>
#include <QLabel>

#include <qlayout.h>
#include <qpushbutton.h>
#include <qtabwidget.h>
#include <qpainter.h>

class EvaluationGraph : public QWidget
{
public:
  EvaluationGraph(QWidget *parent = 0, const char *name = 0);
  void setResult(const osl::stl::vector<Result>& r) {
    result = r;
    update();
  }
  virtual QSize sizeHint() const {
    return QSize(700, 400);
  }
protected:
  void paintEvent(QPaintEvent *);
private:
  osl::stl::vector<Result> result;
};

EvaluationGraph::EvaluationGraph(QWidget *parent, const char *name)
  : QWidget(parent, name)
{
}

void EvaluationGraph::paintEvent(QPaintEvent *)
{
  QPainter painter(this);
  QBrush brush(QColor("white"));
  painter.fillRect(0, 0, width(), height(), brush);
  painter.setPen(QColor("grey"));
  painter.drawLine(60, 100, 640, 100);
  painter.drawLine(60, 200, 640, 200);
  painter.drawLine(60, 300, 640, 300);
  painter.setPen(QColor("blue"));
  for (int i = 0; i < (int)result.size() - 1; i++)
  {
    int left = std::max(std::min(200 - result[i].value / 10, 400), 0);
    int right = std::max(std::min(200 - result[i + 1].value / 10, 400), 0);
    painter.drawLine(60 + 4 * i,  left,
		     60 + 4 * (i + 1), right);
  }
  painter.setPen(QColor("black"));
  painter.drawText(0, 200 - 30, 50, 60, Qt::AlignRight | Qt::AlignVCenter, QString("%1").arg(0));
  painter.drawText(0, 100 - 30, 50, 60, Qt::AlignRight | Qt::AlignVCenter, QString("%1").arg(1000));
  painter.drawText(0, 300 - 30, 50, 60, Qt::AlignRight | Qt::AlignVCenter, QString("%1").arg(-1000));

  painter.drawText(60 + 4 * (50 - 1), 200 - 30, 60, 60,
                   Qt::AlignLeft | Qt::AlignVCenter, QString("%1").arg(50));
  painter.drawText(60 + 4 * (100 - 1), 200 - 30, 60, 60,
                   Qt::AlignLeft | Qt::AlignVCenter, QString("%1").arg(100));
}

class AnalyzeItem : public Q3ListViewItem
{
public:
  AnalyzeItem(Q3ListView *parent, int i, const Result& result);
  int compare(Q3ListViewItem *i, int col, bool ascending) const;
  int getNumber() const { return number; }
private:
  int number;
};

AnalyzeItem::AnalyzeItem(Q3ListView *parent, int n, const Result& result)
  : Q3ListViewItem(parent), number(n)
{
  int i = 0;
  setText(i++, QString("%1").arg(number));
  setText(i++, gpsshogi::gui::Util::moveToString(result.move));
  setText(i++, gpsshogi::gui::Util::moveToString(result.computed_move));
  setText(i++, QString("%1").arg(result.value));
  setText(i++, QString("%1").arg(result.depth));
  QString pv;
  for (size_t i = 0; i < result.pvs.size(); ++i)
  {
    if (!pv.isEmpty())
    {
      pv.append(", ");
    }
    pv.append(gpsshogi::gui::Util::moveToString(result.pvs[i]));
  }
  setText(i++, pv);
}

int AnalyzeItem::compare(Q3ListViewItem *i, int, bool) const
{
  AnalyzeItem *item = (AnalyzeItem *)i;
  if (number == item->getNumber())
    return 0;
  else if (number < item->getNumber())
    return -1;
  else
    return 1;
}

class AnalyzeThread : public QThread
{
public:
  AnalyzeThread(KifuAnalyzer *w, const osl::state::SimpleState &s,
                const osl::stl::vector<osl::Move> &ms) : widget(w), moves(ms)
  {
    player.reset(new osl::game_playing::AlphaBeta2OpenMidEndingEvalPlayer());
    player->setDepthLimit(1200, 400, 200);
    if (osl::OslConfig::isMemoryLimitEffective()) 
    {
      player->setTableLimit(std::numeric_limits<size_t>::max(), 0);
      player->setNodeLimit(std::numeric_limits<size_t>::max());
    }
    game_state.reset(new osl::game_playing::GameState(s));
    monitor.reset(new ViewerSearchMonitor(widget->currentStatus,
                                          game_state->state()));
    player->addMonitor(monitor);
  }
  void run()
  {
    for (size_t i = 0; i < moves.size() && widget->shouldSearch(); i++)
    {
      monitor->setState(game_state->state());
      player->selectBestMove(*game_state, 0, 0, 5);
      widget->addResult(moves[i], *player, *game_state);
      game_state->pushMove(moves[i]);
    }
    widget->currentStatus->setText("");
  }
private:
  KifuAnalyzer *widget;
  const osl::stl::vector<osl::Move> moves;
  boost::scoped_ptr<osl::game_playing::AlphaBeta2OpenMidEndingEvalPlayer> player;
  boost::scoped_ptr<osl::game_playing::GameState> game_state;
  boost::shared_ptr<ViewerSearchMonitor> monitor;
};

KifuAnalyzer::KifuAnalyzer(const osl::state::SimpleState &state,
			   const osl::stl::vector<osl::Move> &moves,
			   QWidget *parent)
  : QDialog(parent), search(true)
{
  setAttribute(Qt::WA_DeleteOnClose);
  QTabWidget *tab = new QTabWidget(this);

  list = new Q3ListView(tab);
  list->addColumn("");
  list->addColumn("Move");
  list->addColumn("Computed Move");
  list->addColumn("Value");
  list->addColumn("Depth");
  list->addColumn("PV");
  list->setColumnAlignment(0, Qt::AlignRight);
  list->setColumnAlignment(3, Qt::AlignRight);
  list->setColumnAlignment(4, Qt::AlignRight);
  osl::state::NumEffectState estate(state);
  osl::eval::ml::OpenMidEndingEval eval(estate);
  graph = new EvaluationGraph(tab);
  pawnValue = eval.captureValue(osl::newPtypeO(osl::WHITE, osl::PAWN)) / 2;

  QVBoxLayout *layout = new QVBoxLayout(this);
  tab->addTab(graph, "Graph");
  tab->addTab(list, "List");
  layout->addWidget(tab);

  currentStatus = new QLabel(this);
  layout->addWidget(currentStatus);

  QPushButton *button = new QPushButton(this);
  button->setText("&OK");
  layout->addWidget(button);
  connect(button, SIGNAL(clicked()), this, SLOT(accept()));

  thread.reset(new AnalyzeThread(this, state, moves));;
  thread->start();
}

KifuAnalyzer::~KifuAnalyzer()
{
  search = false;
  thread->wait();
}

void KifuAnalyzer::addResult(const osl::Move move,
			     const osl::game_playing::SearchPlayer& player,
			     const osl::game_playing::GameState& state)
{
  const osl::search::SimpleHashTable& table = *player.table();
  const osl::HashKey key(state.state());
  const osl::search::SimpleHashRecord *record = table.find(key);

  osl::container::MoveVector moves;
  if (record)
  {
    int value = record->hasLowerBound(0) ? record->lowerBound()  * 100L / pawnValue :
      (result.empty() ? 0 : result.back().value);
    int limit = (record->lowerLimit() > record->upperLimit()) ?
      record->lowerLimit() : record->upperLimit();
    const osl::Move best_move = record->bestMove().move();
    if (best_move.isValid())
    {
      const osl::HashKey new_key = key.newHashWithMove(best_move);
      if (table.find(new_key))
      {
        moves.push_back(best_move);
        table.getPV(new_key, moves);
      }
    }
    result.push_back(Result(move, record->bestMove().move(),
                            value, limit, moves));
  }
  else
  {
    int value = result.empty() ? 0 : result.back().value;
    result.push_back(Result(move, osl::Move::INVALID(), value, -1, moves));
  }
  graph->setResult(result);
  new AnalyzeItem(list, result.size(), result.back());
}
