/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef _globus_rls_client_h_
#define _globus_rls_client_h_

#include <globus_io.h>
#include <globus_error_string.h>

/**
 * @mainpage
 * The Globus Replica Location Service (RLS) C API provides functions to
 * view and update data in a RLS catalog.  There are 2 types of RLS servers,
 * Local Replica Catalog (LRC) servers, which maintain Logical to Physical
 * File Name mappings (LFN to PFN), and Replica Location Index (RLI) servers,
 * which  maintain LFN to LRC mappings.  Note an RLS server can act as both
 * an LRC and RLI server.
 *
 * Functions are divided into the following groups:
 * - @link globus_rls_client_activation Activation @endlink
 * - @link globus_rls_client_connection Connection Management @endlink
 * - @link globus_rls_client_lrc_operation Operations on an LRC server @endlink
 * - @link globus_rls_client_rli_operation Operations on an RLI server @endlink
 * - @link globus_rls_client_miscellaneous Miscellaneous Types and
 *     Functions @endlink
 * - @link globus_rls_client_queryresult Query Results @endlink
 * - @link globus_rls_client_status Status Codes @endlink
 *
 * Applications using this API should include <B>globus_rls_client.h</B>, and
 * be linked with the library <B>globus_rls_client_FLAVOR</B>.
 */



/**
 * @defgroup globus_rls_client_status Status Codes
 * All of the functions in the API that return status return it
 * in a globus_result_t structure.  Prior to version 2.0.0 an integer
 * status was returned.  The globus_result_t structure includes an
 * integer "type" which is set to one of the status codes defined below (the
 * same values that were returned by earlier versions of the API).
 * The function globus_rls_client_error_info() may be used to extract
 * the status code and/or error message from a globus_result_t.  GLOBUS_SUCCESS
 * is returned when the operation was successful.
 */

/**
 * Operation succeeded.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_SUCCESS		0
/**
 * An error was returned by the Globus I/O module.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_GLOBUSERR		1
/**
 * The ::globus_rls_handle_t handle is invalid.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_INVHANDLE		2
/**
 * The URL could not be parsed.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_BADURL		3
/**
 * Out of memory.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_NOMEMORY		4
/**
 * A result was too large to fit in buffer.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_OVERFLOW		5
/**
 * Bad argument (eg NULL where string pointer expected).
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_BADARG		6
/**
 * Client does not have permission for requested action.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_PERM			7
/**
 * RPC error, invalid method name sent to server.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_BADMETHOD		8
/**
 * LRC request made to RLI server or vice versa.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_INVSERVER		9
/**
 * LFN,PFN (LRC) or LFN,LRC (RLI) mapping doesn't exist.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_MAPPING_NEXIST	10
/**
 * LFN already exists in LRC or RLI database.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_LFN_EXIST		11
/**
 * LFN doesn't exist in LRC or RLI database.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_LFN_NEXIST		12
/**
 * PFN already exists in LRC database.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_PFN_EXIST		13
/**
 * PFN doesn't exist in LRC database.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_PFN_NEXIST		14
/**
 * LRC already exists in LRC or RLI database.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_LRC_EXIST		15
/**
 * LRC doesn't exist in RLI database.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_LRC_NEXIST		16
/**
 * Database error.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_DBERROR		17
/**
 * RLI already exists in LRC database.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_RLI_EXIST		18
/**
 * RLI doesn't exist in LRC.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_RLI_NEXIST		19
/**
 * LFN,PFN (LRC) or LFN,LRC (RLI) mapping already exists.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_MAPPING_EXIST	20
/**
 * Invalid attribute type, see globus_rls_attr_type_t.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_INV_ATTR_TYPE	21
/**
 * Attribute already exists.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_ATTR_EXIST		22
/**
 * Attribute doesn't exist.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_ATTR_NEXIST		23
/**
 * Invalid object type, see globus_rls_obj_type_t.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_INV_OBJ_TYPE		24
/**
 * Invalid attribute search operator, see globus_rls_attr_op_t.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_INV_ATTR_OP		25
/**
 * Operation is unsupported.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_UNSUPPORTED		26
/**
 * IO timeout.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_TIMEOUT		27
/**
 * Too many connections.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_TOO_MANY_CONNECTIONS	28

/**
 * Attribute with specified value not found.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_ATTR_VALUE_NEXIST	29

/**
 * Attribute in use by some object, can't be deleted.
 * @ingroup globus_rls_client_status
 */
#define GLOBUS_RLS_ATTR_INUSE		30

/* Globus LRC server is in the process of starting up. You can issue queries
but cannot update the database. */

#define GLOBUS_RLS_INITIALIZING          31

/**
 * @defgroup globus_rls_client_miscellaneous Miscellaneous
 * Miscellaneous functions and types.
 */

/**
 * Wildcard character style.
 * @ingroup globus_rls_client_miscellaneous
 */
typedef enum {
  rls_pattern_unix,	/**< Unix file globbing chars (*, ?)	*/
  rls_pattern_sql	/**< SQL "like" wildcards (%, _)	*/
} globus_rls_pattern_t;

/* Obsolete, use globus_rls_pattern_t */
#define globus_rls_pattern_type	globus_rls_pattern_t;

#define BUFLEN	1024

/**
 * Attribute Value Types
 * @ingroup globus_rls_client_miscellaneous
 */
typedef enum {
  globus_rls_attr_type_date,	/**< Date (time_t). */
  globus_rls_attr_type_flt,	/**< Floating point (double). */
  globus_rls_attr_type_int,	/**< Integer (int). */
  globus_rls_attr_type_str	/**< String (char *). */
} globus_rls_attr_type_t;

/*
 * Obsolete names for attribute value types, please use enum above.
 */
#define GLOBUS_RLS_ATTR_TYPE_DATE	globus_rls_attr_type_date
#define GLOBUS_RLS_ATTR_TYPE_FLT	globus_rls_attr_type_flt
#define GLOBUS_RLS_ATTR_TYPE_INT	globus_rls_attr_type_int
#define GLOBUS_RLS_ATTR_TYPE_STR	globus_rls_attr_type_str

/**
 * Object types in LRC and RLI databases.
 * @ingroup globus_rls_client_miscellaneous
 */
typedef enum {
  globus_rls_obj_lrc_lfn,	/**< LRC Logical File Name. */
  globus_rls_obj_lrc_pfn,	/**< LRC Physical File Name. */
  globus_rls_obj_rli_lfn,	/**< RLI Logical File Name. */
  globus_rls_obj_rli_lrc	/**< RLI LRC URL. */
} globus_rls_obj_type_t;

/*
 * The following are obsolete, please use the enum above.
 */
#define GLOBUS_RLS_CLIENT_LRC_OBJ_LFN	globus_rls_obj_lrc_lfn
#define GLOBUS_RLS_CLIENT_LRC_OBJ_PFN	globus_rls_obj_lrc_pfn
#define GLOBUS_RLS_CLIENT_RLI_OBJ_LFN	globus_rls_obj_rli_lfn
#define GLOBUS_RLS_CLIENT_RLI_OBJ_LRC	globus_rls_obj_rli_lrc

/**
 * Object (LFN or PFN) attribute type.
 * @ingroup globus_rls_client_miscellaneous
 */
typedef struct {
  char			*name;		/**< Attribute name. */
  globus_rls_obj_type_t	objtype;	/**< Object type. */
  globus_rls_attr_type_t type;		/**< Attribute value type. */ 
  /** Value of attribute (depends on type). */
  union {
    time_t	t;	/**< Date value (unix time). */
    double	d;	/**< Floating point value. */
    int		i;	/**< Integer value. */
    char	*s;	/**< String value. */
  }			val;
} globus_rls_attribute_t;

/**
 * globus_rls_client_lrc_attr_search() returns a list of these structures
 * which include the object name (LFN or PFN) and attribute value found
 * by the query.
 * @ingroup globus_rls_client_queryresult
 */
typedef struct {
  /** Attribute value. */
  globus_rls_attribute_t	attr;
  /** LFN or PFN. */
  char				*key;
  /** Result code, only used in bulk query. */
  int				rc;
} globus_rls_attribute_object_t;

/**
 * Attribute Value Query Operators
 * @ingroup globus_rls_client_miscellaneous
 */
typedef enum {
  globus_rls_attr_op_all,	/**< All values returned.		*/
  globus_rls_attr_op_eq,	/**< Values matching operand 1 returned.*/
  globus_rls_attr_op_ne,	/**< Values not matching operand 1.	*/
  globus_rls_attr_op_gt,	/**< Values greater than operand 1.	*/
  globus_rls_attr_op_ge,	/**< Values greater than or equal to op1*/
  globus_rls_attr_op_lt,	/**< Values less than operand 1.	*/
  globus_rls_attr_op_le,	/**< Values less than or equal to op1.	*/
  globus_rls_attr_op_btw,	/**< Values between operand1 and 2.	*/
  globus_rls_attr_op_like	/**< Strings "like" operand1 (SQL like)	*/
} globus_rls_attr_op_t;

/*
 * Following are obsolete, please use enum above.
 */
#define GLOBUS_RLS_CLIENT_ATTR_OP_ALL globus_rls_attr_op_all
#define GLOBUS_RLS_CLIENT_ATTR_OP_EQ globus_rls_attr_op_eq
#define GLOBUS_RLS_CLIENT_ATTR_OP_NE globus_rls_attr_op_ne
#define GLOBUS_RLS_CLIENT_ATTR_OP_GT globus_rls_attr_op_gt
#define GLOBUS_RLS_CLIENT_ATTR_OP_GE globus_rls_attr_op_ge
#define GLOBUS_RLS_CLIENT_ATTR_OP_LT globus_rls_attr_op_lt
#define GLOBUS_RLS_CLIENT_ATTR_OP_LE globus_rls_attr_op_le
#define GLOBUS_RLS_CLIENT_ATTR_OP_BTW globus_rls_attr_op_btw
#define GLOBUS_RLS_CLIENT_ATTR_OP_LIKE globus_rls_attr_op_like




/**
 * @defgroup globus_rls_client_queryresult Query Results
 * List results are returned as globus_list_t's, list datums depend
 * on the type of query (eg globus_rls_string2_t, globus_rls_attribute_t,
 * etc).  A list result should be freed with globus_rls_client_free_list() when
 * it's no longer needed.  RLS supports limiting the number of results
 * returned by a single query using an offset and reslimit.  The offset
 * specifies which result to begin with, reslimit specifies how many results
 * to return.  Offset should begin at 0 to retrieve all records.  If reslimit
 * is 0 then all results are returned at once, unless the server has
 * a limit on results configured.  If NULL is passed as the offset argument
 * then the API will repeatedly call the query function until are results
 * are retrieved.  The following are equivalent examples of how to
 * print the lfn,pfn pairs returned by globus_rls_client_lrc_get_lfn():
 *
 * @verbatim
   globus_list_t *str2_list;
   globus_list_t *p;
   globus_rls_string2_t *str2;

   // Retrieve all results, API will handle looping through partial results
   // if the server has a limit configured.  Error handling has been omitted.
   globus_rls_client_lrc_get_lfn(h, "somepfn", NULL, 0, &str2_list);
   for (p = str2_list; p; p = globus_list_rest(p)) {
     str2 = (globus_rls_string2_t *) globus_list_first(p);
     printf("lfn: %s pfn:%s\n", str2->s1, str2->s2);
   }
   globus_rls_client_free_list(str2_list);

   // This code fragment retrieves results 5 at a time.  Note offset is set
   // to -1 when the server has no more results to return.
   int offset = 0;

   while (globus_rls_client_lrc_get_lfn(h, "somepfn", &offset, 5, &str2_list) == GLOBUS_SUCCESS) {
     for (p = str2_list; p; p = globus_list_rest(p)) {
       str2 = (globus_rls_string2_t *) globus_list_first(p);
       printf("lfn: %s pfn:%s\n", str2->s1, str2->s2);
     }
     globus_rls_client_free_list(str2_list);
     if (offset == -1)
       break;
   }
   @endverbatim
 */

/**
 * @ingroup globus_rls_client_queryresult
 * String pair result.  Many of the query functions use this to return
 * pairs of strings (eg LFN,PFN or LFN,LRC).
 */
typedef struct {
  char	*s1;	/**< First string in pair (eg LFN). */
  char	*s2;	/**< Second string in pair (eg PFN or LRC). */
} globus_rls_string2_t;

/**
 * @ingroup globus_rls_client_queryresult
 * String pair result with return code, returned by bulk query operations.
 */
typedef struct {
  int			rc;
  globus_rls_string2_t	str2;
} globus_rls_string2_bulk_t;

/**
 * Free result list returned by one of the query functions.
 * @ingroup globus_rls_client_queryresult
 *
 * @param list
 *	List returned by one of the query functions.
 *
 * @retval GLOBUS_SUCCESS
 *	List and contents successfully freed.
 */
extern globus_result_t globus_rls_client_free_list(
  globus_list_t                         *list);



/**
 * @defgroup globus_rls_client_activation Activation
 * This module must be activated before any functions in this API may
 * be used.  This module depends on other Globus modules GLOBUS_COMMON_MODULE
 * and GLOBUS_IO_MODULE, which should be activated first:
 * @verbatim
   globus_module_activate(GLOBUS_COMMON_MODULE);
   globus_module_activate(GLOBUS_IO_MODULE);
   globus_module_activate(GLOBUS_RLS_CLIENT_MODULE);
   @endverbatim
 * When finished modules should be deactivated in reverse order.
 */

/*@{*/
/**
 * RLS module.
 * @ingroup globus_rls_client_activation
 */
extern globus_module_descriptor_t	globus_rls_client_module;
/**
 * RLS Module Name.
 */
#define GLOBUS_RLS_CLIENT_MODULE	(&globus_rls_client_module)
/*@}*/




/**
 * @defgroup globus_rls_client_connection Connection Management
 * Functions to open and close connections to an RLS server.
 */

/**
 * URL scheme to use when connecting to RLS server.
 * @ingroup globus_rls_client_connection
 */
#define GLOBUS_RLS_URL_SCHEME		"rls"
/**
 * URL scheme when connecting to RLS server without authentication.
 * @ingroup globus_rls_client_connection
 */
#define GLOBUS_RLS_URL_SCHEME_NOAUTH	"rlsn"
/**
 * Default port number that RLS server listens on.
 * @ingroup globus_rls_client_connection
 */
#define GLOBUS_RLS_SERVER_DEFPORT	39281
/**
 * Maximum length of error messages returned by server.
 * @ingroup globus_rls_client_connection
 */
#define MAXERRMSG	1024

/**
 * RLS Client Handle.
 */
typedef struct {
  globus_url_t		url; /**< URL of RLS server (RLS://host[:port]). */
  globus_io_handle_t	handle; /**< Globus IO handle. */
  int			flags; /**< See FH_xxx flags below. */
} globus_rls_handle_t;
#define FH_IOERROR	0x1	/* If set don't send "close" to server */
#define FH_CLOSED	0x2	/* Globus handle closed (set when	*/
				/* retrying failed IO operation)	*/

/**
 * Set certificate used in authentication.  Sets environment variables
 * X509_USER_CERT, X509_USER_KEY, and clears X509_USER_PROXY.
 * @ingroup globus_rls_client_connection
 *
 * @param certfile
 *	Name of X509 certificate file.
 *
 * @param keyfile
 *	Name of X509 key file.
 */
void
globus_rls_client_certificate(
  char					*certfile,
  char					*keyfile);

/**
 * Set X509_USER_PROXY environment variable to specified file.
 * @ingroup globus_rls_client_connection
 *
 * @param proxy
 *	Name of X509 proxy certificate file.  If NULL clears X509_USER_PROXY.
 */
void
globus_rls_client_proxy_certificate(
  char					*proxy);

/**
 * Open connection to RLS server.
 * @ingroup globus_rls_client_connection
 *
 * @param url
 *	URL of server to connect to.  URL scheme should be <B>RLS</B> or
 *	<B>RLSN</B>, eg <B>RLS://my.host</B>.
 *	If the URL scheme is <B>RLSN</B> then no authentication is performed
 *	(the RLS server must be started with authentication disabled as well,
 *	this option is primarily intended for testing).
 *
 * @param h
 *	If the connection is successful @a *h will be set to the connection
 *	handle.  This handle is required by all other functions in the API.
 *
 * @retval GLOBUS_SUCCESS
 *	Handle @a h now connected to RLS server identified by @a url.
 */
extern globus_result_t globus_rls_client_connect(
  char					*url,
  globus_rls_handle_t			**h);

/**
 * Close connection to RLS server.
 * @ingroup globus_rls_client_connection
 *
 * @param h
 *	Connection handle to be closed, previously allocated by
 *	globus_rls_client_connect().
 *
 * @retval GLOBUS_SUCCESS
 *	Connection closed, @a h is no longer valid.
 */
extern globus_result_t globus_rls_client_close(
  globus_rls_handle_t		*h);

/**
 * Get timeout for IO calls to RLS server.  If 0 IO calls do not timeout.
 * The default is 30 seconds.
 * @ingroup globus_rls_client_connection
 *
 * @retval timeout
 *	Seconds to wait before timing out an IO operation.
 */
extern int globus_rls_client_get_timeout();

/**
 * Set timeout for IO calls to RLS server.
 * @ingroup globus_rls_client_connection
 *
 * @param seconds
 *	Seconds to wait before timing out an IO operation.  If 0 IO calls do
 *	not timeout.  The default is 30 seconds.
 */
extern void globus_rls_client_set_timeout(
  int				seconds);



/**
 * @defgroup globus_rls_client_lrc_operation LRC Operations
 * Functions to view and update data managed by a LRC server.
 */

/**
 * Add an attribute to an object in the LRC database.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param key
 *	Logical or Physical File Name (LFN or PFN) that identifies object
 *	attribute should be added to.
 *
 * @param attr
 *	Attribute to be added to object.  @link globus_rls_attribute_t name,
 *	objtype, type and val @endlink should be set in @a attr.
 *
 * @retval GLOBUS_SUCCESS
 *	Attribute successfully associated with object.
 */
extern globus_result_t globus_rls_client_lrc_attr_add(
  globus_rls_handle_t		*h,
  char				*key,
  globus_rls_attribute_t	*attr);

/**
 * Bulk add attributes to objects in the LRC database.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param attr_obj_list
 *	List of object names (LFN or PFN) and attributes to be added.  Each
 *	list datum should be of type globus_rls_attribute_object_t.
 *
 * @param str2bulk_list
 *	List of failed updates.  Each list datum is a
 *	globus_rls_string2_bulk_t structure.  <B>str2.s1</B> will
 *	be the oject name, <B>str2.s2</B> the attribute name, and
 *	<B>rc</B> will be the result code from the failed update.  Only
 *	failed updates will be returned.
 */
extern globus_result_t globus_rls_client_lrc_attr_add_bulk(
  globus_rls_handle_t		*h,
  globus_list_t			*attr_obj_list,
  globus_list_t			**str2bulk_list);

/**
 * Define new attribute in LRC database.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an LRC server.
 *
 * @param name
 *	Name of attribute.
 *
 * @param objtype
 *	Object (LFN or PFN) type that attribute applies to.
 *
 * @param type
 *	Type of attribute value.
 *
 * @retval GLOBUS_SUCCESS
 *	Attribute successfully created.
 */
extern globus_result_t globus_rls_client_lrc_attr_create(
  globus_rls_handle_t		*h,
  char				*name,
  globus_rls_obj_type_t		objtype,
  globus_rls_attr_type_t	type);

/**
 * Undefine attribute in LRC database, previously created with
 * globus_rls_client_lrc_attr_create().
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an LRC server.
 *
 * @param name
 *	Name of attribute.
 *
 * @param objtype
 *	Object (LFN or PFN) type that attribute applies to.
 *
 * @param clearvalues
 *	If GLOBUS_TRUE then any any values for this attribute are first
 *	removed from the objects they're associated with.  If GLOBUS_FALSE
 *	and any values exist then ::GLOBUS_RLS_ATTR_EXIST is returned.
 *
 * @retval GLOBUS_SUCCESS
 *	Attribute successfully removed.
 */
extern globus_result_t globus_rls_client_lrc_attr_delete(
  globus_rls_handle_t		*h,
  char				*name,
  globus_rls_obj_type_t		objtype,
  globus_bool_t			clearvalues);

/**
 * Return definitions of attributes in LRC database.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param name
 *	Name of attribute.  If name is NULL all attributes of the specified
 *	@a objtype are returned.
 *
 * @param objtype
 *	Object (LFN or PFN) type that attribute applies to.
 *
 * @param attr_list
 *	Any attribute definitions found will be returned as a list of
 *	globus_rls_attribute_t structures.
 *
 * @retval GLOBUS_SUCCESS
 *	Attribute definitions successfully retrieved.  @a attr_list
 *	should be freed with globus_rls_client_free_list() when it is
 *	no longer needed.
 */
extern globus_result_t globus_rls_client_lrc_attr_get(
  globus_rls_handle_t		*h,
  char				*name,
  globus_rls_obj_type_t		objtype,
  globus_list_t			**attr_list);

/**
 * Modify an attribute value.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param key
 *	Name of object (LFN or PFN).
 *
 * @param attr
 *	Attribute to be modified.  The @link globus_rls_attribute_t
 *	objtype @endlink, @link globus_rls_attribute_t name @endlink
 *	and @link globus_rls_attribute_t type @endlink
 *	fields should be set in @a attr to identify the attribute, the
 *	@link globus_rls_attribute_t val @endlink field should be the
 *	new value.
 *
 * @retval GLOBUS_SUCCESS
 *	Attribute successfully modified.
 */
extern globus_result_t globus_rls_client_lrc_attr_modify(
  globus_rls_handle_t		*h,
  char				*key,
  globus_rls_attribute_t	*attr);

/**
 * Remove an attribute from an object (LFN or PFN) in the LRC database.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param key
 *	Name of object (LFN or PFN).
 *
 * @param attr
 *	Attribute to be removed.  The @link globus_rls_attribute_t
 *	objtype @endlink and @link globus_rls_attribute_t name @endlink
 *	fields should be set in @a attr to identify the attribute.
 *
 * @retval GLOBUS_SUCCESS
 *	Attribute successfully removed.
 */
extern globus_result_t globus_rls_client_lrc_attr_remove(
  globus_rls_handle_t		*h,
  char				*key,
  globus_rls_attribute_t	*attr);
  
/**
 * Bulk remove attributes from objects in the LRC database.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param attr_obj_list
 *	List of object names (LFN or PFN) and attributes to be removed.
 *	It is not necessary to set the attribute type or value.  Each list
 *	datum should be of type globus_rls_attribute_object_t.
 *
 * @param str2bulk_list
 *	List of failed updates.  Each list datum is a
 *	globus_rls_string2_bulk_t structure.  <B>str2.s1</B> will
 *	be the oject name, <B>str2.s2</B> the attribute name, and
 *	<B>rc</B> will be the result code from the failed update.  Only
 *	failed updates will be returned.
 */
extern globus_result_t globus_rls_client_lrc_attr_remove_bulk(
  globus_rls_handle_t		*h,
  globus_list_t			*attr_obj_list,
  globus_list_t			**str2bulk_list);

/**
 * Search for objects (LFNs or PFNs) in a LRC database that have
 * the specified attribute whose value matches a boolean expression.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param name
 *	Name of attribute.
 *
 * @param objtype
 *	Object (LFN or PFN) type that attribute applies to.
 *
 * @param op
 *	Operator to be used in searching for values.
 *
 * @param operand1
 *	First operand in boolean expression.  @link globus_rls_attribute_t type
 *	@endlink and @link globus_rls_attribute_t val @endlink should be
 *	set in globus_rls_attribute_t.
 *
 * @param operand2
 *	Second operand in boolean expression, only used when @a op is
 *	globus_rls_client_attr_op_btw.  @link globus_rls_attribute_t type
 *	@endlink and @link globus_rls_attribute_t val @endlink should be
 *	set in globus_rls_attribute_t.
 *
 * @param offset
 *	Offset into result list.  Used in conjunction with @a reslimit
 *	to retrieve results a few at a time.  Use 0 to begin with first
 *	result.  If NULL then API will handle accumulating partial results
 *	transparently.
 *
 * @param reslimit
 *	Maximum number of results to return.  Used in conjunction with
 *	@a offset to retrieve results a few at a time.  Use 0 to retrieve
 *	all results.
 *
 * @param attr_obj_list
 *	Any objects with the specified attribute will be returned, with
 *	the attribute value, in a list of globus_rls_attribute_object_t
 *	structures.
 *
 * @retval GLOBUS_SUCCESS
 *	Objects with specified attribute returned in @a attr_obj_list.
 *	@a attr_obj_list should be freed with globus_rls_client_free_list()
 *	when it is no longer needed.  See @ref globus_rls_client_queryresult.
 */
extern globus_result_t globus_rls_client_lrc_attr_search(
  globus_rls_handle_t		*h,
  char				*name,
  globus_rls_obj_type_t		objtype,
  globus_rls_attr_op_t		op,
  globus_rls_attribute_t	*operand1,
  globus_rls_attribute_t	*operand2,
  int				*offset,
  int				reslimit,
  globus_list_t			**attr_obj_list);

/**
 * Return attributes in LRC database for specified object (LFN or PFN).
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param key
 *	Logical or Physical File Name (LFN or PFN) that identifies object
 *	attributes should be retrieved for.
 *
 * @param name
 *	Name of attribute to retrieve.  If NULL all attributes for
 *	@a key, @a objtype are returned.
 *
 * @param objtype
 *	Object (LFN or PFN) type that attribute applies to.
 *
 * @param attr_list
 *	Any attributes found will be returned in this list
 *	of globus_rls_attribute_t structures.
 *
 * @retval GLOBUS_SUCCESS
 *	Attributes successfully retrieved.  @a attr_list
 *	should be freed with globus_rls_client_free_list() when it is
 *	no longer needed.
 */
extern globus_result_t globus_rls_client_lrc_attr_value_get(
  globus_rls_handle_t		*h,
  char				*key,
  char				*name,
  globus_rls_obj_type_t		objtype,
  globus_list_t			**attr_list);

/**
 * Return attributes in LRC database for specified objects (LFN or PFN).
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param keylist
 *	Logical or Physical File Names (LFNs or PFNs) that identify object
 *	attributes should be retrieved for.  Each list datum should be a string
 *	containing the LFN or PFN.
 *
 * @param name
 *	Name of attribute to retrieve.  If NULL all attributes for
 *	@a key, @a objtype are returned.
 *
 * @param objtype
 *	Object (LFN or PFN) type that attribute applies to.
 *
 * @param attr_obj_list
 *	Any attributes found will be returned in this list
 *	of globus_rls_attribute_object_t structures.
 *
 * @retval GLOBUS_SUCCESS
 *	Attributes successfully retrieved.  @a attr_obj_list
 *	should be freed with globus_rls_client_free_list() when it is
 *	no longer needed.
 */
extern globus_result_t globus_rls_client_lrc_attr_value_get_bulk(
  globus_rls_handle_t		*h,
  globus_list_t			*keylist,
  char				*name,
  globus_rls_obj_type_t		objtype,
  globus_list_t			**attr_obj_list);

/**
 * Add mapping to PFN to an existing LFN.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param lfn
 *	LFN to add @a pfn mapping to, should already exist.
 *
 * @param pfn
 *	PFN that @a lfn should map to.
 *
 * @retval GLOBUS_SUCCESS
 *	New mapping created.
 */
extern globus_result_t globus_rls_client_lrc_add(
  globus_rls_handle_t		*h,
  char				*lfn,
  char				*pfn);

/**
 * Bulk add LFN,PFN mappings in LRC database.  LFNs must already exist.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param str2_list
 *	LFN,PFN pairs to add mappings.
 *
 * @param str2bulk_list
 *	List of failed updates.  Each list datum is a
 *	globus_rls_string2_bulk_t structure.  <B>str2.s1</B> will
 *	be the LFN, and <B>str2.s2</B> the PFN it maps to, and
 *	<B>rc</B> will be the result code from the failed update.  Only
 *	failed updates will be returned.
 */
extern globus_result_t globus_rls_client_lrc_add_bulk(
  globus_rls_handle_t			*h,
  globus_list_t				*str2_list,
  globus_list_t				**str2bulk_list);

/**
 * Clear all mappings from LRC database.  User needs both ADMIN and LRCUPDATE
 * privileges to perform this operation.  Note that if the LRC is cleared
 * this will not be reflected in any RLI servers updated by the LRC until
 * the next softstate update, even if immediate updates are enabled.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @retval GLOBUS_SUCCESS
 *	Mappings cleared.
 */
extern globus_result_t globus_rls_client_lrc_clear(
  globus_rls_handle_t		*h);

/**
 * Create mapping between a LFN and PFN.  LFN should not exist yet.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param lfn
 *	LFN to add @a pfn mapping to, should not already exist.
 *
 * @param pfn
 *	PFN that @a lfn should map to.
 *
 * @retval GLOBUS_SUCCESS
 *	New mapping created.
 */
extern globus_result_t globus_rls_client_lrc_create(
  globus_rls_handle_t		*h,
  char				*lfn,
  char				*pfn);

/**
 * Bulk create LFN,PFN mappings in LRC database.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param str2_list
 *	LFN,PFN pairs to create mappings for.
 *
 * @param str2bulk_list
 *	List of failed updates.  Each list datum is a
 *	globus_rls_string2_bulk_t structure.  <B>str2.s1</B> will
 *	be the LFN, and <B>str2.s2</B> the PFN it maps to, and
 *	<B>rc</B> will be the result code from the failed update.  Only
 *	failed updates will be returned.
 */
extern globus_result_t globus_rls_client_lrc_create_bulk(
  globus_rls_handle_t			*h,
  globus_list_t				*str2_list,
  globus_list_t				**str2bulk_list);

/**
 * Delete mapping between LFN and PFN.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param lfn
 *	LFN to remove mapping from.
 *
 * @param pfn
 *	PFN that @a lfn maps to that is being removed.
 *
 * @retval GLOBUS_SUCCESS
 *	Mapping removed.
 */
extern globus_result_t globus_rls_client_lrc_delete(
  globus_rls_handle_t		*h,
  char				*lfn,
  char				*pfn);

/**
 * Bulk delete LFN,PFN mappings in LRC database.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param str2_list
 *	LFN,PFN pairs to add mappings.
 *
 * @param str2bulk_list
 *	List of failed updates.  Each list datum is a
 *	globus_rls_string2_bulk_t structure.  <B>str2.s1</B> will
 *	be the LFN, and <B>str2.s2</B> the PFN it maps to, and
 *	<B>rc</B> will be the result code from the failed update.  Only
 *	failed updates will be returned.
 */
extern globus_result_t globus_rls_client_lrc_delete_bulk(
  globus_rls_handle_t			*h,
  globus_list_t				*str2_list,
  globus_list_t				**str2bulk_list);

/**
 * Check if an object exists in the LRC database.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param key
 *	LFN or PFN that identifies object.
 *
 * @param objtype
 *	Type of object @a key refers to (::globus_rls_obj_lrc_lfn or
 *	::globus_rls_obj_lrc_pfn).
 *
 * @retval GLOBUS_SUCCESS
 *	Object exists.
 */
extern globus_result_t globus_rls_client_lrc_exists(
  globus_rls_handle_t		*h,
  char				*key,
  globus_rls_obj_type_t		objtype);

/**
 * Bulk check if objects exist in the LRC database.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param keylist
 *	LFNs or PFNs that identify objects.
 *
 * @param objtype
 *	Type of object @a key refers to (::globus_rls_obj_lrc_lfn or
 *	::globus_rls_obj_lrc_pfn).
 *
 * @param str2bulk_list
 *	Results of existence check.  Each list datum will be a
 *	globus_rls_string2_bulk_t structure.  <B>str2.s1</B> will
 *	be the LFN or PFN, and <B>str2.s2</B> empty, and
 *	<B>rc</B> will be the result code indicating existence.
 */
extern globus_result_t globus_rls_client_lrc_exists_bulk(
  globus_rls_handle_t		*h,
  globus_list_t			*keylist,
  globus_rls_obj_type_t		objtype,
  globus_list_t			**str2bulk_list);

/**
 * Return LFNs mapped to PFN in the LRC database.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param pfn
 *	PFN to search for.
 *
 * @param offset
 *	Offset into result list.  Used in conjunction with @a reslimit
 *	to retrieve results a few at a time.  Use 0 to begin with first
 *	result.  If NULL then API will handle accumulating partial results
 *	transparently.
 *
 * @param reslimit
 *	Maximum number of results to return.  Used in conjunction with
 *	@a offset to retrieve results a few at a time.  Use 0 to retrieve
 *	all results.
 *
 * @param str2_list
 *	List of LFNs that map to @pfn.  Each list datum will be a
 *	globus_rls_string2_t structure.  <B>s1</B> will
 *	be the LFN, and <B>s2</B> the PFN it maps to.
 *
 * @retval GLOBUS_SUCCESS
 *	List of LFNs that map to @a pfn in @a str2_list.  See
 *	@ref globus_rls_client_queryresult.
 */
extern globus_result_t globus_rls_client_lrc_get_lfn(
  globus_rls_handle_t			*h,
  char					*pfn,
  int					*offset,
  int					reslimit,
  globus_list_t				**str2_list);

/**
 * Bulk return LFNs mapped to PFN in the LRC database.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param pfnlist
 *	PFNs to search for.
 *
 * @param str2bulk_list
 *	Results of queries.  Each list datum will be a
 *	globus_rls_string2_bulk_t structure.  <B>str2.s1</B> will
 *	be the LFN, and <B>str2.s2</B> the PFN it maps to, and
 *	<B>rc</B> will be the result code from the query.
 */
extern globus_result_t globus_rls_client_lrc_get_lfn_bulk(
  globus_rls_handle_t			*h,
  globus_list_t				*pfnlist,
  globus_list_t				**str2bulk_list);

/**
 * Return LFNs mapped to wildcarded PFN in the LRC database.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param pfn_pattern
 *	PFN pattern to search for.
 *
 * @param type
 *	Identifies wildcard characters used in @a pfn_pattern.  Wildcard
 *	chars can be Unix file globbing chars (* matches 0 or more characters,
 *	? matches any single character) or SQL "like" wildcard characters
 *	(% matches 0 or more charactes, _ matches any single character).
 *
 * @param offset
 *	Offset into result list.  Used in conjunction with @a reslimit
 *	to retrieve results a few at a time.  Use 0 to begin with first
 *	result.  If NULL then the API will handle accumulating partial
 *	results transparently.
 *
 * @param reslimit
 *	Maximum number of results to return.  Used in conjunction with
 *	@a offset to retrieve results a few at a time.  Use 0 to retrieve
 *	all results.
 *
 * @param str2_list
 *	List of LFNs that map to @pfn_pattern.  Each list datum will be a
 *	globus_rls_string2_t structure.  <B>s1</B> will
 *	be the LFN, and <B>s2</B> the PFN it maps to.
 *
 * @retval GLOBUS_SUCCESS
 *	List of LFNs that map to @a pfn_pattern in @a str2_list.  See
 *	@ref globus_rls_client_queryresult.
 */
extern globus_result_t globus_rls_client_lrc_get_lfn_wc(
  globus_rls_handle_t			*h,
  char					*pfn_pattern,
  globus_rls_pattern_t			type,
  int					*offset,
  int					reslimit,
  globus_list_t				**str2_list);

/**
 * Return PFNs mapped to LFN in the LRC database.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param lfn
 *	LFN to search for.
 *
 * @param offset
 *	Offset into result list.  Used in conjunction with @a reslimit
 *	to retrieve results a few at a time.  Use 0 to begin with first
 *	result.
 *
 * @param reslimit
 *	Maximum number of results to return.  Used in conjunction with
 *	@a offset to retrieve results a few at a time.  Use 0 to retrieve
 *	all results.
 *
 * @param str2_list
 *	List of PFNs that map to @lfn.  Each list datum will be a
 *	globus_rls_string2_t structure.  <B>s1</B> will
 *	be the LFN, and <B>s2</B> the PFN it maps to.
 *
 * @retval GLOBUS_SUCCESS
 *	List of PFNs that map to @a lfn in @a str2_list.
 */
extern globus_result_t globus_rls_client_lrc_get_pfn(
  globus_rls_handle_t			*h,
  char					*lfn,
  int					*offset,
  int					reslimit,
  globus_list_t				**str2_list);

/**
 * Bulk return PFNs mapped to LFN in the LRC database.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param lfnlist
 *	LFNs to search for.
 *
 * @param str2bulk_list
 *	Results of queries.  Each list datum will be a
 *	globus_rls_string2_bulk_t structure.  <B>str2.s1</B> will
 *	be the LFN, and <B>str2.s2</B> the PFN it maps to, and
 *	<B>rc</B> will be the result code from the query.
 */
extern globus_result_t globus_rls_client_lrc_get_pfn_bulk(
  globus_rls_handle_t			*h,
  globus_list_t				*lfnlist,
  globus_list_t				**str2bulk_list);

/**
 * Return PFNs mapped to wildcarded LFN in the LRC database.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param lfn_pattern
 *	LFN pattern to search for.
 *
 * @param type
 *	Identifies wildcard characters used in @a lfn_pattern.  Wildcard
 *	chars can be Unix file globbing chars (* matches 0 or more characters,
 *	? matches any single character) or SQL "like" wildcard characters
 *	(% matches 0 or more charactes, _ matches any single character).
 *
 * @param offset
 *	Offset into result list.  Used in conjunction with @a reslimit
 *	to retrieve results a few at a time.  Use 0 to begin with first
 *	result.
 *
 * @param reslimit
 *	Maximum number of results to return.  Used in conjunction with
 *	@a offset to retrieve results a few at a time.  Use 0 to retrieve
 *	all results.
 *
 * @param str2_list
 *	List of PFNs that map to @lfn_pattern.  Each list datum will be a
 *	globus_rls_string2_t structure.  <B>s1</B> will
 *	be the LFN, and <B>s2</B> the PFN it maps to.
 *
 * @retval GLOBUS_SUCCESS
 *	List of PFNs that map to @a lfn_pattern in @a str2_list.  See
 *	@ref globus_rls_client_queryresult.
 */
extern globus_result_t globus_rls_client_lrc_get_pfn_wc(
  globus_rls_handle_t			*h,
  char					*lfn_pattern,
  globus_rls_pattern_t			type,
  int					*offset,
  int					reslimit,
  globus_list_t				**str2_list);

/**
 * Check if a mapping exists in the LRC database.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param lfn
 *	LFN of mapping.
 *
 * @param pfn
 *	PFN of mapping.
 *
 * @retval GLOBUS_SUCCESS
 *	Object exists.
 */
extern globus_result_t globus_rls_client_lrc_mapping_exists(
  globus_rls_handle_t		*h,
  char				*lfn,
  char				*pfn);

/**
 * Rename LFN. If immediate mode is ON, the LRC will send
 * update messages to associated RLIs.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param oldname
 *	Existing LFN name, to be renamed.
 *
 * @param newname
 *	New LFN name, to replace existing name.
 *
 * @retval GLOBUS_SUCCESS
 *	LFN renamed.
 */
extern globus_result_t globus_rls_client_lrc_renamelfn(
  globus_rls_handle_t		*h,
  char				*oldname,
  char				*newname);

/**
 * Bulk rename LFN names in LRC database.  LFNs must already exist.
 * If immediate mode is ON, the LRC will send update messages to 
 * associated RLIs.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param str2_list
 *	oldname,newname pairs such that newname replaces oldname for LFNs.
 *
 * @param str2bulk_list
 *	List of failed updates.  Each list datum is a
 *	globus_rls_string2_bulk_t structure.  <B>str2.s1</B> will
 *	be the old LFN name, and <B>str2.s2</B> the new LFN name, and
 *	<B>rc</B> will be the result code from the failed update.  Only
 *	failed updates will be returned.
 */
extern globus_result_t globus_rls_client_lrc_renamelfn_bulk(
  globus_rls_handle_t			*h,
  globus_list_t				*str2_list,
  globus_list_t				**str2bulk_list);

/**
 * Rename PFN.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param oldname
 *	Exisintg PFN name, to be renamed.
 *
 * @param newname
 *	New PFN name, to replace existing name.
 *
 * @retval GLOBUS_SUCCESS
 *	PFN renamed.
 */
extern globus_result_t globus_rls_client_lrc_renamepfn(
  globus_rls_handle_t		*h,
  char				*oldname,
  char				*newname);

/**
 * Bulk rename PFN names in LRC database.  PFNs must already exist.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param str2_list
 *	oldname,newname pairs such that newname replaces oldname for PFNs.
 *
 * @param str2bulk_list
 *	List of failed updates.  Each list datum is a
 *	globus_rls_string2_bulk_t structure.  <B>str2.s1</B> will
 *	be the old PFN name, and <B>str2.s2</B> the new PFN name, and
 *	<B>rc</B> will be the result code from the failed update.  Only
 *	failed updates will be returned.
 */
extern globus_result_t globus_rls_client_lrc_renamepfn_bulk(
  globus_rls_handle_t			*h,
  globus_list_t				*str2_list,
  globus_list_t				**str2bulk_list);

/**
 * LRC servers send information about LFNs in their database to the
 * the list of RLI servers in the database, added with the following
 * function.  Updates may be partitioned amongst multiple RLIs by
 * specifying one or more patterns for an RLI.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param rli_url
 *	URL of RLI server that LRC should send updates to.
 *
 * @param flags
 *	Should be zero or ::FRLI_BLOOMFILTER.
 *
 * @param pattern
 *	If not NULL used to filter which LFNs are sent to @a rli_url.
 *	Standard Unix wildcard characters (*, ?) may be used to do wildcard
 *	matches.
 *
 * @retval GLOBUS_SUCCESS
 *	RLI (with pattern if not NULL) added to LRC database.
 */
extern globus_result_t globus_rls_client_lrc_rli_add(
  globus_rls_handle_t		*h,
  char				*rli_url,
  int				flags,
  char				*pattern);
/**
 * Update RLI using bloom filters (see globus_rls_client_lrc_rli_add()).
 * @ingroup globus_rls_client_lrc_operation
 */
#define FRLI_BLOOMFILTER	0x1
#define FRLI_RLI		0x2	/* Internal use only	*/

/**
 * Delete an entry from the LRC rli/partition tables.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param rli_url
 *	URL of RLI server to remove from LRC partition table.
 *
 * @param pattern
 *	If not NULL then only the specific @a rli_url/pattern is removed,
 *	else all partition information for @a rli_url is removed.
 *
 * @retval GLOBUS_SUCCESS
 *	RLI and pattern (if specified) removed from LRC partition table.
 */
extern globus_result_t globus_rls_client_lrc_rli_delete(
  globus_rls_handle_t		*h,
  char				*rli_url,
  char				*pattern);

/**
 * Get RLI update partitions from LRC server.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param rli_url
 *	If not NULL identifies RLI that partition data will be retrieved for.
 *	If NULL then all RLIs are retrieved.
 *
 * @param pattern
 *	If not NULL returns only partitions with matching patterns,
 *	otherwise all patterns are retrieved.
 *
 * @param str2_list
 *	Results added to list.  Datums in @a str2_list are of type
 *	globus_rls_string2_t structure.  <B>s1<B> will be the rli url,
 *	<B>s2<B> an  empty string or the pattern used to partition
 *	updates.  See @ref globus_rls_client_queryresult.
 *
 * @retval GLOBUS_SUCCESS
 *	Partition data retrieved from server, written to @a str2_list.
 */
extern globus_result_t globus_rls_client_lrc_rli_get_part(
  globus_rls_handle_t			*h,
  char					*rli_url,
  char					*pattern,
  globus_list_t				**str2_list);

/**
 * Maximum length of URL string.
 * @ingroup globus_rls_client_lrc_operation
 */
#define MAXURL	256

/**
 * Information about RLI server, returned by globus_rls_client_lrc_rli_info()
 * and globus_rls_client_lrc_rli_list().
 * @ingroup globus_rls_client_lrc_operation
 */
typedef struct {
  char		url[MAXURL];	/**< URL of server.			*/
  int		updateinterval;	/**< Interval between softstate updates.*/
  int		flags;		/**< RLI flags (see ::FRLI_BLOOMFILTER).*/
  time_t	lastupdate;	/**< Time of last softstate update.	*/
} globus_rls_rli_info_t;

/**
 * Information about server sending updates to an rli, returned by
 * globus_rls_client_rli_sender_list().
 * @ingroup globus_rls_client_rli_operation
 */
typedef struct {
  char		url[MAXURL];	/**< URL of server.			*/
  time_t	lastupdate;	/**< Time of last softstate update.	*/
} globus_rls_sender_info_t;

#define globus_rls_lrc_info_t	globus_rls_sender_info_t


/**
 * Get info about RLI server updated by an LRC server.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param rli_url
 *	URL of RLI server to retrieve info for.
 *
 * @param info
 *	Data about RLI server will be written here.
 *
 * @retval GLOBUS_SUCCESS
 *	Info about RLI server successfully retrieved.
 */
extern globus_result_t globus_rls_client_lrc_rli_info(
  globus_rls_handle_t			*h,
  char					*rli_url,
  globus_rls_rli_info_t			*info);

/**
 * Return URLs of RLIs that LRC sends updates to.
 * @ingroup globus_rls_client_lrc_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param rliinfo_list
 *	List of RLIs updated by this LRC returned in this list.  Each
 *	list datum is of type globus_rls_rli_info_t.  @a rliinfo_list
 *	should be freed with globus_rls_client_free_list() when no longer
 *	needed.
 *
 * @retval GLOBUS_SUCCESS
 *	List of RLIs updated by this LRC returned in @a rliinfo_list.
 */
extern globus_result_t globus_rls_client_lrc_rli_list(
  globus_rls_handle_t			*h,
  globus_list_t				**rliinfo_list);


/**
 * @defgroup globus_rls_client_rli_operation RLI Operations
 * Functions to view and update data managed by a RLI server.
 */

/**
 * Check if an object exists in the RLI database.
 * @ingroup globus_rls_client_rli_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param key
 *	LFN or LRC that identifies object.
 *
 * @param objtype
 *	Type of object @a key refers to (::globus_rls_obj_rli_lfn or
 *	::globus_rls_obj_rli_lrc).
 *
 * @retval GLOBUS_SUCCESS
 *	Object exists.
 */
extern globus_result_t globus_rls_client_rli_exists(
  globus_rls_handle_t		*h,
  char				*key,
  globus_rls_obj_type_t		objtype);

/**
 * Bulk check if objects exist in the RLI database.
 * @ingroup globus_rls_client_rli_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param keylist
 *	LFNs or LRCs that identify objects.
 *
 * @param objtype
 *	Type of object @a key refers to (::globus_rls_obj_rli_lfn or
 *	::globus_rls_obj_rli_lrc).
 *
 * @param str2bulk_list
 *	Results of existence check.  Each list datum will be a
 *	globus_rls_string2_bulk_t structure.  <B>str2.s1</B> will
 *	be the LFN or LRC, and <B>str2.s2</B> empty, and
 *	<B>rc</B> will be the result code indicating existence.
 */
extern globus_result_t globus_rls_client_rli_exists_bulk(
  globus_rls_handle_t		*h,
  globus_list_t			*keylist,
  globus_rls_obj_type_t		objtype,
  globus_list_t			**str2bulk_list);

/**
 * Return LRCs mapped to LFN in the RLI database.
 * @ingroup globus_rls_client_rli_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param lfn
 *	LFN whose list of LRCs is desired.
 *
 * @param offset
 *	Offset into result list.  Used in conjunction with @a reslimit
 *	to retrieve results a few at a time.  Use 0 to begin with first
 *	result.
 *
 * @param reslimit
 *	Maximum number of results to return.  Used in conjunction with
 *	@a offset to retrieve results a few at a time.  Use 0 to retrieve
 *	all results.
 *
 * @param str2_list
 *	List of LRCs that @lfn maps to.  Each list datum will be a
 *	globus_rls_string2_t structure.  <B>s1</B> will
 *	be the LFN, and <B>s2</B> the LRC it maps to.  @a str2_list
 *	should be freed with globus_rls_client_free_list().
 *
 * @retval GLOBUS_SUCCESS
 *	List of LRCs that map to @a lfn in @a str2_list.  See
 *	@ref globus_rls_client_queryresult.
 */
extern globus_result_t globus_rls_client_rli_get_lrc(
  globus_rls_handle_t			*h,
  char					*lfn,
  int					*offset,
  int					reslimit,
  globus_list_t				**str2_list);

/**
 * Bulk return LRCs mapped to LFN in the RLI database.
 * @ingroup globus_rls_client_rli_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param lfnlist
 *	LFNs to search for.
 *
 * @param str2bulk_list
 *	Results of queries.  Each list datum will be a
 *	globus_rls_string2_bulk_t structure.  <B>str2.s1</B> will
 *	be the LFN, and <B>str2.s2</B> the LRC it maps to, and
 *	<B>rc</B> will be the result code from the query.
 */
extern globus_result_t globus_rls_client_rli_get_lrc_bulk(
  globus_rls_handle_t			*h,
  globus_list_t				*lfnlist,
  globus_list_t				**str2bulk_list);

/**
 * Return LRCs mapped to wildcarded LFN in the RLI database.
 * @ingroup globus_rls_client_rli_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param lfn_pattern
 *	LFN pattern to search for.
 *
 * @param type
 *	Identifies wildcard characters used in @a lfn_pattern.  Wildcard
 *	chars can be Unix file globbing chars (* matches 0 or more characters,
 *	? matches any single character) or SQL "like" wildcard characters
 *	(% matches 0 or more charactes, _ matches any single character).
 *
 * @param offset
 *	Offset into result list.  Used in conjunction with @a reslimit
 *	to retrieve results a few at a time.  Use 0 to begin with first
 *	result.
 *
 * @param reslimit
 *	Maximum number of results to return.  Used in conjunction with
 *	@a offset to retrieve results a few at a time.  Use 0 to retrieve
 *	all results.
 *
 * @param str2_list
 *	List of LRCs that map to @lfn_pattern.  Each list datum will be a
 *	globus_rls_string2_t.  <B>s1</B> will
 *	be the LFN, and <B>s2</B> the LRC it maps to.  @a str2_list
 *	should be freed with globus_rls_client_free_list().
 *
 * @retval GLOBUS_SUCCESS
 *	List of LRCs that map to @a lfn_pattern in @a str2_list.  See
 *	@ref globus_rls_client_queryresult.
 */
extern globus_result_t globus_rls_client_rli_get_lrc_wc(
  globus_rls_handle_t			*h,
  char					*lfn_pattern,
  globus_rls_pattern_t			type,
  int					*offset,
  int					reslimit,
  globus_list_t				**str2_list);

/**
 * Get list of servers updating this RLI server.  Similar to
 * globus_rls_client_rli_get_part() except no partition information is
 * returned.
 * @ingroup globus_rls_client_rli_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param senderinfo_list
 *	Datums in senderinfo_list will be of type globus_rls_sender_info_t.
 *	@a senderinfo_list should be freed with globus_rls_client_free_list().
 *
 * @retval GLOBUS_SUCCESS
 *	List of LRCs updating RLI added to @a senderinfo_list.
 */
extern globus_result_t globus_rls_client_rli_sender_list(
  globus_rls_handle_t			*h,
  globus_list_t				**senderinfo_list);

/**
 * Deprecated, use globus_rls_client_rli_sender_list().
 * @ingroup globus_rls_client_rli_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param lrcinfo_list
 *	Datums in lrcinfo_list will be of type globus_rls_lrc_info_t.
 *	@a lrcinfo_list should be freed with globus_rls_client_free_list().
 *
 * @retval GLOBUS_SUCCESS
 *	List of LRCs updating RLI added to @a lrcinfo_list.
 */
extern globus_result_t globus_rls_client_rli_lrc_list(
  globus_rls_handle_t			*h,
  globus_list_t				**lrcinfo_list);

/**
 * Check if a mapping exists in the RLI database.
 * @ingroup globus_rls_client_rli_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param lfn
 *	LFN of mapping.
 *
 * @param lrc
 *	LRC of mapping.
 *
 * @retval GLOBUS_SUCCESS
 *	Mapping exists.
 */
extern globus_result_t globus_rls_client_rli_mapping_exists(
  globus_rls_handle_t		*h,
  char				*lfn,
  char				*lrc);

/**
 * RLI servers send information about LFNs in their database to the
 * the list of RLI servers in the database, added with the following
 * function.  Updates may be partitioned amongst multiple RLIs by
 * specifying one or more patterns for an RLI.
 * @ingroup globus_rls_client_rli_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param rli_url
 *	URL of RLI server that LRC should send updates to.
 *
 * @param pattern
 *	If not NULL used to filter which LFNs are sent to @a rli_url.
 *	Standard Unix wildcard characters (*, ?) may be used to do wildcard
 *	matches.
 *
 * @retval GLOBUS_SUCCESS
 *	RLI (with pattern if not NULL) added to RLI database.
 */
extern globus_result_t globus_rls_client_rli_rli_add(
  globus_rls_handle_t		*h,
  char				*rli_url,
  char				*pattern);

/**
 * Delete an entry from the RLI rli/partition tables.
 * @ingroup globus_rls_client_rli_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param rli_url
 *	URL of RLI server to remove from RLI partition table.
 *
 * @param pattern
 *	If not NULL then only the specific @a rli_url/pattern is removed,
 *	else all partition information for @a rli_url is removed.
 *
 * @retval GLOBUS_SUCCESS
 *	RLI and pattern (if specified) removed from LRC partition table.
 */
extern globus_result_t globus_rls_client_rli_rli_delete(
  globus_rls_handle_t		*h,
  char				*rli_url,
  char				*pattern);

/**
 * Get RLI update partitions from RLI server.
 * @ingroup globus_rls_client_rli_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param rli_url
 *	If not NULL identifies RLI that partition data will be retrieved for.
 *	If NULL then all RLIs are retrieved.
 *
 * @param pattern
 *	If not NULL returns only partitions with matching patterns,
 *	otherwise all patterns are retrieved.
 *
 * @param str2_list
 *	Results added to list.  Datums in @a str2_list are of type
 *	globus_rls_string2_t structure.  <B>s1<B> will be the rli url,
 *	<B>s2<B> an  empty string or the pattern used to partition
 *	updates.  See @ref globus_rls_client_queryresult.
 *
 * @retval GLOBUS_SUCCESS
 *	Partition data retrieved from server, written to @a str2_list.
 */
extern globus_result_t globus_rls_client_rli_rli_get_part(
  globus_rls_handle_t			*h,
  char					*rli_url,
  char					*pattern,
  globus_list_t				**str2_list);

/**
 * Return URLs of RLIs that RLI sends updates to.
 * @ingroup globus_rls_client_rli_operation
 *
 * @param h
 *	Handle connected to an RLS server.
 *
 * @param rliinfo_list
 *	List of RLIs updated by this RLI returned in this list.  Each
 *	list datum is of type globus_rls_rli_info_t.  @a rliinfo_list
 *	should be freed with globus_rls_client_free_list() when no longer
 *	needed.
 *
 * @retval GLOBUS_SUCCESS
 *	List of RLIs updated by this LRC returned in @a rliinfo_list.
 */
extern globus_result_t globus_rls_client_rli_rli_list(
  globus_rls_handle_t			*h,
  globus_list_t				**rliinfo_list);

/**
 * globus_rls_client_admin() commands.
 * @ingroup globus_rls_client_miscellaneous
 */
typedef enum {
  globus_rls_admin_cmd_ping,	/**< Verify RLS server responding.	*/
  globus_rls_admin_cmd_quit,	/**< Tell RLS server to exit.		*/
  globus_rls_admin_cmd_ssu	/**< Tell LRC server to do softstate update.*/
} globus_rls_admin_cmd_t;

/*
 * The following are obsolete, please use enum above.
 */
#define GLOBUS_RLS_ADMIN_PING	globus_rls_admin_ping
#define GLOBUS_RLS_ADMIN_QUIT	globus_rls_admin_quit
#define GLOBUS_RLS_ADMIN_SSU	globus_rls_admin_ssu

#define MAXVERSION 16
/**
 * Various configuration options and statistics about an RLS server
 * returned in the following structures by globus_rls_client_stats().
 * See ::RLS_LRCSERVER for possible flags values.
 * @ingroup globus_rls_client_miscellaneous
 */
typedef struct {
  char		version[MAXVERSION];
  time_t	uptime;
  int		flags;		/**< See ::RLS_LRCSERVER.	*/
  int		lrc_bloomfilterui;
  int		lrc_lfnlistui;
  int		lrc_numlfn;
  int		lrc_numpfn;
  int		lrc_nummap;
  int		rli_numlfn;
  int		rli_numlrc;
  int		rli_numsender;
  int		rli_nummap;
} globus_rls_stats_t;

/**
 * Server is LRC server.
 * @ingroup globus_rls_client_miscellaneous
 */
#define RLS_LRCSERVER		 0x1
/**
 * Server is RLI server.
 * @ingroup globus_rls_client_miscellaneous
 */
#define RLS_RLISERVER		 0x2
/**
 * RLI accepts LFN list updates.
 * @ingroup globus_rls_client_miscellaneous
 */
#define RLS_RCVLFNLIST		 0x4
/**
 * RLI accepts Bloom filter updates.
 * @ingroup globus_rls_client_miscellaneous
 */
#define RLS_RCVBLOOMFILTER	 0x8
/**
 * LRC sends LFN list updates.
 * @ingroup globus_rls_client_miscellaneous
 */
#define RLS_SNDLFNLIST		0x10
/**
 * LRC sends Bloom filter updates.
 * @ingroup globus_rls_client_miscellaneous
 */
#define RLS_SNDBLOOMFILTER	0x20
/**
 * RLC is fully initialized.
 * @ingroup globus_rls_client_miscellaneous
 */
#define RLS_INITIALIZED		0x40

/**
 * Miscellaneous administrative operations.  Most operations require the
 * admin privilege.
 * @ingroup globus_rls_client_miscellaneous
 *
 * @param h
 *	Handle connected to RLS server.
 *
 * @param cmd
 * 	Command to be sent to RLS server.
 *
 * @retval GLOBUS_SUCCESS
 *	Command succeeded.
 */
extern globus_result_t globus_rls_client_admin(
  globus_rls_handle_t		*h,
  globus_rls_admin_cmd_t	cmd);


/**
 * Get server configuration.  Client needs admin privilege.
 * @ingroup globus_rls_client_miscellaneous
 *
 * @param h
 *	Handle connected to RLS server.
 *
 * @param option
 *	Configuration option to get.  If NULL all options are retrieved.
 *
 * @retval conf_list
 *	List of configuration options.
 *
 * @retval GLOBUS_SUCCESS
 *	List of retrieved config options returned in @a conf_list, each datum
 *	is of type globus_rls_string2_t.  @a conf_list should be freed
 *	with globus_rls_client_free_list().  There may be multiple
 *	"acl" entries in the list, since the access control list can
 *	include more than one entry.  Each acl configuration value is
 *	consists of a regular expression (matched against grid-mapfile
 *	users or DNs), a colon, and space separated list of permissions
 *	the matching users are granted.
 */
extern globus_result_t globus_rls_client_get_configuration(
  globus_rls_handle_t		*h,
  char				*option,
  globus_list_t			**conf_list);

/**
 * Set server configuration option.  Client needs admin privilege.
 * @ingroup globus_rls_client_miscellaneous
 *
 * @param h
 *	Handle connected to RLS server.
 *
 * @param option
 *	Configuration option to set.
 *
 * @param value
 *	New value for option.
 *
 * @retval GLOBUS_SUCCESS
 *	Option set on server.
 */
extern globus_result_t globus_rls_client_set_configuration(
  globus_rls_handle_t		*h,
  char				*option,
  char				*value);

/**
 * Retrieve various statistics from RLS server.  Requires stats privilege.
 * @ingroup globus_rls_client_miscellaneous
 *
 * @param h
 *	Handle connected to RLS server.
 *
 * @param rlsstats
 *	Stats returned here.
 *
 * @retval GLOBUS_SUCCESS
 *	Stats returned in @a rlsstats.
 */
extern globus_result_t globus_rls_client_stats(
  globus_rls_handle_t		*h,
  globus_rls_stats_t		*rlsstats);

/**
 * Map attribute value to string.
 * @ingroup globus_rls_client_miscellaneous
 *
 * @param attr
 *	Attribute to convert.  If @a attr->type is ::globus_rls_attr_type_date
 *	then the resulting string will be in the format MySQL uses by
 *	default, which is YYYYMMDDHHMMSS.
 *
 * @param buf
 *	Buffer to write string value to.  Note if @a attr->type is
 *	::globus_rls_attr_type_str then @a attr->val.s is returned, and
 *	@a buf is unused.
 *
 * @param buflen
 *	Size of @a buf in bytes.
 *
 * @retval "String Value"
 *	Attribute value converted to a string.
 */
extern char *globus_rls_client_attr2s(
  globus_rls_attribute_t		*attr,
  char					*buf,
  int					buflen);

/**
 * Set globus_rls_attribute_t type and val fields from a type and string value.
 * @ingroup globus_rls_client_miscellaneous
 *
 * @param type
 *	Attribute value type.
 *
 * @param sval
 *	String value to convert to binary.  If type is
 *	::globus_rls_attr_type_date @a sval should be in the form
 *	YYYY-MM-DD HH:MM:SS.
 *
 * @param attr
 *	Attribute whose type and val fields are to be set.
 *
 * @retval GLOBUS_SUCCESS
 *	@a attr->type and @a attr->val successfully set.
 */
extern globus_result_t globus_rls_client_s2attr(
  globus_rls_attr_type_t		type,
  char					*sval,
  globus_rls_attribute_t		*attr);

/**
 * Get error code and message from globus_result_t returned by this API.
 * @ingroup globus_rls_client_miscellaneous
 *
 * @param r
 *	Result returned by RLS API function.  @a r is freed by this call
 *	and should not be referenced again.  If @preserve is set then a
 *	new globus_result_t is constructed with the same values and
 *	returned as the function value.
 *
 * @param rc
 *	Address to store error code at.  If NULL error code is not returned.
 *
 * @param buf
 *	Address to store error message at.  If NULL error message is not
 *	returned.
 *
 * @param preserve
 *	If GLOBUS_TRUE then a new globus_result_t is contructed with the
 *	same values as the old and returned as the function value.
 *
 * @param buflen
 *	Size of @a buf.
 *
 * @retval globus_result_t
 *	If @a preserve is set a new globus_result_t identical to @a r is
 *	returned, otherwise GLOBUS_SUCCESS.
 */
extern globus_result_t globus_rls_client_error_info(
  globus_result_t		r,
  int				*rc,
  char				*buf,
  int				buflen,
  globus_bool_t			preserve);

/**
 * Compute length of list.  globus_list_size() is implemented using recursion,
 * besides being inefficient it can run out of stack space when the list is
 * large.
 * @ingroup globus_rls_client_miscellaneous
 */
extern int globus_list_len(
  globus_list_t			*len);

/**
 * Map RLS status code to error string.
 * @ingroup globus_rls_client_miscellaneous
 *
 * @param rc
 *	Status code.
 *
 * @param specificmsg
 *	If not NULL prepended (with a colon) to error string.
 *
 * @param buf
 *	Buffer to write error message to.
 *
 * @param buflen
 *	Length of @a buf.  Message will be truncated to fit if too long.
 *
 * @retval char *
 *	Returns @a buf, error message written to @a buf.
 */
extern char *
globus_rls_errmsg(
  int				rc,
  char				*specificmsg,
  char				*buf,
  int				buflen);

extern int
mycftime(
  char	*s,
  int	len,
  char	*fmt,
  time_t clock);
#endif
