/*
 * $Id: glue-audio_esd.c,v 1.3 2009-01-28 11:00:33 potyra Exp $
 *
 * eSound audio plugin driver for FAUmachine
 *
 * Derived from MPlayer (libao2 - ao_esd),
 * originally written by:
 *
 * Copyright (c) FAUmachine Team.
 * Copyright (c) Juergen Keil <jk@tools.de>
 * Copyright (c) MPlayer Team
 *
 * This driver is distributed under the terms of the GPL
 *
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 *
 * TODO / known problems:
 * - does not work well when the esd daemon has autostandby disabled
 *   (workaround: run esd with option "-as 2" - fortunatelly this is
 *    the default)
 * - plays noise on a linux 2.4.4 kernel with a SB16PCI card, when using
 *   a local tcp connection to the esd daemon; there is no noise when using
 *   a unix domain socket connection.
 *   (there are EIO errors reported by the sound card driver, so this is
 *   most likely a linux sound card driver problem)
 */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <dlfcn.h>
#include <errno.h>
#include <fcntl.h>
#include <time.h>
#ifdef	__svr4__
#include <stropts.h>
#endif
#include <esd.h>

#include "config.h"
#include "glue-audio_internal.h"

#define	ESD_CLIENT_NAME	"FAUmachine"
#define	ESD_MAX_DELAY	(1.0f)	/* max amount of data buffered in esd (#sec) */

static ao_info_t info =
{
	"EsounD audio output",
	"esd",
	"Juergen Keil <jk@tools.de>",
	""
};

LIBAO_EXTERN(esd);

int (*e_open_sound)(const char *host);
esd_server_info_t * (*e_get_server_info)(int esd);
int (*e_get_latency)(int esd);
int (*e_play_stream_fallback)(esd_format_t format, int rate, const char *host, const char *name);
int (*e_close)(int esd);
void (*e_free_server_info)(esd_server_info_t *server_info);

static struct {
	const char * name;
	void * funcpp;
} esd_array[] = {
	{ "esd_open_sound", &e_open_sound },
	{ "esd_get_server_info", &e_get_server_info },
	{ "esd_get_latency", &e_get_latency },
	{ "esd_play_stream_fallback", &e_play_stream_fallback },
	{ "esd_close", &e_close },
	{ "esd_free_server_info", &e_free_server_info },
	{ NULL, NULL }
};

static void * esd_lib_handle = NULL;

static int esd_bps;
static int esd_samplerate;
static int esd_fd = -1;
static int esd_play_fd = -1;
static esd_server_info_t *esd_svinfo;
static int esd_bytes_per_sample;
static unsigned long esd_samples_written;
static struct timeval esd_play_start;


/*
 * open & setup audio device
 * return: 1=success 0=fail
 */
static int
init(void)
{
	esd_format_t esd_fmt;
	int i;
	int fl;
	char *server = NULL;  /* NULL for localhost */

	esd_lib_handle = dlopen("libesd.so", RTLD_NOW | RTLD_GLOBAL | RTLD_NODELETE);
	if(esd_lib_handle == NULL) {
		return 0;
	}

	for(i=0; esd_array[i].name; i++) {
		dlerror();
		*(void**) esd_array[i].funcpp = dlsym(esd_lib_handle,esd_array[i].name);
		if(dlerror() != NULL) {
			return 0;
		}
	}

	if (esd_fd < 0) {
		esd_fd = e_open_sound(server);
		if (esd_fd < 0) {
			return 0;
		}
		esd_svinfo = e_get_server_info(esd_fd);
	}

	esd_fmt = ESD_STREAM | ESD_PLAY;
	esd_samplerate = GLUE_AUDIO_RATE;

#if 0 /* !defined(ESD_RESAMPLES) */
	/* FIXME: let audio filter convert the sample rate */
	if (esd_svinfo != NULL)
		esd_sampelrate = esd_svinfo->rate;
#endif

#if GLUE_AUDIO_CHANNELS == 1
	esd_fmt |= ESD_MONO;
	esd_bytes_per_sample = 1;
#elif GLUE_AUDIO_CHANNELS == 2
	esd_fmt |= ESD_STEREO;
	esd_bytes_per_sample = 2;
#else
#error unsupported number of channels
#endif

#if GLUE_AUDIO_FORMAT == AFMT_U8 || GLUE_AUDIO_FORMAT == AFMT_S8
	esd_fmt |= ESD_BITS8;
#else
	esd_fmt |= ESD_BITS16;
	esd_bytes_per_sample *= 2;
#endif
	esd_play_fd = e_play_stream_fallback(esd_fmt, esd_samplerate,
			server, ESD_CLIENT_NAME);
	if (esd_play_fd < 0) {
		return 0;
	}

	/* enable non-blocking i/o on the socket connection to the esd server */
	if ((fl = fcntl(esd_play_fd, F_GETFL)) >= 0)
		fcntl(esd_play_fd, F_SETFL, O_NDELAY|fl);

	esd_bps = esd_bytes_per_sample * esd_samplerate;

	esd_play_start.tv_sec = 0;
	esd_samples_written = 0;

	return 1;
}


/*
 * close audio device
 */
static void
uninit(int immed)
{
	if (esd_play_fd >= 0) {
		e_close(esd_play_fd);
		esd_play_fd = -1;
	}

	if (esd_svinfo) {
		e_free_server_info(esd_svinfo);
		esd_svinfo = NULL;
	}

	if (esd_fd >= 0) {
		e_close(esd_fd);
		esd_fd = -1;
	}
}


/*
 * plays 'len' bytes of 'data'
 * it should round it down to outburst*n
 * return: number of bytes played
 */
static int
play(void* data, int len)
{
	int offs;
	int nwritten;
	int nsamples;
	int n;

	/* round down buffersize to a multiple of ESD_BUF_SIZE bytes */
	len = len / ESD_BUF_SIZE * ESD_BUF_SIZE;
	if (len <= 0)
		return 0;

#define	SINGLE_WRITE 0
#if	SINGLE_WRITE
	nwritten = write(esd_play_fd, data, len);
#else
	for (offs = 0, nwritten=0; offs + ESD_BUF_SIZE <= len; offs += ESD_BUF_SIZE) {
		/*
		 * note: we're writing to a non-blocking socket here.
		 * A partial write means, that the socket buffer is full.
		 */
		n = write(esd_play_fd, (char*)data + offs, ESD_BUF_SIZE);
		if ( n < 0 ) {
			break;
		} else if ( n != ESD_BUF_SIZE ) {
			nwritten += n;
			break;
		} else
			nwritten += n;
	}
#endif

	if (nwritten > 0) {
		if (!esd_play_start.tv_sec)
			gettimeofday(&esd_play_start, NULL);
		nsamples = nwritten / esd_bytes_per_sample;
		esd_samples_written += nsamples;
	}

	return nwritten;
}


/*
 * return: how many bytes can be played without blocking
 */
static int
get_space(void)
{
	struct timeval tmout;
	fd_set wfds;
	float current_delay;
	int space;

	/* 
	 * Don't buffer too much data in the esd daemon.
	 *
	 * If we send too much, esd will block in write()s to the sound
	 * device, and the consequence is a huge slow down for things like
	 * e_get_all_info().
	 */
	if ((current_delay = get_delay()) >= ESD_MAX_DELAY) {
		return 0;
	}

	FD_ZERO(&wfds);
	FD_SET(esd_play_fd, &wfds);
	tmout.tv_sec = 0;
	tmout.tv_usec = 0;

	if (select(esd_play_fd + 1, NULL, &wfds, NULL, &tmout) != 1)
		return 0;

	if (!FD_ISSET(esd_play_fd, &wfds))
		return 0;

	/* try to fill 50% of the remaining "free" buffer space */
	space = (ESD_MAX_DELAY - current_delay) * esd_bps * 0.5f;

	/* round up to next multiple of ESD_BUF_SIZE */
	space = (space + ESD_BUF_SIZE-1) / ESD_BUF_SIZE * ESD_BUF_SIZE;

	return space;
}


/*
 * return: delay in seconds between first and last sample in buffer
 */
static float
get_delay(void)
{
	struct timeval now;
	double buffered_samples_time;
	double play_time;

	if (!esd_play_start.tv_sec)
		return 0;

	buffered_samples_time = (float)esd_samples_written / esd_samplerate;
	gettimeofday(&now, NULL);
	play_time  =  now.tv_sec  - esd_play_start.tv_sec;
	play_time += (now.tv_usec - esd_play_start.tv_usec) / 1000000.;

	if (play_time > buffered_samples_time) {
		esd_play_start.tv_sec = 0;
		esd_samples_written = 0;
		return 0;
	}

	return buffered_samples_time - play_time;
}
