C Copyright 1981-2012 ECMWF.
C
C This software is licensed under the terms of the Apache Licence 
C Version 2.0 which can be obtained at http://www.apache.org/licenses/LICENSE-2.0.
C
C In applying this licence, ECMWF does not waive the privileges and immunities 
C granted to it by virtue of its status as an intergovernmental organisation 
C nor does it submit to any jurisdiction.
C

      INTEGER FUNCTION IRGTOG (PIFELD, KSTART, KINS, KOWE, KONS,
     1   KWEIND, KNSIND, PWFACT, POFELD, KPR, KERR)
C
C---->
C**** *IRGTOG*
C
C     Purpose
C     -------
C
C     Perform basic interpolation between the input and output fields
C     for a quasi regular input field and a regular output field.
C
C     Interface
C     ---------
C
C     IERR = IRGTOG (PIFELD, KSTART, KINS, KOWE, KONS, KWEIND,
C    1   KNSIND, PWFACT, POFELD, KPR, KERR)
C
C     Input parameters
C     ----------------
C
C     PIFELD     - The input field provided by the calling routine.
C
C     KSTART     - The array offset for each line of latitude within
C                  the quasi regular field.
C
C     KINS       - The number of points in the North-South direction
C                  in the input field.
C
C     KOWE       - The number of points in the West-East direction in
C                  the output field.
C
C     KONS       - The number of points in the North-South direction
C                  in the output field.
C
C     KWEIND     - This array contains the array offsets of the West
C                  and East points in the input array required for
C                  interpolation.
C
C     KNSIND     - This array contains the array offsets of the North
C                  and South points in the input array required for
C                  interpolation.
C
C     PWFACT     - The array of interpolating weights to the four
C                  neighbouring points for every output point.
C
C     KPR        - The debug print switch.
C                  0  , No debugging output.
C                  1  , Produce debugging output.
C
C     KERR       - The error control flag.
C                  -ve, No error message. Return error code.
C                  0  , Hard failure with error message.
C                  +ve, Print error message. Return error code.
C
C     Output parameters
C     -----------------
C
C     POFELD     - The output field returned to the calling routine.
C
C     Return value
C     ------------
C
C     The error indicator (INTEGER).
C
C     Error and Warning Return Values
C     -------------------------------
C
C     None
C
C     Common block usage
C     ------------------
C
C     None
C
C     Externals
C     ---------
C
C     INTLOG(R)    - Logs messages.
C
C     Method
C     ------
C
C     This routine performs basic linear interpolation using the four
C     neighbouring points in the quasi regular input array to
C     generate the output array.
C
C     Reference
C     ---------
C
C     None
C
C     Comments
C     --------
C
C     None
C
C     Author
C     ------
C
C     K. Fielding      *ECMWF*      Nov 1993
C
C     Modifications
C     -------------
C
C     Allow for missing data values
C     J.D.Chambers      ECMWF       August 2000
C
C     Force nearest neighbour processing with env variable or
C     INTOUT parameter
C     S.Curic           ECMWF       September 2005
C
C----<
C     -----------------------------------------------------------------|
C*    Section 0. Definition of variables.
C     -----------------------------------------------------------------|
C
      IMPLICIT NONE
C
#include "parim.h"
#include "nifld.common"
#include "nofld.common"
C
C     Function arguments
C
      INTEGER KINS, KOWE, KONS, KPR, KERR
      INTEGER KSTART (KINS)
      INTEGER KWEIND (2, KOWE, 2, KONS), KNSIND (2, KONS)
      REAL PIFELD (*), POFELD (KOWE, KONS)
      REAL PWFACT (4, KOWE, KONS)
C
C     Local variables
C
      INTEGER ILATN, ILATS, INORTH, ISOUTH, JOLAT, JOLON, COUNT
      REAL NEAREST
      CHARACTER*12 YFLAG
      LOGICAL LVEGGY
C
C     Statement function
C
      REAL A, B
      LOGICAL NOTEQ
      NOTEQ(A,B) = (ABS((A)-(B)).GT.(ABS(A)*1E-3))
C
C     -----------------------------------------------------------------|
C*    Section 1. Initialisation
C     -----------------------------------------------------------------|
C
  100 CONTINUE
C
      IRGTOG = 0
C
      IF (KPR .GE. 1) THEN
        CALL INTLOG(JP_DEBUG,'IRGTOG: Section 1.',JPQUIET)
        CALL INTLOG(JP_DEBUG,'IRGTOG: No. input fld lats = ',KINS)
        CALL INTLOG(JP_DEBUG,'IRGTOG: No.output fld lats = ',KONS)
        CALL INTLOG(JP_DEBUG,'IRGTOG: No.output fld longs = ',KOWE)
      ENDIF
C
      LVEGGY = (NITABLE.EQ.128).AND.
     X         ((NIPARAM.EQ.27).OR.
     X          (NIPARAM.EQ.28).OR.
     X          (NIPARAM.EQ.29).OR.
     X          (NIPARAM.EQ.30).OR.
     X          (NIPARAM.EQ.43) )

C     Force nearest neighbour processing with env variable
        CALL GETENV('NEAREST_NEIGHBOUR', YFLAG)
        IF( YFLAG(1:1).EQ.'1' ) LVEGGY = .TRUE.

C     Force nearest neighbour processing with INTOUT parameter
      IF( LMETHOD ) LVEGGY = .TRUE.

      IF( LVEGGY ) CALL INTLOG(JP_DEBUG,
     X  'IRGTOG: nearest neighbour processing (vegetation)',JPQUIET)
C
C     -----------------------------------------------------------------|
C*    Section 2. Basic interpolation
C     -----------------------------------------------------------------|
C
  200 CONTINUE
C
      IF( KPR.GE.1 ) CALL INTLOG(JP_DEBUG,'IRGTOG: Section 2.',JPQUIET)
C
      DO JOLAT = 1, KONS
C
        ILATN = KNSIND(JP_I_N,JOLAT)
        ILATS = KNSIND(JP_I_S,JOLAT)
C
        INORTH = KSTART(ILATN) - 1
        ISOUTH = KSTART(ILATS) - 1
C
        DO JOLON = 1, KOWE
C
C         Count non-missing data values
C
          COUNT = 0
          IF( NOTEQ(PIFELD(KWEIND(JP_I_W,JOLON,JP_I_N,JOLAT)+INORTH),
     X          RMISSGV) ) COUNT = COUNT + 1
          IF( NOTEQ(PIFELD(KWEIND(JP_I_E,JOLON,JP_I_N,JOLAT)+INORTH),
     X          RMISSGV) ) COUNT = COUNT + 1
          IF( NOTEQ(PIFELD(KWEIND(JP_I_W,JOLON,JP_I_S,JOLAT)+ISOUTH),
     X          RMISSGV) ) COUNT = COUNT + 1
          IF( NOTEQ(PIFELD(KWEIND(JP_I_E,JOLON,JP_I_S,JOLAT)+ISOUTH),
     X          RMISSGV) ) COUNT = COUNT + 1
C
C         Interpolate using four neighbours if none are missing
C
          IF( (COUNT.EQ.4).AND.(.NOT.LVEGGY) ) THEN
            POFELD(JOLON,JOLAT) =
     X        PIFELD(KWEIND(JP_I_W,JOLON,JP_I_N,JOLAT)+INORTH) *
     X          PWFACT(JP_I_NW,JOLON,JOLAT) +
     X        PIFELD(KWEIND(JP_I_E,JOLON,JP_I_N,JOLAT)+INORTH) *
     X          PWFACT(JP_I_NE,JOLON,JOLAT) +
     X        PIFELD(KWEIND(JP_I_W,JOLON,JP_I_S,JOLAT)+ISOUTH) *
     X          PWFACT(JP_I_SW,JOLON,JOLAT) +
     X        PIFELD(KWEIND(JP_I_E,JOLON,JP_I_S,JOLAT)+ISOUTH) *
     X          PWFACT(JP_I_SE,JOLON,JOLAT)
C
C         Set missing if all neighbours are missing
C
          ELSE IF( COUNT.EQ.0 ) THEN
            POFELD(JOLON,JOLAT) = RMISSGV
C
C         Otherwise, use the nearest neighbour
C
          ELSE
            NEAREST = PWFACT(JP_I_NW,JOLON,JOLAT)
            POFELD(JOLON,JOLAT) = 
     X        PIFELD(KWEIND(JP_I_W,JOLON,JP_I_N,JOLAT)+INORTH)
C
            IF( PWFACT(JP_I_NE,JOLON,JOLAT).GT.NEAREST ) THEN
              NEAREST = PWFACT(JP_I_NE,JOLON,JOLAT)
              POFELD(JOLON,JOLAT) = 
     X          PIFELD(KWEIND(JP_I_E,JOLON,JP_I_N,JOLAT)+INORTH)
            ENDIF
C
            IF( PWFACT(JP_I_SW,JOLON,JOLAT).GT.NEAREST ) THEN
              NEAREST = PWFACT(JP_I_SW,JOLON,JOLAT)
              POFELD(JOLON,JOLAT) = 
     X          PIFELD(KWEIND(JP_I_W,JOLON,JP_I_S,JOLAT)+ISOUTH)
            ENDIF
C
            IF( PWFACT(JP_I_SE,JOLON,JOLAT).GT.NEAREST ) THEN
              NEAREST = PWFACT(JP_I_SE,JOLON,JOLAT)
              POFELD(JOLON,JOLAT) = 
     X          PIFELD(KWEIND(JP_I_E,JOLON,JP_I_S,JOLAT)+ISOUTH)
            ENDIF
C
          ENDIF
C
        ENDDO
C
      ENDDO
C
C     -----------------------------------------------------------------|
C*    Section 9. Return to calling routine. Format statements
C     -----------------------------------------------------------------|
C
  900 CONTINUE
C
      IF( KPR.GE.1 ) CALL INTLOG(JP_DEBUG,'IRGTOG: Section 9.',JPQUIET)
C
      RETURN
      END
