/*-
 * Copyright (c) 2008, 2013  Peter Pentchev
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Ringlet$
 */

#include <err.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "confget.h"
#include "confget_common.h"
#include "confget_http_get.h"

static int		readfile_http_get(void);

confget_backend	 confget_http_get_backend = {
	"http_get",
	common_openfile_null,
	readfile_http_get,
	common_closefile,
};

static char		*strdup_urldecode(const char * const _s, size_t _len);

/***
 * Function:
 *	readfile_http_get	- scan CGI GET data for the requested variable
 * Inputs:
 *	None.
 * Returns:
 *	0 on success, -1 on error.
 * Modifies:
 *	May write to standard output if the variable has been found.
 */
static int
readfile_http_get(void)
{
	const char *qstr, *start;
	char *vname, *vvalue;
	int res;

	if (section == NULL)
		section = "QUERY_STRING";
	qstr = getenv(section);
	if (qstr == NULL)
		errx(1, "No CGI GET data - undefined environment variable %s",
		    section);
	while (*qstr != '\0') {
		start = qstr;
		while (*qstr != '\0' && *qstr != '&' && *qstr != '=')
			qstr++;
		vname = strdup_urldecode(start, qstr - start);
		if (vname == NULL)
			return (-1);

		if (*qstr != '=') {
			res = foundvar("", vname, "");
			free(vname);
			if (res != 0)
				return (res);
			else if (*qstr == '\0')
				return (0);
			if (!strncmp(qstr + 1, "amp;", 4))
				qstr += 5;
			else
				qstr++;
			continue;
		}

		start = ++qstr;
		while (*qstr != '\0' && *qstr != '&')
			qstr++;
		vvalue = strdup_urldecode(start, qstr - start);
		if (vvalue == NULL) {
			free(vname);
			return (-1);
		}
		res = foundvar("", vname, vvalue);
		free(vname);
		free(vvalue);
		if (res != 0)
			return (res);
		else if (!morevars || *qstr == '\0')
			return (0);
		if (!strncmp(qstr + 1, "amp;", 4))
			qstr += 5;
		else
			qstr++;
	}
	return (0);
}

/*
 * Function:
 *	strdup_urldecode	- duplicate a string, decoding the URL-style
 *				  %XX escape sequences
 * Inputs:
 *	s			- the string to duplicate
 * Returns:
 *	A pointer to the duplicated string on success, NULL on error.
 * Modifies:
 *	Allocates memory, may write to stderr.
 */
static char *
strdup_urldecode(const char * const s, size_t len)
{
	const char *p;
	char *q, *t, *v, *eh;
	char h[3];
	long l;

	t = malloc(len + 1);
	if (t == NULL) {
		warnx("Out of memory");
		return (NULL);
	}

	p = s;
	q = t;
	while (len > 0) {
		v = memchr(p, '%', len);
		if (v == NULL) {
			memcpy(q, p, len);
			q += len;
			p += len;
			len = 0;
			break;
		} else if (v != p) {
			memcpy(q, p, v - p);
			q += v - p;
			len -= v - p;
			p = v;
		}
		p++;
		if (len < 2) {
			free(t);
			warnx("Not enough hex digits in a %%XX URL escape");
			return (NULL);
		}
		h[0] = p[0];
		h[1] = p[1];
		h[2] = '\0';
		l = strtol(h, &eh, 16);
		if (eh != h + 2) {
			free(t);
			warnx("Invalid hex string in a %%XX URL escape");
			return (NULL);
		} else if (l < 0 || l > 255) {
			free(t);
			warnx("Invalid hex value in a %%XX URL escape");
			return (NULL);
		}
		p += 2;
		*q++ = l;
		len -= 3;
	}
	*q++ = '\0';
	return (t);
}
