/**
 *    Copyright 2011 Peter Murray-Rust et. al.
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package org.xmlcml.cml.element;

import java.util.ArrayList;
import java.util.List;

import nu.xom.Element;
import nu.xom.Elements;
import nu.xom.Node;

import org.xmlcml.cml.base.CMLElement;
import org.xmlcml.cml.base.CMLElements;

/**
 * user-modifiable class supporting propertyList. * autogenerated from schema
 * use as a shell which can be edited
 *
 */
public class CMLPropertyList extends AbstractPropertyList {

	/** namespaced element name.*/
	public final static String NS = C_E+TAG;

    /**
     * constructor.
     */
    public CMLPropertyList() {
    }

    /**
     * constructor.
     *
     * @param old
     */
    public CMLPropertyList(CMLPropertyList old) {
        super((AbstractPropertyList) old);

    }

    /**
     * copy node .
     *
     * @return Node
     */
    public Element copy() {
        return new CMLPropertyList(this);

    }

    /**
     * create new instance in context of parent, overridable by subclasses.
     *
     * @param parent
     *            parent of element to be constructed (ignored by default)
     * @return CMLPropertyList
     */
    public CMLElement makeElementInContext(Element parent) {
        return new CMLPropertyList();

    }

    /**
     * gets all nested property descendants of arbitrary CMLElement.
     *
     * recurses through all propertyList and property children, adding the
     * latter to the list. Order is not preserved.
     *
     * @return list of all descendants
     */
    public List<CMLProperty> getPropertyDescendants() {
        List<CMLProperty> mList = new ArrayList<CMLProperty>();
        CMLElements<CMLPropertyList> propertyLists = this
                .getPropertyListElements();
        for (CMLPropertyList ml : propertyLists) {
            mList.addAll(ml.getPropertyDescendants());
        }
        CMLElements<CMLProperty> propertys = this.getPropertyElements();
        for (CMLProperty m : propertys) {
            mList.add(m);
        }
        return mList;
    }

    /**
     * gets all nested property descendants of arbitrary CMLElement.
     *
     * recurses through all propertyList and property children, adding the
     * latter to the list. Order is not preserved.
     *
     * @param parent
     *            with property(List) children
     * @return list of all descendants
     */
    public static List<CMLProperty> getPropertyDescendants(CMLElement parent) {
        List<CMLProperty> mList = new ArrayList<CMLProperty>();
        Elements propertyLists = parent
                .getChildCMLElements(CMLPropertyList.TAG);
        for (int i = 0; i < propertyLists.size(); i++) {
            CMLPropertyList propertyList = (CMLPropertyList) propertyLists
                    .get(i);
            mList.addAll(propertyList.getPropertyDescendants());
        }
        // generally property should be under propertyList parent
        Elements propertys = parent.getChildCMLElements(CMLProperty.TAG);
        for (int i = 0; i < propertys.size(); i++) {
            mList.add((CMLProperty) propertys.get(i));
        }
        return mList;
    }

    /**
     * gets property elements with a given dictRef. if either param is null,
     * returns empty list
     *
     * @param propertyList
     *            list to filter
     * @param dictRef
     *            value of dictRef attribute
     * @return filtered list
     */
    public static List<CMLProperty> getPropertyDescendantsByDictRef(
            List<CMLProperty> propertyList, String dictRef) {
        List<CMLProperty> newPropertyList = new ArrayList<CMLProperty>();
        if (dictRef != null && propertyList != null) {
            for (CMLProperty property : propertyList) {
                if (dictRef.equals(property.getDictRef())) {
                    newPropertyList.add(property);
                }
            }
        }
        return newPropertyList;
    }

    /**
     * gets property elements with a given dictRef. if dictRef is null returns
     * empty list
     *
     * @param dictRef
     *            value of dictRef attribute
     * @return filtered list
     */
    public List<CMLProperty> getPropertyDescendantsByName(String dictRef) {
        return getPropertyDescendantsByDictRef(this.getPropertyDescendants(),
                dictRef);
    }

}
