/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2012 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>

// #include "cdw_processwin.h"
#include "cdw_read_disc_info.h"
#include "cdw_disc.h"
#include "cdw_window.h"
#include "cdw_main_window.h"
#include "cdw_widgets.h"
// #include "cdw_drive.h"
// #include "cdw_fs.h"
#include "gettext.h"
#include "cdw_debug.h"


/* use to check if there is a disc in drive and if we can operate on it */
//static bool cdw_disc_is_disc_ready(void);

#if 0
static bool using_local_processwin = false;



/**
   \brief Re-read and display optical disc information, show it in text window if required

   Read meta-information form optical disc that is currently in drive
   and display summary in 'Disc info' area of ui.

   TODO: this function needs some review and improvement.

   \return CDW_OK if media info was read properly
   \return CDW_NO if media info was not read properly
*/
cdw_rv_t cdw_read_disc_info(void)
{
	const char *drive = cdw_drive_get_drive_fullpath();
	if (drive == (char *) NULL) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Can't find any drives, please check your hardware."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_NO;
	}

	if (cdw_processwin_is_active()) {
		using_local_processwin = false;
	} else {
		using_local_processwin = true;
		/* 2TRANS: this is title of process window; reading
		   disc meta information */
		cdw_processwin_create(_("Read disc information"),
				      /* 2TRANS: this is message in
					 process window */
				      _("Reading disc information"),
				      false);
	}

	cdw_disc_t *disc = cdw_disc_get__();
	cdw_main_ui_disc_info_view_display_data(disc);

	cdw_rv_t retval = CDW_OK;
	if (!cdw_drive_is_disc_in_drive()) {
		cdw_vdm ("INFO: no disc in drive\n");
		cdw_main_ui_disc_info_view_display_data(disc);
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Cannot show disc info. No disc in drive."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		retval = CDW_NO;
		goto cdw_read_disc_info_return;
	}

	drive = cdw_drive_get_drive_fullpath();
	cdw_rv_t mounted = cdw_fs_check_device_mounted(drive);
	if (mounted == CDW_ERROR) {
		cdw_vdm ("WARNING: checking if disc is mounted ended with error\n");
		/* let's try and proceed with further checks anyway */
	} else if (mounted == CDW_OK) { /* device is mounted */
		cdw_vdm ("INFO: optical disc is mounted\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Cannot check media. Please unmount your drive."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		retval = CDW_NO;
		goto cdw_read_disc_info_return;
	} else { /* mounted == CDW_NO, not mounted */
		cdw_vdm ("INFO: disc is not mounted\n");
	}


	/* get_meta_info() may fail only partially; display what you can */
	cdw_main_ui_disc_info_view_display_data(disc);
	if (!disc->has_data) {
		cdw_vdm ("ERROR: can't get disc meta info\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Cannot get media info. Consult log file ('L' hotkey in main window)."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		retval = CDW_NO;
		goto cdw_read_disc_info_return;
	} else { /* mediainfo == CDW_OK, we have proper media info */
		cdw_vdm ("INFO: disc meta info read properly\n");
	}


	/* at this point everything ends well: disc is in drive, it
	   is not mounted, we have read its meta information properly */
	retval = CDW_OK;

cdw_read_disc_info_return:
	if (using_local_processwin) {
		cdw_processwin_destroy("", false);
	}
	return retval;
}

#endif




#if 0 /* unused */

/**
   \brief Check if disc is in drive and if it is unmounted

   Check if disc is ready for further investigations and actions,
   i.e. if it is in drive at all and if it is unmounted. This is
   important when you want to read metadata from disc or when you
   want to burn to disc or erase disc (perhaps read disc as well).

   \return true when disc is ready, or when this is not certain, but user wants to continue
   \return false when disc is not ready, or when disc is probably not ready, but user don't want to continue
*/
bool cdw_disc_is_disc_ready(void)
{
	/* this function only checks if there is media in drive */
	if (!cdw_drive_is_disc_in_drive()) {
		/* cdw_dialogbox() was called by the function to
		   inform about problems */
		// current_disc.disc_type = CDW_DISC_NONE;
		// cdw_disc_resolve();
		return false;
	}

	const char *drive = cdw_drive_get_drive_fullpath();
	int tries = 5;
	for (int i = 0; i < tries; i++) {
		/* can't write to disc that is mounted */
		cdw_rv_t crv = cdw_fs_check_device_mounted(drive);
		if (crv == CDW_ERROR) {
			/* 2TRANS: this is title of dialog window */
			cdw_rv_t d = cdw_buttons_dialog(_("Error"),
							/* 2TRANS: this is message in dialog
							   window, user can select "OK" or
							   "Cancel" button */
							_("Cannot check if disc is unmounted. You should unmount your drive. Continue?"),
							CDW_BUTTONS_OK_CANCEL, CDW_COLORS_ERROR);
			if (d == CDW_OK) {
				/* disc may be mounted, but user wants to try anyway */
				return true;

			} else {
				/* disc may be mounted, and user don't want
				   to try to continue */
				return false;

			}
		} else if (crv == CDW_OK) { /* device is mounted */
			if (i == tries - 1) {
				/* 2TRANS: this is title of dialog window */
				cdw_buttons_dialog(_("Error"),
						   /* 2TRANS: this is message in dialog window: an
						      operation cannot be executed, because optical
						      disc is mounted in file system */
						   _("Cannot continue with mounted drive."),
						   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
				return false;
			}

			/* 2TRANS: this is title of dialog window */
			cdw_rv_t m = cdw_buttons_dialog(_("Error"),
							/* 2TRANS: this is message in dialog window: an
							   operation cannot be executed, because optical
							   disc is mounted in file system */
							_("Cannot continue. Please unmount your drive and press OK."),
							CDW_BUTTONS_OK_CANCEL, CDW_COLORS_ERROR);
			if (m == CDW_OK) {
				; /* try again */
			} else {
				return false;
			}
		} else { /* crv == CDW_NO, not mounted */
			return true;
		}
	}

	return false;
}
#endif




void cdw_read_and_display_disc_info(void)
{
	cdw_rv_t crv = CDW_OK;

	cdw_disc_t *disc = cdw_disc_new();
	if (disc) {
		crv = cdw_disc_get(disc);
	} else {
		crv = CDW_ERROR;
	}

	cdw_main_ui_disc_info_view_display_data(disc);

	int n_lines = 20;
	int n_cols = 70;
	int begin_y = (LINES - n_lines) / 2;
	int begin_x = (COLS - n_cols) / 2;
	WINDOW *window = cdw_window_new((WINDOW *) NULL,
					n_lines, n_cols,
					begin_y, begin_x,
					CDW_COLORS_DIALOG,
					/* 2TRANS: this is a title of a window */
					_("Disc info"),
					/* 2TRANS: this is message at the bottom of a window */
					_("Press any key to close window"));
	if (window == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: can't create window\n");
		return;
	}
	WINDOW *subwindow = cdw_window_new(window,
					   n_lines - 2, n_cols - 2, 1, 1,
					   CDW_COLORS_DIALOG,
					   (char *) NULL, (char *) NULL);
	if (subwindow == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: can't create window\n");
		delwin(window);
		window = (WINDOW *) NULL;
		return;
	}

	wrefresh(window);

	int col = 2;
	if (crv != CDW_OK) {
		/* 2TRANS: this is a message in dialog window */
		mvwprintw(subwindow, 1, col, _("Can't get disc information"));
	} else {
		/* 2TRANS: this is a message/label in dialog window: "disc information"
		   meta information read from/about an optical disc */
		mvwprintw(subwindow, 1, col, "Disc information:");
		/* 2TRANS: this is a label in dialog window, "%s" is for label
		   like "CD-R" or "DVD+RW" */
		mvwprintw(subwindow, 3, col, "Disc type: %s", disc->type_label);
		if (disc->state_empty == CDW_TRUE) {
			/* 2TRANS: this is a message in dialog window */
			mvwprintw(subwindow, 4, col, "Disc is empty");
		} else {
			/* 2TRANS: this is a label in dialog window, "%s" is
			   for label like "ISO 9660" */
			mvwprintw(subwindow, 4, col, "File system: %s", disc->cdio->ofs->type_label);
			/* 2TRANS: this is a label in dialog window, "%zd" is
			   for number of tracks on optical disc */
			mvwprintw(subwindow, 5, col, "Number of tracks: %zd", disc->cdio->n_tracks);

			/* 2TRANS: this is a message in dialog window,
			   "write to the disc" = "burn data to the disc";
			   this message will be followed by a message stating
			   if you can write to disc */
			char *label = _("Can write to the disc: ");
			mvwprintw(subwindow, 6, col, "%s", label);
			int c = (int) strlen(label);
			if (disc->type_writable == CDW_TRUE) {
				if (disc->state_writable == CDW_TRUE) {
					/* 2TRANS: this is a label in dialog
					   window, displayed after "Can write
					   to the disc: " label; "yes" as in
					   "yes, you can write to the disc" */
					mvwprintw(subwindow, 6, c + 2, _("yes"));
				} else if (disc->state_writable == CDW_FALSE) {
					/* 2TRANS: this is a label in dialog
					   window, displayed after "Can write
					   to the disc: "; "no" as in "no, you
					   can't write to the disc" */
					mvwprintw(subwindow, 6, c + 2, _("no, disc is closed"));
				} else {
					/* 2TRANS: this is a label in dialog
					   window, displayed after "Can write
					   to the disc: " label; "unknown" as
					   in "I don't know if you can write
					   to this disc" */
					mvwprintw(subwindow, 6, c + 2, _("unknown, probably no"));
				}
			} else if (disc->type_writable == CDW_FALSE) {
				/* 2TRANS: this is message in dialog window,
				   displayed after "Can write to the disc: "
				   label; it means "no, you can't write to
				   this disc because it is 'read only' type
				   of disc" */
				mvwprintw(subwindow, 6, c + 2, _("no, disc type is \"read only\""));
			} else {
				/* 2TRANS: this is message in dialog window,
				   displayed after "Can write to the disc: "
				   label; it means "I don't know if you can
				   write to this disc" */
				mvwprintw(subwindow, 6, c + 2, _("unknown"));
			}


			/* 2TRANS: this is a label in dialog window,
			   it will be followed by a string describing
			   if a disc can be erased or no */
			label = _("Can erase this disc: ");
			c = (int) strlen(label);
			mvwprintw(subwindow, 7, col, "%s", label);

			if (disc->type_erasable == CDW_TRUE) {
				/* 2TRANS: this is a label in dialog window,
				   "yes" as in "yes, you can erase the disc" */
				mvwprintw(subwindow, 7, c + 2, _("yes"));
			} else if (disc->type_erasable == CDW_FALSE) {
				/* 2TRANS: this is a label in dialog window,
				   "no" as in "no, you can't erase the disc" */
				mvwprintw(subwindow, 7, c + 2, _("no"));
			} else {
				/* 2TRANS: this is a label in dialog
				   window, displayed after "Can erase this
				   disc: " label; "unknown" as in "I don't
				   know if you can erase this disc" */
				mvwprintw(subwindow, 7, c + 2, _("unknown, probably no"));
			}


			/* 2TRANS: this is a label in dialog window, it will
			   be followed by string representing volume ID
			   (a disc label) */
			mvwprintw(subwindow, 9, col, _("Volume id:"));
			mvwprintw(subwindow, 10, col, "\"%s\"", disc->cdio->ofs->volume_id);
		}
	}

	cdw_disc_delete(&disc);

	wrefresh(subwindow);
	wgetch(subwindow);

	delwin(subwindow);
	subwindow = (WINDOW *) NULL;
	delwin(window);
	window = (WINDOW *) NULL;

	return;

}
