#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sstream>

#include "opensslutil.h"
#include "canlxx.h"

namespace AuthN {

using namespace AuthN::OpenSSL;

/*
  static std::string tostring(unsigned int val) {
    std::stringstream ss;
    ss << val;
    return ss.str();
  }
*/

  CredentialsRequest::CredentialsRequest(void) :
      context_(NULL), csrctx_(NULL), keyctx_(NULL), valid_from_(-1), valid_till_(-1) {
  }

  CredentialsRequest::CredentialsRequest(const CredentialsRequest& arg) :
      context_(NULL), csrctx_(NULL), keyctx_(NULL), valid_from_(-1), valid_till_(-1) {
    if(arg.context_) context_ = &arg.context_->Copy();
    Assign(arg);
  }

  CredentialsRequest& CredentialsRequest::operator=(const CredentialsRequest& arg) {
    Assign(arg);
    return *this;
  }

  Status CredentialsRequest::Assign(const CredentialsRequest& arg) {
    if(!arg.csrctx_) return (last_error_ = Status(-1, "Failed to find CertContext object"));
    if(!arg.keyctx_) return (last_error_ = Status(-1, "Failed to find KeyContext object"));
    std::string req, key;
    arg.csrctx_->csrToPEM(req);
    arg.keyctx_->privateToPEM(key);

    return AssignRequest(req, key);
  }

  CredentialsRequest::CredentialsRequest(const AuthN::Context& ctx) : csrctx_(NULL), keyctx_(NULL),
      valid_from_(-1), valid_till_(-1) {
    context_ = &ctx.Copy();
    csrctx_ = new AuthN::OpenSSL::CSRContext(*context_);
    keyctx_ = new AuthN::OpenSSL::KeyContext(*context_);
  }

  CredentialsRequest::~CredentialsRequest() {
    if(context_) delete context_;
    if(csrctx_) delete csrctx_; 
    if(keyctx_) delete keyctx_;
  }

  CredentialsRequest& CredentialsRequest::Copy(void) const {
    CredentialsRequest* req = new CredentialsRequest(context_?*context_:Context());
    std::string req_str, priv_str;
    GetRequest(req_str);
    GetPrivateKey(priv_str);
    req->AssignRequest(req_str, priv_str);
    return *req;
  }

  Status CredentialsRequest::GetStatus(void) const {
    return last_error_;
  }

  AuthN::Status CredentialsRequest::MakeKeys(int bits) {
    if(!keyctx_) return (last_error_ = Status(-1, "Failed to find KeyContext object"));
    return (keyctx_->createKey(bits));
  }

  Status CredentialsRequest::GetPrivateKey(std::string& str, bool encrypt) const{
    if(!keyctx_) return Status(-1, "KeyContext is empty");
    keyctx_->privateToPEM(str, encrypt);
    return Status(0); //TODO: return more status;
  }

  Status CredentialsRequest::GetPrivateKey(std::ostream& o, bool encrypt) const{
    std::string s;
    Status err = GetPrivateKey(s,encrypt);
    o << s;
    return err;
  }

  Status CredentialsRequest::GetPublicKey(std::string& str) const{
    if(!keyctx_) return Status(-1, "KeyContext is empty");
    keyctx_->publicToPEM(str);
    return Status(0); //TODO: return more status;
  }

  Status CredentialsRequest::GetPublicKey(std::ostream& o) const{
    std::string s;
    Status err = GetPublicKey(s);
    o << s;
    return err;
  }

  AuthN::Status CredentialsRequest::MakeRequest(void) {
    if(!csrctx_) return (last_error_ = Status(-1, "Failed to find CSRContext object"));
    if(!keyctx_) return (last_error_ = Status(-1, "Failed to find KeyContext object"));

    AuthN::OpenSSL::CertificateOptions req_opts;
    std::string sub_str = convert_rfc2253_to_ossdn(subject_name_);
    req_opts.setSubject(sub_str);
    req_opts.setAsUser();
    AuthN::Utils::Time start, end;
    if(valid_from_ != -1) start = valid_from_;
    else start = AuthN::Utils::Time();
    if(valid_till_ != -1) end = valid_till_;
    if(start < end)req_opts.setValidityPeriod(start, end);

    if(!(csrctx_->createRequest(req_opts, *keyctx_))) 
      return (last_error_ = Status(-1, "Failed to create X509 request"));
    return Status(0);
  }

  //const X509_REQ* CredentialsRequest::GetRequest(void) const {
  //  return req_;
  //}

  Status CredentialsRequest::GetRequest(std::string& str) const {
    if(!csrctx_) return Status(-1, "Failed to find CSRContext object"); 
    csrctx_->csrToPEM(str);
    return Status(0);
  }

  Status CredentialsRequest::GetRequest(std::ostream& o) const {
    std::string s;
    Status err = GetRequest(s);
    o << s;
    return err;
  }

  AuthN::Status CredentialsRequest::AssignKeys(std::istream& priv) {
    std::string priv_str;

    std::getline<char>(priv, priv_str, 0);
    if (!priv) return Status(-1);
    return(AssignKeys(priv_str));
  }

  AuthN::Status CredentialsRequest::AssignKeys(const std::string& priv) {
    if(!keyctx_) return (last_error_ = Status(-1, "Failed to find KeyContext object"));
    if(!(keyctx_->privateFromStr(priv))) return (last_error_ = Status(-1, "Failed to assign private key"));
    return Status(0);
  }

  AuthN::Status CredentialsRequest::AssignRequest(std::istream& req, std::istream& priv) {
    std::string req_str;
    std::string priv_str;

    std::getline<char>(req, req_str, 0);
    if (!req) return Status(-1);

    std::getline<char>(priv, priv_str, 0);
    if (!priv) return Status(-1);
    return(AssignRequest(req_str, priv_str));
  }

  AuthN::Status CredentialsRequest::AssignRequest(const std::string& req, const std::string& priv) {
    if(!csrctx_) return (last_error_ = Status(-1, "Failed to find CertContext object"));
    if(!keyctx_) return (last_error_ = Status(-1, "Failed to find KeyContext object"));
    if(!(csrctx_->csrFromPEM(req))) return (last_error_ = Status(-1, "Failed to load CSR"));
    if(!priv.empty() && !(keyctx_->privateFromStr(priv))) return (last_error_ = Status(-1, "Failed to load private key"));
    subject_name_ = convert_ossdn_to_rfc2253(csrctx_->getProps()->subject);
    valid_from_ = csrctx_->getProps()->start.GetTime();
    valid_till_ = csrctx_->getProps()->end.GetTime();
    return Status(0);
  }

  AuthN::Status CredentialsRequest::SetSubjectName(const std::string sn) {
    if(sn.find("/") != std::string::npos)
      subject_name_ = convert_ossdn_to_rfc2253(sn);
    else subject_name_ = sn;
    return Status(0);
  }

  std::string CredentialsRequest::GetSubjectName(void) const {
    return subject_name_;
  }

  time_t CredentialsRequest::GetValidFrom(void) const {
    return valid_from_;
  }

  void CredentialsRequest::SetValidFrom(time_t sec) {
    valid_from_ = sec;
  }

  time_t CredentialsRequest::GetValidTill(void) const {
    return valid_till_;
  }

  void CredentialsRequest::SetValidTill(time_t sec) {
    valid_till_ = sec;
  }

  Status CredentialsRequest::GetExtension(int pos, Credentials::Extension& ext) const {
    if(!csrctx_) return Status(-1);
    return csrctx_->getCSRExtension(pos, ext);
  }

  Status CredentialsRequest::GetExtension(const std::string& name, Credentials::Extension& ext) const {
    if(!csrctx_) return Status(-1);
    return csrctx_->getCSRExtension(name, ext);
  }

  Status CredentialsRequest::AddExtension(Credentials::Extension& ext) {
	Status status;
	if(!csrctx_) return Status(-1);
    // Check if the private key is contained
    if(keyctx_->isPrivate()) {
      if(csrctx_->isSigned())
        status = csrctx_->setCSRExtension(ext, *keyctx_);
      else status = csrctx_->setCSRExtension(ext);
    }
    else status = csrctx_->setCSRExtension(ext);
    return status;
  }

  Status CredentialsRequest::GetAttributes(const std::string& name, 
    std::list<std::string>& attrs) const {
    Status status;
    status = csrctx_->getCSRAttribute(name, attrs);
    return status;
  }

  Status CredentialsRequest::AddAttributes(const std::string& name, 
      const std::string& attrs) {
    Status status;
    if(!csrctx_) return Status(-1);
    // Check if the private key is contained
    if(keyctx_->isPrivate()) {
      if(csrctx_->isSigned())
        status = csrctx_->setCSRAttribute(name, attrs, *keyctx_);
      else status = csrctx_->setCSRAttribute(name, attrs);
    }
    else status = csrctx_->setCSRAttribute(name, attrs);
    return status;
  }

  CACredentialsRequest::CACredentialsRequest(const AuthN::Context& ctx) : CredentialsRequest(ctx), certctx_(NULL) {
  }

  CACredentialsRequest::~CACredentialsRequest(void) {
    if(certctx_) delete certctx_;
  }

  CredentialsRequest& CACredentialsRequest::Copy(void) const {
    CACredentialsRequest* req = new CACredentialsRequest(context_?*context_:Context());
    std::string req_str, priv_key_str;
    GetRequest(req_str);
    GetPrivateKey(priv_key_str);
    req->CredentialsRequest::AssignRequest(req_str, priv_key_str);
    std::string cert;
    if(certctx_) { certctx_->certToPEM(cert); req->Assign(cert); }
    return *req;
  }

  AuthN::Status CACredentialsRequest::MakeRequest(void) {
    if(!csrctx_) {
      context_->Log(Context::LogError,"Failed to find CSRContext object");
      return (last_error_ = Status(-1, "Failed to find CSRContext object"));
    }
    if(!keyctx_) {
      context_->Log(Context::LogError,"Failed to find KeyContext object");
      return (last_error_ = Status(-1, "Failed to find KeyContext object"));
    }

    AuthN::OpenSSL::CertificateOptions req_opts;
    std::string sub_str = convert_rfc2253_to_ossdn(subject_name_);
    req_opts.setSubject(sub_str);
    req_opts.setAsCA();
    AuthN::Utils::Time start, end;
    if(valid_from_ != -1) start = valid_from_;
    else start = AuthN::Utils::Time();
    if(valid_till_ != -1) end = valid_till_;
    if(start < end)req_opts.setValidityPeriod(start, end);

    if(!(csrctx_->createRequest(req_opts, *keyctx_))) {
      context_->Log(Context::LogError,"Failed to create X509 request");
      return (last_error_ = Status(-1, "Failed to create X509 request"));
    }
    return Status(0);

  }

  Status CACredentialsRequest::GetCertificate(std::string& str) {
    if(!certctx_) return (last_error_ = Status(-1, "The CertContext if empty"));
    certctx_->certToPEM(str);
    if(str.empty()) return (last_error_ = Status(-1,"The X509 certificate is empty"));
    return Status(0);
  }

  Status CACredentialsRequest::GetCertificate(std::ostream& o) {
    std::string s;
    Status err = GetCertificate(s);
    o << s;
    return err;
  }

  AuthN::Status CACredentialsRequest::Assign(const std::string& cert) {
    if(!certctx_) return (last_error_ = Status(-1, "Failed to find CertContext object"));
    if(!(certctx_->certFromPEM(cert))) return (last_error_ = Status(-1, "Failed to load X509 Certificate"));
    return Status(0);     
  }

  ProxyCredentialsRequest::ProxyCredentialsRequest(const AuthN::Context& ctx) :
      CredentialsRequest(ctx), limited_(false) { };

  ProxyCredentialsRequest::~ProxyCredentialsRequest(void) { };

  CredentialsRequest& ProxyCredentialsRequest::Copy(void) const {
    ProxyCredentialsRequest* req = new ProxyCredentialsRequest(context_?*context_:Context());
    std::string req_str, priv_key_str;
    GetRequest(req_str);
    GetPrivateKey(priv_key_str);
    req->CredentialsRequest::AssignRequest(req_str, priv_key_str);
    return *req;
  }

  bool ProxyCredentialsRequest::GetLimited(void) {
    return limited_;
  }

  void ProxyCredentialsRequest::SetLimited(bool limited) {
    limited_ = limited;
  }

  Status ProxyCredentialsRequest::GetPolicy(Credentials::Extension& policy) {
    const CertContextProps* props = csrctx_->getProps();
    policy.oid = props->proxyPolicy.oid;
    policy.critical = props->proxyPolicy.critical;
    policy.value = props->proxyPolicy.value;
    return policy.oid.empty()?false:true;

  }

  Status ProxyCredentialsRequest::SetPolicy(const Credentials::Extension& policy) {
    // The method could cover two cases: 
    // 1. On the request side, SetPolicy should be called before MakeRequest, 
    //    so that the policy will be insert into X509_REQ
    // 2. On the signer side, the call of SetPolicy will only cause the change 
    //    of CertContextProps.
    CertContextProps* props = &(csrctx_->props_);
    props->proxyPolicy.value = policy.value;
     
    return Status(0);
  }

}//AuthN


