//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Numeric/DoubleSpinBox.h
//! @brief     Defines class DoubleSpinBox
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_VIEW_NUMERIC_DOUBLESPINBOX_H
#define BORNAGAIN_GUI_VIEW_NUMERIC_DOUBLESPINBOX_H
#include "GUI/Model/Descriptor/DoubleProperty.h"
#include <QDoubleSpinBox>

//! SpinBox for DoubleProperties, supporting units.
class DoubleSpinBox : public QDoubleSpinBox {
    Q_OBJECT
public:
    //! Create a DoubleSpinBox with the information found in a DoubleProperty.
    //!
    //! The spin box will be fully initialized (tooltip, limits, unit, current value, size
    //! policy). Furthermore, the spin box will prohibit accidental changes by the mouse wheel.
    //! Otherwise it would be dangerous if the spin box is on a scrollable form - unintended and
    //! unnoticed changes would take place when just scrolling through the form.
    DoubleSpinBox(DoubleProperty& d, bool easyScrollable = false, QWidget* parent = nullptr);
    //! Set a display unit.
    //!
    //! The caller has to make sure that the new display unit has a conversion to/from the contained
    //! base value unit.
    void setDisplayUnit(Unit displayUnit);

    //! Set the base value (unit is the one of the contained descriptor).
    void setBaseValue(double baseValue);
    //! The display unit as human readable string.
    QString displayUnitAsString() const;

    QString uid() const;

    void setPropertyValue(double v);

    //! Returns the unit of the contained DoubleProperty.
    //!
    //! If the unit is defined as a string, this method returns Unit::other. To get the string, use
    //! valueProperty().unit
    Unit baseUnit() const;

    //! Update the shown value to the one contained in the value descriptor.
    //!
    //! No signal will be emitted if the new value has changed.
    void updateValue();

signals:
    //! Emitted whenever the value changes.
    //!
    //! newBaseValue is in the unit of the valueProperty.
    void baseValueChanged(double newBaseValue);

protected:
    void wheelEvent(QWheelEvent* event) override;

private:
    void onDisplayValueChanged(double newDisplayValue);

    double toDisplayValue(double baseValue) const;
    double toBaseValue(double displayValue) const;

    using QDoubleSpinBox::setValue; // To hide from usage

private:
    Unit m_displayUnit = Unit::unitless;

    DoubleProperty& m_valueProperty;
    bool m_easyScrollable;

    //! it was decided to not show the unit as a suffix. However, this may be user
    //! selectable once, therefore the code is kept and controlled by this flag
    bool m_showUnitAsSuffix = false;
};


#endif // BORNAGAIN_GUI_VIEW_NUMERIC_DOUBLESPINBOX_H
