//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Fit/FitParameterWidget.cpp
//! @brief     Implements class FitParameterWidget
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Fit/FitParameterWidget.h"
#include "Base/Util/Assert.h"
#include "GUI/Model/Job/FitParameterContainerItem.h"
#include "GUI/Model/Job/ParameterTreeItems.h"
#include "GUI/Model/Model/FitParameterModel.h"
#include "GUI/Model/Project/ProjectDocument.h"
#include "GUI/Support/Util/CustomEventFilters.h"
#include "GUI/View/Fit/FitParameterDelegate.h"
#include "GUI/View/Fit/ParameterTuningWidget.h"
#include "GUI/View/Info/OverlayLabelController.h"
#include <QAction>
#include <QHeaderView>
#include <QMenu>
#include <QTreeView>
#include <QVBoxLayout>
#include <boost/polymorphic_cast.hpp>


FitParameterWidget::FitParameterWidget(QWidget* parent)
    : DataAccessWidget(parent)
    , m_treeView(new QTreeView)
    , m_tuningWidget(nullptr)
    , m_createFitParAction(nullptr)
    , m_removeFromFitParAction(nullptr)
    , m_removeFitParAction(nullptr)
    , m_keyboardFilter(new DeleteEventFilter(this))
    , m_infoLabel(new OverlayLabelController(this))
{
    auto* layout = new QVBoxLayout(this);
    layout->addWidget(m_treeView);
    layout->setContentsMargins(0, 0, 0, 0);
    layout->setSpacing(0);
    init_actions();

    m_treeView->setSelectionMode(QAbstractItemView::ExtendedSelection);
    m_treeView->setSelectionBehavior(QAbstractItemView::SelectRows);
    m_treeView->setContextMenuPolicy(Qt::CustomContextMenu);
    m_treeView->setItemDelegate(new FitParameterDelegate(this));
    m_treeView->setDragEnabled(true);
    m_treeView->setDragDropMode(QAbstractItemView::DragDrop);
    m_treeView->installEventFilter(m_keyboardFilter);
    m_treeView->setAlternatingRowColors(true);
    m_treeView->setStyleSheet("alternate-background-color: #EFF0F1;");
    m_treeView->header()->setSectionResizeMode(QHeaderView::Stretch);

    connect(m_treeView, &QTreeView::customContextMenuRequested, this,
            &FitParameterWidget::onFitParameterTreeContextMenu);

    m_infoLabel->setArea(m_treeView);
    m_infoLabel->setText("Drop parameter(s) to fit here");
}

void FitParameterWidget::setJobOrRealItem(QObject* job_item)
{
    DataAccessWidget::setJobOrRealItem(job_item);
    ASSERT(jobItem());

    init_fit_model();
}

//! Sets ParameterTuningWidget to be able to provide it with context menu and steer
//! its behaviour in the course of fit settings or fit running

void FitParameterWidget::setParameterTuningWidget(ParameterTuningWidget* tuningWidget)
{
    if (tuningWidget == m_tuningWidget)
        return;

    if (m_tuningWidget)
        disconnect(m_tuningWidget, &ParameterTuningWidget::itemContextMenuRequest, this,
                   &FitParameterWidget::onTuningWidgetContextMenu);

    m_tuningWidget = tuningWidget;
    if (!m_tuningWidget)
        return;

    connect(m_tuningWidget, &ParameterTuningWidget::itemContextMenuRequest, this,
            &FitParameterWidget::onTuningWidgetContextMenu, Qt::UniqueConnection);
    connect(tuningWidget, &QObject::destroyed, [this] { m_tuningWidget = nullptr; });
}

//! Creates context menu for ParameterTuningWidget

void FitParameterWidget::onTuningWidgetContextMenu(const QPoint& point)
{
    QMenu menu;

    if (jobItem()->status() == JobStatus::Fitting) {
        setActionsEnabled(false);
        return;
    }

    m_removeFromFitParAction->setEnabled(canRemoveFromFitParameters());
    m_createFitParAction->setEnabled(canCreateFitParameter());

    menu.addAction(m_createFitParAction);
    QMenu* addToFitParMenu = menu.addMenu("Add to existing fit parameter");
    addToFitParMenu->setEnabled(true);

    const bool allow_one_fit_parameter_to_have_more_than_one_link = true;
    if (allow_one_fit_parameter_to_have_more_than_one_link) {
        QStringList fitParNames = fitContainerItem()->fitParameterNames();
        if (fitParNames.isEmpty() || !canCreateFitParameter())
            addToFitParMenu->setEnabled(false);
        for (int i = 0; i < fitParNames.count(); ++i) {
            auto* action = new QAction(QString("to ").append(fitParNames.at(i)), addToFitParMenu);
            connect(action, &QAction::triggered, [this, i] { onAddToFitParAction(i); });
            addToFitParMenu->addAction(action);
        }
    }
    menu.addSeparator();
    menu.addAction(m_removeFromFitParAction);

    menu.exec(point);
    setActionsEnabled(true);
}

//! Creates context menu for the tree with fit parameters

void FitParameterWidget::onFitParameterTreeContextMenu(const QPoint& point)
{
    if (jobItem()->status() == JobStatus::Fitting) {
        setActionsEnabled(false);
        return;
    }
    if (fitContainerItem()->isEmpty())
        return;

    QMenu menu;
    menu.addAction(m_removeFitParAction);
    menu.exec(m_treeView->viewport()->mapToGlobal(point));
    setActionsEnabled(true);
}

//! Propagates selection form the tree with fit parameters to the tuning widget

void FitParameterWidget::onFitParametersSelectionChanged(const QItemSelection& selection)
{
    if (selection.indexes().isEmpty())
        return;

    for (auto index : selection.indexes()) {
        m_tuningWidget->selectionModel()->clearSelection();
        QObject* item = m_fitParameterModel->itemForIndex(index);
        if (auto* fitLinkItem = dynamic_cast<FitParameterLinkItem*>(item->parent())) {
            QString link = fitLinkItem->link();
            m_tuningWidget->makeSelected(
                jobItem()->parameterContainerItem()->findParameterItem(link));
        }
    }
}

//! Creates fit parameters for all selected ParameterItem's in tuning widget

void FitParameterWidget::onCreateFitParAction()
{
    for (auto* item : m_tuningWidget->selectedParameterItems())
        if (!fitContainerItem()->fitParameterItem(item))
            fitContainerItem()->createFitParameter(item);
}

//! All ParameterItem's selected in tuning widget will be removed from link section of
//! corresponding fitParameterItem.

void FitParameterWidget::onRemoveFromFitParAction()
{
    for (auto* item : m_tuningWidget->selectedParameterItems())
        fitContainerItem()->removeLink(item);

    for (auto* item : emptyFitParameterItems())
        fitContainerItem()->removeFitParameter(item);

    emit fitContainerItem()->fitItemChanged();
}

//! All selected FitParameterItem's of FitParameterItemLink's will be removed

void FitParameterWidget::onRemoveFitParAction()
{
    // retrieve both, selected FitParameterItem and FitParameterItemLink
    QStringList linksToRemove = selectedFitParameterLinks();
    QVector<FitParameterItem*> itemsToRemove = selectedFitParameterItems();

    for (const auto& link : linksToRemove)
        for (auto* fitParItem : fitContainerItem()->fitParameterItems())
            fitParItem->removeLink(link);

    // remove parameters that lost their links
    itemsToRemove = itemsToRemove + emptyFitParameterItems();

    for (auto* item : itemsToRemove)
        fitContainerItem()->removeFitParameter(item);

    emit fitContainerItem()->fitItemChanged();
}

//! Add all selected parameters to fitParameter with given index

void FitParameterWidget::onAddToFitParAction(int ipar)
{
    const QString fitParName = fitContainerItem()->fitParameterNames().at(ipar);
    for (auto* item : m_tuningWidget->selectedParameterItems())
        fitContainerItem()->addToFitParameter(item, fitParName);
}

void FitParameterWidget::onFitParameterModelChange()
{
    spanParameters();
    updateInfoLabel();
}

void FitParameterWidget::init_actions()
{
    m_createFitParAction = new QAction("Create fit parameter", this);
    connect(m_createFitParAction, &QAction::triggered, this,
            &FitParameterWidget::onCreateFitParAction);

    m_removeFromFitParAction = new QAction("Remove from fit parameters", this);
    connect(m_removeFromFitParAction, &QAction::triggered, this,
            &FitParameterWidget::onRemoveFromFitParAction);

    m_removeFitParAction = new QAction("Remove fit parameter", this);
    connect(m_removeFitParAction, &QAction::triggered, this,
            &FitParameterWidget::onRemoveFitParAction);

    connect(m_keyboardFilter, &DeleteEventFilter::removeItem, this,
            &FitParameterWidget::onRemoveFitParAction);
}

//! Initializes FitParameterModel and its tree.

void FitParameterWidget::init_fit_model()
{
    m_treeView->setModel(nullptr);

    m_fitParameterModel.reset(new FitParameterModel(fitContainerItem(), jobItem()));
    m_treeView->setModel(m_fitParameterModel.get());

    connect(m_fitParameterModel.get(), &FitParameterModel::dataChanged, this,
            &FitParameterWidget::onFitParameterModelChange, Qt::UniqueConnection);
    connect(m_fitParameterModel.get(), &FitParameterModel::modelReset, this,
            &FitParameterWidget::onFitParameterModelChange, Qt::UniqueConnection);

    connect(fitContainerItem(), &FitParameterContainerItem::fitItemChanged,
            gProjectDocument.value(), &ProjectDocument::setModified, Qt::UniqueConnection);

    onFitParameterModelChange();
    connectFitParametersSelection(true);
}

FitParameterContainerItem* FitParameterWidget::fitContainerItem() const
{
    return jobItem()->fitParameterContainerItem();
}

//! Returns true if tuning widget contains selected ParameterItem's which can be used to create
//! a fit parameter (i.e. it is not linked with some fit parameter already).

bool FitParameterWidget::canCreateFitParameter()
{
    QVector<ParameterItem*> selected = m_tuningWidget->selectedParameterItems();
    for (auto* item : selected) {
        if (fitContainerItem()->fitParameterItem(item) == nullptr)
            return true;
    }
    return false;
}

//! Returns true if tuning widget contains selected ParameterItem's which can be removed from
//! fit parameters.

bool FitParameterWidget::canRemoveFromFitParameters()
{
    QVector<ParameterItem*> selected = m_tuningWidget->selectedParameterItems();
    for (auto* item : selected) {
        if (fitContainerItem()->fitParameterItem(item))
            return true;
    }
    return false;
}

//! Enables/disables all context menu actions.

void FitParameterWidget::setActionsEnabled(bool value)
{
    m_createFitParAction->setEnabled(value);
    m_removeFromFitParAction->setEnabled(value);
    m_removeFitParAction->setEnabled(value);
}

//! Returns list of FitParameterItem's currently selected in FitParameterItem tree

QVector<FitParameterItem*> FitParameterWidget::selectedFitParameterItems()
{
    QVector<FitParameterItem*> result;
    QModelIndexList indexes = m_treeView->selectionModel()->selectedIndexes();
    for (auto index : indexes) {
        QObject* item = static_cast<QObject*>(index.internalPointer());
        if (auto* fitParItem = dynamic_cast<FitParameterItem*>(item))
            result.push_back(fitParItem);
    }
    return result;
}

//! Returns list of FitParameterItem's which doesn't have any links attached.

QVector<FitParameterItem*> FitParameterWidget::emptyFitParameterItems()
{
    QVector<FitParameterItem*> result;
    for (auto* fitParItem : fitContainerItem()->fitParameterItems())
        if (fitParItem->linkItems().empty())
            result.push_back(fitParItem);

    return result;
}

//! Returns links of FitParameterLink's item selected in FitParameterItem tree

QStringList FitParameterWidget::selectedFitParameterLinks()
{
    QStringList result;
    QModelIndexList indexes = m_treeView->selectionModel()->selectedIndexes();
    for (QModelIndex index : indexes) {
        QObject* item = static_cast<QObject*>(index.internalPointer());
        if (auto* linkItem = dynamic_cast<LinkItem*>(item))
            result.push_back(linkItem->link());
    }
    return result;
}

//! Makes first column in FitParameterItem's tree related to ParameterItem link occupy whole space.

void FitParameterWidget::spanParameters()
{
    m_treeView->expandAll();
    for (int i = 0; i < m_fitParameterModel->rowCount(QModelIndex()); i++) {
        QModelIndex parameter = m_fitParameterModel->index(i, 0, QModelIndex());
        if (!parameter.isValid())
            break;
        int childRowCount = m_fitParameterModel->rowCount(parameter);
        if (childRowCount > 0) {
            for (int j = 0; j < childRowCount; j++)
                m_treeView->setFirstColumnSpanned(j, parameter, true);
        }
    }
}

//! Places overlay label on top of tree view, if there is no fit parameters
void FitParameterWidget::updateInfoLabel()
{
    if (!jobItem())
        return;

    m_infoLabel->setShown(fitContainerItem()->isEmpty());
}

void FitParameterWidget::connectFitParametersSelection(bool active)
{
    if (active) {
        connect(m_treeView->selectionModel(), &QItemSelectionModel::selectionChanged, this,
                &FitParameterWidget::onFitParametersSelectionChanged, Qt::UniqueConnection);
    } else {
        disconnect(m_treeView->selectionModel(), &QItemSelectionModel::selectionChanged, this,
                   &FitParameterWidget::onFitParametersSelectionChanged);
    }
}
