//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Support/Data/SimulationOptionsItem.cpp
//! @brief     Defines class SimulationOptionsItem
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Support/Data/SimulationOptionsItem.h"
#include "GUI/Support/XML/DeserializationException.h"
#include "GUI/Support/XML/UtilXML.h"
#include <thread>

namespace {
namespace Tag {

const QString RunImmediately("RunImmediately");
const QString NumberOfThreads("NumberOfThreads");
const QString Analytical("Analytical");
const QString NumberOfMonteCarloPoints("NumberOfMonteCarloPoints");
const QString UseAverageMaterials("UseAverageMaterials");
const QString IncludeSpecularPeak("IncludeSpecularPeak");
const QString SelectedInstrumentIndex("SelectedInstrumentIndex");
const QString SelectedSampleIndex("SelectedSampleIndex");
const QString SelectedDataIndex("SelectedDataIndex");

} // namespace Tag
} // namespace


SimulationOptionsItem::SimulationOptionsItem()
{
    m_runImmediately = true;
    m_numberOfThreads = std::thread::hardware_concurrency();
    m_computationMethodAnalytical = true;
    m_numberOfMonteCarloPoints = 100;
    m_useAverageMaterials = false;
    m_includeSpecularPeak = false;
    m_selectedInstrumentIndex = -1;
    m_selectedSampleIndex = -1;
    m_selectedDataIndex = -1;
}

bool SimulationOptionsItem::useMonteCarloIntegration() const
{
    return !useAnalytical();
}

void SimulationOptionsItem::setUseMonteCarloIntegration(unsigned numberOfPoints)
{
    m_computationMethodAnalytical = false;
    m_numberOfMonteCarloPoints = numberOfPoints;
}

void SimulationOptionsItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // run immediately?
    w->writeStartElement(Tag::RunImmediately);
    XML::writeAttribute(w, XML::Attrib::value, m_runImmediately);
    w->writeEndElement();

    // number of threads
    w->writeStartElement(Tag::NumberOfThreads);
    XML::writeAttribute(w, XML::Attrib::value, m_numberOfThreads);
    w->writeEndElement();

    // computation method
    w->writeStartElement(Tag::Analytical);
    XML::writeAttribute(w, XML::Attrib::value, m_computationMethodAnalytical);
    w->writeEndElement();

    // number of Monte Carlo points
    w->writeStartElement(Tag::NumberOfMonteCarloPoints);
    XML::writeAttribute(w, XML::Attrib::value, m_numberOfMonteCarloPoints);
    w->writeEndElement();

    // use average materials
    w->writeStartElement(Tag::UseAverageMaterials);
    XML::writeAttribute(w, XML::Attrib::value, m_useAverageMaterials);
    w->writeEndElement();

    // include specular peak
    w->writeStartElement(Tag::IncludeSpecularPeak);
    XML::writeAttribute(w, XML::Attrib::value, m_includeSpecularPeak);
    w->writeEndElement();

    // selected instrument index
    w->writeStartElement(Tag::SelectedInstrumentIndex);
    XML::writeAttribute(w, XML::Attrib::value, m_selectedInstrumentIndex);
    w->writeEndElement();

    // selected sample index
    w->writeStartElement(Tag::SelectedSampleIndex);
    XML::writeAttribute(w, XML::Attrib::value, m_selectedSampleIndex);
    w->writeEndElement();

    // selected data index
    w->writeStartElement(Tag::SelectedDataIndex);
    XML::writeAttribute(w, XML::Attrib::value, m_selectedDataIndex);
    w->writeEndElement();
}

void SimulationOptionsItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // run immediately?
        if (tag == Tag::RunImmediately) {
            XML::readAttribute(r, XML::Attrib::value, &m_runImmediately);
            XML::gotoEndElementOfTag(r, tag);

            // number of threads
        } else if (tag == Tag::NumberOfThreads) {
            XML::readAttribute(r, XML::Attrib::value, &m_numberOfThreads);
            XML::gotoEndElementOfTag(r, tag);

            // computation method
        } else if (tag == Tag::Analytical) {
            XML::readAttribute(r, XML::Attrib::value, &m_computationMethodAnalytical);
            XML::gotoEndElementOfTag(r, tag);

            // number of Monte Carlo points
        } else if (tag == Tag::NumberOfMonteCarloPoints) {
            XML::readAttribute(r, XML::Attrib::value, &m_numberOfMonteCarloPoints);
            XML::gotoEndElementOfTag(r, tag);

            // use average materials
        } else if (tag == Tag::UseAverageMaterials) {
            XML::readAttribute(r, XML::Attrib::value, &m_useAverageMaterials);
            XML::gotoEndElementOfTag(r, tag);

            // include specular peak
        } else if (tag == Tag::IncludeSpecularPeak) {
            XML::readAttribute(r, XML::Attrib::value, &m_includeSpecularPeak);
            XML::gotoEndElementOfTag(r, tag);

            // selected instrument index
        } else if (tag == Tag::SelectedInstrumentIndex) {
            XML::readAttribute(r, XML::Attrib::value, &m_selectedInstrumentIndex);
            XML::gotoEndElementOfTag(r, tag);

            // selected sample index
        } else if (tag == Tag::SelectedSampleIndex) {
            XML::readAttribute(r, XML::Attrib::value, &m_selectedSampleIndex);
            XML::gotoEndElementOfTag(r, tag);

            // selected data index
        } else if (tag == Tag::SelectedDataIndex) {
            XML::readAttribute(r, XML::Attrib::value, &m_selectedDataIndex);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}
