/* File: FtpListingParser.java
 * Created by jogoodma
 * Created on Feb 4, 2004 12:41:33 PM
 * Project: citrina
 * 
 * CVS Info:
 * $Id: FtpListingParser.java,v 1.9 2004/08/03 17:46:46 jogoodma Exp $
 * $Author: jogoodma $
 * $Date: 2004/08/03 17:46:46 $
 *
 * Copyright (c) 2004, Indiana University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  * Neither the name of Indiana University, Bloomington nor the names
 *    of its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */

package org.gmod.biomaj.ant.task.net;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.tools.ant.Project;
import org.inria.biomaj.ant.task.net.ListingParser;
import org.inria.biomaj.internal.ant.task.net.RemoteFile;
import org.inria.biomaj.utils.BiomajBuildException;

/**
 * This class parses a single line from an ftp listing generated by Ant's FTP task. After parsing you can retrieve the filename and the date in milliseconds.
 * @author  Josh Goodman
 * @version  CVS $Revision: 1.9 $
 */
public class FtpListingParser implements ListingParser {

	//*                                                           -r--r--r--   1 IUBio    archive  46700108 Jan  1 07:48 nc0101.flat.gz
	private static final Pattern DATEWITHTIME  = Pattern.compile("(\\d+)\\s+(\\w+\\s+\\d+\\s+\\d+:\\d+)\\s+([\\d\\w\\.\\/\\\\\\-]+)(\\s+->\\s+)?([\\d\\w\\.\\/\\\\\\-]+)?$");
	private static final Pattern DATEWITHYEAR = Pattern.compile("(\\d+)\\s+(\\w+\\s+\\d+\\s+\\d+)\\s+([\\d\\w\\.\\/\\\\\\-]+)(\\s+->\\s+)?([\\d\\w\\.\\/\\\\\\-]+)?$");
	
	/**
	 * @uml.property  name="line"
	 */
	private String line;
	/**
	 * @uml.property  name="currentDate"
	 */
	private Calendar currentDate;
	/**
	 * @uml.property  name="remoteFileDate"
	 */
	private Calendar remoteFileDate;
	/**
	 * @uml.property  name="fileName"
	 */
	private String fileName;
    /**
	 * @uml.property  name="linkName"
	 */
    private String linkName;
    /**
	 * @uml.property  name="fileSize"
	 */
    private long fileSize;
    /**
	 * @uml.property  name="isLink"
	 */
    private boolean isLink;
	/**
	 * @uml.property  name="unixTime"
	 */
	private long unixTime;
	/**
	 * @uml.property  name="currentProject"
	 * @uml.associationEnd  multiplicity="(1 1)"
	 */
	private Project currentProject;
	
	/**
	 * @uml.property  name="matchDateWithTime"
	 */
	private Matcher matchDateWithTime;
	/**
	 * @uml.property  name="matchDateWithYear"
	 */
	private Matcher matchDateWithYear;
	
	/**
	 * Constructor that takes the ftp listing line, a java regex pattern, and the current ant project for access
	 * to the log stream.  
	 * 
	 * @param p The current ant project usually accessed via <code>org.apache.tools.ant.Task.getProject()</code>
	 */
	public FtpListingParser(Project p) {
		currentProject = p;
		init();
	}
	
	/**
	 * Method to parse out the filename and file date in miliseconds since the epoc.
	 * 
	 * @return A boolean indicating weather the line was parsed correctly.
	 */
	public boolean parse() throws ParseException {
		if (line==null)
			throw new BiomajBuildException(currentProject,"ftplistingParser.error.init",new Exception());
		
//		Parse the line using the time format.
		if (parseDateWithTime()) {
			currentProject.log("Matched date with time for line: " + line, Project.MSG_DEBUG);
		}
		//Parse the line using the year format if the time match fails.
		else if (parseDateWithYear()) {
			currentProject.log("Matched date with year for line: " + line,Project.MSG_DEBUG);
		}
		//Log the error is no matches were found.
		else {
			throw new BiomajBuildException(currentProject,"ftplistingParser.error.match",line,new Exception());
		}
		return true;
	}
	
	/**
	 * @param line  the line to set
	 * @uml.property  name="line"
	 */
	public void setLine(String s) {
		line = s;
	}
	
	/**
	 * Returns the remote filename parsed out of the ftp listing file.
	 * @return  The remote filename.
	 * @uml.property  name="fileName"
	 */
	public String getFileName() {
		return fileName;
	}
	
    /**
	 * Returns the remote file that the link points to.
	 * @return  String with the name of the relative link target pointed to by the link file.
	 * @uml.property  name="linkName"
	 */
    public String getLinkName() {
       return linkName; 
    }
    
    /**
	 * Returns the size of the current parsed ftp file.
	 * @return  An int indicating the size of the file in bytes.
	 * @uml.property  name="fileSize"
	 */
    public long getFileSize() {
    	return fileSize;
    }
    
    private void setFileSize(String size) {
    	fileSize = Long.parseLong(size);
    }
    
    /**
	 * Returns whether or not the current file being parsed is a link or not.
	 * @return  A boolean indicating whether the file is a link or not.
	 * @uml.property  name="isLink"
	 */
    public boolean isLink() {
    	return isLink;
    }
    
    private void setIsLink(Matcher m) {
        if (m.groupCount() == 5) {
            String txt = m.group(4);
            if (txt != null && txt.matches("\\s+->\\s+")) {
                linkName = m.group(5);
                isLink = true;
            }
        }
    }
    
	/**
	 * Returns the last modified date in miliseconds for the file parsed out of the ftp listing file.
	 * 
	 * @return The remote file last modified date in miliseconds.
	 */
	public long getTime() {
		return unixTime;
	}
	
	/**
	 * Parses the ftp listing file line that has a format similar to 
	 * -r--r--r--   1 IUBio    archive  46700108 Jan  1 07:48 nc0101.flat.gz
	 * 
	 * @return A boolean indicating whether it matched or not.
	 */
	private boolean parseDateWithTime() {
		boolean found = false;			  //Initialize the return var.
		matchDateWithTime.reset(line);    //Pass the line to the date with time matcher object.
		
		//If it matches do this.
		if (matchDateWithTime.find()) {
			found = true;    //Set return value.
			//Get the filename from the second regex group as defined in the pattern.
			fileName = matchDateWithTime.group(3);	
		    setFileSize(matchDateWithTime.group(1));
            setIsLink(matchDateWithTime);
			
			try {
				//Create a new date parser object and use it to set the date of the file.
				SimpleDateFormat dateFormat = new SimpleDateFormat("MMM dd HH:mm");
				remoteFileDate.setTime(dateFormat.parse(matchDateWithTime.group(2)));
			} catch (ParseException pe) {
				throw new BiomajBuildException(currentProject,pe);
			}
			
			//Set the current year since no year is specified.
			remoteFileDate.set(Calendar.YEAR, currentDate.get(Calendar.YEAR));
			
			//If the date is in the future then the remote ftp server hasn't switched
			//formats and the file is from last year.
			if (remoteFileDate.getTimeInMillis() > currentDate.getTimeInMillis()) {
				remoteFileDate.add(Calendar.YEAR, -1);
			}
			//Set the time.
			unixTime = remoteFileDate.getTimeInMillis();
		}
		
		return found;
	}
	
	/**
	 * Parses the ftp listing file line that has a format similar to 
	 * -r--r--r--   1 IUBio    archive  46700108 Jan  1 2004 nc0101.flat.gz
	 * 
	 * @return A boolean indicating whether it matched or not.
	 */
	private boolean parseDateWithYear() {
		boolean found = false;					//Initialize return var.
		matchDateWithYear.reset(line);	//Pass the current line to this matcher object.
		
		//If we match do this.
		if (matchDateWithYear.find()) {
			//Set return value and store the filename.
			found = true;
			fileName = matchDateWithYear.group(3);
            setIsLink(matchDateWithYear);
            setFileSize(matchDateWithYear.group(1));
            
			//Log some info.
			currentProject.log("Filename found: " + fileName,Project.MSG_DEBUG);
			try {
				//Parse the date from the ftp listing line.
				SimpleDateFormat dateFormat = new SimpleDateFormat("MMM dd yyyy");
				remoteFileDate.setTime(dateFormat.parse(matchDateWithYear.group(2)));
			} catch (ParseException pe) {
				throw new BiomajBuildException(currentProject,pe);
			}
			//Set the time.
			unixTime = remoteFileDate.getTimeInMillis();
		}
		return found;
	}
	
	/**
	 * Method to initialize the matchers used for various patterns, Calendar objects,
	 * and the filename.
	 */
	private void init() {
		//Initialize matchers for each pattern.
		matchDateWithTime = DATEWITHTIME.matcher("");
		matchDateWithYear = DATEWITHYEAR.matcher("");
		
		//Create calendar objects.
		currentDate = Calendar.getInstance();
		remoteFileDate = Calendar.getInstance();
		
		//Create filename object.
		fileName = null;
        isLink = false;
	}

	public RemoteFile getRemoteFile() {
		// TODO Auto-generated method stub
		return null;
	}
	
	
	
}
