/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

#include "alarm_thread.h"
#include "configuration.h"
#include "main_window.h"
#include "main_list.h"
#include "alarm_runner.h"
#include "translate.h"
#include "modify_alarm.h"
#include "status_icon.h"
#include <time.h>

gchar
*format_time (gint usecs)
{
	gchar *tleft, *ghours, *gminutes, *gseconds;
	gint int_buf;
	gint days = usecs / 86400;
	int_buf = days * 86400;
	gint hours = (usecs - int_buf) / 3600;
	int_buf = int_buf + hours * 3600;
	gint minutes = (usecs - int_buf) / 60;
	int_buf = int_buf + minutes * 60;
	gint seconds = (usecs -int_buf);
	
	if (hours < 10) ghours = g_strdup_printf("0%i", hours);
	else ghours = g_strdup_printf("%i", hours);

	if (minutes < 10) gminutes = g_strdup_printf("0%i", minutes);
	else gminutes = g_strdup_printf("%i", minutes);
	
	if (seconds < 10) gseconds = g_strdup_printf("0%i", seconds);
	else gseconds = g_strdup_printf("%i", seconds);	

	
	if (days < 1)
	{
		tleft = g_strdup_printf("%s:%s:%s", ghours, gminutes, gseconds);
	}
	else if (days < 365)
	{
		tleft = g_strdup_printf("%id %s:%s:%s", days, ghours, gminutes, gseconds);
	}
	else
	{
		gint years = days / 365;
		days = days % 365;
		tleft = g_strdup_printf("%iy %id %s:%s:%s", years, days, ghours, gminutes, gseconds);
	}
	
	g_free(ghours);
	g_free(gminutes);
	g_free(gseconds);
	
	return tleft;
}

gboolean
is_date_excluded(gchar *alarm_name, gint day, gint month)
{
	gchar *buffer, **split, **date_split;
	gint d, m, count = 0;

	buffer = g_key_file_get_string(loaded_alarms, alarm_name, "ScheduleDateExclude", NULL);
	if (buffer == NULL) 
		return FALSE;

	split = g_strsplit(buffer, ";", -1);
	g_free(buffer);
	while (TRUE)
	{
		if (split[count] == NULL) break;
		date_split = g_strsplit(split[count], "/", 2);

		d = g_ascii_strtoll(date_split[0], NULL, 10);
		m = g_ascii_strtoll(date_split[1], NULL, 10);


		if (d == day && m == month) 
		{
			g_strfreev(split);
			g_strfreev(date_split);
			return TRUE;
		}

		g_strfreev(date_split);
		count++;
	}

	g_strfreev(split);
	
	return FALSE;
}

gchar *
get_iso_date_from_scheduled(gchar *alarm_name)
{
	time_t rawtime;
	struct tm * timeinfo;
	time ( &rawtime );
	timeinfo = localtime ( &rawtime );
	gchar *buffer, **split, **buffer_split, *date_buffer, **weekdays, **months;
	GTimeVal included_date, smallest_date, smallest_weekday;
	gint count;
	GDate *check_date = g_date_new();
	GDate *current_date = g_date_new();
	gint day, weekday, month;
	gboolean ok = TRUE, got_date = FALSE, ok_date;
	gint time_min, time_hour;
	
	/* First - need to check if we have included date today */

	buffer = g_key_file_get_string(loaded_alarms, alarm_name, "ScheduleDateInclude", NULL);
	if (g_strcmp0(buffer, "") != 0) got_date = TRUE;
	split = g_strsplit(buffer, ";", -1);
	g_free(buffer);

	count = 0;
	while (TRUE)
	{
		if (split[count] == NULL) break;
		buffer = g_key_file_get_string(loaded_alarms, alarm_name, "DateTime", NULL);
		buffer_split = g_strsplit(buffer, "T", 2);
		g_free(buffer);
		buffer = g_strdup(buffer_split[1]);
		g_strfreev(buffer_split);

		buffer_split = g_strsplit(split[count], "/", 2);

		date_buffer = g_strdup_printf("%i-%s-%sT%s", timeinfo->tm_year + 1900, 
		                              buffer_split[1], buffer_split[0], buffer);


		g_strfreev(buffer_split);
		g_free(buffer);

		
		g_time_val_from_iso8601(date_buffer, &included_date);

		if (count == 0 && date_buffer != NULL) g_time_val_from_iso8601(date_buffer, &smallest_date);

		if (included_date.tv_sec < smallest_date.tv_sec) smallest_date = included_date;

		count++;
	    g_free(date_buffer);
	}


	g_strfreev(split);
	
	/* Then, we have to check if we have the alarm because of weekday selected 
	   How the fuck I'll do it?:( */
	
	buffer = g_key_file_get_string(loaded_alarms, alarm_name, "ScheduleWeekdays", NULL);
	if (g_strcmp0(buffer, "F:F:F:F:F:F:F") == 0) ok = FALSE;
	weekdays = g_strsplit(buffer, ":", 7);
	
	g_free(buffer);

	buffer = g_key_file_get_string(loaded_alarms, alarm_name, "DateTime", NULL);
	buffer_split = g_strsplit(buffer, "T", 2);
	g_free(buffer);

	buffer = g_strdup(buffer_split[1]);
	g_strfreev(buffer_split);

	buffer_split = g_strsplit(buffer, ":", 2);
	time_hour = g_ascii_strtoll(buffer_split[0], NULL, 10);
	time_min = g_ascii_strtoll(buffer_split[1], NULL, 10);

	g_free(buffer);
	g_strfreev(buffer_split);


	buffer = g_key_file_get_string(loaded_alarms, alarm_name, "ScheduleMonths", NULL);
	months = g_strsplit(buffer, ":", 12);
	g_free(buffer);	
	
	g_date_set_time_t(check_date, rawtime);
	g_date_set_time_t(current_date, rawtime);

	while (TRUE && ok)
	{
		day = g_date_get_day(check_date);
		weekday = g_date_get_weekday(check_date);
		month = g_date_get_month(check_date);

		ok_date = FALSE;

		if (g_date_compare(current_date, check_date) == 0)
			if (timeinfo->tm_hour < time_hour)
				ok_date = TRUE;
			if (timeinfo->tm_hour == time_hour)
				if (timeinfo->tm_min < time_min)
					ok_date = TRUE;
				

		if (g_date_compare(current_date, check_date) < 0) ok_date = TRUE;

		if (g_strcmp0(weekdays[weekday - 1], "T") == 0 && g_strcmp0(months[month - 1], "T") == 0 && ok_date)
		{
			buffer = g_key_file_get_string(loaded_alarms, alarm_name, "DateTime", NULL);
			buffer_split = g_strsplit(buffer, "T", 2);
			g_free(buffer);
			buffer = g_strdup_printf("%i-%i-%iT%s", g_date_get_year(check_date),
			                         month, day, buffer_split[1]);
			g_time_val_from_iso8601(buffer, &smallest_weekday);
			g_free(buffer);
			if (!is_date_excluded(alarm_name, day, month)) break;
		}

		g_date_add_days(check_date, 1);

	}
	/* We have to compare which one is earlier */

	
	if (got_date) return g_time_val_to_iso8601(&smallest_date);
	else return g_time_val_to_iso8601(&smallest_weekday);





	
	
}

void
move_to_missed(gchar *name)
{
	gint key = 0;
	gchar *buffer, **keys;
	GKeyFile *temp_key = g_key_file_new();
	
	g_key_file_load_from_file(temp_key, config_missed, G_KEY_FILE_NONE, NULL);
	
	keys = g_key_file_get_keys(loaded_alarms, name, NULL, NULL);
	
	while (TRUE)
	{
		if (keys[key] == NULL) break;
		buffer = g_key_file_get_string(loaded_alarms, name, keys[key], NULL);

		g_key_file_set_string(temp_key, name, keys[key], buffer);
		
		key++;
		g_free(buffer);
	}
	
	buffer = g_key_file_to_data(temp_key, NULL, NULL);
	
	g_file_set_contents(config_missed, buffer, -1, NULL);
	
	g_key_file_free(temp_key);
	g_free(buffer);
	g_strfreev(keys);
	
}

void
manage_old_alarm(gchar *name)
{
	gchar *type = g_key_file_get_string(loaded_alarms, name, "AlarmType", NULL);
	gboolean snooze = g_key_file_get_boolean(loaded_alarms, name, "snoozed", NULL);
	gboolean single = TRUE;

	if (g_strcmp0(type, "Schedule") == 0) single = FALSE;
	if (snooze)  snooze = FALSE;

	g_free(type);
	
	if (need_check_first)
	{
		move_to_missed(name);
		if (single) 
		{
			remove_alarm(name);
			reload_alarms();
			update_list_entries();
		}
	}
	else
	{
		if (single && !snooze && !running_snooze)
		{
			g_usleep(G_USEC_PER_SEC);
			remove_alarm(name);
			reload_alarms();
			update_list_entries();
		}
	}
}


void
update_alarm_thread(void)
{
	time_t rawtime;
	struct tm * timeinfo;
	gint diff;
	GTimeVal alarm_time, current_time;
	gchar *iso_date, *time_left, *alarm_type, **alarms;
	guint count;
	GtkTreeIter iter;
	gchar *name;
	count = 0;
	gboolean first = TRUE;

	time ( &rawtime );
	timeinfo = localtime ( &rawtime );
	gboolean snoozed;

	g_get_current_time(&current_time);

	alarms = g_key_file_get_groups(loaded_alarms, NULL);
	while(TRUE)
	{
	if (alarms[count] == NULL) break;
		alarm_type = g_key_file_get_string(loaded_alarms, alarms[count], "AlarmType", NULL);
		snoozed = g_key_file_get_boolean(loaded_alarms, alarms[count], "Snoozed", NULL);

		if (g_strcmp0(alarm_type, "Single") == 0 || g_strcmp0(alarm_type, "Counter") == 0)
		{
			iso_date = g_key_file_get_string(loaded_alarms, alarms[count], "DateTime", NULL);
		}
		else if (g_strcmp0(alarm_type, "Schedule") == 0)
		{
			iso_date = get_iso_date_from_scheduled(alarms[count]);
		}
		else
		{
			g_free(alarm_type);
			return;
		}
		
		g_time_val_from_iso8601(iso_date, &alarm_time);
		g_free(iso_date);

		g_get_current_time(&current_time);

		iso_date = g_time_val_to_iso8601(&alarm_time);
		g_time_val_from_iso8601(iso_date, &alarm_time);

		g_free(iso_date);
		
		diff = alarm_time.tv_sec - current_time.tv_sec;
		gtk_tree_model_get_iter_first(GTK_TREE_MODEL(store), &iter);

		while (TRUE)
		{
			gtk_tree_model_get (GTK_TREE_MODEL(store), &iter, 4, &name, -1);
			if (g_strcmp0(name, alarms[count]) == 0) break;
			if (gtk_tree_model_iter_next(GTK_TREE_MODEL(store), &iter) == FALSE) break;
		}
		// Tutaj formatujemy timeleft

		if (diff > 1)
		{
			if (snoozed)
			{
				gchar *buf = format_time(diff);
				time_left = g_strdup_printf("<i>%s</i>", buf);
				gtk_list_store_set(store, &iter, 5, time_left, -1);
				g_free(buf);
				g_free(time_left);
			}
			else
			{
				time_left = format_time(diff);
				gtk_list_store_set(store, &iter, 5, time_left, -1);
				g_free(time_left);
			}

		}


		if (diff == 1)
		{
			gdk_threads_enter();
			run_alarm(alarms[count]);
			manage_old_alarm(alarms[count]);
			update_list_entries();
			gdk_threads_leave();
		}

		if (diff < 1)
		{
			manage_old_alarm(alarms[count]);
		}
		if (first) {
			first = FALSE;
			continue;
		}
		else {
		count++; }
		g_free(alarm_type);
	}
	g_strfreev(alarms);
}

void
process_alarm_thread(void)
{
	time_t rawtime;
	struct tm * timeinfo;
	gint diff;
	GTimeVal alarm_time, current_time;
	gchar *iso_date, **alarms;

	time ( &rawtime );
	timeinfo = localtime ( &rawtime );
	
	guint count;
	count = 0;

	

	if (timeinfo->tm_hour == 0 && timeinfo->tm_min == 0 && timeinfo->tm_sec == 0)
	{
		gdk_threads_enter();
		check_current_birthdays ();
		gdk_threads_leave();
	}

	alarms = g_key_file_get_groups(loaded_alarms, NULL);
	
	// Checking if we should play the alarm or not
		
	while(TRUE)
	{	
		if (alarms[count] == NULL) break;
		
		iso_date = g_key_file_get_string(loaded_alarms, alarms[count], "DateTime", NULL);
		g_time_val_from_iso8601(iso_date, &alarm_time);
		g_free(iso_date);
		iso_date = g_strdup_printf("%i-%i-%iT%i:%i:%i", timeinfo->tm_year + 1900, timeinfo->tm_mon + 1,
								   timeinfo->tm_mday, timeinfo->tm_hour, timeinfo->tm_min, timeinfo->tm_sec);
		g_time_val_from_iso8601(iso_date, &current_time);
		g_free(iso_date);
		diff = alarm_time.tv_sec - current_time.tv_sec;
		count++;	
	}
	update_alarm_thread();

}

void
check_current_birthdays(void)
{
	gchar *iso_date, *month, *day, *processed_date, *buffer;
	GString *names = g_string_new(NULL);
	gboolean got_birthday = FALSE, first_name = TRUE;
	GKeyFile *birthdays_key;
	gchar **birthdays;
	gint count = 0;
	time_t rawtime;
	struct tm * timeinfo;
	time ( &rawtime );
	timeinfo = localtime ( &rawtime );

	if (timeinfo->tm_mon + 1 < 10)
	{
		month = g_strdup_printf("0%i", timeinfo->tm_mon + 1);
	}
	else
	{
		month = g_strdup_printf("%i", timeinfo->tm_mon + 1);
	}

	if (timeinfo->tm_mday < 10)
	{
		day = g_strdup_printf("0%i", timeinfo->tm_mday);
	}
	else
	{
		day = g_strdup_printf("%i", timeinfo->tm_mday);
	}
	
	iso_date = g_strdup_printf("%i-%s-%s", timeinfo->tm_year + 1900, month, day);

	birthdays_key = g_key_file_new();
	g_key_file_load_from_file(birthdays_key, config_birthdays, G_KEY_FILE_NONE, NULL);
	birthdays = g_key_file_get_groups(birthdays_key, NULL);
	
	while (TRUE)
	{
		if (birthdays[count] == NULL) break;
		processed_date = g_strdup(g_key_file_get_string(birthdays_key, birthdays[count], "Date", NULL));
		if (g_strcmp0(iso_date, processed_date) == 0)
		{
			got_birthday = TRUE;
			if (first_name)
			{
				g_string_append(names, g_key_file_get_string(birthdays_key, birthdays[count], "Name", NULL));
				first_name = FALSE;
			}
			else
			{
				g_string_append(names, "\n");
				g_string_append(names, g_key_file_get_string(birthdays_key, birthdays[count], "Name", NULL));
			}
		}
		g_free(processed_date);
		count++;
	}
	change_birthday_status(got_birthday);
	if (got_birthday)
	{
	buffer = g_strdup_printf(_("Today's birthdays:\n\n%s"), names->str);
	gtk_status_icon_set_tooltip(GTK_STATUS_ICON(status_icon), buffer);
	g_free(buffer);
	}
	else
	{
		gtk_status_icon_set_tooltip(GTK_STATUS_ICON(status_icon), _("Alarm Clock"));
	}


	g_free(iso_date);
	g_free(month);
	g_free(day);
	g_strfreev(birthdays);
	g_key_file_free(birthdays_key);
	g_string_free(names, TRUE);
}

void
alarm_thread(void)
{
	need_check_first = TRUE;
	
	gdk_threads_enter();
	check_current_birthdays();
	update_list_entries();
	check_missed_alarms();
	gdk_threads_leave();
	
	while (TRUE)
	{
		process_alarm_thread ();
		g_usleep(G_USEC_PER_SEC);
	}
}

void
reload_alarms(void)
{
	if (loaded_alarms != NULL)
		g_key_file_free(loaded_alarms);
	
	loaded_alarms = g_key_file_new();
	g_key_file_load_from_file(loaded_alarms, config_alarms, G_KEY_FILE_NONE, NULL);
}
