#! /usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Copyright 2022, Nils Hilbricht, Germany ( https://www.hilbricht.net )

This file is part of the Laborejo Software Suite ( https://www.laborejo.org ),

This application is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""

import logging; logger = logging.getLogger(__name__); logger.info("import")


#Standard Library
import pathlib
import os

#Third Party
from PyQt5 import QtCore, QtWidgets

#QtGui
from .designer.projectname import Ui_ProjectName
from .designer.newsession import Ui_NewSession

#Engine
from engine.config import METADATA  #includes METADATA only. No other environmental setup is executed.
import engine.api as api


class ProjectNameWidget(QtWidgets.QDialog):
    """Ask the user for a project name. Either for renaming, new or copy.
    Will have a live-detection """

    def __init__(self, parent, startwith:str="", start=True, alsoDisable=None):
        super().__init__(parent)
        self.ui =  Ui_ProjectName()
        self.ui.setupUi(self)

        self.alsoDisable = alsoDisable #in case we are embedded

        #self.ui.labelDescription

        self.ui.labelError.setText("")
        self.ui.name.setText(startwith)
        self.check(startwith)
        self.ui.name.textEdited.connect(self.check) #not called when text is changed programatically

        self.result = None

        self.ui.buttonBox.accepted.connect(self.process)
        self.ui.buttonBox.rejected.connect(self.reject)

        if start:            
            self.setWindowFlag(QtCore.Qt.Popup, True)
            self.setModal(True)
            self.setFocus(True)
            self.ui.name.setFocus(True)
            self.exec_()            

    def check(self, currentText):
        """Called every keypress.
        We do preliminary error and collision checking here, so the engine does not have to throw
        an error """        
        if currentText.endswith("/"):
            currentText = currentText[:-1]

        fullpath = pathlib.Path(api.sessionRoot(), currentText) 
        path = pathlib.Path(currentText)        

        errorMessage = ""
        if currentText == "":
            errorMessage = QtCore.QCoreApplication.translate("ProjectNameWidget", "Name must not be empty.")
        elif pathlib.PurePosixPath(path).match("/*") or str(pathlib.PurePosixPath(path)).startswith("/"):
            errorMessage = QtCore.QCoreApplication.translate("ProjectNameWidget", "Name must be a relative path.")
        elif pathlib.PurePosixPath(path).match("../*") or pathlib.PurePosixPath(path).match("*..*"):
            errorMessage = QtCore.QCoreApplication.translate("ProjectNameWidget", "Moving to parent directory not allowed.")
        elif "/" in currentText and fullpath.parent.exists() and not os.access(fullpath.parent, os.W_OK):
            errorMessage = QtCore.QCoreApplication.translate("ProjectNameWidget", "Writing in this directory is not permitted.")
        elif fullpath.exists():
            errorMessage = QtCore.QCoreApplication.translate("ProjectNameWidget", "Name is already in use.")

        ok = self.ui.buttonBox.button(QtWidgets.QDialogButtonBox.Ok)
        if errorMessage:
            if self.alsoDisable:
                self.alsoDisable.setEnabled(False)
            ok.setEnabled(False)
            self.ui.labelError.setText("<i>"+errorMessage+"</i>")
        else:
            #print (path)
            if self.alsoDisable:
                self.alsoDisable.setEnabled(True)
            self.ui.labelError.setText("")
            ok.setEnabled(True)

    def process(self):
        """Careful! Calling this eats python errors without notice. Make sure your objects exists
        and your syntax is correct"""
        self.result = self.ui.name.text()
        self.done(True)

class NewSessionDialog(QtWidgets.QDialog):

    def __init__(self, parent, startwith:str=""):
        super().__init__(parent)
        self.ui =  Ui_NewSession()
        self.ui.setupUi(self)

        #send our childWidget our own ok button so it can disable it for the name check
        self.ok = self.ui.buttonBox.button(QtWidgets.QDialogButtonBox.Ok)
        #Embed a project name widget
        self.projectName = ProjectNameWidget(self, startwith, start=False, alsoDisable=self.ok)
        self.projectName.ui.buttonBox.hide()        
        self.ui.nameGroupBox.layout().addWidget(self.projectName)
        self.projectName.ui.name.returnPressed.connect(self.ok.click) #We want to accept everything when return is pressed.
        self.projectName.reject = self.reject #forward escape to our reject

        self.result = None

        nsmExecutables = api.getNsmExecutables() #type set, cached, very fast.
        data = METADATA["preferredClients"]["data"]
        con = METADATA["preferredClients"]["connections"]       
        
        self.ui.checkBoxJack.setEnabled(con in nsmExecutables)
        self.ui.checkBoxJack.setVisible(con in nsmExecutables)
        self.ui.checkBoxData.setEnabled(data in nsmExecutables)
        self.ui.checkBoxData.setVisible(data in nsmExecutables)           

        self.ui.buttonBox.accepted.connect(self.process)
        self.ui.buttonBox.rejected.connect(self.reject)

        self.setModal(True)
        self.setWindowFlag(QtCore.Qt.Popup, True)
        self.projectName.ui.name.setFocus(True)
        self.exec_()

    def process(self):
        """Careful! Calling this eats python errors without notice. Make sure your objects exists
        and your syntax is correct"""
        assert self.ok.isEnabled()        

        data = METADATA["preferredClients"]["data"]
        con = METADATA["preferredClients"]["connections"]        
        startclients = []
        if self.ui.checkBoxJack.isChecked(): startclients.append(con)
        if self.ui.checkBoxData.isChecked(): startclients.append(data)
        self.result = {
            "name" : self.projectName.ui.name.text(),
            "startclients" : startclients,
        }        
        self.done(True) 
