; Java Library
;
; Copyright (C) 2020 Kestrel Institute (http://www.kestrel.edu)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Author: Alessandro Coglio (coglio@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "JAVA")

(include-book "aij-notions")
(include-book "java-primitives")
(include-book "java-primitive-arrays")
(include-book "test-structures")

(include-book "kestrel/error-checking/ensure-list-has-no-duplicates" :dir :system)
(include-book "kestrel/error-checking/ensure-value-is-boolean" :dir :system)
(include-book "kestrel/error-checking/ensure-value-is-function-name" :dir :system)
(include-book "kestrel/error-checking/ensure-value-is-in-list" :dir :system)
(include-book "kestrel/error-checking/ensure-value-is-string" :dir :system)
(include-book "kestrel/error-checking/ensure-value-is-untranslated-term" :dir :system)
(include-book "kestrel/event-macros/xdoc-constructors" :dir :system)
(include-book "kestrel/std/system/check-list-call" :dir :system)
(include-book "kestrel/std/system/known-packages-plus" :dir :system)
(include-book "kestrel/std/system/pure-raw-p" :dir :system)
(include-book "kestrel/std/system/rawp" :dir :system)
(include-book "kestrel/std/system/ubody" :dir :system)
(include-book "kestrel/std/system/unquote-term" :dir :system)
(include-book "kestrel/utilities/doublets" :dir :system)
(include-book "kestrel/utilities/er-soft-plus" :dir :system)
(include-book "kestrel/utilities/error-checking/top" :dir :system)
(include-book "oslib/catpath" :dir :system)
(include-book "oslib/file-types" :dir :system)
(include-book "std/typed-alists/symbol-symbollist-alistp" :dir :system)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(xdoc::evmac-topic-input-processing
 atj
 (xdoc::p
  "As part of input processing,
   we collect the names of all the ACL2 functions to be translated to Java,
   as determined by @('fn1'), ..., @('fnp').
   As we do that,
   we also check that they satisfy the constraints
   stated in the user documentation.")
 (xdoc::p
  "This collection and checking of the ACL2 functions
   is realized via a worklist algorithm.
   The worklist is initialized with @('fn1'), ..., @('fnp').
   At each step, a function @('fn') is taken from the worklist and processed.
   If @('fn') satisfies all the necessary constraints,
   it is added to a list of collected functions (which is initially empty);
   otherwise, we stop with an error.
   If @('fn') is defined,
   we collect the functions that occur in its defining body
   and add them to the worklist,
   except for those that are already in the worklist or in the collected list
   (so that we do not process the same function twice).
   Note that by adding @('fn') to the collected list
   before examining the functions that occur in its defining body,
   we ensure termination in the presence of
   (singly or mutually) recursive functions.
   We proceed like this until the worklist is empty (or an error occurs).
   If there are no errors, at the end
   we will have checked all the functions
   transitively called by @('fn1'), ..., @('fnp'),
   and the collected list will contain all the functions
   that must be translated to Java.
   This is the basic algorithm, but there are some complications,
   described in the following.")
 (xdoc::p
  "A complication arises from
   calls of @(tsee return-last) whose first argument is @('\'acl2::mbe1-raw'),
   which are calls of @(tsee mbe) in translated form.
   As explained in the user documentation,
   when the @(':guards') input of ATJ is @('nil'),
   the Java code generated by ATJ executes ``in the logic'',
   and in particular executes the @(':logic') parts of @(tsee mbe)s;
   when instead the @(':guards') input of ATJ is @('t'),
   the Java code generated by ATJ assumes the satisfaction of the guards,
   and in particular executes the @(':exec') parts of @(tsee mbe)s.
   Thus, when we recursively collect the functions
   from the body of a defined function,
   when we encounter these calls of @(tsee return-last),
   we selectively descend into the @(':logic') or @(':exec') part
   (based on the value of the @(':guards') input),
   ignoring the other part.")
 (xdoc::p
  "Another complication arises from
   calls of @(tsee return-last) whose first argument is @('\'acl2::progn'),
   which are calls of @(tsee prog2$) and @(tsee progn$) in translated form.
   As explained in the documentation,
   code is generated from the last argument only,
   but the other arguments must be checked to satisfy constraints as well.
   Thus, we use two worklists and two collected lists:
   one worklist and one collected list for the functions
   for which Java code must be generated,
   and one worklist and one collected list for the functions
   that must be only checked to satisfy the constraints.
   At the end of the iteration,
   the first collected list is used to generate Java code,
   while the second collected list is discarded;
   however, this second collected list is used during the iteration,
   to keep track of the functions already checked
   that do not appear in the worklists or in the first collected list.
   The function @('fn') is always taken from the first worklist,
   unless this worklist is empty, in which case it is taken from the second:
   in other words, the first worklist is processed first,
   and then the second one;
   the iteration terminates when both worklists are empty.")
 (xdoc::p
  "Yet another complication arises from
   calls of functions in
   @(tsee *atj-jprim-fns*) and @(tsee *atj-jprimarr-fns*),
   which are translated directly to suitable Java constructs
   when @(':deep') is @('nil') and @(':guards') is @('t').
   Under these conditions, when @('fn') is taken from a worklist,
   its defining body is not examined;
   i.e. it is treated like a natively implemented function,
   which it is in some sense.")
 (xdoc::p
  "Further details and complications of the worklist algorithm
   are explained in the implementing functions."))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-process-targets ((targets true-listp) deep guards ctx state)
  :returns (mv erp (result null) state)
  :short "Process the @('fn1'), ..., @('fnp') inputs."
  :long
  (xdoc::topstring
   (xdoc::p
    "Here we only check @('fn1'), ..., @('fnp') themselves.
     We collect and check the called functions
     after checking the remaining inputs;
     see @(tsee atj-process-inputs)."))
  (b* (((er &) (case (len targets)
                 (0 (er-soft+ ctx t nil
                              "At least one target function must be supplied."))
                 (1 (ensure-value-is-function-name$
                     (car targets)
                     (msg "The ~x0 input" (car targets))
                     t
                     nil))
                 (t (ensure-list-functions$ targets
                                            (msg "The ~&0 inputs" targets)
                                            t nil))))
       ((er &) (ensure-list-has-no-duplicates$ targets
                                               (msg "The target functions ~&0"
                                                    targets)
                                               t nil))
       ((when (or deep (not guards))) (value nil))
       (target-prims (intersection-eq targets
                                      (union-eq *atj-jprim-fns*
                                                *atj-jprimarr-fns*)))
       ((when (null target-prims)) (value nil)))
    (er-soft+ ctx t nil
              "Since the :DEEP input is (perhaps by default) NIL ~
               and the :GUARDS input is (perhaps by default) T, ~
               ~@0."
              (if (= (len target-prims) 1)
                  (msg "the function ~x0 cannot be specified as target"
                       (car target-prims))
                (msg "the functions ~&0 cannot be specified as targets")))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-process-java-package ((java-package) ctx state)
  :returns (mv erp (nothing null) state)
  :short "Process the @(':java-package') input."
  (b* (((er &) (ensure-string-or-nil$ java-package
                                      "The :JAVA-PACKAGE input"
                                      t nil))
       ((unless (or (null java-package)
                    (atj-string-ascii-java-package-name-p java-package)))
        (er-soft+ ctx t nil
                  "The :JAVA-PACKAGE input ~x0 is not ~
                   NIL or a valid Java package name ~
                   consisting of only ASCII characters."
                  java-package))
       ((when (equal java-package *aij-package*))
        (er-soft+ ctx t nil
                  "The :JAVA-PACKAGE input ~x0 must differ from ~
                   the name of the Java package of AIJ ~x1."
                  java-package *aij-package*)))
    (value nil))
  :guard-hints (("Goal" :in-theory (enable acl2::ensure-string-or-nil))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defval *atj-default-java-class*
  :short "Default Java class name to use if @(':java-class') is @('nil')."
  "Acl2Code"
  ///
  (assert-event (stringp *atj-default-java-class*)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-process-java-class (java-class ctx state)
  :returns (mv erp
               (java-class$ (implies (not erp) (stringp java-class$)))
               state)
  :short "Process the @(':java-class') input."
  (b* (((er &) (ensure-string-or-nil$ java-class
                                      "The :JAVA-CLASS input"
                                      t nil))
       ((unless (or (null java-class)
                    (atj-string-ascii-java-identifier-p java-class)))
        (er-soft+ ctx t nil
                  "The :JAVA-CLASS input ~x0 is not ~
                   NIL or a valid Java class name ~
                   consisting of only ASCII characters."
                  java-class))
       (name (or java-class *atj-default-java-class*)))
    (value name))
  :prepwork ((local (in-theory (enable acl2::ensure-string-or-nil)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-process-test-input-jprim-value
  ((input pseudo-termp)
   (type primitive-typep)
   (fn symbolp "Just for error messages.")
   (call pseudo-termp "Just for error messages.")
   ctx
   state)
  :returns (mv erp value state)
  :short "Process a Java primitive input, or part of an input,
          of a test for a function call."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is used only if @(':deep') is @('nil') and @(':guards') is @('t').")
   (xdoc::p
    "The @('input') argument could be a sub-term @('in')
     of a term @('(fn ... in ...)') specified in the @(':tests') input
     when that argument of @('fn') has a Java primitive type,
     or it could be a further sub-term of such an @('in') sub-term
     when the latter denotes a Java primitive array."))
  (b* (((when (member-eq (primitive-type-kind type) '(:float :double)))
        (er-soft+ ctx t nil "Internal error: type ~x0 not supported." type))
       (irrelevant (primitive-type-case type
                                        :boolean (boolean-value nil)
                                        :char (char-value 0)
                                        :byte (byte-value 0)
                                        :short (short-value 0)
                                        :int (int-value 0)
                                        :long (long-value 0)
                                        :float (impossible)
                                        :double (impossible)))
       (constructor (primitive-type-case type
                                         :boolean 'boolean-value
                                         :char 'char-value
                                         :byte 'byte-value
                                         :short 'short-value
                                         :int 'int-value
                                         :long 'long-value
                                         :float nil
                                         :double nil))
       (err-msg (msg "The term ~x0 that is (possibly part of) an argument of ~
                      the function call (~x1 ...) that translates ~
                      the test term ~x2 in the :TESTS input, ~
                      must be a call (~x3 X) where X is ~s4."
                     input
                     fn
                     call
                     constructor
                     (primitive-type-case type
                                          :boolean "a boolean"
                                          :char "an unsigned 16-bit integer"
                                          :byte "a signed 8-bit integer"
                                          :short "a signed 16-bit integer"
                                          :int "a signed 32-bit integer"
                                          :long "a signed 64-bit integer"
                                          :float nil
                                          :double nil)))
       ((unless (ffn-symb-p input constructor))
        (er-soft+ ctx t irrelevant "~@0" err-msg))
       (args (fargs input))
       ((unless (= (len args) 1))
        (er-soft+ ctx t irrelevant "~@0" err-msg))
       (arg (car args))
       ((unless (quotep arg))
        (er-soft+ ctx t irrelevant "~@0" err-msg))
       (arg (unquote-term arg))
       ((unless (primitive-type-case type
                                     :boolean (booleanp arg)
                                     :char (ubyte16p arg)
                                     :byte (sbyte8p arg)
                                     :short (sbyte16p arg)
                                     :int (sbyte32p arg)
                                     :long (sbyte64p arg)
                                     :float nil
                                     :double nil))
        (er-soft+ ctx t irrelevant "~@0" err-msg)))
    (value
     (primitive-type-case type
                          :boolean (boolean-value arg)
                          :char (char-value arg)
                          :byte (byte-value arg)
                          :short (short-value arg)
                          :int (int-value arg)
                          :long (long-value arg)
                          :float nil
                          :double nil)))
  ///

  (more-returns
   (value boolean-valuep :hyp (primitive-type-case type :boolean))
   (value char-valuep :hyp (primitive-type-case type :char))
   (value byte-valuep :hyp (primitive-type-case type :byte))
   (value short-valuep :hyp (primitive-type-case type :short))
   (value int-valuep :hyp (primitive-type-case type :int))
   (value long-valuep :hyp (primitive-type-case type :long))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-process-test-input-jprim-values
  ((inputs pseudo-term-listp)
   (type primitive-typep)
   (fn symbolp "Just for error messages.")
   (call pseudo-termp "Just for error messages.")
   ctx
   state)
  :returns (mv erp values state)
  :short "Lift @(tsee atj-process-test-input-jprim-value) to lists."
  (b* (((when (endp inputs)) (value nil))
       ((cons input inputs) inputs)
       ((mv erp value state)
        (atj-process-test-input-jprim-value input type fn call ctx state))
       ((when erp) (mv erp nil state))
       ((er values)
        (atj-process-test-input-jprim-values inputs type fn call ctx state)))
    (value (cons value values)))
  ///

  (more-returns
   (values boolean-value-listp :hyp (primitive-type-case type :boolean))
   (values char-value-listp :hyp (primitive-type-case type :char))
   (values byte-value-listp :hyp (primitive-type-case type :byte))
   (values short-value-listp :hyp (primitive-type-case type :short))
   (values int-value-listp :hyp (primitive-type-case type :int))
   (values long-value-listp :hyp (primitive-type-case type :long))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-process-test-input ((input pseudo-termp)
                                (type atj-typep)
                                (fn symbolp "Just for error messages.")
                                (call pseudo-termp "Just for error messages.")
                                (deep$ booleanp)
                                (guards$ booleanp)
                                ctx
                                state)
  :returns (mv erp
               (test-input atj-test-valuep)
               state)
  :short "Process the input of a test for a function call."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is some sub-term @('in') of a term @('(fn ... in ...)')
     specified in the @(':tests') input.
     The requirements on @('in'), as expained in the user documentation,
     depend on the @(':deep') and @(':guards') inputs,
     as well as on the ATJ type assigned to
     the parameter of @('fn') that corresponds to @('in'):
     these three values are passed as inputs to this function,
     which checks these requirements, thus validating @('in').
     If the checks succeed, we turn @('in') into
     the corresponding test value.
     Note that these checks imply that @('in') is ground."))
  (b* ((irrelevant (atj-test-value-acl2 :irrelevant))
       ((when (or deep$
                  (not guards$)
                  (atj-type-case type :acl2)))
        (if (quotep input)
            (value (atj-test-value-acl2 (unquote-term input)))
          (er-soft+ ctx t irrelevant
                    "The term ~x0 that is an argument of ~
                     the function call (~x1 ...) that translates ~
                     the test term ~x2 in the :TESTS input, ~
                     must be a quoted constant."
                    input fn call)))
       ((when (atj-type-case type :jprim))
        (b* ((ptype (atj-type-jprim->get type))
             ((mv erp value state)
              (atj-process-test-input-jprim-value
               input ptype fn call ctx state))
             ((when erp) (mv erp irrelevant state)))
          (value (primitive-type-case
                  ptype
                  :boolean (atj-test-value-jboolean value)
                  :char (atj-test-value-jchar value)
                  :byte (atj-test-value-jbyte value)
                  :short (atj-test-value-jshort value)
                  :int (atj-test-value-jint value)
                  :long (atj-test-value-jlong value)
                  :float (atj-test-value-acl2 :irrelevant)
                  :double (atj-test-value-acl2 :irrelevant)))))
       (ptype (atj-type-jprimarr->comp type))
       ((when (or (primitive-type-case ptype :float)
                  (primitive-type-case ptype :double)))
        (er-soft+ ctx t irrelevant
                  "Internal error: type of ~x0 arrays not supported." ptype))
       (constructor (primitive-type-case
                     ptype
                     :boolean 'boolean-array-new-init
                     :char 'char-array-new-init
                     :byte 'byte-array-new-init
                     :short 'short-array-new-init
                     :int 'int-array-new-init
                     :long 'long-array-new-init
                     :float (impossible)
                     :double (impossible)))
       (err-msg (msg "The term ~x0 that is an argument of ~
                      the function call (~x1 ...) that translates ~
                      the test term ~x2 in the :TESTS input, ~
                      must be a call (~x3 X) where X is ~
                      a translated (LIST ...) term ~
                      (i.e. a nest of CONSes ending with a quoted NIL) ~
                      with fewer than 2^32 terms."
                     input
                     fn
                     call
                     constructor))
       ((unless (ffn-symb-p input constructor))
        (er-soft+ ctx t irrelevant "~@0" err-msg))
       (args (fargs input))
       ((unless (= (len args) 1))
        (er-soft+ ctx t irrelevant "~@0" err-msg))
       (arg (car args))
       ((mv okp elements) (check-list-call arg))
       ((unless okp)
        (er-soft+ ctx t irrelevant "~@0" err-msg))
       ((mv erp values state) (atj-process-test-input-jprim-values
                               elements ptype fn input ctx state))
       ((when erp) (mv erp irrelevant state))
       ((unless (< (len values) (expt 2 31)))
        (er-soft+ ctx t irrelevant "~@0" err-msg)))
    (value
     (primitive-type-case
      ptype
      :boolean (atj-test-value-jboolean[] (boolean-array-new-init values))
      :char (atj-test-value-jchar[] (char-array-new-init values))
      :byte (atj-test-value-jbyte[] (byte-array-new-init values))
      :short (atj-test-value-jshort[] (short-array-new-init values))
      :int (atj-test-value-jint[] (int-array-new-init values))
      :long (atj-test-value-jlong[] (long-array-new-init values))
      :float irrelevant
      :double irrelevant))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-process-test-inputs ((inputs pseudo-term-listp)
                                 (types atj-type-listp)
                                 (fn symbolp "Just for error messages.")
                                 (call pseudo-termp "Just for error messages.")
                                 (deep$ booleanp)
                                 (guards$ booleanp)
                                 ctx
                                 state)
  :guard (= (len types) (len inputs))
  :returns (mv erp
               (test-inputs atj-test-value-listp)
               state)
  :short "Lift @(tsee atj-process-test-input) to lists."
  :long
  (xdoc::topstring-p
   "This is used to process all the inputs of a test.")
  (b* (((when (endp inputs)) (value nil))
       ((mv erp test-input state)
        (atj-process-test-input (car inputs)
                                (car types)
                                fn call
                                deep$ guards$
                                ctx state))
       ((when erp) (mv t nil state))
       ((er test-inputs) (atj-process-test-inputs (cdr inputs)
                                                  (cdr types)
                                                  fn call
                                                  deep$ guards$
                                                  ctx state)))
    (value (cons test-input test-inputs))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-process-test (name
                          call
                          (targets$ symbol-listp)
                          (deep$ booleanp)
                          (guards$ booleanp)
                          ctx
                          state)
  :returns (mv erp
               (test$ "An @(tsee atj-testp).")
               state)
  :mode :program ; because of TRANS-EVAL
  :short "Process a test from the @(':tests') input."
  :long
  (xdoc::topstring
   (xdoc::p
    "The first two arguments of this function are the two components
     of a pair in the alist computed from @(':tests').
     These two components are the name of the test and the call of the test.")
   (xdoc::p
    "We first ensure that the name is a non-empty string
     consisting only of letters and digits.
     Then we translate the term (ensuring that the translation succeeds),
     and we ensure that it has the form @('(fn in1 in2 ...)'),
     where @('fn') is one of the target functions.
     We check that all the arguments of the function call
     satisfy the needed requirements (via @(tsee atj-process-test-inputs)),
     obtaining the corresponding input test values.
     If the @(':guards') input is @('t'),
     we ensure that the inputs satisfy the guard of the function.
     We evaluate the call @('(fn in1 in2 ...)'),
     obtaining either a single result value (if @('fn') is single-valued)
     or a list of result values (if @('fn') is multi-valued).
     If @(':deep') is @('nil') and @(':guards') is @('t'),
     we ensure that the inputs will select an overloaded methods,
     and we obtain the corresponding output types
     to contruct the appropriate kind of output test values.
     We create and return an @(tsee atj-test) record.")
   (xdoc::p
    "Note that a single-valued function may return a list.
     So we need to look at the number of results returned by the function
     to recognize the result of the function call from @(tsee trans-eval)
     as either a single list result or a list of multiple results."))
  (b* (((er &) (ensure-value-is-string$
                name
                (msg "The test name ~x0 in the :TESTS input" name)
                t nil))
       ((when (equal name ""))
        (er-soft+ ctx t nil "The test name ~x0 in the :TESTS input ~
                             cannot be the empty string." name))
       ((unless (chars-in-charset-p (explode name) (alpha/digit-chars)))
        (er-soft+ ctx t nil "The test name ~x0 in the :TESTS input ~
                             must contain only letters and digits." name))
       ((er (list term$ &))
        (ensure-value-is-untranslated-term$
         call
         (msg "The test term ~x0 in the :TESTS input" call)
         t nil))
       ((when (or (variablep term$)
                  (fquotep term$)
                  (flambda-applicationp term$)))
        (er-soft+ ctx t nil
                  "The test term ~x0 in the :TESTS input ~
                   must translate to ~
                   the call of a named function." call))
       (fn (ffn-symb term$))
       ((er &) (ensure-value-is-in-list$
                fn
                targets$
                (msg "among the target functions ~&0." targets$)
                (msg "The function ~x0 called by ~
                      the test term ~x1 in the :TESTS input"
                     fn call)
                t nil))
       (inputs (fargs term$))
       (fn-info (atj-get-function-type-info fn guards$ (w state)))
       (main-fn-type (atj-function-type-info->main fn-info))
       (other-fn-types (atj-function-type-info->others fn-info))
       ((er test-inputs)
        (atj-process-test-inputs inputs
                                 (atj-function-type->inputs main-fn-type)
                                 fn
                                 term$
                                 deep$
                                 guards$
                                 ctx
                                 state))
       ((er &) (if guards$
                   (b* ((guard (subcor-var (formals fn (w state))
                                           inputs
                                           (uguard fn (w state))))
                        ((er (cons & guard-satisfied))
                         (trans-eval guard ctx state nil)))
                     (if (not guard-satisfied)
                         (er-soft+ ctx t nil
                                   "The test term ~x0 in the :TESTS input ~
                                    must translate to a function call ~
                                    where the guards are satisfied, ~
                                    because the :GUARDS input ~
                                    is (perhaps by default) T."
                                   call)
                       (value nil)))
                 (value nil)))
       ((er (cons & output/outputs)) (trans-eval term$ ctx state nil))
       (nresults (atj-number-of-results fn (w state)))
       ((when (and (>= nresults 2)
                   (or (not (true-listp output/outputs))
                       (not (equal (len output/outputs)
                                   nresults)))))
        (value (raise "Internal error: ~
                       the function ~x0 returns ~x1 results, ~
                       but evaluating its call returns ~x2, ~
                       which is not a true list of length ~x1."
                      fn nresults output/outputs)))
       (outputs (if (= nresults 1)
                    (list output/outputs)
                  output/outputs))
       ((when (or deep$ (not guards$)))
        (b* ((test-outputs (atj-test-value-acl2-list outputs)))
          (value (atj-test name fn test-inputs test-outputs))))
       (in-types (atj-test-values-to-types test-inputs))
       (all-fn-types (cons main-fn-type other-fn-types))
       (fn-type? (atj-function-type-of-min-input-types in-types all-fn-types))
       ((when (null fn-type?))
        (value (raise "Internal error: ~
                       the test term ~x0 in the :TESTS input ~
                       does not have a corresponding Java overloaded method."
                      call)))
       (out-types (atj-function-type->outputs fn-type?))
       ((unless (= (len outputs) (len out-types)))
        (value (raise "Internal error: ~
                       the number of results ~x0 of ~x1 ~
                       does not match the number ~x2 of its output types."
                      (len outputs) fn (len out-types))))
       (test-outputs (atj-test-values-of-types outputs out-types)))
    (value (atj-test name fn test-inputs test-outputs))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-process-tests (tests
                           (targets$ symbol-listp)
                           (deep$ booleanp)
                           (guards$ booleanp)
                           ctx
                           state)
  :returns (mv erp
               (tests$ "An @(tsee atj-test-listp).")
               state)
  :mode :program ; because of TRANS-EVAL
  :short "Process the @(':tests') input."
  :long
  (xdoc::topstring
   (xdoc::p
    "After evaluating @(':tests')
     and ensuring that the result is a list of doublets,
     we convert it into an alist and we ensure that the keys are unique.
     Then we process each pair in the alist, via an auxiliary function."))
  (b* (((er (cons & tests)) (trans-eval tests ctx state nil))
       (description "The :TESTS input")
       ((er &) (ensure-doublet-list$ tests description t nil))
       (alist (doublets-to-alist tests))
       (names (strip-cars alist))
       (description (msg
                     "The list ~x0 of names of the tests in the :TESTS input"
                     names))
       ((er &) (ensure-list-has-no-duplicates$ names description t nil)))
    (atj-process-tests-aux alist targets$ deep$ guards$ ctx state))

  :prepwork
  ((define atj-process-tests-aux ((tests-alist alistp)
                                  (targets$ symbol-listp)
                                  (deep$ booleanp)
                                  (guards$ booleanp)
                                  ctx
                                  state)
     :returns (mv erp
                  tests$ ; ATJ-TEST-LISTP
                  state)
     :mode :program ; because of TRANS-EVAL in ATJ-PROCESS-TEST
     :parents nil
     (b* (((when (endp tests-alist)) (value nil))
          ((cons (cons name call) tests-alist) tests-alist)
          ((er test$)
           (atj-process-test name call targets$ deep$ guards$ ctx state))
          ((er tests$)
           (atj-process-tests-aux tests-alist
                                  targets$ deep$ guards$ ctx state)))
       (value (cons test$ tests$))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-process-output-dir (output-dir
                                (java-class$ stringp)
                                (tests$ atj-test-listp)
                                ctx
                                state)
  :returns (mv erp
               (result "A tuple
                        @('(output-file$ output-file-env$ output-file-test$)')
                        satisfying
                        @('(typed-tuplep stringp stringp maybe-stringp)'),
                        where @('output-file$') is the path
                        of the generated main Java file,
                        @('output-file-env$') is the path
                        of the generated environment-building Java file,
                        and @('output-file-test$') is
                        @('nil') if the @(':tests') input is @('nil'),
                        otherwise it is the path
                        of the generated test Java file.")
               state)
  :short "Process the @(':output-dir') input."
  (b* (((er &)
        (ensure-value-is-string$ output-dir "The :OUTPUT-DIR input" t nil))
       ((mv err/msg kind state) (oslib::file-kind output-dir))
       ((when (or err/msg
                  (not (eq kind :directory))))
        (er-soft+ ctx t nil
                  "The output directory ~x0 is invalid."
                  output-dir))
       (file (oslib::catpath output-dir
                             (concatenate 'string java-class$ ".java")))
       ((er &) (b* (((mv err/msg exists state) (oslib::path-exists-p file))
                    ((when err/msg)
                     (er-soft+ ctx t nil
                               "The existence of the output path ~x0 ~
                                cannot be tested." file))
                    ((when (not exists)) (value :this-is-irrelevant))
                    ((mv err/msg kind state) (oslib::file-kind file))
                    ((when err/msg)
                     (er-soft+ ctx t nil
                               "The kind of the output path ~x0 ~
                                cannot be tested." file))
                    ((when (not (eq kind :regular-file)))
                     (er-soft+ ctx t nil
                               "The output path ~x0 ~
                                exists but is not a regular file." file)))
                 (value :this-is-irrelevant)))
       (file-env (oslib::catpath output-dir
                                 (concatenate 'string
                                              java-class$
                                              "Environment.java")))
       ((er &) (b* (((mv err/msg exists state) (oslib::path-exists-p file-env))
                    ((when err/msg)
                     (er-soft+ ctx t nil
                               "The existence of the output path ~x0 ~
                                cannot be tested." file-env))
                    ((when (not exists)) (value :this-is-irrelevant))
                    ((mv err/msg kind state) (oslib::file-kind file-env))
                    ((when err/msg)
                     (er-soft+ ctx t nil
                               "The kind of the output path ~x0 ~
                                cannot be tested." file-env))
                    ((when (not (eq kind :regular-file)))
                     (er-soft+ ctx t nil
                               "The output path ~x0 ~
                                exists but is not a regular file." file-env)))
                 (value :this-is-irrelevant)))
       (file-test (if tests$
                      (oslib::catpath output-dir
                                      (concatenate 'string
                                                   java-class$
                                                   "Tests.java"))
                    nil))
       ((er &) (b* (((when (null file-test)) (value :this-is-irrelevant))
                    ((mv err/msg exists state) (oslib::path-exists-p file-test))
                    ((when err/msg)
                     (er-soft+ ctx t nil
                               "The existence of the output path ~x0 ~
                                cannot be tested." file-test))
                    ((when (not exists)) (value :this-is-irrelevant))
                    ((mv err/msg kind state) (oslib::file-kind file-test))
                    ((when err/msg)
                     (er-soft+ ctx t nil
                               "The kind of the output path ~x0 ~
                                cannot be tested." file-test))
                    ((when (not (eq kind :regular-file)))
                     (er-soft+ ctx t nil
                               "The output path ~x0 ~
                                exists but is not a regular file." file-test)))
                 (value :this-is-irrelevant))))
    (value (list file file-env file-test)))
  :guard-hints (("Goal" :in-theory (enable acl2::ensure-value-is-string))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defines atj-collect-fns-in-term
  :short "Collect all the functions in a term,
          in the course of the worklist algorithm."
  :long
  (xdoc::topstring
   (xdoc::p
    "See the "
    (xdoc::seetopic "atj-input-processing" "overview")
    " of the worklist algorithm first.")
   (xdoc::p
    "This is called on the defining body
     of the function removed from the worklist,
     and recursively on subterms of the defining body.")
   (xdoc::p
    "Besides the term, this function takes as arguments
     the two worklists and the two collected lists:
     the @('-gen') suffix stands for `generation',
     i.e. the functions for which Java code must be generated;
     the @('-chk') suffix stands for `checking',
     i.e. the functions that must be just checked.
     The collected lists are only used to see
     which of the functions encountered in the term
     have already been processed by the worklist algorithm.
     The worklists are updated as appropriate,
     and eventually returned.")
   (xdoc::p
    "This function also takes an argument flag @('gen?') saying whether
     we are examining a term from (a function from) the first worklist
     or a term from a function from (a function from) the second worklist.")
   (xdoc::p
    "Since variables and quoted constants contain no functions,
     we return the worklists unchanged in these cases.")
   (xdoc::p
    "A term @('(mbe :logic a :exec b)')
     is translated to @('(return-last \'acl2::mbe1-raw b a)').
     When @(':guards') is @('nil'), we translate @('a') to Java,
     but still need to check @('b') for side effects:
     thus, we recursively descend into @('a') with the current @('gen?') flag,
     and we recursively descend into @('b') with @('gen?') set to @('nil').
     When @(':guards') is @('t'),
     the treatment of @('a') and @('b') is reversed.")
   (xdoc::p
    "A term @('(prog2$ a b)')
     is translated to @('(return-last \'acl2::progn a b)')
     (and @(tsee progn$) is translated into a nest of @(tsee prog2$)s).
     Thus, when we encounter this kind of call,
     when we descend into the argument @('a')
     we set the @('gen?') flag to @('nil'),
     while when we descend into the argument @('b')
     we leave the @('gen?') flag unchanged.")
   (xdoc::p
    "If we encounter a call of @(tsee return-last) of some other form,
     we immediately return because such other forms are not supported.
     In this case, the third result of the function is set to @('t'),
     so that the caller can immediately recognize the situation
     and cause the iteration to terminate.")
   (xdoc::p
    "If we encounter a call of anything other than @(tsee return-last),
     we recursively process the arguments,
     propagating any error signaled by the third result.")
   (xdoc::p
    "If the call is of a lambda expression,
     we conclude by recursively processing
     the body of the lambda expression.")
   (xdoc::p
    "Otherwise, the call is of a named function (not @(tsee return-last)).
     We add the function to the appropriate worklist
     (the exact worklist is determined by the @('gen?') flag),
     unless it is already there or in a collected list.
     If @('gen?') is @('t') and the function is already
     in @('worklist-chk') or @('collected-chk')
     but not in @('worklist-gen') or @('collected-gen'),
     we need to add it to @('worklist-gen') nonetheless,
     because it must eventually end up in @('collected-gen')
     in order to generate code for it.
     Thus, if @('gen?') is @('t'),
     we only check it against @('worklist-gen') and @('collected-gen'),
     and if we add it to @('worklist-gen')
     we also remove it from @('worklist-chk') if present there
     (with @(tsee remove1) because worklists never have duplicates;
     if it is not present, no change to @('worklist-chk') occurs),
     so that the function is not processed again.
     We do not need to remove the function from @('collected-chk')
     because, when @('gen?') is @('t'), that collected list is always empty:
     the reason is that the iteration
     first processes @('worklist-gen') completely
     (during this processing @('gen?') is @('t')),
     keeping @('collected-chk') empty,
     and then it processes @('worklist-chk'),
     and it is during this processing (when @('gen?') is thus @('nil'))
     that @('collected-chk') gets populated.")
   (xdoc::p
    "We also return a duplicate-free list of
     the function symbols called by the term
     for which Java code must be generated."))

  (define atj-collect-fns-in-term ((term pseudo-termp)
                                   (gen? booleanp)
                                   (worklist-gen symbol-listp)
                                   (worklist-chk symbol-listp)
                                   (called-fns symbol-listp)
                                   (collected-gen symbol-listp)
                                   (collected-chk symbol-listp)
                                   (deep$ booleanp)
                                   (guards$ booleanp))
    :returns (mv (new-worklist-gen symbol-listp :hyp :guard)
                 (new-worklist-chk symbol-listp :hyp :guard)
                 (new-called-fns symbol-listp :hyp :guard)
                 (unsuppported-return-last? booleanp))
    (b* (((when (member-eq (pseudo-term-kind term)
                           '(:null :var :quote)))
          (mv worklist-gen worklist-chk called-fns nil))
         (fn (pseudo-term-call->fn term))
         (args (pseudo-term-call->args term))
         ((when (eq fn 'return-last))
          (b* ((1st-arg (first args))
               ((unless (pseudo-term-case 1st-arg :quote))
                (raise "Internal error: ~
                        the first argument of ~x0 is not a quoted constant."
                       term)
                (mv worklist-gen worklist-chk called-fns nil))) ; irrelevant
            (case (pseudo-term-quote->val 1st-arg)
              (acl2::mbe1-raw
               (if guards$
                   (b* (((mv worklist-gen
                             worklist-chk
                             called-fns
                             unsuppported-return-last?)
                         (atj-collect-fns-in-term (third args)
                                                  nil
                                                  worklist-gen
                                                  worklist-chk
                                                  called-fns
                                                  collected-gen
                                                  collected-chk
                                                  deep$
                                                  guards$))
                        ((when unsuppported-return-last?)
                         (mv worklist-gen worklist-chk called-fns t)))
                     (atj-collect-fns-in-term (second args)
                                              gen?
                                              worklist-gen
                                              worklist-chk
                                              called-fns
                                              collected-gen
                                              collected-chk
                                              deep$
                                              guards$))
                 (b* (((mv worklist-gen
                           worklist-chk
                           called-fns
                           unsuppported-return-last?)
                       (atj-collect-fns-in-term (second args)
                                                nil
                                                worklist-gen
                                                worklist-chk
                                                called-fns
                                                collected-gen
                                                collected-chk
                                                deep$
                                                guards$))
                      ((when unsuppported-return-last?)
                       (mv worklist-gen worklist-chk called-fns t)))
                   (atj-collect-fns-in-term (third args)
                                            gen?
                                            worklist-gen
                                            worklist-chk
                                            called-fns
                                            collected-gen
                                            collected-chk
                                            deep$
                                            guards$))))
              (acl2::progn
               (b* (((mv worklist-gen
                         worklist-chk
                         called-fns
                         unsuppported-return-last?)
                     (atj-collect-fns-in-term (second args)
                                              nil
                                              worklist-gen
                                              worklist-chk
                                              called-fns
                                              collected-gen
                                              collected-chk
                                              deep$
                                              guards$))
                    ((when unsuppported-return-last?)
                     (mv worklist-gen worklist-chk called-fns t)))
                 (atj-collect-fns-in-term (third args)
                                          gen?
                                          worklist-gen
                                          worklist-chk
                                          called-fns
                                          collected-gen
                                          collected-chk
                                          deep$
                                          guards$)))
              (t (mv worklist-gen worklist-chk called-fns t)))))
         ((mv worklist-gen worklist-chk called-fns unsupported-return-last?)
          (atj-collect-fns-in-terms args
                                    gen?
                                    worklist-gen
                                    worklist-chk
                                    called-fns
                                    collected-gen
                                    collected-chk
                                    deep$
                                    guards$))
         ((when unsupported-return-last?)
          (mv worklist-gen worklist-chk called-fns t))
         ((when (consp fn))
          (atj-collect-fns-in-term (pseudo-lambda->body fn)
                                   gen?
                                   worklist-gen
                                   worklist-chk
                                   called-fns
                                   collected-gen
                                   collected-chk
                                   deep$
                                   guards$)))
      (if gen?
          (b* ((called-fns (add-to-set-eq fn called-fns)))
            (if (or (member-eq fn worklist-gen)
                    (member-eq fn collected-gen))
                (mv worklist-gen worklist-chk called-fns nil)
              (mv (cons fn worklist-gen)
                  (remove1-eq fn worklist-chk)
                  called-fns
                  nil)))
        (if (or (member-eq fn worklist-gen)
                (member-eq fn worklist-chk)
                (member-eq fn collected-gen)
                (member-eq fn collected-chk))
            (mv worklist-gen worklist-chk called-fns nil)
          (mv worklist-gen
              (cons fn worklist-chk)
              called-fns
              nil))))
    :measure (pseudo-term-count term))

  (define atj-collect-fns-in-terms ((terms pseudo-term-listp)
                                    (gen? booleanp)
                                    (worklist-gen symbol-listp)
                                    (worklist-chk symbol-listp)
                                    (called-fns symbol-listp)
                                    (collected-gen symbol-listp)
                                    (collected-chk symbol-listp)
                                    (deep$ booleanp)
                                    (guards$ booleanp))
    :returns (mv (new-worklist-gen symbol-listp :hyp :guard)
                 (new-worklist-chk symbol-listp :hyp :guard)
                 (new-called-fns symbol-listp :hyp :guard)
                 (unsuppported-return-last? booleanp))
    (b* (((when (endp terms)) (mv worklist-gen worklist-chk called-fns nil))
         ((mv worklist-gen worklist-chk called-fns unsuppported-return-last?)
          (atj-collect-fns-in-term (car terms)
                                   gen?
                                   worklist-gen
                                   worklist-chk
                                   called-fns
                                   collected-gen
                                   collected-chk
                                   deep$
                                   guards$))
         ((when unsuppported-return-last?)
          (mv worklist-gen worklist-chk called-fns t)))
      (atj-collect-fns-in-terms (cdr terms)
                                gen?
                                worklist-gen
                                worklist-chk
                                called-fns
                                collected-gen
                                collected-chk
                                deep$
                                guards$))
    :measure (pseudo-term-list-count terms))

  :prepwork ((local (include-book "std/typed-lists/symbol-listp" :dir :system))
             (local (in-theory
                     ;; just to speed up the proofs:
                     (disable pseudo-termp
                              member-equal
                              acl2::member-of-cons
                              acl2::symbol-listp-when-subsetp-equal))))

  :verify-guards nil ; done below
  ///
  (verify-guards atj-collect-fns-in-term
    :hints (("Goal"
             :expand (pseudo-termp term)
             :in-theory (enable member-equal acl2::member-of-cons)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-worklist-iterate ((worklist-gen symbol-listp)
                              (worklist-chk symbol-listp)
                              (collected-gen symbol-listp)
                              (collected-chk symbol-listp)
                              (call-graph symbol-symbollist-alistp)
                              (deep$ booleanp)
                              (guards$ booleanp)
                              (ignore-whitelist$ booleanp)
                              (verbose$ booleanp)
                              ctx
                              state)
  :returns (mv erp
               (result "A tuple @('(fns new-call-graph)') satisfying
                        @('(typed-tuplep symbol-listp
                                         symbol-symbollist-alistp
                                         result)').")
               state)
  :mode :program ; until termination is proved (which will take a bit of work)
  :short "Worklist algorithm iteration."
  :long
  (xdoc::topstring
   (xdoc::p
    "See the "
    (xdoc::seetopic "atj-input-processing" "overview")
    " of the worklist algorithm first.")
   (xdoc::p
    "The iteration ends when both worklists are empty.
     When that happens, we return the collected list of functions
     for which code must be generated.
     We also return a call graph of these functions,
     as an alist from each function
     to a list of its directly called functions.")
   (xdoc::p
    "We always pick the next function from @('worklist-gen'),
     until it is empty; then we switch to @('worklist-chk').
     Since we start with all empty lists except @('worklist-gen')
     (see the caller of this function),
     as we go through @('worklist-gen') we may populate
     @('collected-gen') and @('worklist-chk'),
     but not @('collected-chk').
     When we go through @('worklist-chk'),
     we may further populate @('collected-chk'),
     but not @('worklist-gen') or @('collected-gen').")
   (xdoc::codeblock
    "Initial lists:"
    "  (...) () () ()"
    "After processing the first worklist:"
    "  () (...) (...) ()"
    "After processing the second worklist:"
    "  () () (...) (...)")
   (xdoc::p
    "The iteration terminates because
     there is a finite number of functions in the ACL2 world,
     but for simplicity we leave this function in program mode
     to avoid having to articulate the termination proof for now.")
   (xdoc::p
    "When we encounter a function that is natively implemented in AIJ,
     we do not examine its body
     (which the ACL2 primitive functions,
     all of which are natively implemented in AIJ,
     do not have anyhow):
     we just remove it from the worklist,
     and, if @('gen?') is @('t'),
     we add it to @('collected-gen'),
     i.e. we include among the functions for which code must be generated.
     When @(':deep') is @('nil') and @(':guards') is @('t'),
     we apply the same treatment to the functions
    in @(tsee *atj-jprim-fns*) and @(tsee *atj-jprimarr-fns*).")
   (xdoc::p
    "If the function satisfies all the needed constraints,
     its name is printed when verbose mode is on.
     The caller of this function precedes this printing
     with a suitable message (see the caller).")
   (xdoc::p
    "It should be an invariant that there are no duplicate function symbols
     in the four lists (worklists and collected lists) altogether;
     i.e. each list is free of duplicates,
     and the lists are pairwise disjoint.")
   (xdoc::p
    "Note that since @(tsee atj-collect-fns-in-term)
     extends the worklists via @(tsee cons),
     and since the fixpoint iteration picks the next function via @(tsee car),
     we visit the call graph depth-first;
     the worklists are used as stacks."))
  (b* (((when (and (endp worklist-gen)
                   (endp worklist-chk)))
        (value (list collected-gen call-graph)))
       ((mv fn
            gen?
            worklist-gen
            worklist-chk)
        (if (consp worklist-gen)
            (mv (car worklist-gen)
                t
                (cdr worklist-gen)
                worklist-chk)
          (mv (car worklist-chk)
              nil
              worklist-gen
              (cdr worklist-chk))))
       ((when (or (aij-nativep fn)
                  (and (not deep$)
                       guards$
                       (or (atj-jprim-fn-p fn)
                           (atj-jprimarr-fn-p fn)))))
        (b* (((mv collected-gen collected-chk)
              (if gen?
                  (mv (cons fn collected-gen) collected-chk)
                (mv collected-gen collected-chk))))
          (atj-worklist-iterate worklist-gen
                                worklist-chk
                                collected-gen
                                collected-chk
                                call-graph
                                deep$
                                guards$
                                ignore-whitelist$
                                verbose$
                                ctx state)))
       ((when (and (rawp fn state)
                   (not ignore-whitelist$)
                   (not (pure-raw-p fn))))
        (er-soft+ ctx t nil
                  "The function ~x0 has raw Lisp code ~
                   and is not in the whitelist; ~
                   therefore, code generation cannot proceed." fn))
       ((unless (no-stobjs-p fn (w state)))
        (er-soft+ ctx t nil
                  "The function ~x0 has input or output stobjs; ~
                   therefore, code generation cannot proceed." fn))
       (body (atj-fn-body fn (w state)))
       ((unless body)
        (er-soft+ ctx t nil
                  "The function ~x0 has no unnormalized body ~
                   and no suitable attachment; ~
                   therefore, code generation cannot proceed." fn))
       ((run-when verbose$)
        (cw "  ~x0~%" fn))
       ((mv collected-gen collected-chk)
        (if gen?
            (mv (cons fn collected-gen) collected-chk)
          (mv collected-gen (cons fn collected-chk))))
       ((mv worklist-gen worklist-chk called-fns unsuppported-return-last?)
        (atj-collect-fns-in-term body
                                 gen?
                                 worklist-gen
                                 worklist-chk
                                 nil
                                 collected-gen
                                 collected-chk
                                 deep$
                                 guards$))
       ((when unsuppported-return-last?)
        (er-soft+ ctx t nil
                  "The function RETURN-LAST is used ~
                   with an unsupported first argument; ~
                   therefore, code generation cannot proceed."))
       (call-graph (acons fn called-fns call-graph)))
    (atj-worklist-iterate worklist-gen
                          worklist-chk
                          collected-gen
                          collected-chk
                          call-graph
                          deep$
                          guards$
                          ignore-whitelist$
                          verbose$
                          ctx state)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-fns-to-translate ((targets$ symbol-listp)
                              (deep$ booleanp)
                              (guards$ booleanp)
                              (ignore-whitelist$ booleanp)
                              (verbose$ booleanp)
                              ctx
                              state)
  :returns (mv erp
               (result "A tuple @('(fns-to-translate call-graph')) satisfying
                        @('(typed-tuplep symbol-listp
                                         symbol-symbollist-alistp
                                         result)').")
               state)
  :mode :program ; because of ATJ-WORKLIST-ITERATE
  :short "Collect the names of all the ACL2 functions to be translated to Java,
          checking that they satisfy all the necessary constraints."
  :long
  (xdoc::topstring
   (xdoc::p
    "See the "
    (xdoc::seetopic "atj-input-processing" "overview")
    " of the worklist algorithm first.")
   (xdoc::p
    "We start the worklist iteration with the targets supplied by the user.")
   (xdoc::p
    "The returned list of function names should have no duplicates,
     but we double-check that for robustness.
     The list is in no particular order.")
   (xdoc::p
    "We also return the call graph of those functions."))
  (b* (((run-when verbose$)
        (cw "~%ACL2 functions to translate to Java:~%"))
       (worklist-gen targets$)
       ((er (list fns call-graph))
        (atj-worklist-iterate worklist-gen
                              nil
                              nil
                              nil
                              nil
                              deep$
                              guards$
                              ignore-whitelist$
                              verbose$
                              ctx
                              state))
       ((unless (no-duplicatesp-eq fns))
        (value (raise "Internal error: ~
                       the list ~x0 of collected function names ~
                       has duplicates."
                      fns))))
    (value (list fns call-graph))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-pkgs-to-translate ((verbose$ booleanp) state)
  :returns (pkgs string-listp)
  :short "Collect all the ACL2 packages to be translated to Java."
  :long
  (xdoc::topstring
   (xdoc::p
    "Here `translate to Java' really means `build a Java representation of'.")
   (xdoc::p
    "For now we return all the current packages.
     In the future, it might be possible to reduce them
     to just the ones referenced by the functions to be translated to Java."))
  (b* ((pkgs (known-packages+ state))
       ((run-when verbose$)
        (cw "~%Known ACL2 packages:~%")
        (atj-show-pkgs pkgs)))
    pkgs)

  :prepwork
  ((define atj-show-pkgs ((pkgs string-listp))
     :returns (nothing null)
     :parents nil
     (if (endp pkgs)
         nil
       (b* ((- (cw "  ~s0~%" (car pkgs))))
         (atj-show-pkgs (cdr pkgs)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defval *atj-allowed-options*
  :short "Keyword options accepted by @(tsee atj)."
  (list :deep
        :guards
        :java-package
        :java-class
        :output-dir
        :tests
        :ignore-whitelist
        :verbose)
  ///
  (assert-event (symbol-listp *atj-allowed-options*))
  (assert-event (no-duplicatesp-eq *atj-allowed-options*)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-process-inputs ((args true-listp) ctx state)
  :returns (mv erp
               (result "A tuple @('(fns-to-translate
                                    call-graph
                                    pkgs
                                    deep$
                                    guards$
                                    java-package$
                                    java-class$
                                    output-file$
                                    output-file-env$
                                    output-file-test$
                                    tests$
                                    verbose$)')
                        satisfying
                        @('(typed-tuplep symbol-listp
                                         symbol-symbollist-alistp
                                         string-listp
                                         booleanp
                                         booleanp
                                         maybe-stringp
                                         stringp
                                         stringp
                                         stringp
                                         maybe-stringp
                                         atj-test-listp
                                         booleanp
                                         result)').")
               state)
  :mode :program ; because of ATJ-FNS-TO-TRANSLATE and ATJ-PROCESS-TESTS
  :short "Ensure that the inputs to @(tsee atj) are valid."
  :long
  (xdoc::topstring
   (xdoc::p
    "We process the inputs in order,
     except that @(':output-dir') is processed after @(':tests')
     because the result of processing the latter
     is used in processing the former.")
   (xdoc::p
    "We also collect, check, and return the functions
     for which code must be generated.
     We also collect and return the packages
     whose representation must be built in Java;
     for now these are all the current packages,
     but it might be possible to reduce them
     to just the ones referenced by the functions."))
  (b* (((mv erp targets options) (partition-rest-and-keyword-args
                                  args *atj-allowed-options*))
       ((when erp) (er-soft+ ctx t nil
                             "The inputs must be the names of ~
                              one or more target functions ~
                              followed by the options ~&0."
                             *atj-allowed-options*))
       (deep (cdr (assoc-eq :deep options)))
       (guards (b* ((pair? (assoc-eq :guards options)))
                 (if (consp pair?)
                     (cdr pair?)
                   t)))
       (java-package (cdr (assoc-eq :java-package options)))
       (java-class (cdr (assoc-eq :java-class options)))
       (output-dir (or (cdr (assoc-eq :output-dir options)) "."))
       (tests (cdr (assoc-eq :tests options)))
       (ignore-whitelist (cdr (assoc-eq :ignore-whitelist options)))
       (verbose (cdr (assoc-eq :verbose options)))
       ((er &) (atj-process-targets targets deep guards ctx state))
       ((er &) (ensure-value-is-boolean$ deep "The :DEEP intput" t nil))
       ((er &) (ensure-value-is-boolean$ guards "The :GUARDS intput" t nil))
       ((er &) (atj-process-java-package java-package ctx state))
       ((er java-class$) (atj-process-java-class java-class ctx state))
       ((er tests$) (atj-process-tests tests targets deep guards ctx state))
       ((er (list output-file$
                  output-file-env$
                  output-file-test$))
        (atj-process-output-dir output-dir java-class$ tests$ ctx state))
       ((er &) (ensure-value-is-boolean$ ignore-whitelist
                                         "The :IGNORE-WHITELIST input" t nil))
       ((er &) (ensure-value-is-boolean$ verbose "The :VERBOSE input" t nil))
       ((er (list fns-to-translate call-graph))
        (atj-fns-to-translate
         targets deep guards ignore-whitelist verbose ctx state))
       (pkgs (atj-pkgs-to-translate verbose state)))
    (value (list fns-to-translate
                 call-graph
                 pkgs
                 deep
                 guards
                 java-package
                 java-class$
                 output-file$
                 output-file-env$
                 output-file-test$
                 tests$
                 verbose))))
