/*
 * Copyright 2015 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors:
 *     Lars Uebernickel <lars.uebernickel@canonical.com>
 */

#include <gio/gio.h>
#include <locale.h>
#include <geonames.h>

static void
assert_first (const gchar *query,
              const gchar *expected_city,
              const gchar *expected_country,
              const gchar *expected_country_code,
              gdouble expected_latitude,
              gdouble expected_longitude,
              guint expected_population)
{
  g_autofree gint *indices;
  guint len;
  g_autoptr(GeonamesCity) city = NULL;

  indices = geonames_query_cities_sync (query, GEONAMES_QUERY_DEFAULT, &len, NULL, NULL);
  g_assert (indices);
  g_assert_cmpint (indices[len], ==, -1);

  g_assert_cmpint (len, >, 0);
  city = geonames_get_city (indices[0]);

  g_assert_cmpstr (geonames_city_get_name (city), ==, expected_city);
  g_assert_cmpstr (geonames_city_get_country (city), ==, expected_country);
  g_assert_cmpstr (geonames_city_get_country_code (city), ==, expected_country_code);
  g_assert_cmpfloat (geonames_city_get_latitude (city), ==, expected_latitude);
  g_assert_cmpfloat (geonames_city_get_longitude (city), ==, expected_longitude);
  g_assert_cmpint (geonames_city_get_population (city), ==, expected_population);
}

static void
test_common_cities (void)
{
  assert_first ("berlin", "Berlin", "Germany", "DE", 52.52437, 13.41053, 3426354);
  assert_first ("new york", "New York City", "United States", "US", 40.71427, -74.00597, 8175133);
  assert_first ("san fran", "San Francisco", "United States", "US", 37.77493, -122.41942, 805235);
  assert_first ("amster", "Amsterdam", "Netherlands", "NL", 52.37403, 4.88969, 741636);
  assert_first ("montreal", "Montréal", "Canada", "CA", 45.50884, -73.58781, 3268513);
  assert_first ("montréal", "Montréal", "Canada", "CA", 45.50884, -73.58781, 3268513);
}

static void
test_edge_cases (void)
{
  gint *indices;
  guint len;

  indices = geonames_query_cities_sync ("", GEONAMES_QUERY_DEFAULT, &len, NULL, NULL);
  g_assert_cmpint (len, ==, 0);
  g_assert_cmpint (indices[0], ==, -1);
  g_free (indices);

  indices = geonames_query_cities_sync ("a city that doesn't exist", GEONAMES_QUERY_DEFAULT, &len, NULL, NULL);
  g_assert_cmpint (len, ==, 0);
  g_assert_cmpint (indices[0], ==, -1);
  g_free (indices);
}

int
main (int argc, char **argv)
{
  setlocale (LC_ALL, "");
  g_test_init (&argc, &argv, NULL);

  g_test_add_func ("/common-cities", test_common_cities);
  g_test_add_func ("/edge-cases", test_edge_cases);

  return g_test_run ();
}
