/*
 *  Exhaustive test for case conversions of String.prototype for single
 *  character BMP inputs.  This test does not cover context sensitive rules,
 *  which only apply when a character is (or is not) surrounded by specified
 *  other characters.
 */

/*---
{
    "slow": true
}
---*/

/*
 *  Helper to generate case conversion tables for all 16-bit codepoints
 *  for an existing implementation.  Encoding is packed to minimize test
 *  file size.
 *
 *  NOTE: toLocaleUpperCase() and toLocaleLowerCase() tables are dependent
 *  on the current locale (at least in principle).
 */

function encode_unicode(x) {
    var i;
    var t;
    var res = [];

    for (i = 0; i < x.length; i++) {
        t = x.charCodeAt(i);
        res.push(t.toString(16));
    }
    return res.join('-');
}

function generate(funcname) {
    var i;
    var t, s;
    var tab = [];

    for (i = 0; i < 65536; i++) {
        t = String.fromCharCode(i);
        t = t[funcname]();
        if (t.length === 1 && t.charCodeAt(0) === i) {
            // Majority of codepoints map to themselves, so shorthand empty
            // string to mean as-is.
            tab.push('');
        } else {
            tab.push(encode_unicode(t));
        }
    }

    var res = [];
    var perline = 256;
    for (i = 0; i < 65536; i += perline) {
        var tmp = "'";
        for (j = 0; j < perline; j++) {
            tmp += tab[i + j] + '=';
        }
        tmp += "'";
        res.push(tmp);
    }

    // Result has a trailing '=' but it's ignored when parsing.

    return res.join(' +\n');
}

function parse_table(x) {
    var tab = x.split('=').map(function (v, idx) {
        if (v === '') {
            return String.fromCharCode(idx);
        } else {
            return v.split('-').map(function (v) { return String.fromCharCode(parseInt(v, 16)); }).join('');
        }
    });
    // tab.length is 65537 due to trailing '=', drop it.
    tab.length = 65536;
    return tab;
}

/*
var uc = generate('toUpperCase');
print('var uc = ' + uc + ';');
var luc = generate('toLocaleUpperCase');
print('var luc = ' + luc + ';');
var lc = generate('toLowerCase');
print('var lc = ' + lc + ';');
var llc = generate('toLocaleLowerCase');
print('var llc = ' + llc + ';');
*/

/* Generated from:
 * $ nodejs
 * > process.version
 * 'v11.2.0'
 */

var expected_uc = parse_table('=================================================================================================41=42=43=44=45=46=47=48=49=4a=4b=4c=4d=4e=4f=50=51=52=53=54=55=56=57=58=59=5a===========================================================39c==========================================53-53=c0=c1=c2=c3=c4=c5=c6=c7=c8=c9=ca=cb=cc=cd=ce=cf=d0=d1=d2=d3=d4=d5=d6==d8=d9=da=db=dc=dd=de=178=' +
'=100==102==104==106==108==10a==10c==10e==110==112==114==116==118==11a==11c==11e==120==122==124==126==128==12a==12c==12e==49==132==134==136===139==13b==13d==13f==141==143==145==147=2bc-4e==14a==14c==14e==150==152==154==156==158==15a==15c==15e==160==162==164==166==168==16a==16c==16e==170==172==174==176===179==17b==17d=53=243===182==184===187====18b======191===1f6====198=23d====220===1a0==1a2==1a4===1a7=====1ac===1af====1b3==1b5===1b8====1bc==1f7======1c4=1c4==1c7=1c7==1ca=1ca==1cd==1cf==1d1==1d3==1d5==1d7==1d9==1db=18e==1de==1e0==1e2==1e4==1e6==1e8==1ea==1ec==1ee=4a-30c==1f1=1f1==1f4====1f8==1fa==1fc==1fe=' +
'=200==202==204==206==208==20a==20c==20e==210==212==214==216==218==21a==21c==21e====222==224==226==228==22a==22c==22e==230==232=========23b===2c7e=2c7f==241=====246==248==24a==24c==24e=2c6f=2c6d=2c70=181=186==189=18a==18f==190=a7ab====193=a7ac==194==a78d=a7aa==197=196=a7ae=2c62=a7ad===19c==2c6e=19d===19f========2c64===1a6===1a9====a7b1=1ae=244=1b1=1b2=245======1b7===========a7b2=a7b0==================================================================================================' +
'=====================================================================399============================================370==372====376====3fd=3fe=3ff===================399-308-301============================386=388=389=38a=3a5-308-301=391=392=393=394=395=396=397=398=399=39a=39b=39c=39d=39e=39f=3a0=3a1=3a3=3a3=3a4=3a5=3a6=3a7=3a8=3a9=3aa=3ab=38c=38e=38f==392=398====3a6=3a0=3cf==3d8==3da==3dc==3de==3e0==3e2==3e4==3e6==3e8==3ea==3ec==3ee=39a=3a1=3f9=37f==395===3f7===3fa=====' +
'================================================410=411=412=413=414=415=416=417=418=419=41a=41b=41c=41d=41e=41f=420=421=422=423=424=425=426=427=428=429=42a=42b=42c=42d=42e=42f=400=401=402=403=404=405=406=407=408=409=40a=40b=40c=40d=40e=40f==460==462==464==466==468==46a==46c==46e==470==472==474==476==478==47a==47c==47e==480==========48a==48c==48e==490==492==494==496==498==49a==49c==49e==4a0==4a2==4a4==4a6==4a8==4aa==4ac==4ae==4b0==4b2==4b4==4b6==4b8==4ba==4bc==4be===4c1==4c3==4c5==4c7==4c9==4cb==4cd=4c0==4d0==4d2==4d4==4d6==4d8==4da==4dc==4de==4e0==4e2==4e4==4e6==4e8==4ea==4ec==4ee==4f0==4f2==4f4==4f6==4f8==4fa==4fc==4fe=' +
'=500==502==504==506==508==50a==50c==50e==510==512==514==516==518==51a==51c==51e==520==522==524==526==528==52a==52c==52e==================================================531=532=533=534=535=536=537=538=539=53a=53b=53c=53d=53e=53f=540=541=542=543=544=545=546=547=548=549=54a=54b=54c=54d=54e=54f=550=551=552=553=554=555=556=535-552=========================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================1c90=1c91=1c92=1c93=1c94=1c95=1c96=1c97=1c98=1c99=1c9a=1c9b=1c9c=1c9d=1c9e=1c9f=1ca0=1ca1=1ca2=1ca3=1ca4=1ca5=1ca6=1ca7=1ca8=1ca9=1caa=1cab=1cac=1cad=1cae=1caf=1cb0=1cb1=1cb2=1cb3=1cb4=1cb5=1cb6=1cb7=1cb8=1cb9=1cba===1cbd=1cbe=1cbf=' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'========================================================================================================================================================================================================================================================13f0=13f1=13f2=13f3=13f4=13f5===' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================412=414=41e=421=422=422=42a=462=a64a========================================================================================================================' +
'=========================================================================================================================a77d====2c63===================================================================================================================================' +
'=1e00==1e02==1e04==1e06==1e08==1e0a==1e0c==1e0e==1e10==1e12==1e14==1e16==1e18==1e1a==1e1c==1e1e==1e20==1e22==1e24==1e26==1e28==1e2a==1e2c==1e2e==1e30==1e32==1e34==1e36==1e38==1e3a==1e3c==1e3e==1e40==1e42==1e44==1e46==1e48==1e4a==1e4c==1e4e==1e50==1e52==1e54==1e56==1e58==1e5a==1e5c==1e5e==1e60==1e62==1e64==1e66==1e68==1e6a==1e6c==1e6e==1e70==1e72==1e74==1e76==1e78==1e7a==1e7c==1e7e==1e80==1e82==1e84==1e86==1e88==1e8a==1e8c==1e8e==1e90==1e92==1e94=48-331=54-308=57-30a=59-30a=41-2be=1e60======1ea0==1ea2==1ea4==1ea6==1ea8==1eaa==1eac==1eae==1eb0==1eb2==1eb4==1eb6==1eb8==1eba==1ebc==1ebe==1ec0==1ec2==1ec4==1ec6==1ec8==1eca==1ecc==1ece==1ed0==1ed2==1ed4==1ed6==1ed8==1eda==1edc==1ede==1ee0==1ee2==1ee4==1ee6==1ee8==1eea==1eec==1eee==1ef0==1ef2==1ef4==1ef6==1ef8==1efa==1efc==1efe=' +
'1f08=1f09=1f0a=1f0b=1f0c=1f0d=1f0e=1f0f=========1f18=1f19=1f1a=1f1b=1f1c=1f1d===========1f28=1f29=1f2a=1f2b=1f2c=1f2d=1f2e=1f2f=========1f38=1f39=1f3a=1f3b=1f3c=1f3d=1f3e=1f3f=========1f48=1f49=1f4a=1f4b=1f4c=1f4d===========3a5-313=1f59=3a5-313-300=1f5b=3a5-313-301=1f5d=3a5-313-342=1f5f=========1f68=1f69=1f6a=1f6b=1f6c=1f6d=1f6e=1f6f=========1fba=1fbb=1fc8=1fc9=1fca=1fcb=1fda=1fdb=1ff8=1ff9=1fea=1feb=1ffa=1ffb===1f08-399=1f09-399=1f0a-399=1f0b-399=1f0c-399=1f0d-399=1f0e-399=1f0f-399=1f08-399=1f09-399=1f0a-399=1f0b-399=1f0c-399=1f0d-399=1f0e-399=1f0f-399=1f28-399=1f29-399=1f2a-399=1f2b-399=1f2c-399=1f2d-399=1f2e-399=1f2f-399=1f28-399=1f29-399=1f2a-399=1f2b-399=1f2c-399=1f2d-399=1f2e-399=1f2f-399=1f68-399=1f69-399=1f6a-399=1f6b-399=1f6c-399=1f6d-399=1f6e-399=1f6f-399=1f68-399=1f69-399=1f6a-399=1f6b-399=1f6c-399=1f6d-399=1f6e-399=1f6f-399=1fb8=1fb9=1fba-399=391-399=386-399==391-342=391-342-399=====391-399==399====1fca-399=397-399=389-399==397-342=397-342-399=====397-399====1fd8=1fd9=399-308-300=399-308-301===399-342=399-308-342=========1fe8=1fe9=3a5-308-300=3a5-308-301=3a1-313=1fec=3a5-342=3a5-308-342===========1ffa-399=3a9-399=38f-399==3a9-342=3a9-342-399=====3a9-399====' +
'================================================================================================================================================================================================================================================================' +
'==============================================================================2132==================================2160=2161=2162=2163=2164=2165=2166=2167=2168=2169=216a=216b=216c=216d=216e=216f=====2183============================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================24b6=24b7=24b8=24b9=24ba=24bb=24bc=24bd=24be=24bf=24c0=24c1=24c2=24c3=24c4=24c5=24c6=24c7=24c8=24c9=24ca=24cb=24cc=24cd=24ce=24cf=======================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================2c00=2c01=2c02=2c03=2c04=2c05=2c06=2c07=2c08=2c09=2c0a=2c0b=2c0c=2c0d=2c0e=2c0f=2c10=2c11=2c12=2c13=2c14=2c15=2c16=2c17=2c18=2c19=2c1a=2c1b=2c1c=2c1d=2c1e=2c1f=2c20=2c21=2c22=2c23=2c24=2c25=2c26=2c27=2c28=2c29=2c2a=2c2b=2c2c=2c2d=2c2e===2c60====23a=23e==2c67==2c69==2c6b=======2c72===2c75===========2c80==2c82==2c84==2c86==2c88==2c8a==2c8c==2c8e==2c90==2c92==2c94==2c96==2c98==2c9a==2c9c==2c9e==2ca0==2ca2==2ca4==2ca6==2ca8==2caa==2cac==2cae==2cb0==2cb2==2cb4==2cb6==2cb8==2cba==2cbc==2cbe==2cc0==2cc2==2cc4==2cc6==2cc8==2cca==2ccc==2cce==2cd0==2cd2==2cd4==2cd6==2cd8==2cda==2cdc==2cde==2ce0==2ce2=========2ceb==2ced=====2cf2=============' +
'10a0=10a1=10a2=10a3=10a4=10a5=10a6=10a7=10a8=10a9=10aa=10ab=10ac=10ad=10ae=10af=10b0=10b1=10b2=10b3=10b4=10b5=10b6=10b7=10b8=10b9=10ba=10bb=10bc=10bd=10be=10bf=10c0=10c1=10c2=10c3=10c4=10c5==10c7======10cd===================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'=================================================================a640==a642==a644==a646==a648==a64a==a64c==a64e==a650==a652==a654==a656==a658==a65a==a65c==a65e==a660==a662==a664==a666==a668==a66a==a66c====================a680==a682==a684==a686==a688==a68a==a68c==a68e==a690==a692==a694==a696==a698==a69a=====================================================================================================' +
'===================================a722==a724==a726==a728==a72a==a72c==a72e====a732==a734==a736==a738==a73a==a73c==a73e==a740==a742==a744==a746==a748==a74a==a74c==a74e==a750==a752==a754==a756==a758==a75a==a75c==a75e==a760==a762==a764==a766==a768==a76a==a76c==a76e===========a779==a77b===a77e==a780==a782==a784==a786=====a78b=====a790==a792====a796==a798==a79a==a79c==a79e==a7a0==a7a2==a7a4==a7a6==a7a8============a7b4==a7b6==a7b8=======================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'===================================================================================a7b3=============================13a0=13a1=13a2=13a3=13a4=13a5=13a6=13a7=13a8=13a9=13aa=13ab=13ac=13ad=13ae=13af=13b0=13b1=13b2=13b3=13b4=13b5=13b6=13b7=13b8=13b9=13ba=13bb=13bc=13bd=13be=13bf=13c0=13c1=13c2=13c3=13c4=13c5=13c6=13c7=13c8=13c9=13ca=13cb=13cc=13cd=13ce=13cf=13d0=13d1=13d2=13d3=13d4=13d5=13d6=13d7=13d8=13d9=13da=13db=13dc=13dd=13de=13df=13e0=13e1=13e2=13e3=13e4=13e5=13e6=13e7=13e8=13e9=13ea=13eb=13ec=13ed=13ee=13ef=================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'46-46=46-49=46-4c=46-46-49=46-46-4c=53-54=53-54=============544-546=544-535=544-53b=54e-546=544-53d=========================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'=================================================================ff21=ff22=ff23=ff24=ff25=ff26=ff27=ff28=ff29=ff2a=ff2b=ff2c=ff2d=ff2e=ff2f=ff30=ff31=ff32=ff33=ff34=ff35=ff36=ff37=ff38=ff39=ff3a======================================================================================================================================================================');

var expected_lc = parse_table('=================================================================61=62=63=64=65=66=67=68=69=6a=6b=6c=6d=6e=6f=70=71=72=73=74=75=76=77=78=79=7a======================================================================================================e0=e1=e2=e3=e4=e5=e6=e7=e8=e9=ea=eb=ec=ed=ee=ef=f0=f1=f2=f3=f4=f5=f6==f8=f9=fa=fb=fc=fd=fe==================================' +
'101==103==105==107==109==10b==10d==10f==111==113==115==117==119==11b==11d==11f==121==123==125==127==129==12b==12d==12f==69-307==133==135==137===13a==13c==13e==140==142==144==146==148===14b==14d==14f==151==153==155==157==159==15b==15d==15f==161==163==165==167==169==16b==16d==16f==171==173==175==177==ff=17a==17c==17e====253=183==185==254=188==256=257=18c===1dd=259=25b=192==260=263==269=268=199====26f=272==275=1a1==1a3==1a5==280=1a8==283===1ad==288=1b0==28a=28b=1b4==1b6==292=1b9====1bd========1c6=1c6==1c9=1c9==1cc=1cc==1ce==1d0==1d2==1d4==1d6==1d8==1da==1dc===1df==1e1==1e3==1e5==1e7==1e9==1eb==1ed==1ef===1f3=1f3==1f5==195=1bf=1f9==1fb==1fd==1ff==' +
'201==203==205==207==209==20b==20d==20f==211==213==215==217==219==21b==21d==21f==19e==223==225==227==229==22b==22d==22f==231==233========2c65=23c==19a=2c66===242==180=289=28c=247==249==24b==24d==24f==================================================================================================================================================================================' +
'================================================================================================================371==373====377=========3f3=======3ac==3ad=3ae=3af==3cc==3cd=3ce==3b1=3b2=3b3=3b4=3b5=3b6=3b7=3b8=3b9=3ba=3bb=3bc=3bd=3be=3bf=3c0=3c1==3c3=3c4=3c5=3c6=3c7=3c8=3c9=3ca=3cb====================================3d7=========3d9==3db==3dd==3df==3e1==3e3==3e5==3e7==3e9==3eb==3ed==3ef======3b8===3f8==3f2=3fb===37b=37c=37d=' +
'450=451=452=453=454=455=456=457=458=459=45a=45b=45c=45d=45e=45f=430=431=432=433=434=435=436=437=438=439=43a=43b=43c=43d=43e=43f=440=441=442=443=444=445=446=447=448=449=44a=44b=44c=44d=44e=44f=================================================461==463==465==467==469==46b==46d==46f==471==473==475==477==479==47b==47d==47f==481==========48b==48d==48f==491==493==495==497==499==49b==49d==49f==4a1==4a3==4a5==4a7==4a9==4ab==4ad==4af==4b1==4b3==4b5==4b7==4b9==4bb==4bd==4bf==4cf=4c2==4c4==4c6==4c8==4ca==4cc==4ce===4d1==4d3==4d5==4d7==4d9==4db==4dd==4df==4e1==4e3==4e5==4e7==4e9==4eb==4ed==4ef==4f1==4f3==4f5==4f7==4f9==4fb==4fd==4ff==' +
'501==503==505==507==509==50b==50d==50f==511==513==515==517==519==51b==51d==51f==521==523==525==527==529==52b==52d==52f===561=562=563=564=565=566=567=568=569=56a=56b=56c=56d=56e=56f=570=571=572=573=574=575=576=577=578=579=57a=57b=57c=57d=57e=57f=580=581=582=583=584=585=586==========================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================2d00=2d01=2d02=2d03=2d04=2d05=2d06=2d07=2d08=2d09=2d0a=2d0b=2d0c=2d0d=2d0e=2d0f=2d10=2d11=2d12=2d13=2d14=2d15=2d16=2d17=2d18=2d19=2d1a=2d1b=2d1c=2d1d=2d1e=2d1f=2d20=2d21=2d22=2d23=2d24=2d25==2d27======2d2d===================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================ab70=ab71=ab72=ab73=ab74=ab75=ab76=ab77=ab78=ab79=ab7a=ab7b=ab7c=ab7d=ab7e=ab7f=ab80=ab81=ab82=ab83=ab84=ab85=ab86=ab87=ab88=ab89=ab8a=ab8b=ab8c=ab8d=ab8e=ab8f=ab90=ab91=ab92=ab93=ab94=ab95=ab96=ab97=ab98=ab99=ab9a=ab9b=ab9c=ab9d=ab9e=ab9f=aba0=aba1=aba2=aba3=aba4=aba5=aba6=aba7=aba8=aba9=abaa=abab=abac=abad=abae=abaf=abb0=abb1=abb2=abb3=abb4=abb5=abb6=abb7=abb8=abb9=abba=abbb=abbc=abbd=abbe=abbf=13f8=13f9=13fa=13fb=13fc=13fd===========' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================10d0=10d1=10d2=10d3=10d4=10d5=10d6=10d7=10d8=10d9=10da=10db=10dc=10dd=10de=10df=10e0=10e1=10e2=10e3=10e4=10e5=10e6=10e7=10e8=10e9=10ea=10eb=10ec=10ed=10ee=10ef=10f0=10f1=10f2=10f3=10f4=10f5=10f6=10f7=10f8=10f9=10fa===10fd=10fe=10ff=================================================================' +
'================================================================================================================================================================================================================================================================' +
'1e01==1e03==1e05==1e07==1e09==1e0b==1e0d==1e0f==1e11==1e13==1e15==1e17==1e19==1e1b==1e1d==1e1f==1e21==1e23==1e25==1e27==1e29==1e2b==1e2d==1e2f==1e31==1e33==1e35==1e37==1e39==1e3b==1e3d==1e3f==1e41==1e43==1e45==1e47==1e49==1e4b==1e4d==1e4f==1e51==1e53==1e55==1e57==1e59==1e5b==1e5d==1e5f==1e61==1e63==1e65==1e67==1e69==1e6b==1e6d==1e6f==1e71==1e73==1e75==1e77==1e79==1e7b==1e7d==1e7f==1e81==1e83==1e85==1e87==1e89==1e8b==1e8d==1e8f==1e91==1e93==1e95==========df==1ea1==1ea3==1ea5==1ea7==1ea9==1eab==1ead==1eaf==1eb1==1eb3==1eb5==1eb7==1eb9==1ebb==1ebd==1ebf==1ec1==1ec3==1ec5==1ec7==1ec9==1ecb==1ecd==1ecf==1ed1==1ed3==1ed5==1ed7==1ed9==1edb==1edd==1edf==1ee1==1ee3==1ee5==1ee7==1ee9==1eeb==1eed==1eef==1ef1==1ef3==1ef5==1ef7==1ef9==1efb==1efd==1eff==' +
'========1f00=1f01=1f02=1f03=1f04=1f05=1f06=1f07=========1f10=1f11=1f12=1f13=1f14=1f15===========1f20=1f21=1f22=1f23=1f24=1f25=1f26=1f27=========1f30=1f31=1f32=1f33=1f34=1f35=1f36=1f37=========1f40=1f41=1f42=1f43=1f44=1f45============1f51==1f53==1f55==1f57=========1f60=1f61=1f62=1f63=1f64=1f65=1f66=1f67=========================1f80=1f81=1f82=1f83=1f84=1f85=1f86=1f87=========1f90=1f91=1f92=1f93=1f94=1f95=1f96=1f97=========1fa0=1fa1=1fa2=1fa3=1fa4=1fa5=1fa6=1fa7=========1fb0=1fb1=1f70=1f71=1fb3============1f72=1f73=1f74=1f75=1fc3============1fd0=1fd1=1f76=1f77=============1fe0=1fe1=1f7a=1f7b=1fe5============1f78=1f79=1f7c=1f7d=1ff3====' +
'================================================================================================================================================================================================================================================================' +
'======================================3c9====6b=e5=======214e==============================================2170=2171=2172=2173=2174=2175=2176=2177=2178=2179=217a=217b=217c=217d=217e=217f====================2184=============================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'======================================================================================================================================================================================24d0=24d1=24d2=24d3=24d4=24d5=24d6=24d7=24d8=24d9=24da=24db=24dc=24dd=24de=24df=24e0=24e1=24e2=24e3=24e4=24e5=24e6=24e7=24e8=24e9=================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'2c30=2c31=2c32=2c33=2c34=2c35=2c36=2c37=2c38=2c39=2c3a=2c3b=2c3c=2c3d=2c3e=2c3f=2c40=2c41=2c42=2c43=2c44=2c45=2c46=2c47=2c48=2c49=2c4a=2c4b=2c4c=2c4d=2c4e=2c4f=2c50=2c51=2c52=2c53=2c54=2c55=2c56=2c57=2c58=2c59=2c5a=2c5b=2c5c=2c5d=2c5e==================================================2c61==26b=1d7d=27d===2c68==2c6a==2c6c==251=271=250=252==2c73===2c76=========23f=240=2c81==2c83==2c85==2c87==2c89==2c8b==2c8d==2c8f==2c91==2c93==2c95==2c97==2c99==2c9b==2c9d==2c9f==2ca1==2ca3==2ca5==2ca7==2ca9==2cab==2cad==2caf==2cb1==2cb3==2cb5==2cb7==2cb9==2cbb==2cbd==2cbf==2cc1==2cc3==2cc5==2cc7==2cc9==2ccb==2ccd==2ccf==2cd1==2cd3==2cd5==2cd7==2cd9==2cdb==2cdd==2cdf==2ce1==2ce3=========2cec==2cee=====2cf3==============' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================a641==a643==a645==a647==a649==a64b==a64d==a64f==a651==a653==a655==a657==a659==a65b==a65d==a65f==a661==a663==a665==a667==a669==a66b==a66d====================a681==a683==a685==a687==a689==a68b==a68d==a68f==a691==a693==a695==a697==a699==a69b======================================================================================================' +
'==================================a723==a725==a727==a729==a72b==a72d==a72f====a733==a735==a737==a739==a73b==a73d==a73f==a741==a743==a745==a747==a749==a74b==a74d==a74f==a751==a753==a755==a757==a759==a75b==a75d==a75f==a761==a763==a765==a767==a769==a76b==a76d==a76f===========a77a==a77c==1d79=a77f==a781==a783==a785==a787=====a78c==265===a791==a793====a797==a799==a79b==a79d==a79f==a7a1==a7a3==a7a5==a7a7==a7a9==266=25c=261=26c=26a==29e=287=29d=ab53=a7b5==a7b7==a7b9========================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'================================================================================================================================================================================================================================================================' +
'=================================ff41=ff42=ff43=ff44=ff45=ff46=ff47=ff48=ff49=ff4a=ff4b=ff4c=ff4d=ff4e=ff4f=ff50=ff51=ff52=ff53=ff54=ff55=ff56=ff57=ff58=ff59=ff5a======================================================================================================================================================================================================');

/* XXX: for now, no locale sensitive tables */
var expected_luc = expected_uc;
var expected_llc = expected_lc;

/*===
exec
===*/

print('exec');

/*
 *  Comparison of differences between the various tables.
 *
 *  This just documents the expected differences in the implementations
 *  under test based on static tables and is not really a compliance test.
 */

/*===
===*/

function comparetables(ctx, t1, t2) {
    var i;

    for (i = 0; i < 65536; i++) {
        if (t1[i] !== t2[i]) {
            print(ctx, 'diff', i);
        }
    }
}

/*
 *  Generate actual case conversion tables for the current
 *  implementation and compare them against expected values.
 */

/*===
uc mismatch at 642 input 282 got a7c5 expected 282
luc mismatch at 642 input 282 got a7c5 expected 282
uc mismatch at 7566 input 1d8e got a7c6 expected 1d8e
luc mismatch at 7566 input 1d8e got a7c6 expected 1d8e
uc mismatch at 42900 input a794 got a7c4 expected a794
luc mismatch at 42900 input a794 got a7c4 expected a794
lc mismatch at 42938 input a7ba got a7bb expected a7ba
llc mismatch at 42938 input a7ba got a7bb expected a7ba
uc mismatch at 42939 input a7bb got a7ba expected a7bb
luc mismatch at 42939 input a7bb got a7ba expected a7bb
lc mismatch at 42940 input a7bc got a7bd expected a7bc
llc mismatch at 42940 input a7bc got a7bd expected a7bc
uc mismatch at 42941 input a7bd got a7bc expected a7bd
luc mismatch at 42941 input a7bd got a7bc expected a7bd
lc mismatch at 42942 input a7be got a7bf expected a7be
llc mismatch at 42942 input a7be got a7bf expected a7be
uc mismatch at 42943 input a7bf got a7be expected a7bf
luc mismatch at 42943 input a7bf got a7be expected a7bf
lc mismatch at 42946 input a7c2 got a7c3 expected a7c2
llc mismatch at 42946 input a7c2 got a7c3 expected a7c2
uc mismatch at 42947 input a7c3 got a7c2 expected a7c3
luc mismatch at 42947 input a7c3 got a7c2 expected a7c3
lc mismatch at 42948 input a7c4 got a794 expected a7c4
llc mismatch at 42948 input a7c4 got a794 expected a7c4
lc mismatch at 42949 input a7c5 got 282 expected a7c5
llc mismatch at 42949 input a7c5 got 282 expected a7c5
lc mismatch at 42950 input a7c6 got 1d8e expected a7c6
llc mismatch at 42950 input a7c6 got 1d8e expected a7c6
===*/

function compliancetest() {
    var i;
    var c;
    var uc, lc, luc, llc;
    var e_uc, e_lc, e_luc, e_llc;

    function err(ctx, idx, got, expect) {
        print(ctx, 'mismatch at', idx,
              'input', encode_unicode(String.fromCharCode(idx)),
              'got', encode_unicode(got),
              'expected', encode_unicode(expect));
    }

    e_uc = expected_uc;
    e_lc = expected_lc;
    e_luc = expected_luc;
    e_llc = expected_llc;

    for (i = 0; i < 65536; i++) {
        c = String.fromCharCode(i);
        uc = c.toUpperCase();
        lc = c.toLowerCase();
        luc = c.toLocaleUpperCase();
        llc = c.toLocaleLowerCase();

        if (uc !== e_uc[i]) {
            err('uc', i, uc, e_uc[i]);
        }
        if (lc !== e_lc[i]) {
            err('lc', i, lc, e_lc[i]);
        }
        if (luc !== e_luc[i]) {
            err('luc', i, luc, e_luc[i]);
        }
        if (llc !== e_llc[i]) {
            err('llc', i, llc, e_llc[i]);
        }
    }
}

compliancetest();

/*===
done
===*/

print('done');
